/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%        CCCC   OOO  M   M  PPPP    OOO   SSSSS  IIIII  TTTTT  EEEEE          %
%       C      O   O MM MM  P   P  O   O  SS       I      T    E              %
%       C      O   O M M M  PPPP   O   O   SSS     I      T    EEE            %
%       C      O   O M   M  P      O   O     SS    I      T    E              %
%        CCCC   OOO  M   M  P       OOO   SSSSS  IIIII    T    EEEEE          %
%                                                                             %
%                                                                             %
%                    ImageMagick Image Composite Methods                      %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 July 1992                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/client.h"
#include "magick/color.h"
#include "magick/composite.h"
#include "magick/composite_private.h"
#include "magick/constitute.h"
#include "magick/fx.h"
#include "magick/gem.h"
#include "magick/geometry.h"
#include "magick/image.h"
#include "magick/list.h"
#include "magick/log.h"
#include "magick/memory_.h"
#include "magick/mogrify.h"
#include "magick/option.h"
#include "magick/resource_.h"
#include "magick/string_.h"
#include "magick/utility.h"
#include "magick/version.h"

/*
  Typedef declarations.
*/
typedef struct _CompositeOptions
{
  char
    *blend_geometry,
    *displace_geometry,
    *dissolve_geometry,
    *geometry,
    *unsharp_geometry,
    *watermark_geometry;

  CompositeOperator
    compose;

  GravityType
    gravity;

  long
    stegano;

  MagickBooleanType
    stereo,
    tile;
} CompositeOptions;

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C o m p o s i t e I m a g e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CompositeImage() returns the second image composited onto the first at the
%  specified offsets.
%
%  The format of the CompositeImage method is:
%
%      MagickBooleanType CompositeImage(Image *image,
%        const CompositeOperator compose,const Image *composite_image,
%        const long x_offset,const long y_offset)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o compose: This operator affects how the composite is applied to
%      the image.  The operators and how they are utilized are listed here:
%      http://www.w3.org/TR/SVG12/#compositing
%
%    o composite_image: The composite image.
%
%    o x_offset: The column offset of the composited image.
%
%    o y_offset: The row offset of the composited image.
%
%
*/

static inline PixelPacket MagickCompositeAdd(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    pixel;

  PixelPacket
    composite;

  pixel=p->red+(MagickRealType) q->red;
  if (pixel > MaxRGB)
    pixel-=(MaxRGB+1.0);
  composite.red=RoundToQuantum(pixel);
  pixel=p->green+(MagickRealType) q->green;
  if (pixel > MaxRGB)
    pixel-=(MaxRGB+1.0);
  composite.green=RoundToQuantum(pixel);
  pixel=p->blue+(MagickRealType) q->blue;
  if (pixel > MaxRGB)
    pixel-=(MaxRGB+1.0);
  composite.blue=RoundToQuantum(pixel);
  pixel=alpha+beta;
  if (pixel > MaxRGB)
    pixel-=(MaxRGB+1.0);
  composite.opacity=RoundToQuantum(pixel);
  return(composite);
}

static inline PixelPacket MagickCompositeBumpmap(const PixelPacket *p,
  MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  PixelPacket
    composite;

  composite.red=(Quantum)
    ((QuantumScale*PixelIntensityToQuantum(p)*q->red)+0.5);
  composite.green=(Quantum)
    ((QuantumScale*PixelIntensityToQuantum(p)*q->green)+0.5);
  composite.blue=(Quantum)
    ((QuantumScale*PixelIntensityToQuantum(p)*q->blue)+0.5);
  composite.opacity=(Quantum)
    ((QuantumScale*PixelIntensityToQuantum(p)*p->opacity)+0.5);
  return(composite);
}

static inline PixelPacket MagickCompositeClear(void)
{
  PixelPacket
    composite;

  composite.red=0;
  composite.green=0;
  composite.blue=0;
  composite.opacity=TransparentOpacity;
  return(composite);
}

static PixelPacket MagickCompositeColorBurn(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    delta,
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  delta=(1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->red*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  else
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      ((1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)+
      (1.0-QuantumScale*beta)*q->red*(1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta))/
      (1.0-QuantumScale*alpha)*p->red+(1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  delta=(1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->green*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->green*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  else
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      ((1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)+
      (1.0-QuantumScale*beta)*q->green*(1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta))/
      (1.0-QuantumScale*alpha)*p->green+(1.0-QuantumScale*alpha)*p->green*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  delta=(1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->blue*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->blue*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  else
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      ((1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)+
      (1.0-QuantumScale*beta)*q->blue*(1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta))/
      (1.0-QuantumScale*alpha)*p->blue+(1.0-QuantumScale*alpha)*p->blue*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static PixelPacket MagickCompositeColorDodge(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    delta,
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  delta=(1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->red*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)+(1.0-QuantumScale*alpha)*p->red*QuantumScale*
      beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  else
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->red*
      (1.0-QuantumScale*alpha)/(1-(1.0-QuantumScale*alpha)*p->red/
      (1.0-QuantumScale*alpha))+(1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  delta=(1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->green*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)+(1.0-QuantumScale*alpha)*p->green*QuantumScale*
      beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  else
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->green*
      (1.0-QuantumScale*alpha)/(1-(1.0-QuantumScale*alpha)*p->green/
      (1.0-QuantumScale*alpha))+(1.0-QuantumScale*alpha)*p->green*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  delta=(1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)+
    (1.0-QuantumScale*beta)*q->blue*(1.0-QuantumScale*alpha);
  if (delta >= ((1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta)))
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)+(1.0-QuantumScale*alpha)*p->blue*QuantumScale*
      beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  else
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->blue*
      (1.0-QuantumScale*alpha)/(1-(1.0-QuantumScale*alpha)*p->blue/
      (1.0-QuantumScale*alpha))+(1.0-QuantumScale*alpha)*p->blue*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static inline PixelPacket MagickCompositeDarken(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  if (((1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)) <
      ((1.0-QuantumScale*beta)*q->red*(1.0-QuantumScale*alpha)))
    composite.red=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->red+
      (1.0-QuantumScale*beta)*q->red*QuantumScale*alpha)+0.5);
  else
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->red+
      (1.0-QuantumScale*alpha)*p->red*QuantumScale*beta));
  if (((1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)) <
      ((1.0-QuantumScale*beta)*q->green*(1.0-QuantumScale*alpha)))
    composite.green=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->green+
      (1.0-QuantumScale*beta)*q->green*QuantumScale*alpha)+0.5);
  else
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->green+
      (1.0-QuantumScale*alpha)*p->green*QuantumScale*beta));
  if (((1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)) <
      ((1.0-QuantumScale*beta)*q->blue*(1.0-QuantumScale*alpha)))
    composite.blue=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->blue+
      (1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha)+0.5);
  else
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->blue+
      (1.0-QuantumScale*alpha)*p->blue*QuantumScale*beta));
  return(composite);
}

static inline PixelPacket MagickCompositeDifference(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->red+
    (1.0-QuantumScale*beta)*q->red-2*Min((1.0-QuantumScale*alpha)*p->red*
    (1.0-QuantumScale*beta),(1.0-QuantumScale*beta)*q->red*
    (1.0-QuantumScale*alpha))));
  composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->green+
    (1.0-QuantumScale*beta)*q->green-2*Min((1.0-QuantumScale*alpha)*p->green*
    (1.0-QuantumScale*beta),(1.0-QuantumScale*beta)*q->green*
    (1.0-QuantumScale*alpha))));
  composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->blue+
    (1.0-QuantumScale*beta)*q->blue-2*Min((1.0-QuantumScale*alpha)*p->blue*
    (1.0-QuantumScale*beta),(1.0-QuantumScale*beta)*q->blue*
    (1.0-QuantumScale*alpha))));
  return(composite);
}

static PixelPacket MagickCompositeExclusion(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*(((1.0-QuantumScale*alpha)*p->red*
    (1.0-QuantumScale*beta)+(1.0-QuantumScale*beta)*q->red*
    (1.0-QuantumScale*alpha)-2*QuantumScale*(1.0-QuantumScale*alpha)*p->red*
    (1.0-QuantumScale*beta)*q->red)+(1.0-QuantumScale*alpha)*p->red*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  composite.green=RoundToQuantum(gamma*(((1.0-QuantumScale*alpha)*p->green*
    (1.0-QuantumScale*beta)+(1.0-QuantumScale*beta)*q->green*
    (1.0-QuantumScale*alpha)-2*QuantumScale*(1.0-QuantumScale*alpha)*p->green*
    (1.0-QuantumScale*beta)*q->green)+(1.0-QuantumScale*alpha)*p->green*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  composite.blue=RoundToQuantum(gamma*(((1.0-QuantumScale*alpha)*p->blue*
    (1.0-QuantumScale*beta)+(1.0-QuantumScale*beta)*q->blue*
    (1.0-QuantumScale*alpha)-2*QuantumScale*(1.0-QuantumScale*alpha)*p->blue*
    (1.0-QuantumScale*beta)*q->blue)+(1.0-QuantumScale*alpha)*p->blue*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static PixelPacket MagickCompositeHardLight(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  if ((2*(1.0-QuantumScale*alpha)*p->red) < (1.0-QuantumScale*alpha))
    composite.red=RoundToQuantum(gamma*(2*QuantumScale*(1.0-QuantumScale*alpha)*
      p->red*(1.0-QuantumScale*beta)*q->red+(1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  else
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
      (1.0-QuantumScale*beta)*q->red)*((1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*p->red)+(1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  if ((2*(1.0-QuantumScale*alpha)*p->green) < (1.0-QuantumScale*alpha))
    composite.green=RoundToQuantum(gamma*(2*QuantumScale*
      (1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)*q->green+
      (1.0-QuantumScale*alpha)*p->green*QuantumScale*beta+
      (1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  else
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
      (1.0-QuantumScale*beta)*q->green)*((1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*p->green)+(1.0-QuantumScale*alpha)*p->green*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  if ((2*(1.0-QuantumScale*alpha)*p->blue) < (1.0-QuantumScale*alpha))
    composite.blue=RoundToQuantum(gamma*(2*QuantumScale*
      (1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)*q->blue+
      (1.0-QuantumScale*alpha)*p->blue*QuantumScale*beta+
      (1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  else
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
      (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
      (1.0-QuantumScale*beta)*q->blue)*((1.0-QuantumScale*alpha)-
      (1.0-QuantumScale*alpha)*p->blue)+(1.0-QuantumScale*alpha)*p->blue*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static inline PixelPacket MagickCompositeIn(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=(1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->red*
    (1.0-QuantumScale*beta))+0.5);
  composite.green=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->green*
    (1.0-QuantumScale*beta))+0.5);
  composite.blue=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->blue*
    (1.0-QuantumScale*beta))+0.5);
  return(composite);
}

static inline PixelPacket MagickCompositeLighten(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  if (((1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)) >
      ((1.0-QuantumScale*beta)*q->red*(1.0-QuantumScale*alpha)))
    composite.red=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->red+
      (1.0-QuantumScale*beta)*q->red*QuantumScale*alpha)+0.5);
  else
    composite.red=(Quantum) (gamma*((1.0-QuantumScale*beta)*q->red+
      (1.0-QuantumScale*alpha)*p->red*QuantumScale*beta)+0.5);
  if (((1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)) >
      ((1.0-QuantumScale*beta)*q->green*(1.0-QuantumScale*alpha)))
    composite.green=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->green+
      (1.0-QuantumScale*beta)*q->green*QuantumScale*alpha)+0.5);
  else
    composite.green=(Quantum) (gamma*((1.0-QuantumScale*beta)*q->green+
      (1.0-QuantumScale*alpha)*p->green*QuantumScale*beta)+0.5);
  if (((1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)) >
      ((1.0-QuantumScale*beta)*q->blue*(1.0-QuantumScale*alpha)))
    composite.blue=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->blue+
      (1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha)+0.5);
  else
    composite.blue=(Quantum) (gamma*((1.0-QuantumScale*beta)*q->blue+
      (1.0-QuantumScale*alpha)*p->blue*QuantumScale*beta)+0.5);
  return(composite);
}

static inline PixelPacket MagickCompositeMinus(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=(MagickRealType) MaxRGB;
  composite.opacity=(Quantum) (MaxRGB-gamma+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*((MaxRGB-alpha)*p->red-
    (MaxRGB-beta)*q->red));
  composite.green=RoundToQuantum(gamma*((MaxRGB-alpha)*p->green-
    (MaxRGB-beta)*q->green));
  composite.blue=RoundToQuantum(gamma*((MaxRGB-alpha)*p->blue-
    (MaxRGB-beta)*q->blue));
  return(composite);
}

static inline PixelPacket MagickCompositeMultiply(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*(QuantumScale*(1.0-QuantumScale*alpha)*
    p->red*(1.0-QuantumScale*beta)*q->red+(1.0-QuantumScale*alpha)*p->red*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  composite.green=RoundToQuantum(gamma*(QuantumScale*(1.0-QuantumScale*alpha)*
    p->green*(1.0-QuantumScale*beta)*q->green+(1.0-QuantumScale*alpha)*p->green*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  composite.blue=RoundToQuantum(gamma*(QuantumScale*(1.0-QuantumScale*alpha)*
    p->blue*(1.0-QuantumScale*beta)*q->blue+(1.0-QuantumScale*alpha)*p->blue*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static inline PixelPacket MagickCompositeOut(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=(1.0-QuantumScale*alpha)*QuantumScale*beta;
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->red*
    QuantumScale*beta));
  composite.green=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->green*
    QuantumScale*beta));
  composite.blue=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->blue*
    QuantumScale*beta));
  return(composite);
}

static PixelPacket MagickCompositeOverlay(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  if ((2*(1.0-QuantumScale*beta)*q->red) < ((1.0-QuantumScale*beta)))
    {
      composite.red=RoundToQuantum(gamma*(2*QuantumScale*
        (1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)*q->red+
        (1.0-QuantumScale*alpha)*p->red*QuantumScale*beta+
        (1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
      composite.green=RoundToQuantum(gamma*(2*QuantumScale*
        (1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)*q->green+
        (1.0-QuantumScale*alpha)*p->green*QuantumScale*beta+
        (1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
      composite.blue=RoundToQuantum(gamma*(2*QuantumScale*
        (1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)*q->blue+
        (1.0-QuantumScale*alpha)*p->blue*QuantumScale*beta+
        (1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
      return(composite);
    }
  composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
    (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
    (1.0-QuantumScale*beta)*q->red)*((1.0-QuantumScale*alpha)-
    (1.0-QuantumScale*alpha)*p->red)+(1.0-QuantumScale*alpha)*p->red*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
    (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
    (1.0-QuantumScale*beta)*q->green)*((1.0-QuantumScale*alpha)-
    (1.0-QuantumScale*alpha)*p->green)+(1.0-QuantumScale*alpha)*p->green*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*
    (1.0-QuantumScale*beta)-2*((1.0-QuantumScale*beta)-
    (1.0-QuantumScale*beta)*q->blue)*((1.0-QuantumScale*alpha)-
    (1.0-QuantumScale*alpha)*p->blue)+(1.0-QuantumScale*alpha)*p->blue*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static inline PixelPacket MagickCompositePlus(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->red+
    (1.0-QuantumScale*beta)*q->red));
  composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->green+
    (1.0-QuantumScale*beta)*q->green));
  composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->blue+
    (1.0-QuantumScale*beta)*q->blue));
  return(composite);
}

static inline PixelPacket MagickCompositeScreen(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->red+
    (1.0-QuantumScale*beta)*q->red-QuantumScale*
    (1.0-QuantumScale*alpha)*p->red*(1.0-QuantumScale*beta)*q->red));
  composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->green+
    (1.0-QuantumScale*beta)*q->green-QuantumScale*
    (1.0-QuantumScale*alpha)*p->green*(1.0-QuantumScale*beta)*q->green));
  composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*alpha)*p->blue+
    (1.0-QuantumScale*beta)*q->blue-QuantumScale*
    (1.0-QuantumScale*alpha)*p->blue*(1.0-QuantumScale*beta)*q->blue));
  return(composite);
}

static PixelPacket MagickCompositeSoftLight(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=Max(Min((1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    (1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta),1.0),0.0);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  if ((2*(1.0-QuantumScale*alpha)*p->red) < (1.0-QuantumScale*alpha))
    composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->red*
      ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->red/
      (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->red-
      (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->red*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  else
    if (((1.0-QuantumScale*beta)*q->red*8) <= (1.0-QuantumScale*beta))
      composite.red=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->red*
        ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->red/
        (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->red-
        (1.0-QuantumScale*alpha))*(3-8*(1.0-QuantumScale*beta)*q->red/
        (1.0-QuantumScale*beta)))+(1.0-QuantumScale*alpha)*p->red*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
    else
      composite.red=RoundToQuantum(gamma*(((1.0-QuantumScale*beta)*q->red*
        (1.0-QuantumScale*alpha)+(pow((1.0-QuantumScale*beta)*q->red/
        (1.0-QuantumScale*beta),0.5*(1.0-QuantumScale*beta))-
        (1.0-QuantumScale*beta)*q->red)*(2*(1.0-QuantumScale*alpha)*p->red-
        (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->red*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*QuantumScale*alpha));
  if ((2*(1.0-QuantumScale*alpha)*p->green) < (1.0-QuantumScale*alpha))
    composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->green*
      ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->green/
      (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->green-
      (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->green*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  else
    if (((1.0-QuantumScale*beta)*q->green*8) <= (1.0-QuantumScale*beta))
      composite.green=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->green*
        ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->green/
        (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->green-
        (1.0-QuantumScale*alpha))*(3-8*(1.0-QuantumScale*beta)*q->green/
        (1.0-QuantumScale*beta)))+(1.0-QuantumScale*alpha)*p->green*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
    else
      composite.green=RoundToQuantum(gamma*(((1.0-QuantumScale*beta)*q->green*
        (1.0-QuantumScale*alpha)+(pow((1.0-QuantumScale*beta)*q->green/
        (1.0-QuantumScale*beta),0.5*(1.0-QuantumScale*beta))-
        (1.0-QuantumScale*beta)*q->green)*(2*(1.0-QuantumScale*alpha)*p->green-
        (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->green*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*QuantumScale*alpha));
  if ((2*(1.0-QuantumScale*alpha)*p->blue) < (1.0-QuantumScale*alpha))
    composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->blue*
      ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->blue/
      (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->blue-
      (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->blue*
      QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  else
    if (((1.0-QuantumScale*beta)*q->blue*8) <= (1.0-QuantumScale*beta))
      composite.blue=RoundToQuantum(gamma*((1.0-QuantumScale*beta)*q->blue*
        ((1.0-QuantumScale*alpha)-(1-(1.0-QuantumScale*beta)*q->blue/
        (1.0-QuantumScale*beta))*(2*(1.0-QuantumScale*alpha)*p->blue-
        (1.0-QuantumScale*alpha))*(3-8*(1.0-QuantumScale*beta)*q->blue/
        (1.0-QuantumScale*beta)))+(1.0-QuantumScale*alpha)*p->blue*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
    else
      composite.blue=RoundToQuantum(gamma*(((1.0-QuantumScale*beta)*q->blue*
        (1.0-QuantumScale*alpha)+(pow((1.0-QuantumScale*beta)*q->blue/
        (1.0-QuantumScale*beta),0.5*(1.0-QuantumScale*beta))-
        (1.0-QuantumScale*beta)*q->blue)*(2*(1.0-QuantumScale*alpha)*p->blue-
        (1.0-QuantumScale*alpha)))+(1.0-QuantumScale*alpha)*p->blue*
        QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*QuantumScale*alpha));
  return(composite);
}

static inline PixelPacket MagickCompositeSubtract(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    pixel;

  PixelPacket
    composite;

  pixel=p->red-(MagickRealType) q->red;
  if (pixel < 0.0)
    pixel+=(MaxRGB+1.0);
  composite.red=RoundToQuantum(pixel);
  pixel=p->green-(MagickRealType) q->green;
  if (pixel < 0.0)
    pixel+=(MaxRGB+1.0);
  composite.green=RoundToQuantum(pixel);
  pixel=p->blue-(MagickRealType) q->blue;
  if (pixel < 0.0)
    pixel+=(MaxRGB+1.0);
  composite.blue=RoundToQuantum(pixel);
  pixel=alpha-beta;
  if (pixel < 0.0)
    pixel+=(MaxRGB+1.0);
  composite.opacity=RoundToQuantum(pixel);
  return(composite);
}

static inline PixelPacket MagickCompositeXor(const PixelPacket *p,
  const MagickRealType alpha,const PixelPacket *q,const MagickRealType beta)
{
  MagickRealType
    gamma;

  PixelPacket
    composite;

  gamma=(1.0-QuantumScale*alpha)+(1.0-QuantumScale*beta)-
    2*(1.0-QuantumScale*alpha)*(1.0-QuantumScale*beta);
  composite.opacity=(Quantum) (MaxRGB*(1.0-gamma)+0.5);
  gamma=1.0/(gamma <= MagickEpsilon ? 1.0 : gamma);
  composite.red=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->red*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->red*
    QuantumScale*alpha)+0.5);
  composite.green=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->green*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->green*
    QuantumScale*alpha)+0.5);
  composite.blue=(Quantum) (gamma*((1.0-QuantumScale*alpha)*p->blue*
    QuantumScale*beta+(1.0-QuantumScale*beta)*q->blue*
    QuantumScale*alpha)+0.5);
  return(composite);
}

MagickExport MagickBooleanType CompositeImage(Image *image,
  const CompositeOperator compose,const Image *composite_image,
  const long x_offset,const long y_offset)
{
  const PixelPacket
    *pixels;

  double
    brightness,
    hue,
    sans,
    saturation;

  GeometryInfo
    geometry_info;

  IndexPacket
    *composite_indexes,
    *indexes;

  long
    y;

  MagickBooleanType
    modify_outside_overlay;

  MagickPixelPacket
    pixel;

  MagickRealType
    amount,
    dst_dissolve,
    midpoint,
    percent_brightness,
    percent_saturation,
    src_dissolve,
    threshold;

  MagickStatusType
    flags;

  PixelPacket
    destination,
    source;

  register const PixelPacket
    *p;

  register long
    x;

  register PixelPacket
    *q;

  /*
    Prepare composite image.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(composite_image != (Image *) NULL);
  assert(composite_image->signature == MagickSignature);
  if (compose == NoCompositeOp)
    return(MagickTrue);
  image->storage_class=DirectClass;
  amount=0.5;
  dst_dissolve=1.0;
  modify_outside_overlay=MagickFalse;
  percent_brightness=100.0;
  percent_saturation=100.0;
  src_dissolve=1.0;
  threshold=0.05f;
  switch (compose)
  {
    case ClearCompositeOp:
    case SrcCompositeOp:
    case SrcInCompositeOp:
    case InCompositeOp:
    case SrcOutCompositeOp:
    case OutCompositeOp:
    case DstInCompositeOp:
    case DstAtopCompositeOp:
    {
      /*
        These operators modify destination outside the overlaid region.
      */
      modify_outside_overlay=MagickTrue;
      break;
    }
    case CopyOpacityCompositeOp:
    {
      if (image->matte == MagickFalse)
        SetImageOpacity(image,OpaqueOpacity);
      modify_outside_overlay=MagickTrue;
      break;
    }
    case DisplaceCompositeOp:
    {
      Image
        *displace_image;

      MagickRealType
        horizontal_scale,
        vertical_scale,
        x_displace,
        y_displace;

      register PixelPacket
        *r;

      /*
        Allocate the displace image.
      */
      displace_image=CloneImage(composite_image,0,0,MagickTrue,
        &image->exception);
      if (displace_image == (Image *) NULL)
        return(MagickFalse);
      horizontal_scale=20.0;
      vertical_scale=20.0;
      if (composite_image->geometry != (char *) NULL)
        {
          /*
            Determine the horizontal and vertical displacement scale.
          */
          SetGeometryInfo(&geometry_info);
          flags=ParseGeometry(composite_image->geometry,&geometry_info);
          horizontal_scale=geometry_info.rho;
          vertical_scale=geometry_info.sigma;
          if ((flags & SigmaValue) == 0)
            vertical_scale=horizontal_scale;
        }
      /*
        Shift image pixels as defined by a displacement map.
      */
      for (y=0; y < (long) composite_image->rows; y++)
      {
        if (((y+y_offset) < 0) || ((y+y_offset) >= (long) image->rows))
          continue;
        p=AcquireImagePixels(composite_image,0,y,composite_image->columns,1,
          &image->exception);
        q=GetImagePixels(image,0,y+y_offset,image->columns,1);
        r=GetImagePixels(displace_image,0,y,displace_image->columns,1);
        if ((p == (const PixelPacket *) NULL) || (q == (PixelPacket *) NULL) ||
            (r == (PixelPacket *) NULL))
          break;
        q+=x_offset;
        for (x=0; x < (long) composite_image->columns; x++)
        {
          if (((x_offset+x) < 0) || ((x_offset+x) >= (long) image->columns))
            {
              p++;
              q++;
              continue;
            }
          x_displace=(horizontal_scale*(PixelIntensityToQuantum(p)-
            (((MagickRealType) MaxRGB+1.0)/2)))/(((MagickRealType) MaxRGB+1.0)/
            2.0);
          y_displace=x_displace;
          if (composite_image->matte != MagickFalse)
            y_displace=(vertical_scale*(p->opacity-(((MagickRealType) MaxRGB+
              1.0)/2)))/(((MagickRealType) MaxRGB+1.0)/2);
          *r=InterpolateColor(image,(MagickRealType) (x_offset+x+x_displace),
            (MagickRealType) (y_offset+y+y_displace),&image->exception);
          p++;
          q++;
          r++;
        }
        if (SyncImagePixels(displace_image) == MagickFalse)
          break;
      }
      composite_image=displace_image;
      break;
    }
    case DissolveCompositeOp:
    {
      if (composite_image->geometry != (char *) NULL)
        {
          /*
            Geometry arguments to dissolve factors.
          */
          flags=ParseGeometry(composite_image->geometry,&geometry_info);
          src_dissolve=geometry_info.rho/100.0;
          dst_dissolve=1.0;
          if ((src_dissolve-MagickEpsilon) < 0.0)
            src_dissolve=0.0;
          if ((src_dissolve+MagickEpsilon) > 1.0)
            {
              dst_dissolve=2.0-src_dissolve;
              src_dissolve=1.0;
            }
          if ((flags & SigmaValue) != 0)
            dst_dissolve=geometry_info.sigma/100.0;
          if ((dst_dissolve-MagickEpsilon) < 0.0)
            dst_dissolve=0.0;
          modify_outside_overlay=MagickTrue;
          if ((dst_dissolve+MagickEpsilon) > 1.0 )
            {
              dst_dissolve=1.0;
              modify_outside_overlay=MagickFalse;
            }
        }
      break;
    }
    case BlendCompositeOp:
    {
      if (composite_image->geometry != (char *) NULL)
        {
          flags=ParseGeometry(composite_image->geometry,&geometry_info);
          src_dissolve=geometry_info.rho/100.0;
          if ((src_dissolve-MagickEpsilon) < 0.0)
            src_dissolve=0.0;
          if ((src_dissolve+MagickEpsilon) > 1.0)
            src_dissolve=1.0;
          dst_dissolve=1.0-src_dissolve;
          if ((flags & SigmaValue) != 0)
            dst_dissolve=geometry_info.sigma/100.0;
          if ((dst_dissolve-MagickEpsilon) < 0.0)
            dst_dissolve=0.0;
          modify_outside_overlay=MagickTrue;
          if ((dst_dissolve+MagickEpsilon) > 1.0)
            {
              dst_dissolve=1.0;
              modify_outside_overlay=MagickFalse;
            }
        }
      break;
    }
    case ModulateCompositeOp:
    {
      if (composite_image->geometry != (char *) NULL)
        {
          /*
            Determine the brightness and saturation scale.
          */
          flags=ParseGeometry(composite_image->geometry,&geometry_info);
          percent_brightness=geometry_info.rho;
          if ((flags & SigmaValue) != 0)
            percent_saturation=geometry_info.sigma;
        }
      break;
    }
    case ThresholdCompositeOp:
    {
      /*
        Determine the amount and threshold.
      */
      if (composite_image->geometry != (char *) NULL)
        {
          flags=ParseGeometry(composite_image->geometry,&geometry_info);
          amount=geometry_info.rho;
          threshold=geometry_info.sigma;
          if ((flags & SigmaValue) == 0)
            threshold=0.05f;
        }
      threshold*=MaxRGB;
      break;
    }
    default:
      break;
  }
  /*
    Composite image.
  */
  hue=0.0;
  saturation=0.0;
  brightness=0.0;
  midpoint=((MagickRealType) MaxRGB+1.0)/2;
  for (y=0; y < (long) image->rows; y++)
  {
    if (modify_outside_overlay == MagickFalse)
      {
        if (y < y_offset)
          continue;
        if ((y-y_offset) >= (long) composite_image->rows)
          break;
      }
    /*
      If pixels is NULL, y is outside overlay region.
    */
    pixels=p=(PixelPacket *) NULL;
    if ((y >= y_offset) && ((y-y_offset) < (long) composite_image->rows))
      {
        p=AcquireImagePixels(composite_image,0,y-y_offset,
          composite_image->columns,1,&image->exception);
        if (p == (const PixelPacket *) NULL)
          break;
        pixels=p;
        if (x_offset < 0)
          p-=x_offset;
      }
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    indexes=GetIndexes(image);
    composite_indexes=GetIndexes(composite_image);
    for (x=0; x < (long) image->columns; x++)
    {
      if (modify_outside_overlay == MagickFalse)
        {
          if (x < x_offset)
            {
              q++;
              continue;
            }
          if ((x-x_offset) >= (long) composite_image->columns)
            break;
        }
      destination=(*q);
      if (image->matte == MagickFalse)
        destination.opacity=OpaqueOpacity;
      /*
        Handle destination modifications outside overlaid region.
      */
      if ((pixels == (PixelPacket *) NULL) || (x < x_offset) ||
          ((x-x_offset) >= (long) composite_image->columns))
        {
          switch (compose)
          {
            case DissolveCompositeOp:
            case BlendCompositeOp:
            {
              destination.opacity=(Quantum)
                (MaxRGB-dst_dissolve*(MaxRGB-destination.opacity)+0.5);
              break;
            }
            case ClearCompositeOp:
            case SrcCompositeOp:
            {
              destination=MagickCompositeClear();
              break;
            }
            case InCompositeOp:
            case SrcInCompositeOp:
            case SrcOutCompositeOp:
            case DstInCompositeOp:
            case DstAtopCompositeOp:
            case CopyOpacityCompositeOp:
            {
              destination.opacity=TransparentOpacity;
              break;
            }
            default:
              break;

          }
          q->red=destination.red;
          q->green=destination.green;
          q->blue=destination.blue;
          if (image->matte != MagickFalse)
            q->opacity=destination.opacity;
          q++;
          continue;
        }
      /*
        Handle normal overlay of source onto destination.
      */
      source=(*p);
      if (composite_image->matte == MagickFalse)
        source.opacity=OpaqueOpacity;
      switch (compose)
      {
        case ClearCompositeOp:
        {
          destination=MagickCompositeClear();
          break;
        }
        case SrcCompositeOp:
        case CopyCompositeOp:
        case ReplaceCompositeOp:
        {
          destination=source;
          break;
        }
        case DstCompositeOp:
          break;
        case OverCompositeOp:
        case SrcOverCompositeOp:
        {
          destination=MagickCompositeOver(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DstOverCompositeOp:
        {
          destination=MagickCompositeOver(&destination,(MagickRealType)
            destination.opacity,&source,(MagickRealType) source.opacity);
          break;
        }
        case SrcInCompositeOp:
        case InCompositeOp:
        {
          destination=MagickCompositeIn(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DstInCompositeOp:
        {
          destination=MagickCompositeIn(&destination,(MagickRealType)
            destination.opacity,&source,(MagickRealType) source.opacity);
          break;
        }
        case OutCompositeOp:
        case SrcOutCompositeOp:
        {
          destination=MagickCompositeOut(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DstOutCompositeOp:
        {
          destination=MagickCompositeOut(&destination,(MagickRealType)
            destination.opacity,&source,(MagickRealType) source.opacity);
          break;
        }
        case AtopCompositeOp:
        case SrcAtopCompositeOp:
        {
          destination=MagickCompositeAtop(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DstAtopCompositeOp:
        {
          destination=MagickCompositeAtop(&destination,(MagickRealType)
            destination.opacity,&source,(MagickRealType) source.opacity);
          break;
        }
        case XorCompositeOp:
        {
          destination=MagickCompositeXor(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case PlusCompositeOp:
        {
          destination=MagickCompositePlus(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case MultiplyCompositeOp:
        {
          destination=MagickCompositeMultiply(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case ScreenCompositeOp:
        {
          destination=MagickCompositeScreen(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case OverlayCompositeOp:
        {
          destination=MagickCompositeOverlay(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DarkenCompositeOp:
        {
          destination=MagickCompositeDarken(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case LightenCompositeOp:
        {
          destination=MagickCompositeLighten(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case ColorDodgeCompositeOp:
        {
          destination=MagickCompositeColorDodge(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case ColorBurnCompositeOp:
        {
          destination=MagickCompositeColorBurn(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case HardLightCompositeOp:
        {
          destination=MagickCompositeHardLight(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case SoftLightCompositeOp:
        {
          destination=MagickCompositeSoftLight(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DifferenceCompositeOp:
        {
          destination=MagickCompositeDifference(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case ExclusionCompositeOp:
        {
          destination=MagickCompositeExclusion(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case MinusCompositeOp:
        {
          destination=MagickCompositeMinus(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case BumpmapCompositeOp:
        {
          if (source.opacity == TransparentOpacity)
            break;
          destination=MagickCompositeBumpmap(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case DissolveCompositeOp:
        {
          destination=MagickCompositeOver(&source,(MagickRealType)
            MaxRGB-src_dissolve*(MaxRGB-source.opacity),&destination,
            MaxRGB-dst_dissolve*(MaxRGB-destination.opacity));
          break;
        }
        case BlendCompositeOp:
        {
          destination=MagickCompositePlus(&source,(MagickRealType)
            MaxRGB-src_dissolve*(MaxRGB-source.opacity),&destination,
            MaxRGB-dst_dissolve*(MaxRGB-destination.opacity));
          break;
        }
        case DisplaceCompositeOp:
        {
          destination=source;
          break;
        }
        case ThresholdCompositeOp:
        {
          pixel.red=destination.red-(MagickRealType) source.red;
          if (fabs(2.0*pixel.red) < threshold)
            pixel.red=(MagickRealType) destination.red;
          else
            pixel.red=destination.red+(pixel.red*amount);
          pixel.green=destination.green-(MagickRealType) source.green;
          if (fabs(2.0*pixel.green) < threshold)
            pixel.green=(MagickRealType) destination.green;
          else
            pixel.green=destination.green+(pixel.green*amount);
          pixel.blue=destination.blue-(MagickRealType) source.blue;
          if (fabs(2.0*pixel.blue) < threshold)
            pixel.blue=(MagickRealType) destination.blue;
          else
            pixel.blue=destination.blue+(pixel.blue*amount);
          pixel.opacity=destination.opacity-(MagickRealType) source.opacity;
          if (fabs(2.0*pixel.opacity) < threshold)
            pixel.opacity=(MagickRealType) destination.opacity;
          else
            pixel.opacity=(MagickRealType) (destination.opacity+
              (pixel.opacity*amount));
          destination.red=RoundToQuantum(pixel.red);
          destination.green=RoundToQuantum(pixel.green);
          destination.blue=RoundToQuantum(pixel.blue);
          destination.opacity=MaxRGB-RoundToQuantum(pixel.opacity);
          break;
        }
        case ModulateCompositeOp:
        {
          long
            offset;

          if (source.opacity == TransparentOpacity)
            break;
          offset=(long) (PixelIntensityToQuantum(&source)-midpoint);
          if (offset == 0)
            break;
          TransformHSB(destination.red,destination.green,destination.blue,&hue,
            &saturation,&brightness);
          brightness+=(0.01*percent_brightness*offset)/midpoint;
          saturation*=0.01*percent_saturation;
          HSBTransform(hue,saturation,brightness,&destination.red,
            &destination.green,&destination.blue);
          break;
        }
        case HueCompositeOp:
        {
          if (source.opacity == TransparentOpacity)
            break;
          if (destination.opacity == TransparentOpacity)
            {
              destination=source;
              break;
            }
          TransformHSB(destination.red,destination.green,destination.blue,&hue,
            &saturation,&brightness);
          TransformHSB(source.red,source.green,source.blue,&hue,&sans,&sans);
          HSBTransform(hue,saturation,brightness,&destination.red,
            &destination.green,&destination.blue);
          if (source.opacity < destination.opacity)
            destination.opacity=source.opacity;
          break;
        }
        case SaturateCompositeOp:
        {
          if (source.opacity == TransparentOpacity)
            break;
          if (destination.opacity == TransparentOpacity)
            {
              destination=source;
              break;
            }
          TransformHSB(destination.red,destination.green,destination.blue,&hue,
            &saturation,&brightness);
          TransformHSB(source.red,source.green,source.blue,&sans,&saturation,
            &sans);
          HSBTransform(hue,saturation,brightness,&destination.red,
            &destination.green,&destination.blue);
          if (source.opacity < destination.opacity)
            destination.opacity=source.opacity;
          break;
        }
        case LuminizeCompositeOp:
        {
          if (source.opacity == TransparentOpacity)
            break;
          if (destination.opacity == TransparentOpacity)
            {
              destination=source;
              break;
            }
          TransformHSB(destination.red,destination.green,destination.blue,&hue,
            &saturation,&brightness);
          TransformHSB(source.red,source.green,source.blue,&sans,&sans,
            &brightness);
          HSBTransform(hue,saturation,brightness,&destination.red,
            &destination.green,&destination.blue);
          if (source.opacity < destination.opacity)
            destination.opacity=source.opacity;
          break;
        }
        case ColorizeCompositeOp:
        {
          if (source.opacity == TransparentOpacity)
            break;
          if (destination.opacity == TransparentOpacity)
            {
              destination=source;
              break;
            }
          TransformHSB(destination.red,destination.green,destination.blue,&sans,
            &sans,&brightness);
          TransformHSB(source.red,source.green,source.blue,&hue,&saturation,
            &sans);
          HSBTransform(hue,saturation,brightness,&destination.red,
            &destination.green,&destination.blue);
          if (source.opacity < destination.opacity)
            destination.opacity=source.opacity;
          break;
        }
        case AddCompositeOp:  /* deprecated */
        {
          destination=MagickCompositeAdd(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case SubtractCompositeOp:  /* deprecated */
        {
          destination=MagickCompositeSubtract(&source,(MagickRealType)
            source.opacity,&destination,(MagickRealType) destination.opacity);
          break;
        }
        case CopyRedCompositeOp:  /* deprecated */
        case CopyCyanCompositeOp:
        {
          destination.red=source.red;
          break;
        }
        case CopyGreenCompositeOp:  /* deprecated */
        case CopyMagentaCompositeOp:
        {
          destination.green=source.green;
          break;
        }
        case CopyBlueCompositeOp:  /* deprecated */
        case CopyYellowCompositeOp:
        {
          destination.blue=source.blue;
          break;
        }
        case CopyOpacityCompositeOp:  /* deprecated */
        {
          if (composite_image->matte == MagickFalse)
            {
              destination.opacity=(Quantum)
                (MaxRGB-PixelIntensityToQuantum(&source));
              break;
            }
          destination.opacity=source.opacity;
          break;
        }
        case CopyBlackCompositeOp:  /* deprecated */
        {
          if ((image->colorspace == CMYKColorspace) &&
              (composite_image->colorspace == CMYKColorspace))
            indexes[x]=(*composite_indexes++);
          break;
        }
        default:
          break;
      }
      q->red=destination.red;
      q->green=destination.green;
      q->blue=destination.blue;
      if (image->matte != MagickFalse)
        q->opacity=destination.opacity;
      p++;
      if (p >= (pixels+composite_image->columns))
        p=pixels;
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
  }
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+  C o m p o s i t e I m a g e C o m m a n d                                  %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CompositeImageCommand() reads one or more images and an optional mask and
%  composites them into a new image.
%
%  The format of the CompositeImageCommand method is:
%
%      MagickBooleanType CompositeImageCommand(ImageInfo *image_info,int argc,
%        char **argv,char **metadata,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o argc: The number of elements in the argument vector.
%
%    o argv: A text array containing the command line arguments.
%
%    o metadata: any metadata is returned here.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

static MagickBooleanType CompositeImageList(ImageInfo *image_info,Image **image,
  Image *composite_image,CompositeOptions *option_info,ExceptionInfo *exception)
{
  long
    x,
    y;

  MagickBooleanType
    status;

  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image != (Image **) NULL);
  assert((*image)->signature == MagickSignature);
  if ((*image)->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),(*image)->filename);
  assert(exception != (ExceptionInfo *) NULL);
  status=MagickTrue;
  if (composite_image != (Image *) NULL)
    {
      assert(composite_image->signature == MagickSignature);
      if (option_info->compose == BlendCompositeOp)
        (void) CloneString(&composite_image->geometry,
          option_info->blend_geometry);
      if (option_info->compose == DisplaceCompositeOp)
        (void) CloneString(&composite_image->geometry,
          option_info->displace_geometry);
      if (option_info->compose == DissolveCompositeOp)
        (void) CloneString(&composite_image->geometry,
          option_info->dissolve_geometry);
      if (option_info->compose == ModulateCompositeOp)
        (void) CloneString(&composite_image->geometry,
          option_info->watermark_geometry);
      if (option_info->compose == ThresholdCompositeOp)
        (void) CloneString(&composite_image->geometry,
          option_info->unsharp_geometry);
      /*
        Composite image.
      */
      if (option_info->stegano != 0)
        {
          Image
            *stegano_image;

          (*image)->offset=option_info->stegano-1;
          stegano_image=SteganoImage(*image,composite_image,exception);
          if (stegano_image != (Image *) NULL)
            {
              DestroyImageList(*image);
              *image=stegano_image;
            }
        }
      else
        if (option_info->stereo != MagickFalse)
          {
            Image
              *stereo_image;

            stereo_image=StereoImage(*image,composite_image,exception);
            if (stereo_image != (Image *) NULL)
              {
                DestroyImageList(*image);
                *image=stereo_image;
              }
          }
        else
          if (option_info->tile != MagickFalse)
            {
              unsigned long
                columns;

              /*
                Tile the composite image.
              */
              columns=composite_image->columns;
              for (y=0; y < (long) (*image)->rows; y+=composite_image->rows)
                for (x=0; x < (long) (*image)->columns; x+=columns)
                {
                  status&=CompositeImage(*image,option_info->compose,
                    composite_image,x,y);
                  GetImageException(*image,exception);
                }
            }
          else
            {
              char
                composite_geometry[MaxTextExtent];

              RectangleInfo
                geometry;

              /*
                Digitally composite image.
              */
              SetGeometry(*image,&geometry);
              (void) ParseAbsoluteGeometry(option_info->geometry,&geometry);
              (void) FormatMagickString(composite_geometry,MaxTextExtent,
                "%lux%lu%+ld%+ld",composite_image->columns,
                composite_image->rows,geometry.x,geometry.y);
              (*image)->gravity=(GravityType) option_info->gravity;
              (void) ParseGravityGeometry(*image,composite_geometry,&geometry);
              status&=CompositeImage(*image,option_info->compose,
                composite_image,geometry.x,geometry.y);
              GetImageException(*image,exception);
            }
    }
  return(status);
}

static void CompositeUsage(void)
{
  const char
    **p;

  static const char
    *options[]=
    {
      "-affine matrix       affine transform matrix",
      "-authenticate value  decrypt image with this password",
      "-blend geometry      blend images",
      "-blue-primary point  chromaticity blue primary point",
      "-channel type        Red, Green, Blue, Opacity, Index, Cyan, Yellow, ",
      "                     Magenta, Black, or All",
      "-colors value        preferred number of colors in the image",
      "-colorspace type     alternate image colorspace",
      "-comment string      annotate image with comment",
      "-compose operator    composite operator",
      "-compress type       image compression type",
      "-debug events        display copious debugging information",
      "-define format:option",
      "                     define one or more image format options",
      "-density geometry    horizontal and vertical density of the image",
      "-depth value         image depth",
      "-displace geometry   shift image pixels defined by a displacement map",
      "-display server      get image or font from this X server",
      "-dispose method      Undefined, None, Background, Previous",
      "-dissolve value      dissolve the two images a given percent",
      "-dither              apply Floyd/Steinberg error diffusion to image",
      "-encoding type       text encoding type",
      "-endian type         LSB or MSB",
      "-extract geometry    extract area from image",
      "-filter type         use this filter when resizing an image",
      "-font name           render text with this font",
      "-geometry geometry   location of the composite image",
      "-gravity type        which direction to gravitate towards",
      "-green-primary point chromaticity green primary point",
      "-help                print program options",
      "-interlace type      None, Line, Plane, or Partition",
      "-label name          ssign a label to an image",
      "-limit type value    Area, Disk, Map, or Memory resource limit",
      "-log format          format of debugging information",
      "-matte               store matte channel if the image has one",
      "-monochrome          transform image to black and white",
      "-negate              replace every pixel with its complementary color ",
      "-page geometry       size and location of an image canvas",
      "-profile filename    add ICM or IPTC information profile to image",
      "-quality value       JPEG/MIFF/PNG compression level",
      "-red-primary point   chromaticity red primary point",
      "-rotate degrees      apply Paeth rotation to the image",
      "-resize geometry     resize the image",
      "-sampling-factor geometry",
      "                     horizontal and vertical sampling factor",
      "-scene value         image scene number",
      "-sharpen geometry    sharpen the image",
      "-size geometry       width and height of image",
      "-stegano offset      hide watermark within an image",
      "-stereo              combine two image to create a stereo anaglyph",
      "-strip               strip image of all profiles and comments",
      "-support factor      resize support: > 1.0 is blurry, < 1.0 is sharp",
      "-thumbnail geometry  create a thumbnail of the image",
      "-tile                repeat composite operation across image",
      "-transform           affine transform image",
      "-treedepth value     color tree depth",
      "-type type           image type",
      "-units type          PixelsPerInch, PixelsPerCentimeter, or Undefined",
      "-unsharp geometry    sharpen the image",
      "-verbose             print detailed information about the image",
      "-version             print version information",
      "-virtual-pixel method",
      "                     Constant, Edge, Mirror, or Tile",
      "-watermark geometry  percent brightness and saturation of a watermark",
      "-white-point point   chromaticity white point",
      "-write filename      write images to this file",
      (char *) NULL
    };

  (void) printf("Version: %s\n",GetMagickVersion((unsigned long *) NULL));
  (void) printf("Copyright: %s\n\n",GetMagickCopyright());
  (void) printf("Usage: %s [options ...] image [options ...] composite\n"
    "  [ [options ...] mask ] [options ...] composite\n",
    GetClientName());
  (void) printf("\nWhere options include:\n");
  for (p=options; *p != (char *) NULL; p++)
    (void) printf("  %s\n",*p);
  Exit(0);
}

static void RelinquishCompositeOptions(CompositeOptions *option_info)
{
  if (option_info->blend_geometry != (char *) NULL)
    option_info->blend_geometry=(char *)
      RelinquishMagickMemory(option_info->blend_geometry);
  if (option_info->displace_geometry != (char *) NULL)
    option_info->displace_geometry=(char *)
      RelinquishMagickMemory(option_info->displace_geometry);
  if (option_info->dissolve_geometry != (char *) NULL)
    option_info->dissolve_geometry=(char *)
      RelinquishMagickMemory(option_info->dissolve_geometry);
  if (option_info->geometry != (char *) NULL)
    option_info->geometry=(char *)
      RelinquishMagickMemory(option_info->geometry);
  if (option_info->unsharp_geometry != (char *) NULL)
    option_info->unsharp_geometry=(char *)
      RelinquishMagickMemory(option_info->unsharp_geometry);
  if (option_info->watermark_geometry != (char *) NULL)
    option_info->watermark_geometry=(char *)
      RelinquishMagickMemory(option_info->watermark_geometry);
}

MagickExport MagickBooleanType CompositeImageCommand(ImageInfo *image_info,
  int argc,char **argv,char **metadata,ExceptionInfo *exception)
{
#define NotInitialized  (unsigned int) (~0)
#define DestroyComposite() \
{ \
  RelinquishCompositeOptions(&option_info); \
  DestroyImageList(image); \
  DestroyImageList(composite_image); \
  DestroyImageList(mask_image); \
  for (i=0; i < (long) argc; i++) \
    argv[i]=(char *) RelinquishMagickMemory(argv[i]); \
  argv=(char **) RelinquishMagickMemory(argv); \
}
#define ThrowCompositeException(severity,tag,option) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),severity,tag,option);\
  DestroyComposite(); \
  return(MagickFalse); \
}
#define ThrowCompositeInvalidArgumentException(option,argument) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),OptionError, \
    "InvalidArgument",argument,option); \
  DestroyComposite(); \
  return(MagickFalse); \
}

  char
    *filename,
    *format,
    *option;

  CompositeOptions
    option_info;

  Image
    *mask_image,
    *composite_image,
    *image;

  MagickBooleanType
    status;

  long
    j;

  register long
    i;

  /*
    Set default.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(exception != (ExceptionInfo *) NULL);
  if (argc < 4)
    CompositeUsage();
  (void) ResetMagickMemory(&option_info,0,sizeof(CompositeOptions));
  option_info.compose=OverCompositeOp;
  composite_image=NewImageList();
  filename=(char *) NULL;
  format=(char *) "%w,%h,%m";
  image=NewImageList();
  mask_image=NewImageList();
  option=(char *) NULL;
  /*
    Check command syntax.
  */
  ReadCommandlLine(argc,&argv);
  status=ExpandFilenames(&argc,&argv);
  if (status == MagickFalse)
    ThrowCompositeException(ResourceLimitError,"MemoryAllocationFailed",
      strerror(errno));
  (void) CopyMagickString(image_info->filename,argv[argc-1],MaxTextExtent);
  (void) SetImageInfo(image_info,MagickTrue,exception);
  j=1;
  for (i=1; i < (long) (argc-1); i++)
  {
    option=argv[i];
    if ((strlen(option) == 1) || ((*option != '-') && (*option != '+')))
      {
        /*
          Read input images.
        */
        filename=argv[i];
        (void) CopyMagickString(image_info->filename,filename,MaxTextExtent);
        if (composite_image == (Image *) NULL)
          {
            composite_image=ReadImage(image_info,exception);
            if (exception->severity != UndefinedException)
              CatchException(exception);
            if (composite_image != (Image *) NULL)
              {
                status&=MogrifyImages(image_info,(int) (i-j),argv+j,
                  &composite_image);
                GetImageException(composite_image,exception);
              }
            j=i+1;
            continue;
          }
        if (mask_image != (Image *) NULL)
          ThrowCompositeException(OptionError,"InputImagesAlreadySpecified",
            filename);
        if (image == (Image *) NULL)
          {
            image=ReadImage(image_info,exception);
            if (exception->severity != UndefinedException)
              CatchException(exception);
            if (image != (Image *) NULL)
              {
                status&=MogrifyImages(image_info,(int) (i-j),argv+j,&image);
                GetImageException(image,exception);
              }
            j=i+1;
            continue;
          }
        mask_image=ReadImage(image_info,exception);
        if (exception->severity != UndefinedException)
          CatchException(exception);
        status&=mask_image != (Image *) NULL;
        j=i+1;
        continue;
      }
    switch (*(option+1))
    {
      case 'a':
      {
        if (LocaleCompare("affine",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("authenticate",option+1) == 0)
          {
            (void) CloneString(&image_info->authenticate,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->authenticate,argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'b':
      {
        if (LocaleCompare("background",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                (void) QueryColorDatabase(argv[i],&image_info->background_color,
                  exception);
              }
            break;
          }
        if (LocaleCompare("blend",option+1) == 0)
          {
            (void) CloneString(&option_info.blend_geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.blend_geometry,argv[i]);
                option_info.compose=BlendCompositeOp;
              }
            break;
          }
        if (LocaleCompare("blue-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'c':
      {
        if (LocaleCompare("cache",option+1) == 0)
          {
            if (*option == '-')
              {
                unsigned long
                  limit;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                (void) SetMagickResourceLimit(MemoryResource,limit);
                (void) SetMagickResourceLimit(MapResource,2*limit);
              }
            break;
          }
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  channel;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                channel=ParseChannelOption(argv[i]);
                if (channel < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedChannelType",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("colors",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                image_info->colors=(unsigned long) atol(argv[i]);
              }
            break;
          }
        if (LocaleCompare("colorspace",option+1) == 0)
          {
            image_info->colorspace=UndefinedColorspace;
            if (*option == '-')
              {
                long
                  colorspace;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                colorspace=ParseMagickOption(MagickColorspaceOptions,
                  MagickFalse,argv[i]);
                if (colorspace < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedColorspace",
                    argv[i]);
                image_info->colorspace=(ColorspaceType) colorspace;
              }
            break;
          }
        if (LocaleCompare("comment",option+1) == 0)
          {
            if (*option == '+')
              {
                status=DeleteImageOption(image_info,"comment");
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"comment",argv[i]);
            if (status == MagickFalse)
              ThrowCompositeException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("compose",option+1) == 0)
          {
            option_info.compose=OverCompositeOp;
            if (*option == '-')
              {
                long
                  compose;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                compose=ParseMagickOption(MagickCompositeOptions,MagickFalse,
                  argv[i]);
                if (compose < 0)
                  ThrowCompositeException(OptionError,
                    "UnrecognizedComposeOperator",argv[i]);
                option_info.compose=(CompositeOperator) compose;
              }
            (void) strcpy(option+1,"sans");
            break;
          }
        if (LocaleCompare("compress",option+1) == 0)
          {
            image_info->compression=UndefinedCompression;
            if (*option == '-')
              {
                long
                  compression;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                compression=ParseMagickOption(MagickCompressionOptions,
                  MagickFalse,argv[i]);
                if (compression < 0)
                  ThrowCompositeException(OptionError,
                    "UnrecognizedImageCompression",argv[i]);
                image_info->compression=(CompressionType) compression;
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'd':
      {
        if (LocaleCompare("debug",option+1) == 0)
          {
            (void) SetLogEventMask("None");
            if (*option == '-')
              {
                LogEventType
                  event_mask;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                event_mask=SetLogEventMask(argv[i]);
                if (event_mask == UndefinedEvents)
                  ThrowCompositeException(OptionError,"UnrecognizedEventType",
                    option);
              }
            image_info->debug=IsEventLogging();
            break;
          }
        if (LocaleCompare("define",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            if (*option == '+')
              {
                char
                  *define;

                define=RemoveImageOption(image_info,argv[i]);
                if (define == (char *) NULL)
                  ThrowCompositeException(OptionError,"NoSuchOption",argv[i]);
                define=(char *) RelinquishMagickMemory(define);
                break;
              }
            status=DefineImageOption(image_info,argv[i]);
            if (status == MagickFalse)
              ThrowCompositeException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("density",option+1) == 0)
          {
            (void) CloneString(&image_info->density,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->density,argv[i]);
              }
            break;
          }
        if (LocaleCompare("depth",option+1) == 0)
          {
            image_info->depth=QuantumDepth;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                image_info->depth=(unsigned long) atol(argv[i]);
              }
            break;
          }
        if (LocaleCompare("displace",option+1) == 0)
          {
            (void) CloneString(&option_info.displace_geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.displace_geometry,argv[i]);
                option_info.compose=DisplaceCompositeOp;
              }
            break;
          }
        if (LocaleCompare("display",option+1) == 0)
          {
            (void) CloneString(&image_info->server_name,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->server_name,argv[i]);
              }
            break;
          }
        if (LocaleCompare("dispose",option+1) == 0)
          {
            long
              dispose;

            if (*option == '+')
              {
                char
                  *dispose;

                dispose=RemoveImageOption(image_info,argv[i]);
                if (dispose != (char *) NULL)
                  dispose=(char *) RelinquishMagickMemory(dispose);
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            dispose=ParseMagickOption(MagickDisposeOptions,MagickFalse,argv[i]);
            if (dispose < 0)
              ThrowCompositeException(OptionError,"UnrecognizedDisposeMethod",
                argv[i]);
            status=SetImageOption(image_info,"dispose",argv[i]);
            if (status == MagickFalse)
              ThrowCompositeException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("dissolve",option+1) == 0)
          {
            (void) CloneString(&option_info.dissolve_geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.dissolve_geometry,argv[i]);
                option_info.compose=DissolveCompositeOp;
              }
            break;
          }
        if (LocaleCompare("dither",option+1) == 0)
          {
            image_info->dither=(MagickBooleanType) (*option == '-');
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'e':
      {
        if (LocaleCompare("encoding",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("endian",option+1) == 0)
          {
            image_info->endian=UndefinedEndian;
            if (*option == '-')
              {
                long
                  endian;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                endian=ParseMagickOption(MagickEndianOptions,MagickFalse,
                  argv[i]);
                if (endian < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedEndianType",
                    argv[i]);
                image_info->endian=(EndianType) endian;
              }
            break;
          }
        if (LocaleCompare("extract",option+1) == 0)
          {
            (void) CloneString(&image_info->extract,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->extract,argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'f':
      {
        if (LocaleCompare("filter",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  filter;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                filter=ParseMagickOption(MagickFilterOptions,MagickFalse,
                  argv[i]);
                if (filter < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedImageFilter",
                    argv[i]);
                break;
              }
            break;
          }
        if (LocaleCompare("font",option+1) == 0)
          {
            (void) CloneString(&image_info->font,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                (void) CloneString(&image_info->font,argv[i]);
              }
            break;
          }
        if (LocaleCompare("format",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                format=argv[i];
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'g':
      {
        if (LocaleCompare("geometry",option+1) == 0)
          {
            (void) CloneString(&option_info.geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.geometry,argv[i]);
              }
            break;
          }
        if (LocaleCompare("gravity",option+1) == 0)
          {
            option_info.gravity=UndefinedGravity;
            if (*option == '-')
              {
                long
                  gravity;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                gravity=ParseMagickOption(MagickGravityOptions,MagickFalse,
                  argv[i]);
                if (gravity < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedGravityType",
                    argv[i]);
                option_info.gravity=(GravityType) gravity;
                break;
              }
            break;
          }
        if (LocaleCompare("green-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'h':
      {
        if (LocaleCompare("help",option+1) == 0)
          CompositeUsage();
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'i':
      {
        if (LocaleCompare("interlace",option+1) == 0)
          {
            image_info->interlace=UndefinedInterlace;
            if (*option == '-')
              {
                long
                  interlace;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                interlace=ParseMagickOption(MagickInterlaceOptions,MagickFalse,
                  argv[i]);
                if (interlace < 0)
                  ThrowCompositeException(OptionError,
                    "UnrecognizedInterlaceType",argv[i]);
                image_info->interlace=(InterlaceType) interlace;
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'l':
      {
        if (LocaleCompare("label",option+1) == 0)
          {
            if (*option == '+')
              {
                status=DeleteImageOption(image_info,"label");
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"label",argv[i]);
            if (status == MagickFalse)
              ThrowCompositeException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("limit",option+1) == 0)
          {
            if (*option == '-')
              {
                char
                  *type;

                long
                  resource;

                unsigned long
                  limit;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                type=argv[i];
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                resource=ParseMagickOption(MagickResourceOptions,MagickFalse,
                  type);
                if (resource < 0)
                  ThrowCompositeException(OptionError,
                    "UnrecognizedResourceType",type);
                (void) SetMagickResourceLimit((ResourceType) resource,limit);
                break;
              }
            break;
          }
        if (LocaleCompare("log",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if ((i == (long) argc) || (strchr(argv[i],'%') == (char *) NULL))
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                (void) SetLogFormat(argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'm':
      {
        if (LocaleCompare("matte",option+1) == 0)
          break;
        if (LocaleCompare("monochrome",option+1) == 0)
          break;
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'n':
      {
        if (LocaleCompare("negate",option+1) == 0)
          break;
        if (LocaleCompare("noop",option+1) == 0)
          break;
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'p':
      {
        if (LocaleCompare("page",option+1) == 0)
          {
            (void) CloneString(&image_info->page,(char *) NULL);
            if (*option == '+')
              {
                char
                  *page;

                page=RemoveImageOption(image_info,argv[i]);
                if (page != (char *) NULL)
                  page=(char *) RelinquishMagickMemory(page);
                break;
              }
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            status=SetImageOption(image_info,"page",argv[i]);
            if (status == MagickFalse)
              ThrowCompositeException(OptionError,"UnrecognizedOption",argv[i]);
            image_info->page=GetPageGeometry(argv[i]);
            break;
          }
        if (LocaleCompare("process",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
              }
            break;
          }
        if (LocaleCompare("profile",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'q':
      {
        if (LocaleCompare("quality",option+1) == 0)
          {
            image_info->quality=UndefinedCompressionQuality;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                image_info->quality=(unsigned long) atol(argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'r':
      {
        if (LocaleCompare("red-primary",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("render",option+1) == 0)
          break;
        if (LocaleCompare("resize",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("rotate",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowCompositeInvalidArgumentException(option,argv[i]);
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 's':
      {
        if (LocaleCompare("sampling-factor",option+1) == 0)
          {
            (void) CloneString(&image_info->sampling_factor,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->sampling_factor,argv[i]);
              }
            break;
          }
        if (LocaleCompare("scene",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("sharpen",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            if (IsGeometry(argv[i]) == MagickFalse)
              ThrowCompositeInvalidArgumentException(option,argv[i]);
            break;
          }
        if (LocaleCompare("size",option+1) == 0)
          {
            (void) CloneString(&image_info->size,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->size,argv[i]);
              }
            break;
          }
        if (LocaleCompare("stegano",option+1) == 0)
          {
            option_info.stegano=0;
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                option_info.stegano=atol(argv[i])+1;
              }
            break;
          }
        if (LocaleCompare("stereo",option+1) == 0)
          {
            option_info.stereo=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("strip",option+1) == 0)
          break;
        if (LocaleCompare("support",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 't':
      {
        if (LocaleCompare("thumbnail",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("tile",option+1) == 0)
          {
            option_info.tile=(MagickBooleanType) (*option == '-');
            (void) strcpy(option+1,"sans");
            break;
          }
        if (LocaleCompare("transform",option+1) == 0)
          break;
        if (LocaleCompare("treedepth",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("type",option+1) == 0)
          {
            image_info->type=UndefinedType;
            if (*option == '-')
              {
                long
                  type;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                type=ParseMagickOption(MagickImageOptions,MagickFalse,argv[i]);
                if (type < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedImageType",
                    argv[i]);
                image_info->type=(ImageType) type;
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'u':
      {
        if (LocaleCompare("units",option+1) == 0)
          {
            image_info->units=UndefinedResolution;
            if (*option == '-')
              {
                long
                  units;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                units=ParseMagickOption(MagickResolutionOptions,MagickFalse,
                  argv[i]);
                if (units < 0)
                  ThrowCompositeException(OptionError,"UnrecognizedUnitsType",
                    argv[i]);
                i++;
              }
            break;
          }
        if (LocaleCompare("unsharp",option+1) == 0)
          {
            (void) CloneString(&option_info.unsharp_geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.unsharp_geometry,argv[i]);
                option_info.compose=ThresholdCompositeOp;
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'v':
      {
        if (LocaleCompare("verbose",option+1) == 0)
          {
            image_info->verbose=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("version",option+1) == 0)
          break;
        if (LocaleCompare("virtual-pixel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  method;

                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                method=ParseMagickOption(MagickVirtualPixelOptions,MagickFalse,
                  argv[i]);
                if (method < 0)
                  ThrowCompositeException(OptionError,
                    "UnrecognizedVirtualPixelMethod",argv[i]);
              }
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case 'w':
      {
        if (LocaleCompare("watermark",option+1) == 0)
          {
            (void) CloneString(&option_info.watermark_geometry,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
                (void) CloneString(&option_info.watermark_geometry,argv[i]);
                option_info.compose=ModulateCompositeOp;
              }
            break;
          }
        if (LocaleCompare("white-point",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == (long) argc)
                  ThrowCompositeException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowCompositeInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        if (LocaleCompare("write",option+1) == 0)
          {
            i++;
            if (i == (long) argc)
              ThrowCompositeException(OptionError,"MissingArgument",option);
            break;
          }
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
      }
      case '?':
        break;
      default:
        ThrowCompositeException(OptionError,"UnrecognizedOption",option)
    }
  }
  if (i != (long) (argc-1))
    ThrowCompositeException(OptionError,"MissingAnImageFilename",argv[i]);
  if (image == (Image *) NULL)
    ThrowCompositeException(OptionError,"MissingAnImageFilename",argv[argc-1]);
  status&=MogrifyImages(image_info,(int) (i-j),argv+j,&image);
  GetImageException(image,exception);
  if (mask_image != (Image *) NULL)
    {
      SetImageType(composite_image,TrueColorMatteType);
      if (composite_image->matte == MagickFalse)
        SetImageOpacity(composite_image,OpaqueOpacity);
      status&=CompositeImage(composite_image,CopyOpacityCompositeOp,mask_image,
        0,0);
      if (status == MagickFalse)
        GetImageException(composite_image,exception);
    }
  status&=CompositeImageList(image_info,&image,composite_image,&option_info,
    exception);
  /*
    Write composite images.
  */
  status&=WriteImages(image_info,image,argv[argc-1],exception);
  if (metadata != (char **) NULL)
    {
      char
        *text;

      text=TranslateText(image_info,image,format);
      if (text == (char *) NULL)
        ThrowCompositeException(ResourceLimitError,"MemoryAllocationFailed",
          strerror(errno));
      (void) ConcatenateString(&(*metadata),text);
      (void) ConcatenateString(&(*metadata),"\n");
      text=(char *) RelinquishMagickMemory(text);
    }
  RelinquishCompositeOptions(&option_info);
  DestroyComposite();
  return(status);
}
