/*****************************************************************************/

/*
 *      usmodules.c  --  pcimodules like utility for the USB bus
 *     
 *	lsusb.c is derived from:
 *
 *	lspci.c					by Thomas Sailer,
 *	pcimodules.c				by Adam J. Richter
 *	linux-2.4.0-test10/include/linux/usb.h	probably by Randy Dunlap
 *
 *	The code in usbmodules not derived from elsewhere was written by
 *	Adam J. Richter.  David Brownell added the --mapfile and --version
 *	options.
 *
 *	Copyright (C) 2000, 2001  Yggdrasil Computing, Inc.
 *      Copyright (C) 1999  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

/*****************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/param.h>
#include <sys/utsname.h>

#include <linux/types.h>
#ifdef HAVE_LINUX_USBDEVICE_FS_H
#include <linux/usbdevice_fs.h>
#else
#include "usbdevice_fs.h"
#endif
#ifdef HAVE_LINUX_USB_H
#include <linux/usb.h>
#else
#include "usb.h"
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "usbmodules.h"
#include "names.h"
#include "devtree.h"

#define _GNU_SOURCE
#include <getopt.h>

#define OPT_STRING "c:d:hm:v"
static struct option long_options[] = {
	{"check",	required_argument,	NULL,	'c'},
	{"device",      required_argument,      NULL,   'd'},
	{"help",	no_argument, 		NULL,	'h'},
	{"mapfile",     required_argument,      NULL,   'm'},
	{"version",	no_argument, 		NULL,	'v'},
	{ 0,            0,                      NULL,   0}
};

#define MODDIR "/lib/modules"
#define USBMAP "modules.usbmap"

#define LINELENGTH     8000 

#define CTRL_RETRIES    50
#define CTRL_TIMEOUT   100     /* milliseconds */

#define USB_DT_CS_DEVICE               0x21
#define USB_DT_CS_CONFIG               0x22
#define USB_DT_CS_STRING               0x23
#define USB_DT_CS_INTERFACE            0x24
#define USB_DT_CS_ENDPOINT             0x25

static char *checkname = NULL;

static int idVendor;
static int idProduct;
static int bcdDevice;
static int bDeviceClass;
static int bDeviceSubClass;
static int bDeviceProtocol;
struct usbmap_entry *usbmap_list;

static void *
xmalloc(unsigned int size) {
       void *result = malloc(size);
       if (result == NULL) {
               fprintf(stderr, "Memory allocation failure.\n");
               exit(1);
       }
       return result;
}

static int
scan_without_flags(const char *line, struct usbmap_entry *entry, char *name) {
	unsigned int driver_info;
	if (sscanf(line,
		   "%s 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
		   name,
		   &entry->idVendor,
		   &entry->idProduct,
		   &entry->bcdDevice_lo,
		   &entry->bcdDevice_hi,
		   &entry->bDeviceClass,
		   &entry->bDeviceSubClass,
		   &entry->bDeviceProtocol,
		   &entry->bInterfaceClass,
		   &entry->bInterfaceSubClass,
		   &entry->bInterfaceProtocol,
		   &driver_info) != 12)
		return 0;

	entry->match_flags = 0;

	/* idVendor==0 is the wildcard for both idVendor and idProduct,
	   because idProduct==0 is a legitimate product ID. */
	if (entry->idVendor)
		entry->match_flags |= USB_MATCH_VENDOR | USB_MATCH_PRODUCT;

	if (entry->bcdDevice_lo)
		entry->match_flags |= USB_MATCH_DEV_LO;

	if (entry->bcdDevice_hi)
		entry->match_flags |= USB_MATCH_DEV_HI;

	if (entry->bDeviceClass)
		entry->match_flags |= USB_MATCH_DEV_CLASS;

	if (entry->bDeviceSubClass)
		entry->match_flags |= USB_MATCH_DEV_SUBCLASS;

	if (entry->bDeviceProtocol)
		entry->match_flags |= USB_MATCH_DEV_PROTOCOL;

	if (entry->bInterfaceClass)
		entry->match_flags |= USB_MATCH_INT_CLASS;

	if (entry->bInterfaceSubClass)
		entry->match_flags |= USB_MATCH_INT_SUBCLASS;

	if (entry->bInterfaceProtocol)
		entry->match_flags |= USB_MATCH_INT_PROTOCOL;

	return 1;
}

static int
scan_with_flags(const char *line, struct usbmap_entry *entry, char *name) {
	unsigned int driver_info;
	return (sscanf(line, "%s 0x%x 0x%x "
		       "0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
		       name,
		       &entry->match_flags,
		       &entry->idVendor,
		       &entry->idProduct,
		       &entry->bcdDevice_lo,
		       &entry->bcdDevice_hi,
		       &entry->bDeviceClass,
		       &entry->bDeviceSubClass,
		       &entry->bDeviceProtocol,
		       &entry->bInterfaceClass,
		       &entry->bInterfaceSubClass,
		       &entry->bInterfaceProtocol,
		       &driver_info) == 13);
}

void
read_modules_usbmap(char *pathname)
{
       char filename[MAXPATHLEN];
       FILE *usbmap_file;
       char line[LINELENGTH];
       struct usbmap_entry *prev;
       struct usbmap_entry *entry;
       char name[LINELENGTH];

       if (pathname == NULL) {
	       struct utsname utsname;
	       if (uname(&utsname) < 0) {
		       perror("uname");
		       exit(1);
	       }
	       sprintf(filename, "%s/%s/%s", MODDIR, utsname.release, USBMAP);
	       pathname = filename;
       }
       if ((usbmap_file = fopen(pathname, "r")) == NULL) {
               perror(pathname);
               exit(1);
       }

       prev = NULL;
       while(fgets(line, LINELENGTH, usbmap_file) != NULL) {
               if (line[0] == '#')
                       continue;

               entry = xmalloc(sizeof(struct usbmap_entry));

	       if (!scan_with_flags(line, entry, name) &&
		   !scan_without_flags(line, entry, name)) {
                       fprintf (stderr,
                               "modules.usbmap unparsable line: %s.\n", line);
                       free(entry);
                       continue;
               }

               /* Optimize memory allocation a bit, in case someday we
                  have Linux systems with ~100,000 modules.  It also
                  allows us to just compare pointers to avoid trying
                  to load a module twice. */
               if (prev == NULL || strcmp(name, prev->name) != 0) {
                       entry->name = xmalloc(strlen(name)+1);
                       strcpy(entry->name, name);
                       entry->selected_ptr = &entry->selected;
                       entry->selected = 0;
                       prev = entry;
               } else {
                       entry->name = prev->name;
                       entry->selected_ptr = prev->selected_ptr;
               }
               entry->next = usbmap_list;
               usbmap_list = entry;
       }
       fclose(usbmap_file);
}

/* Match modules is called once per interface.  We know that
   each device has at least one interface, because, according
   to the USB 2.0 Specification, section 9.6.3, "A USB device has
   one or more configuration descriptors.  Each configuration has
   one or more interfaces and each interface has zero or more endpoints."
   So, there must be at least one interface on a device.
*/

static void
match_modules(int bInterfaceClass,
            int bInterfaceSubClass,
            int bInterfaceProtocol)
{
        struct usbmap_entry *mod;

        for (mod = usbmap_list; mod != NULL; mod = mod->next) {

                if ((mod->match_flags & USB_MATCH_VENDOR) &&
                    mod->idVendor != idVendor)
                        continue;

                if ((mod->match_flags & USB_MATCH_PRODUCT) &&
                    mod->idProduct != idProduct)
                        continue;

                if ((mod->match_flags & USB_MATCH_DEV_LO) &&
                    mod->bcdDevice_lo > bcdDevice)
                        continue;

                if ((mod->match_flags & USB_MATCH_DEV_HI) &&
                    mod->bcdDevice_hi < bcdDevice)
                        continue;

                if ((mod->match_flags & USB_MATCH_DEV_CLASS) &&
                    mod->bDeviceClass != bDeviceClass)
                        continue;

                if ((mod->match_flags & USB_MATCH_DEV_SUBCLASS) &&
                    mod->bDeviceSubClass != bDeviceSubClass)
                        continue;

                if ((mod->match_flags & USB_MATCH_DEV_PROTOCOL) &&
                    mod->bDeviceProtocol != bDeviceProtocol)
                        continue;

                if ((mod->match_flags & USB_MATCH_INT_CLASS) &&
                    mod->bInterfaceClass != bInterfaceClass)
                        continue;

                if ((mod->match_flags & USB_MATCH_INT_SUBCLASS) &&
                    mod->bInterfaceSubClass != bInterfaceSubClass)
                        continue;

                if ((mod->match_flags & USB_MATCH_INT_PROTOCOL) &&
                    mod->bInterfaceProtocol != bInterfaceProtocol)
                        continue;

		if (checkname != NULL) {
			if (strcmp(checkname, mod->name) == 0)
				exit(0);  /* Program returns "success" */
		} else if (!(*mod->selected_ptr)) {
                        *(mod->selected_ptr) = 1;
                        printf ("%s\n", mod->name);
                }
        }
}

static int usb_control_msg(int fd,
                          u_int8_t requesttype,
                          u_int8_t request,
                          u_int16_t value,
                          u_int16_t index,
                          unsigned int size,
                          void *data)
{
       int result;
       int try;

       struct usbdevfs_ctrltransfer ctrl;

       ctrl.requesttype = requesttype;
       ctrl.request = request;
       ctrl.value = value;
       ctrl.index = index;
       ctrl.length = size;
       ctrl.data = data;
       ctrl.timeout = CTRL_TIMEOUT;

       /* At least on UHCI controllers, this ioctl gets a lot of
          ETIMEDOUT errors which can often be retried with success
          one is persistent enough.  So, we try 100 times, which work
          on one machine, but not on my notebook computer.
          --Adam J. Richter (adam@yggdrasil.com) 2000 November 03. */

       try = 0;
       do {
	  	result = ioctl(fd, USBDEVFS_CONTROL, &ctrl);
               try++;
       } while (try < CTRL_RETRIES && result == -1 && errno == ETIMEDOUT);
       return result;
}


static void do_config(int fd, unsigned int nr)
{
       unsigned char buf[1024], *p;
       unsigned int sz;

       if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR,
                           (USB_DT_CONFIG << 8) | nr,
                           0, USB_DT_CONFIG_SIZE, buf) < 0) {
               fprintf(stderr ,"cannot get config descriptor %d, %s (%d)\n",
                       nr, strerror(errno), errno);
               return;
       }
       if (buf[0] < USB_DT_CONFIG_SIZE || buf[1] != USB_DT_CONFIG)
               fprintf(stderr, "Warning: invalid config descriptor\n");
       sz = buf[2] | buf[3] << 8;
       if (sz > sizeof(buf)) {
               fprintf(stderr,
                       "Config %d descriptor too long, truncating\n", nr);
               sz = sizeof(buf);
       }
       if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR,
                           (USB_DT_CONFIG << 8) | nr, 0, sz, buf) < 0) {
               fprintf(stderr, "cannot get config descriptor %d, %s (%d)\n",
                       nr, strerror(errno), errno);
               return;
       }
       p = buf;
       while (sz >= 2 && p[0] >= 2 && p[0] < sz) {
               if (p[1] == USB_DT_INTERFACE) {
                       const int intClass = p[5];
                       const int intSubClass = p[6];
                       const int intProto = p[7];
                       match_modules(intClass, intSubClass, intProto);
               }
               sz -= p[0];
               p += p[0];
       }
}

static void process_device(const char *path)
{
        unsigned char buf[USB_DT_DEVICE_SIZE];
        int fd;
       unsigned int i, maxcfg;

       if ((fd = open(path, O_RDWR)) == -1) {
               fprintf(stderr, "cannot open %s, %s (%d)\n",
                       path, strerror(errno), errno);
               return;
       }
       if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR,
                           (USB_DT_DEVICE << 8), 0, USB_DT_DEVICE_SIZE, buf)
           < 0) {
               perror("cannot get config descriptor");
               goto err;
       }
       bDeviceClass = buf[4];
       bDeviceSubClass = buf[5];
       bDeviceProtocol = buf[6];
       idVendor = buf[8] | (buf[9] << 8);
       idProduct = buf[10] | (buf[11] << 8);
       bcdDevice = buf[12] | (buf[13] << 8);

       maxcfg = buf[17];
       if (buf[0] < 18 || buf[1] != USB_DT_DEVICE)
               maxcfg = 1;

       for (i = 0; i < maxcfg; i++)
               do_config(fd, i);
 err:
       close(fd);
}


int
main (int argc, char **argv)
{
       int opt_index = 0;
       int opt;
       char *device = NULL;
       char *pathname = NULL;

       while ((opt = getopt_long(argc, argv, OPT_STRING, long_options,
                          &opt_index)) != -1) {
               switch(opt) {
			case 'c':
				checkname = optarg;
				break;
                       case 'd':
                               device = optarg;
                               break;
                       case 'h':
				printf ("Usage: usbmodules [--help] [--device /proc/bus/usb/NNN/NNN] [--check module]\n"
				"\t[--mapfile pathname] [--version]\n"
	"  Lists kernel modules corresponding to USB devices currently plugged\n"
                                       "  into the computer.\n");
                               return 0;
		       case 'm':
                               pathname = optarg;
                               break;
		       case 'v':
                               puts (VERSION);
                               return 0;
                       default:
                               fprintf(stderr,
                                       "Unknown argument character \"%c\".\n",
                                       opt);
                               return 1;
               }
       }       

       if (device == NULL) {
               fprintf (stderr,
                        "You must specify a device with something like:\n"
                        "\tusbmodules --device /proc/bus/usb/001/009\n");
               return 1;
       }
       read_modules_usbmap(pathname);
       process_device(device);

	if (checkname != NULL)
		return 1; /* The module being checked was not needed */

       return 0;
}
