/*****************************************************************************
* fpipemon.c	Frame Relay Monitor.
*
* Authors:	Jaspreet Singh	
*
* Copyright:	(c) 1995-1997 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Mar 22, 1997	Jaspreet Singh	Improved Error handling
* Nov 24, 1997	Jaspreet Singh	Added new stats for driver statistics
* Nov 13, 1997	Jaspreet Singh	Fixed descriptions of Global Error Statistics
* Oct 20, 1997 	Jaspreet Singh	Added new commands for driver specific stats
*				and router up time.
* Jul 28, 1997	Jaspreet Singh	Added a new command for running line trace 
*				displaying RAW data.
* Jul 25, 1997	Jaspreet Singh	Added commands for viewing specific DLCI data 
*				including FECN and BECN. 
* Jun 24, 1997	Jaspreet Singh	S508/FT1 test commands		
* Apr 25, 1997	Farhan Thawar	Initial version based on fpipemon for WinNT.
*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include "fapi_m.h"

#define TIMEOUT 1
#define MDATALEN 2024
#define TRUE 1
#define FALSE 0

#pragma pack(1)
typedef struct {
   unsigned char signature[8];
   unsigned char requestreply;
   unsigned char id;
   unsigned char reserved2[6];
   unsigned char opp_flag;
   unsigned char command;
   unsigned short buffer_length;
   unsigned char return_code;
   unsigned short dlci;
   unsigned char FECN_BECN_DE_CR_bits;
   unsigned short rx_frm_disc_int;
   unsigned long rx_frm_disc_app;
   unsigned char reserved[2];
   unsigned char data[MDATALEN];
} CBLOCK;
#pragma pack()

/******************************************************************************
 * Structure for tracing
 *
 * The one frame trace;
 */
typedef struct {
   // equals to 0x00 if the frame is incoming, 0x01 if outgoing
   // for the first frame of the bunch of frames sent to user the first
   // 5 bits of this byte also tells number of frames passed
   unsigned char status;
   unsigned char passed_length;	// 0: no data passed. 1: real bytes passed
   unsigned short real_length;	// real(original) frame length
   unsigned short time_stamp;	// the frame's time stamp
   unsigned char data[1];	// first byte of data

} FRAME_DATA, *PFRAME_DATA;


/* Prototypes */
void ResetCB(CBLOCK *c1);
int MakeConnection( void );
int ObtainConfiguration( void );
unsigned char DoCommand( CBLOCK *cb );
void SaveCB( CBLOCK *dest, CBLOCK *src );
void init( int argv, char *argc[]);
void error( char return_code);

/* global for now */
int sock;
struct sockaddr_in soin;
CBLOCK cb, cbconfig;
char codeversion[10];
unsigned int frame_count;
unsigned char station_config;
int raw_data = FALSE;
int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int off_counter, green_counter, red_counter;
int loop_counter;

/* defines for now */
char ipaddress[16];
int udp_port = 9000;
unsigned short dlci_number = 0;

void ResetCB(CBLOCK *c1)
{
	memset((void *)&c1->opp_flag, 0, sizeof(CBLOCK)-16);
}; //ResetCB

int MakeConnection( void ) 
{
   	sock = socket( PF_INET, SOCK_DGRAM, 0 );
   	if( sock < 0 ) {
      		printf("Error: Unable to open socket!\n");
      		return( FALSE );
   	} /* if */
   	
	soin.sin_family = AF_INET;
   	soin.sin_addr.s_addr = inet_addr(ipaddress);
	if(soin.sin_addr.s_addr < 0){
		printf("Error: Invalid IP address!\n");
		return( FALSE );
	}
   	soin.sin_port = htons((u_short)udp_port);
   
	if( !connect( sock, (struct sockaddr *)&soin, sizeof(soin)) == 0 ) {
      		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
      		return( FALSE );
   	} /* if */
   	if( !ObtainConfiguration() ) {
      		printf("Error: Unable to obtain frame relay information.\nMake sure the IP and UDP port are correct.\n");
      		close( sock );
      		return( FALSE );
   	} /* if */   
   	return( TRUE );
}; /* MakeConnection */

int ObtainConfiguration( void ) 
{
   	unsigned char x;
   
   	x = 0;
   	cb.command = READ_DLCI_CONFIGURATION;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	while (DoCommand(&cb) != 0 && ++x < 4) {
      		if (cb.return_code == 0xaa) {
	 		printf("Error: Command timeout occurred\n"); 
	 		return(FALSE);
      		}
      		if (cb.return_code == 0xCC ) return(FALSE);
   	}
   
	if (x >= 4) return(FALSE);
   	station_config = cb.data[0];
   
   	strcpy(codeversion, "?.??");
   
   	cb.command = READ_CODE_VERSION;
  	cb.buffer_length = 0;
   	DoCommand(&cb);

   	if (cb.return_code == 0) {
      		cb.data[cb.buffer_length] = 0;
      		strcpy(codeversion, cb.data);
   	}
   	return(TRUE);
}; /* ObtainConfiguration */

unsigned char DoCommand( CBLOCK *cb ) 
{
   	static unsigned char id = 0;
   	fd_set ready;
   	struct timeval to;
   	int x;

   	for( x = 0; x < 4; x += 1 ) {
      		cb->opp_flag = 0;
      		cb->requestreply = 0x01;
      		cb->id = id;
      		cb->return_code = 0xaa;	// 0xAA is our special return code indicating packet timeout
      		send(sock, cb, 32+cb->buffer_length,0);
      		FD_ZERO(&ready);
      		FD_SET(sock,&ready);
      		to.tv_sec = 5;
      		to.tv_usec = 0;
      		if(select(sock + 1,&ready, NULL, NULL, &to)) {
	 		recv(sock, cb, 32+MDATALEN,0);
			break;
      		} else {
	 		printf("Error: No Data received from the connected socket.\n"); 
      		} /* if */
   	} 
   	/* make sure the id is correct (returning results from a previous
   	   call may be disastrous if not recognized)
   	   also make sure it is a reply
	 */	
   	if (cb->id != id || cb->requestreply != 0x02) cb->return_code = 0xbb;
   	id++;
   	if (cb->return_code == 0xCC) {
      		printf("Error: Code is not running on the board!\n");
      		exit(1);
   	}
   	return(cb->return_code);
   
}; /* DoCommand */

void init( int argv, char *argc[]){
   	cb.id = 0;

   	/* Different signature for Driver Statistics */
   	if( argc[1][0] == 'd')
   		strcpy( cb.signature, "DRVSTATS");
   	else
		strcpy( cb.signature, "FPIPE8ND");
 			
  	 /* This is for a specific dlci command */
   	if( argc[1][0] == 'i'){
      		if( argv == 5 || argv == 4){
      			strcpy(ipaddress,argc[3]);
        		dlci_number = atoi(argc[2]);
			if (argv == 5)
				udp_port = atoi(argc[4]);
      		} 
		else{
			printf("Error: Invalid number of arguments\n");
			exit(1);
     		}		
   	} else if( (argc[1][0] == 'f') && (argc[2][0] == 'i' )){
      		if( argv == 6 || argv == 5){
      			strcpy(ipaddress,argc[4]);
      			dlci_number = atoi(argc[3]);
			if(argv == 6)
				udp_port = atoi(argc[5]);
      		} else {
			printf("Error: Invalid number of arguments\n");
			exit(1);
		}
  	 } else if( (argc[1][0] == 'f') || (argc[1][0] == 't' )) {
      		if( argv == 5 || argv == 4) {
      			strcpy(ipaddress,argc[3]);
			if( argv == 5)
	 			udp_port = atoi(argc[4]);
      		} else {
			printf("Error: Invalid number of arguments\n");
			exit(1);
		} 
   	} else {
      		if( argv == 4 || argv == 3) {
      			strcpy(ipaddress,argc[2]);
			if( argv == 4 )
	 			udp_port = atoi(argc[3]);
      		} else {	
			printf("Error: Invalid number of arguments\n");
			exit(1);
		}
   	}
};

void error( char return_code ) 
{
	switch( return_code ){
		case 0x04:
			printf("Error: An invalid DLCI was selected\n");
			break;
		case 0x10:
			printf("Error: A modem failure occurred - DCD and/or CTS were found to be unexpectedly low\n");
			break;
		case 0x11:
			printf("Error: The Channel moved from Operative to being Inoperative\n");
			break;
		case 0x12:
			printf("Error: The Channel moved from Inoperative to being Operative\n");
			break;
		case 0x13:
			printf("Error: The Access Node has reported a change in the status of a DLCI or a number of DLCIs\n");
			break;
		case 0x14:
			printf("Error: A Full Status Report included a DLCI or a number of DLCIis which were not included before\n"); 
			break;
		case 0x1F:
			printf("Error: The frame relay command is invalid\n");
			break;
		default:
			break;
   	}		
}; /* error */

void link_status( void ) 
{
	cb.command = READ_DLC_STATUS;
	cb.buffer_length = 0;
	cb.dlci = 0;
	DoCommand(&cb);
	if (cb.return_code == 0) {
		if (cb.data[0])
			printf("Channel status: OPERATIVE\n");
		else
	 		printf("Channel status: INOPERATIVE\n");
	} else {
		error(cb.return_code);	  	
	} 
}; /* link_status */
 
void modem_status( void ) 
{
   	cb.command = READ_MODEM_STATUS;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {
      		if (cb.data[0] & 0x08) 
	 		printf("DCD is HIGH\n");
      		else
	 		printf("DCD is LOW\n");
      	
      		if( cb.data[0] & 0x20) 
	 		printf("CTS is HIGH\n");
      		else 
	 		printf("CTS is LOW\n");
      	} else {
      		error(cb.return_code);
   	} 

}; //modem_status
 

void comm_err() 
{
	ResetCB(&cb);
   	cb.command = READ_COMMS_ERR_STATS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;	// for supervisor display
   
	DoCommand(&cb);
   
	if (cb.return_code == 0 && cb.buffer_length == 0x0A) {
      		printf("               Number of receiver overrun errors: %d\n",cb.data[0]);
      		printf("                   Number of receiver CRC errors: %d\n",cb.data[1]);
      		printf("                 Number of abort frames received: %d\n",cb.data[2]);
      		printf("Number of times receiver disabled (buffers full): %d\n",cb.data[3]);
      		printf("              Number of abort frames transmitted: %d\n",cb.data[5]);
      		printf("   Number of transmit underrun interrupts missed: %d\n",cb.data[6]);
      		printf("        Number of times DCD dropped unexpectedly: %d\n",cb.data[8]);
      		printf("        Number of times CTS dropped unexpectedly: %d\n",cb.data[9]);
   	} else {
      		error(cb.return_code);
   	} 
}; /* comm_err(); */

void flush_comm_err( void ) 
{
	cb.command = FLUSH_COMMS_ERR_STATS;
	cb.buffer_length = 0;
	cb.dlci = 0;
	DoCommand(&cb);
	if (cb.return_code != 0) 
		error(cb.return_code);
}; /* flush_comm_err */

void global_stats( void ) 
{
   	ResetCB(&cb);
   	cb.command = READ_DLC_STATISTICS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {
      		if( station_config == 0 ) {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[12]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[14]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&cb.data[16]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&cb.data[18]);
      		} else {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[20]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[22]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&cb.data[24]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&cb.data[26]);
      		} //if
      		printf("                                     CPE initializations: %d\n", *(unsigned short*)&cb.data[42]);
      		printf("                            current Send Sequence Number: %d\n", *(unsigned short*)&cb.data[44]);
      		printf("                         current Receive Sequence Number: %d\n", *(unsigned short*)&cb.data[46]);
      		printf("                                      current N392 count: %d\n", *(unsigned short*)&cb.data[52]);
      		printf("                                      current N393 count: %d\n", *(unsigned short*)&cb.data[54]);
   	} else {
		error(cb.return_code);	  	
  	 } 
}; /* global_stats */

void flush_global_stats( void ) 
{
   	cb.command = FLUSH_DLC_STATISTICS;
   	cb.buffer_length = 1;
   	cb.dlci = 0;
	cb.data[0] = 0x01;
   	DoCommand(&cb);
   
	if (cb.return_code != 0) { 
		switch(cb.return_code){
			case 0x06:
				printf("Error: Global Statistics not flushed\n");
				break;
			default:
				error(cb.return_code);
				break;	  
		}
	}	
}; /* flush_global_stats */

void error_stats( void ) 
{
	ResetCB(&cb);
	cb.command = READ_DLC_STATISTICS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {
      		printf("  I-frames not transmitted after a tx. int. due to exessive frame length: %d\n",*(unsigned short*)&cb.data[0]);
      		printf("   I-frames not transmitted after a tx. int. due to excessive throughput: %d\n",*(unsigned short*)&cb.data[2]);
      		printf("     Received frames discarded as they were either too short or too long: %d\n",*(unsigned short*)&cb.data[4]);
      		printf("                               discarded I-frames with unconfigured DLCI: %d\n",*(unsigned short*)&cb.data[6]);
      		printf("                                discarded I-frames due to a format error: %d\n",*(unsigned short*)&cb.data[8]);
      		printf("App. didn't respond to the triggered IRQ within the given timeout period: %d\n",*(unsigned short*)&cb.data[10]);
      		printf("            discarded In-channel Signalling frames due to a format error: %d\n",*(unsigned short*)&cb.data[28]);
      		printf("   In-channel frames received with an invalid Send Seq. Numbers received: %d\n",*(unsigned short*)&cb.data[32]);
      		printf("In-channel frames received with an invalid Receive Seq. Numbers received: %d\n",*(unsigned short*)&cb.data[34]);
      		if( station_config == 0 ) {
	 		printf("                    Number of unsolicited responses from the Access Node: %d\n",*(unsigned short*)&cb.data[30]);
	 		printf("                                              timeouts on the T391 timer: %d\n",*(unsigned short*)&cb.data[36]);
	 		printf("                                  consecutive timeouts on the T391 timer: %d\n",*(unsigned short*)&cb.data[48]);
      		} else {
	 		printf("                                              timeouts on the T392 timer: %d\n",*(unsigned short*)&cb.data[38]);
	 		printf("                                  consecutive timeouts on the T392 timer: %d\n",*(unsigned short*)&cb.data[50]);
      		} 
      		printf("times that N392 error threshold was reached during N393 monitored events: %d\n",*(unsigned short*)&cb.data[40]);
   	} 
}; /* error_stats */

void line_trace() 
{
	unsigned char num_frames, num_chars;
   	unsigned short curr_pos = 0, dlci;
   	PFRAME_DATA frame_info;
   	unsigned int i, j;
   	unsigned char outstr[200];
   	int recv_buff = sizeof(CBLOCK) + 100;
  	fd_set ready;
   	struct timeval to;
   
   	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );
   	cb.command = FPIPE_ENABLE_TRACING;
   	cb.buffer_length = 0;
   	DoCommand(&cb);

   	if( cb.return_code == 0 ) { 
      		printf("Starting trace...(Press ENTER to exit)\n");
   	} else if (cb.return_code == 0x1F) {
      		printf("Line Tracing is possible only with S508 board.\n");
      		return;
   	} else if( cb.return_code == 0xCD ) {
      		printf("Cannot Enable Line Tracing from Underneath.\n");
      		return;
   	} else if( cb.return_code == 0x01 ) {
      		printf("Starting trace...(although it's already enabled!)\n");
      		printf("Press ENTER to exit.\n");
   	} else {
      		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", cb.return_code );
      		return;
   	}
   
	for(;;) {
      	    FD_ZERO(&ready);
      	    FD_SET(0,&ready);
      	    to.tv_sec = 1;
      	    to.tv_usec = 0;
     
	    if(select(1,&ready, NULL, NULL, &to)) {
	 	break;
      	    } /* if */
      
	    cb.command = FPIPE_GET_TRACE_INFO;
      	    cb.buffer_length = 0;
      	    DoCommand(&cb);
     
	    if (cb.return_code == 0 && cb.buffer_length) { 
 
		/*  get number of frames */
 		num_frames = cb.data[0] >> 3;
 
		for ( i=0; i<num_frames; i++) {

    		   frame_info = (PFRAME_DATA)(cb.data + curr_pos);
		   
    		   /*  frame type */
    		   if (frame_info->status & 0x01) {
       		   	sprintf(outstr,"OUTGOING\t");
    		   } else {
       		   	sprintf(outstr, "INCOMING\t");
    		   }

    		   /* real length and time stamp */
    		   sprintf(outstr+strlen(outstr), "%d\t%d\t\0", frame_info->real_length, frame_info->time_stamp);
	    			
		   /*  first update curr_pos */
	    	   curr_pos += sizeof(FRAME_DATA);
	    		
	    	   if (frame_info->passed_length == 0) {
	       	   	sprintf( outstr+strlen(outstr), "the frame data is not available" );
	    	   } else {
	      		/*  update curr_pos again */
	       	  	curr_pos += (frame_info->real_length - 1);
	       		num_chars = (unsigned char)((frame_info->real_length <= 25)?frame_info->real_length:25);

       			if (raw_data) {
			   /* show raw data */
	 		   for( j=0; j<num_chars; j++ ) {
	     			sprintf(outstr+strlen(outstr), "%02X \0", (unsigned char)frame_info->data[j]);
	  		   }
	  		   outstr[strlen(outstr)-1] = '\0';
	      	  	} else { 
			   /* show intrptd. data */
	  		   dlci = (unsigned short)frame_info->data[0];
	  		   dlci = (dlci << 2) | (frame_info->data[1] >> 4);
	  		   if (dlci != 0) {
				/* An Inverse Arp packet */
				if (frame_info->data[2] == 0x03 &&
				    frame_info->data[3] == 0x00 &&
				    frame_info->data[4] == 0x80 &&
				    frame_info->data[8] == 0x08 &&
				    frame_info->data[9] == 0x06) {
					/* Request packet */
					if( frame_info->data[17] == 0x08)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tRequest on DLCI %d", dlci);
					else if( frame_info->data[17] == 0x09)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tReply on DLCI %d", dlci);
				} else 
	     				sprintf(outstr+strlen(outstr), "Data Frame on DLCI %d", dlci);
	  		   } else {
	     			if(frame_info->data[9])
		 		    sprintf(outstr+strlen(outstr), "Link Verification");
	     			else
		 	 	    sprintf(outstr+strlen(outstr), "Full Status");
	     			if (frame_info->data[5] == 0x75)
		 		    sprintf(outstr+strlen(outstr), "\tRequest");
	    	 		else
		 		    sprintf(outstr+strlen(outstr), "\tReply");
	     			
				sprintf(outstr+strlen(outstr), "\tSx %02X  Rx %02X  ", (unsigned char)frame_info->data[0x0C], (unsigned char)frame_info->data[0x0D] );
	     			if ( !frame_info->data[9] && (frame_info->data[5]==0x7D) ) {
				   /* full status reply */
				   for( j=0x0E; ( ((j+5)<frame_info->real_length) && (j<18)); j+=5 ) {
		   			dlci = (unsigned short)frame_info->data[j+2];
		   			dlci = (dlci << 4) | ((frame_info->data[j+3]>>3) & 0x0F);
		   			sprintf(outstr+strlen(outstr), "DLCI %d ", dlci);
		   			if (frame_info->data[j+4] & 0x08)
		       			   sprintf(outstr+strlen(outstr), "NEW %d", dlci);
		   			else
		       			   sprintf(outstr+strlen(outstr), "PRS %d", dlci);
		   			if (frame_info->data[j+4] & 0x02)
		       			   sprintf(outstr+strlen(outstr), "ACT\t%d", dlci);
		   			else
		       		  	   sprintf(outstr+strlen(outstr), "INACT\t%d", dlci);
				  }
	     			}
	  		   }
		 	}
	  	   }
	    	   /* we have to stay on even boundary here, so 
		    * update the curr_pos if needed. It's done 
	 	    * exactly the same way in driver. 
		    */
	    	    if (curr_pos & 0x0001) curr_pos++;
	    	    printf("%s\n",outstr);

	 	} /* end of for */
      	   } else if (cb.return_code != 0) 
		error(cb.return_code);
      	
	   curr_pos = 0;
      	   if (!(cb.data[0] & 0x02)) { 
		 sleep(TIMEOUT);
      	   } 
  	}
   	cb.command = FPIPE_DISABLE_TRACING;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
}; //line_trace

void list_dlcis( void )
{
   	int i,cnt;

   	cb.command = LIST_ACTIVE_DLCI;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
   
	if( cb.return_code == 0){	
     		cnt = cb.buffer_length;
     		if( cnt != 0 ){
       			printf("ACTIVE DLCIs:\n");
       			for(i=0; i< cnt; i+=2){
       	  			printf("DLCI %u\n", *(unsigned short*)&cb.data[i]);	
       			}
     		}else{
       			printf("There are NO ACTIVE DLCIs\n"); 
     		}
   	} else {
      		error(cb.return_code);
   	}	
} /* list_dlcis */

void read_dlci_stat( void )
{
     	cb.command = READ_DLC_STATISTICS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 1;
     	cb.data[0] = 0;
     
	DoCommand(&cb);
     	if( cb.return_code != 0 ){
		error(cb.return_code);
		return;
     	}
     	if( (cb.return_code == 0) && (cb.buffer_length == 0x20)){
		printf("                                Information frames transmitted: %ld\n", *(unsigned long*)&cb.data[0]);
		printf("                                 Information bytes transmitted: %ld\n", *(unsigned long*)&cb.data[4]);
		printf("                                   Information frames received: %ld\n", *(unsigned long*)&cb.data[8]);
		printf("                                    Information bytes received: %ld\n", *(unsigned long*)&cb.data[12]);
		printf("              Received I-frames discarded due to inactive DLCI: %ld\n", *(unsigned long*)&cb.data[20]);
		printf(" I-frames received with Discard Eligibility (DE) indicator set: %ld\n", *(unsigned long*)&cb.data[28]); 
     	}
     
	cb.command = READ_ADD_DLC_STATISTICS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 0;
     	cb.data[0] = 0;
     	DoCommand(&cb);
     
	if( cb.return_code == 0 ){
		printf("                       I-frames received with the FECN bit set: %d\n", *(unsigned short*)&cb.data[0]);
		printf("                       I-frames received with the BECN bit set: %d\n", *(unsigned short*)&cb.data[2]);
	
     	} else  
		printf("Error: Please enter a non-zero DLCI\n");

} /* read_dlci_stat */

void flush_dlci_stats( void )
{
     	cb.command = FLUSH_DLC_STATISTICS;
     	cb.dlci = dlci_number;
     	cb.buffer_length = 0;
     	DoCommand(&cb);
     	
	if( cb.return_code != 0 ) {
		switch( cb.return_code ){
			case 0x06:
				printf("DLCI Statistics are not flushed\n");
				break;
			default:
				error(cb.return_code);
		}
	}
} /* flush_dlci_stats */

void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
     	cb.command = FT1_MONITOR_STATUS_CTRL;
      	cb.buffer_length = 1;
      	cb.data[0] = status; 	
      	DoCommand(&cb);
      
	if( cb.return_code != 0 && status){
		fail = 1;
		if( cb.return_code == 0xCD )
                	printf("Error:  Cannot run this command from Underneath.\n");
		else 
			printf("Error:  This command is only possible with S508/FT1 board!");
      	}

} /* set_FT1_monitor_status */

void set_FT1_mode( void ){
 
	for(;;){
		cb.command = SET_FT1_MODE;
		cb.buffer_length = 0;
		DoCommand(&cb);
		if(cb.return_code == 0)
			break;
		else if( cb.return_code == 0xCD ){
                	printf("Error: Cannot run this command from Underneath.\n");
			exit(1);
		}

	}
} /* set_FT1_mode */

void read_FT1_status( void ){
     int i;
     long delay;
     struct timeval tv;

    
     i = gettimeofday(&tv,NULL);
     delay = tv.tv_usec;

     for(;;){
      i = gettimeofday(&tv,NULL);
      if((abs(tv.tv_usec - delay)) > 90000)
	break;  
     }

     cb.command = FPIPE_FT1_READ_STATUS;
     cb.buffer_length = 0;
     DoCommand(&cb); 
     if( cb.return_code == 0 ){
	par_port_A_byte = cb.data[0];
	par_port_B_byte = cb.data[1];
     }
} /* read_FT1_status */

void view_FT1_status( void ){
     
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     /* check for INS light */
     for(;;){
     	read_FT1_status();
     	if((par_port_B_byte & 0x30) == 0x30)
 		off_counter++;
     	if((par_port_B_byte & 0x10) == 0x00)
		red_counter++;
     	if((par_port_B_byte & 0x20) == 0x00)
		green_counter++;
	if(red_counter != 0 && off_counter != 0 && loop_counter == 30 ) {
		printf("Unit is not IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter == 0 && loop_counter == 30){
		printf("Unit is IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter != 0 && loop_counter == 30){
		printf("INS is flashing green\n");
		break;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("INS is off\n");
		break;
	}
	loop_counter++;
     }	
     /* check for ERR light */
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x0C) == 0x0C)
		off_counter++;
	if((par_port_B_byte & 0x08) == 0x00)
		red_counter++;
	if((par_port_B_byte & 0x04) == 0x00)
		green_counter++;
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("No Line Errors: Valid Line\n");
		break;
	}
	if(red_counter != 0 && off_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("Line Errors: Invalid Line\n");
		break;
	}

	loop_counter++;
     }

     /* check TXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x02) == 0x02)
		off_counter++;
	if((par_port_B_byte & 0x02) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Transmit data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Transmit data is present \n");
		break;
	}
	loop_counter++;
     }

     /* check RXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x01) == 0x01)
		off_counter++;
	if((par_port_B_byte & 0x01) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Receive data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Receive data is present\n");
		break;
	}
	loop_counter++;
     }

}/* view_FT1_status */


void FT1_operational_mode(void){
     
     printf("Operational Mode has been selected\n"); 
     printf("Putting S508/FT1 in operational mode....");
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
     	read_FT1_status();
     	/* ST light is OFF */
     	if((par_port_B_byte & 0xc0) == 0xc0 ){
		off_counter++;
     	}
     	/* ST light is GREEN */
     	if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
		red_counter = 0;
     	}
     	/* ST light is RED */
     	if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
		green_counter = 0;
   	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		break;
	}
	if((red_counter != 0 || green_counter != 0) && loop_counter == 20){
		set_FT1_mode();
		break;
	}

   	loop_counter++;
     } /* end of for */
       
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* DL light is OFF */
   	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;
	}
        /* DL light is RED */
   	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++; 
     } 
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* LL light is off */
   	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
   	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++;
     }
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
	/* RT light is OFF */
   	if((par_port_A_byte & 0x03) == 0x03){
		off_counter++;
	}
	/* RT light is RED */
   	if((par_port_A_byte & 0x01) == 0x00){
		red_counter++;
	}
	/* RT light is GREEN */
   	if((par_port_A_byte & 0x02) == 0x00){
   		green_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		printf("Done!\n");
		break;
	}
	if((red_counter != 0 || green_counter != 0) && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		printf("Done!\n");
		break;
	}
	if(off_counter == 0 && green_counter != 0 && loop_counter == 20){
		printf("Failed!\n");
		printf("Remote End is running Remote Test\n");
		printf("Exit Remote Test at remote end\n");
		break;
	}
	loop_counter++;
     }        	

} /* FT1_operational_mode */


void FT1_self_test(void){
     int started = 0; 
     int selftest = 0;
 
     set_FT1_mode();
     off_counter = 0;
     green_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Self Test has been selected\n");
     for(;;){
	  read_FT1_status();
	  /* ST light is OFF */
	  if((par_port_B_byte & 0xc0) == 0xc0){
		off_counter++;
	  }
	  /* ST light is GREEN */
	  if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
	  }
	  /* ST light is RED */
	  if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
	  }
          if(red_counter == 0 && loop_counter == 3){
		set_FT1_mode();    
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
     		printf("Selecting Self Test....\r");
		selftest++;
	  	if( selftest == 10){
		  printf("\t\t\tFailed!\n");
		  printf("Self Test will not operate while connected to a valid line\n");
		  FT1_operational_mode();
		  break;
	        }
	  }
	  if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
		printf("Performing Self Test....\r");
	        started = 1;
	  }
	  if(green_counter != 0 && off_counter != 0 && started){
	   	printf("\t\t\tDone!\n");
		break;
	  }
          loop_counter++;	  
     }/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void ){
     int dl_test=0;

     set_FT1_mode();
     off_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Bi-directional Digital Loop has been selected\n"); 
     for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;  
	}
	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Digital Loop Test....\r");
		dl_test++;
		if(dl_test==10){
	 		printf("\t\t\t\tFailed\n");
			printf("Remote End might be running Remote Test\n");
			break;
		}
	}
	if(red_counter != 0){
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = 0;
		printf("Performing Digital Loop Test....\r");
		for(;;){
			read_FT1_status();
			printf("Performing Digital Loop Test....\r");
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(red_counter != 0 && (par_port_B_byte & 0x08) == 0x00 ){
				printf("\t\t\t\tFailed!\n");
				printf("Bi-directional Digital Loop test has failed\n");
				printf("Either the unit is not connected or the line is Invalid\n");
				break;
			}
			if(green_counter != 0  && (par_port_B_byte & 0x0C) == 0x0C && loop_counter == 100 ){
				printf("\t\t\t\tDone!\n");
				printf("Bi-directional Digital Loop test has been successfully\n");
				printf("completed\n");
				break;
		
			}
			loop_counter++;
		} /* end of for */
		break;	
	} 
	loop_counter++;
     } /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void ){
    int ll_test = 0;

    set_FT1_mode();	
    off_counter = 0;
    red_counter = 0;
    loop_counter = 0;
    printf("Line Loop Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Line Loop Test....\r");
		ll_test++;
		if(ll_test == 10){
			printf("\t\t\t\tFailed!\n");
			printf("Line Loop Test will not operate while connected to a valid line\n");
			FT1_operational_mode();
		 	break;	
		}
	}
	if(red_counter != 0){	
		off_counter = 0;
		red_counter = 0;
		loop_counter = 0;
		for(;;){
			printf("Performing Line Loop Test....\r");
			read_FT1_status();	
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(green_counter != 0 && red_counter == 0){
				printf("\t\t\t\tDone!\n");
				printf("Line Loop Test has been successfully completed\n");
				break;
			}
			if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
				printf("\t\t\t\tFailed!\n");
				break;
			}
			loop_counter++;
		} /* end of for */
		break;
	} /* end of if */
	loop_counter++;
    } /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void ){
 
    off_counter = 0;
    red_counter = 0;
    green_counter = 0;	  	
    loop_counter = 0;
    printf("Remote Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x03) == 0x03)
		off_counter++;
	if((par_port_A_byte & 0x01) == 0x00)
		red_counter++;
	if((par_port_A_byte & 0x02) == 0x00)
		green_counter++;
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
		printf("Selecting Remote Test....\r");		
	}
	if(green_counter != 0 && loop_counter == 10){
	   loop_counter = 0;
	   off_counter = 0;
	   for(;;) {
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
	        if(off_counter == 0 && loop_counter == 20){
		   printf("Remote End is currently sending Remote Test data\n");
		   printf("Exit from the Remote Test mode at the remote end\n");
		   break;
		}
		if(off_counter != 0 && loop_counter == 20) {			
		   printf("This unit is currently in Remote Testing Mode\n");
		   break;
		}
	        loop_counter++;
	   }
	   break;
	}
	if(red_counter != 0){
		printf("Waiting for a valid response from remote end....\r");
		loop_counter = 0;
		for(;;){
			read_FT1_status();
			if((par_port_A_byte & 0x02) == 0x00){
				printf("\t\t\t\t\t\tDone!\n");
				printf("Valid response has been received from remote end\n");
				printf("Remote Test has been successfully completed\n");
				break;	
			}
			if((par_port_B_byte & 0x08) == 0x00){
				printf("\t\t\t\t\t\tFailed!\n");
				printf("The test can only be successful between two Sangoma\n");
				printf("S508/FT1 units configured the SAME WAY\n");

				break;
			}
		} /* end of for */
		break;
	}
    	loop_counter++;	
    } /* end of for */	
} /* FT1_rt_test */



void fr_driver_stat_ifsend( void )
{
      cb.command = FPIPE_DRIVER_STAT_IFSEND;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      
      printf("                                    Total Number of If_Send entries:  %ld\n", *(unsigned long*)&cb.data[0]);
      printf("                          Number of If_Send entries with SKB = NULL:  %ld\n", *(unsigned long*)&cb.data[4]);
      printf("Number of If_Send entries with broadcast addressed packet discarded:  %ld\n", *(unsigned long*)&cb.data[8]);
      printf("Number of If_Send entries with multicast addressed packet discarded:  %ld\n", *(unsigned long*)&cb.data[12]);
      printf("                Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n", *(unsigned long*)&cb.data[16]);
      printf("   Number of If_Send entries with Critical set and packet discarded:  %ld\n", *(unsigned long*)&cb.data[20]);
      printf("                     Number of If_Send entries with Device Busy set:  %ld\n", *(unsigned long*)&cb.data[24]);
      printf("                 Number of If_Send entries with Device Busy Timeout:  %ld\n", *(unsigned long*)&cb.data[28]);
      printf("                Number of If_Send entries with DRIVER STATS Request:  %ld\n", *(unsigned long*)&cb.data[32]);
      printf("               Number of If_Send entries with FPIPE MONITOR Request:  %ld\n", *(unsigned long*)&cb.data[36]);
      printf("                    Number of If_Send entries with WAN Disconnected:  %ld\n", *(unsigned long*)&cb.data[40]);
      printf("                   Number of If_Send entries with DLCI Disconnected:  %ld\n", *(unsigned long*)&cb.data[44]);
      printf("            Number of If_Send entries with check for Buffers failed:  %ld\n", *(unsigned long*)&cb.data[48]);
      printf("                         Number of If_Send entries with Send failed:  %ld\n", *(unsigned long*)&cb.data[52]);
      printf("                         Number of If_Send entries with Send passed:  %ld\n", *(unsigned long*)&cb.data[56]);
      printf("                   Number of Consecutive send failures for a packet:  %ld\n", *(unsigned long*)&cb.data[60]);
      
} /* fr_driver_stat_ifsend */

void fr_driver_stat_intr( void )
{

      cb.command = FPIPE_DRIVER_STAT_INTR;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
       
      printf("                                   Number of ISR entries:    %ld\n" , *(unsigned long*)&cb.data[0]);
      printf("                 Number of ISR entries with Critical Set:    %ld\n" , *(unsigned long*)&cb.data[4]);
      printf("                             Number of Receive Interrupt:    %ld\n" , *(unsigned long*)&cb.data[8]);
      printf("                            Number of Transmit Interrupt:    %ld\n" , *(unsigned long*)&cb.data[12]);
      printf("             Number of ISR entries for Interrupt Testing:    %ld\n" , *(unsigned long*)&cb.data[16]);
      printf("                            Number of Spurious Interrupt:    %ld\n" , *(unsigned long*)&cb.data[20]);
 
      printf("      Number of Times Transmit Interrupts Enabled in ISR:    %ld\n" , *(unsigned long*)&cb.data[24]);
      printf("   Number of Transmit Interrupts with Device Not Started:    %ld\n" , *(unsigned long*)&cb.data[28]);
 
      printf("        Number of Receive Interrupts with Corrupt Buffer:    %ld\n" , *(unsigned long*)&cb.data[32]);
         
      printf("         Number of Receive Interrupts with orphaned DLCI:    %ld\n" , *(unsigned long*)&cb.data[36]);
         
      printf("             Number of Receive Interrupts with No socket:    %ld\n" , *(unsigned long*)&cb.data[40]);

      printf("    Number of Receive Interrupts with Device Not Started:    %ld\n" , *(unsigned long*)&cb.data[44]);
      
      printf("    Number of Receive Interrupts for DRIVER STAT Request:    %ld\n" , *(unsigned long*)&cb.data[48]);
      printf("  Number of Receive Interrupts for FPIPE MONITOR Request:    %ld\n" , *(unsigned long*)&cb.data[52]);
      printf("      Number of Receive Interrupts with Buffer discarded:    %ld\n" , *(unsigned long*)&cb.data[56]);
      printf("Number of Receive Interrupts with Buffer Passed to Stack:    %ld\n" , *(unsigned long*)&cb.data[60]);
     

} /* fr_driver_stat_intr */

void fr_driver_stat_gen( void )
{
      cb.command = FPIPE_DRIVER_STAT_GEN;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      

      printf("            Number of FPIPE Monitor call with kmalloc error:  %ld\n", *(unsigned long*)&cb.data[0]);
      
      printf("       Number of FPIPE Monitor call with Adapter Type error:  %ld\n", *(unsigned long*)&cb.data[4]);

      printf("          Number of FPIPE Monitor call with Direction Error:  %ld\n", *(unsigned long*)&cb.data[8]);

      printf("  Number of FPIPE Monitor call with Adapter Command Timeout:  %ld\n", *(unsigned long*)&cb.data[12]);
      printf("       Number of FPIPE Monitor call with Adapter Command OK:  %ld\n", *(unsigned long*)&cb.data[16]);
      printf("      Number of FPIPE Monitor call with Adapter Send passed:  %ld\n", *(unsigned long*)&cb.data[20]);
      printf("      Number of FPIPE Monitor call with Adapter Send failed:  %ld\n", *(unsigned long*)&cb.data[24]);
      printf("                Number of FPIPE Monitor call with no socket:  %ld\n", *(unsigned long*)&cb.data[28]);
      printf("  Number of FPIPE Monitor call with pkt not passed to stack:  %ld\n", *(unsigned long*)&cb.data[32]);
      printf("      Number of FPIPE Monitor call with pkt passed to stack:  %ld\n", *(unsigned long*)&cb.data[36]);
      printf("              Number of DRIVER STAT call with Kmalloc error:  %ld\n", *(unsigned long*)&cb.data[40]);
      printf("    Number of DRIVER STAT call with Adapter Command Timeout:  %ld\n", *(unsigned long*)&cb.data[44]);
      printf("         Number of DRIVER STAT call with Adapter Command OK:  %ld\n", *(unsigned long*)&cb.data[48]);
      printf("        Number of DRIVER STAT call with Adapter Send passed:  %ld\n", *(unsigned long*)&cb.data[52]);
      printf("        Number of DRIVER STAT call with Adapter Send failed:  %ld\n", *(unsigned long*)&cb.data[56]);
      printf("                  Number of DRIVER STAT call with no socket:  %ld\n", *(unsigned long*)&cb.data[60]);
      printf("Number of DRIVER STAT call with pkt not passed to the stack:  %ld\n", *(unsigned long*)&cb.data[64]);
      printf("    Number of DRIVER STAT call with pkt passed to the stack:  %ld\n", *(unsigned long*)&cb.data[68]);
      
      printf("                                     Number of Poll Entries:  %ld\n", *(unsigned long*)&cb.data[72]);
      printf("                   Number of Poll Entries with Critical set:  %ld\n", *(unsigned long*)&cb.data[76]);
      printf("                           Number of Poll Entries Processed:  %ld\n", *(unsigned long*)&cb.data[80]);
      printf("             Number of times host irq left disabled in Poll:  %ld\n", *(unsigned long*)&cb.data[84]);

} /* fr_driver_stat_gen */

void flush_driver_stats( void )
{

      cb.command = FPIPE_FLUSH_DRIVER_STATS;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);

      printf("All Driver Statistics are Flushed.\n");

} /* flush_driver_stats */

void fr_router_up_time( void )
{
     unsigned long time;
     cb.command = FPIPE_ROUTER_UP_TIME;
     cb.buffer_length = 0;
     cb.data[0] = 0;
     DoCommand(&cb);
     
     time = cb.data[0] + (cb.data[1]*256) + (cb.data[2]*65536) + 
					(cb.data[3]*16777216);
     if (time < 3600) {
	if (time<60) 
     		printf("    Router UP Time:  %ld seconds\n", time);
	else
     		printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     }else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
} /* fr_router_up_time */

void usage( void ) {

	printf("fpipemon [f](lush)/ [t](1 features)  option ip-address [port]\n\n");
	printf("One of the following:\n");
	printf("\ts  Link Status\n");
	printf("\tm  Modem Status\n");
	printf("\ta  List Active DLCIs\n");
	printf("\tl  Line Trace: display Interpreted data\n");
	printf("\tlr Line Trace: display RAW data\n");
	printf("\tds Display If_Send Driver Statistics\n");
	printf("\tdi Display Interrupt Driver Statistics\n");
	printf("\tdg Display General Driver Statistics\n");
	printf("\tu  Display Router UP Time\n");

	printf("\nOptions that can be flushed:\n");
	printf("\tc Comm Error Statistics\n");
	printf("\tg Global Statistics\n");
	printf("\te Global Error Statistics\n");
	printf("\ti Read DLCI Statistic\t:specify your DLCI beside it (...i ##...)\n");
	printf("\td Flush all Driver statistics\n");
	printf("\nOptions for S508/FT1 only\n");
	printf("\tv View Status\n");
	printf("\ts Self Test\n");
	printf("\tl Line Loop Test\n");
	printf("\td Digital Loop Test\n");
	printf("\tr Remote Test\n");	
	printf("\to Operational Mode\n");

}; //usage

void main(int argv, char* argc[])
{
	int proceed;

	printf("\n");
	if( argv > 2 ) {
		init( argv, argc);
		proceed = MakeConnection();
		if(proceed == TRUE){
			switch( argc[1][0] ) {
			case 's':
				link_status();
				break;
			case 'c':
				comm_err();
	 			break;
      			case 'g':
	 			global_stats();
	 			break;
      			case 'm':
				modem_status();
	 			break;
      			case 'e':
	 			error_stats();
	 			break;
      			case 'l':
	 			if( argc[1][1] == 'r' )
	   			raw_data = TRUE;
	 			line_trace();
	 			break;
      			case 'a':
	 			list_dlcis();
	 			break;
      			case 'i':
	 			read_dlci_stat();
	 			break;
      			case 'd':
	 			if( argc[1][1] == 's' )
	 				fr_driver_stat_ifsend();
	 			if( argc[1][1] == 'i' )
					fr_driver_stat_intr();
	 			if( argc[1][1] == 'g' )
				fr_driver_stat_gen();
	 			break;
      			case 'u':
	 			fr_router_up_time();
	 			break;
      			case 'f':
	 			if( argv > 2 ) {
	    				switch( argc[2][0] ) {
	    				case 'c':
	       					flush_comm_err();
	       					comm_err();
	       					break;
	    				case 'g':
	       					flush_global_stats();
	       					global_stats();
	       					break;
	    				case 'e':
	      	 				flush_global_stats();
	       					error_stats();
	      	 				break;
	   		 		case 'i':
	       					flush_dlci_stats();
	       					read_dlci_stat();
	       					break;
	    				case 'd':
	       					flush_driver_stats();
	       					break;
	    				default:
	       					usage();
	    				} //switch
	 			} else {
	    				usage();
	 			}//if
	 			break;
      			case 't':
				if( argv > 2 ){
	   				switch( argc[2][0] ){
	  	 			case 'v': 
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							view_FT1_status();
	     						}
	     					set_FT1_monitor_status(0x00);
	     					break;
	   				case 's':
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_self_test();
	     					}
             					set_FT1_monitor_status(0x00);
	     					break;
  	   				case 'l':
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_ll_test();
	     					}
						set_FT1_monitor_status(0x00);
	    		 			break;
	   				case 'd':
   	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
	     						FT1_dl_test();
	    					}
	     					set_FT1_monitor_status(0x00);
	     					break;
           				case 'r':
	     					set_FT1_monitor_status(0x01);
             					if(!fail){
	     						FT1_rt_test();
	     					}
						set_FT1_monitor_status(0x00);	
	     					break;
	   				case 'o':
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_operational_mode();
	    		 			}
						set_FT1_monitor_status(0x00);
	     					break;
	   				default:
	      					usage();
	   				} /* end of switch */
   				} else{
 	   				usage();
        			} /* end of if */
				break;	
      			default:
	 			usage();
      			} //switch
     		}
     		close( sock );
   	} else {
      		usage();
   	} //if
   	printf("\n");
}; //main

/*
 * EOF fpipemon.c
 */
