/*
 * addgroup - add users to /etc/passwd and /etc/shadow
 *
 *
 * Copyright (C) 1999 by Lineo, inc.
 * Written by John Beppu <beppu@lineo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "internal.h"
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

/* structs __________________________ */

/* data _____________________________ */

static const char *addgroup_usage =
"addgroup [LOGIN NAME] [OPTIONS]...\n\n"
"\t-g\tspecify gid\n"
;

/* defaults : should this be in an external file? */
static char *default_passwd      = "x";


/* FIXME : redundant (adduser uses this little func, too) */
/* move to one char past the first colon you hit */
static void
post_colon_maneuvre(char **str)
{
    while( (**str) != ':') { (*str)++; }
    (*str)++;
}

/* from str, extract gid from a group file*/
static void
get_gid(char *str, gid_t *g)
{
    char *s = str;
    post_colon_maneuvre(&s);
    post_colon_maneuvre(&s);
    g[0] = strtol(s, NULL, 10);
}

/* get the next available gid from a group file
 * if you want, specify a preference for a specific gid
 * in the second arg, desired. 
 */
static gid_t
get_next_gid(const char *filename, gid_t desired)
{
    FILE    *etc_group;
    char    buffer[1024];
    gid_t   tmp_g;
    gid_t   g = 0;

    const int max = 65000;

    etc_group = fopen(filename, "r");
    if (!etc_group) { 
	fprintf(stderr, "addgroup: %s: %s\n", filename, strerror(errno));
	return 0; 
    }
    while (fgets(buffer, 1024, etc_group)) {
	char *s = buffer;
	get_gid(s, &tmp_g);
	if ((desired) && (tmp_g == desired)) {
	    fprintf(stderr, "addgroup: %d: gid has already been allocated\n", desired);
	    return 0;
	}
	if ((tmp_g > g) && (tmp_g < max)) {
	    g = tmp_g;
	}
    }
    fclose(etc_group); g++;
    if (desired) {
	return desired;
    }
    return g;
}

/* append a new user to the passwd file */
static int
addgroup(const char *filename, const char *group, gid_t gid)
{
    FILE    *etc_group;
    FILE    *etc_gshadow;
    gid_t   g;
    char    *gshadow = GSHADOW_FILE;

    /* group:passwd:gid:userlist */
    const char *entryfmt = "%s:%s:%d:%s\n";

    g = get_next_gid(filename, gid);
    if (g == 0) { return 0; }

    /* add entry to group */
    etc_group = fopen(filename, "a");
    if (!etc_group) { 
	fprintf(stderr, "addgroup: %s: %s\n", filename, strerror(errno));
	return 0; 
    }
    fprintf (
	etc_group, entryfmt, 
	group, default_passwd, g, ""
    );
    fclose(etc_group);

    /* add entry to gshadow if necessary */
    if (access(gshadow, W_OK) == 0) {
	etc_gshadow = fopen(gshadow, "a");
	if (!etc_gshadow) { 
	    fprintf(stderr, "addgroup: %s: %s\n", gshadow, strerror(errno));
	    return 0; 
	}
	fprintf(etc_gshadow, "%s:!::\n", group);
	fclose(etc_gshadow);
    }

    return 1;
}

/*
 * addgroup will take a login_name as its first parameter.
 *
 * home
 * shell
 * gecos 
 *
 * can be customized via command-line parameters.
 * ________________________________________________________________________ */
int
addgroup_main(int argc, char **argv)
{
    int	    i;
    char    opt;
    char    *group;
    gid_t   gid = 0;

#if 1
    /* get remaining args */
    for (i = 1; i < argc; i++) {
	if (argv[i][0] == '-') {
	    opt = argv[i][1];
	    switch (opt) {
		case 'h':
		    usage(addgroup_usage);
		    break;
		case 'g':
		    gid = strtol(argv[++i], NULL, 10);
		    break;
		default:
		    fprintf(stderr, "addgroup: invalid option -- %c\n", opt);
		    exit(1);
		    break;
	    }
	} else {
	    break;
	}
    }
#endif

    if (getuid != 0) {
	fprintf(stderr, "addgroup: Only root may add a group to the system.\n");
	exit(1);
    }

    if (i >= argc) {
	fprintf(stderr, "addgroup: no group specified\n");
	exit(1);
    } else {
	group = argv[i];
    }

    /* werk */
    addgroup(GROUP_FILE, group, gid);

    exit(0);
}

/* $Id: addgroup.c,v 1.5 1999/12/29 21:59:57 beppu Exp $ */
