/* $Id: ipfwd.c,v 1.8 1997/10/16 17:53:12 cananian Exp $
 *
 * ipfwd        - a utility for redirecting ip protocol connections.
 *
 * Author:      C. Scott Ananian
 *              <cananian@alumni.princeton.edu>
 *
 * Originally written to support use of Microsoft PPTP across a Linux
 * firewall.  Is general enough to support proxy of any type of non-TCP
 * IP protocol connection.  Will probably work with TCP, too, but 
 * the 'redir' package is much better for that.
 *
 * You need root priviledges to use this program, because it accesses
 * packets at a very low level.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place --- Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright (C) 1997 C. Scott Ananian <cananian@alumni.princeton.edu>
 */
static const char *rcsid="@(#) $Id: ipfwd.c,v 1.8 1997/10/16 17:53:12 cananian Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <setjmp.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef HAVE_SYSLOG_H
# include <syslog.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_STRERROR
# ifdef HAVE_STRING_H
#  include <string.h>
# endif
#else /* !HAVE_STRERROR */
# define strerror(x) "[error msg unknown]"
#endif

/* Maximum possible IP datagram size, as defined in RFC791 */
#define IP_MAXBUF 65536

/*-------------------------------------------------------------*/
/*                    Prototypes                               */

/*   print usage string */
void ipfwd_usage(char *name);
/*   parse command line arguments */
void parse_args(int argc, char **argv, 
		char **remote_host, int *protocol,
		int *masq_flag, int *debug_flag, int *syslog_flag);
/*   handle signals: */
jmp_buf clean_env;         /* stack state to restore upon signal */
RETSIGTYPE clean_up(int);  /* signal handler                     */
/*   compute ip header checksum as per RFC791 */
unsigned short in_cksum(unsigned short *addr,int len);

/*-------------------------------------------------------------*/
/*          Function Definitions                               */

/* Usage information */
void ipfwd_usage(char *name) {
  fprintf(stderr,
	  "%s -- a utility to redirect generic IP protocol connections\n"
	  "Copyright (C) 1997  "
	  "C. Scott Ananian <cananian@alumni.princeton.edu>\n"
	  PACKAGE " is free software, and comes with ABSOLUTELY NO WARRANTY.\n"
	  "\n"
	  "\t%s [options] remote-host protocol\n"
	  "\n"
	  "where options are:\n"
	  "\t--masq   \t masquerade ip datagrams\n"
	  "\t--debug  \t output debugging info\n"
#ifdef HAVE_SYSLOG_H
	  "\t--syslog \t log messages to syslog\n"
#endif
	  "\n"
	  PACKAGE " version is " VERSION "\n", name, name);
  exit(2);
}

/* Parse command line arguments */
void parse_args(int argc, char **argv, 
		char **remote_host, int *protocol,
		int *masq_flag, int *debug_flag, int *syslog_flag) {
  static struct option long_options[] = {
    {"masq",	no_argument,	0, 'm'},
    {"debug",	no_argument,	0, 'd'},
#ifdef HAVE_SYSLOG_H
    {"syslog",	no_argument,	0, 's'},
#endif
    {"help",	no_argument,	0, 'h'},
    {"version",	no_argument,	0, 'v'},
    {"usage",   no_argument,	0, 'u'},
    {0,0,0,0} /* end of option list */
  };
  int option_index = 0;
  int c;

  (*debug_flag) = (*syslog_flag) = (*masq_flag) = 0;

  while ((c = getopt_long(argc, argv, "dvm", 
			  long_options, &option_index))!=-1)
    switch (c) {
    case 'm':
      (*masq_flag)=1;
      break;
    case 'd':
      (*debug_flag)=1;
      break;
#ifdef HAVE_SYSLOG_H
    case 's':
      (*syslog_flag)=1;
      break;
#endif
    case 'h':
    case 'v':
    case 'u':
    default:
      ipfwd_usage(argv[0]);
      break;
    }
  if (argc-optind != 2) ipfwd_usage(argv[0]);
  *remote_host = argv[optind++];
  *protocol = atoi(argv[optind++]);
}

/* Define error-checking helpers */
#ifdef HAVE_SYSLOG_H
#define perhaps_syslog(level, msg) if (syslog_flag) syslog(level, msg)
#else
#define perhaps_syslog(level, msg) ((void) 0)
#endif

#define check(cond, msg)                                     \
	if(cond) {                                           \
	  perhaps_syslog(LOG_ERR, msg);                      \
	  fprintf(stderr, "%s: %s\n", msg, strerror(errno)); \
	  exit(1);                                           \
        }

int main(int argc, char **argv) {
  char buffer[IP_MAXBUF];
  int src_fd, dst_fd, status, ip_len;
  struct sockaddr_in src_addr, dst_addr;
  int addrlen = sizeof(src_addr);
  
  int PROTOCOL;
  char *REMOTE_HOST;
  int masq_flag, debug_flag, syslog_flag;

  parse_args(argc, argv, &REMOTE_HOST, &PROTOCOL, 
	     &masq_flag, &debug_flag, &syslog_flag);

#ifdef HAVE_SYSLOG_H
  /* open log */
  if (syslog_flag) {
    openlog(argv[0], LOG_PID, LOG_DAEMON);
    syslog(LOG_NOTICE, "ipfwd version %s startup: "
	   "forwarding IP protocol %d to %s. MASQ is %s, DEBUG is %s.",
	   VERSION, PROTOCOL, REMOTE_HOST, 
	   (masq_flag)?"ON":"OFF", (debug_flag)?"ON":"OFF");
  }
#endif
  
  /* open listening port */
  src_fd = socket(AF_INET, SOCK_RAW, PROTOCOL);
  check(src_fd<0, "Couldn't open listening socket");
  src_addr.sin_family       = AF_INET;
  src_addr.sin_addr.s_addr  = INADDR_ANY;
  src_addr.sin_port         = 0;
  
  /* open sending port */
  if (masq_flag)
    dst_fd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW); /* RAW RAW RAW! */
  else
    dst_fd = socket(AF_INET, SOCK_RAW, PROTOCOL);    /* kosher */
  check(dst_fd<0, "Couldn't open sending socket");

  /* look up remote host address */
  status = inet_aton(REMOTE_HOST, &(dst_addr.sin_addr));
  check(!status, "Invalid sending address");
  dst_addr.sin_family = AF_INET;
  dst_addr.sin_port   = 0;

  /* set up so that we exit cleanly */
  if (setjmp(clean_env)==0) {
    signal(SIGINT, clean_up);
    signal(SIGTERM, clean_up);

    while (1) {
      status = recvfrom(src_fd, &buffer, IP_MAXBUF, 0, 
			(struct sockaddr *) &src_addr, &addrlen);
      check(status<0, "Error receiving from source");
      if (debug_flag)
	printf("Got %d bytes from %s/%d.\n", status,
	       inet_ntoa(src_addr.sin_addr), (int) ntohs(src_addr.sin_port));
#ifdef HAVE_SYSLOG_H
      if (syslog_flag)
	syslog(LOG_NOTICE, "forwarding ip proto %d from %s to %s",
	       PROTOCOL, inet_ntoa(src_addr.sin_addr), REMOTE_HOST);
#endif

      /* verify that this is IPv4 we're dealing with here (else drop) */
      if ((status<1)||((buffer[0]&0xF0)!=0x40)) continue;  /* IPv4 */
      /* get IP header length */
      ip_len = (buffer[0]&0xF)*4;
      /* we're not obviously broken, are we? */
      if (status<=ip_len) continue; /* broken IP */
      
      if (masq_flag) {
	/* now we just do a little forgery on the IP header... */
	/* rewrite destination: */
	*((struct in_addr *)(buffer+16)) = dst_addr.sin_addr;
	/* zero checksum... */
	*((unsigned short *)(buffer+10)) = 0;
	/* ...and recompute */
	*((unsigned short *)(buffer+10)) = 
	                          in_cksum((unsigned short *)buffer, ip_len);
	/* Ain't IP spoofing fun? */
      } else { /* !masq_flag */
	/* Chop off ip header and let sendto reconstruct */
	status -=ip_len;
      }
      /* now send that puppy on its way! */
      status = sendto(dst_fd, (void*) buffer+(masq_flag?0:ip_len), status, 0, 
		      (struct sockaddr *) &dst_addr, addrlen);
      check(status<0, "Error transmitting to destination");
      if (debug_flag)
	printf("Sent %d bytes from %s/%d.\n", status,
	       inet_ntoa(dst_addr.sin_addr), (int) ntohs(dst_addr.sin_port));
    }
  }
  /* Come here when application is killed. */
  if (debug_flag)
    printf("Closing.\n");
#ifdef HAVE_SYSLOG_H
  if (syslog_flag) {
    syslog(LOG_NOTICE, "Closing ipfwd on protocol %d", PROTOCOL);
    closelog();
  }
#endif
  /* Close those sockets */
  close(src_fd);
  close(dst_fd);
}

RETSIGTYPE clean_up(int i) {
  longjmp(clean_env, 1);
}

/* IP header checksum routine, reputedly originally stolen from ping */
unsigned short in_cksum(unsigned short *addr,int len)
{
  register int nleft = len;
  register unsigned short *w = addr;
  register int sum = 0;
  unsigned short answer = 0;
  
  while (nleft > 1)
    {
      sum += *w++;
      nleft -= 2;
    }
  if (nleft == 1)
    {
      *(u_char *)(&answer) = *(u_char *)w ;
      sum += answer;
    }
  sum = (sum >> 16) + (sum & 0xffff);
  sum += (sum >> 16);
  answer = ~sum;
  return(answer);
}


