/* information about connections between hosts and clients
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: connections.h,v 1.20 1999/04/11 00:44:16 dhr Exp $
 */

/* There are two kinds of connections:
 * - ISAKMP connections, between hosts (for IKE communication)
 * - IPsec connections, between clients (for secure IP communication)
 */

/* An ISAKMP connection looks like:
 *   host<--->host
 *
 * An IPsec connection looks like:
 *   client-subnet<-->host<->nexthop<--->nexthop<->host<-->client-subnet
 *
 * For the connection to be relevant to this instance of Pluto,
 * exactly one of the hosts must be a public interface of our machine
 * known to this instance.
 *
 * The client subnet might simply be the host -- this is a
 * representation of "host mode".
 *
 * Each nexthop defaults to the neighbouring host's IP address.
 * The nexthop is a property of the pair of hosts, not each
 * individually.  It is only needed for IPsec because of the
 * way IPsec is mixed into the kernel routing logic.  Furthermore,
 * only this end's nexthop is actually used.  Eventually, nexthop
 * will be unnecessary.
 *
 * Other information represented:
 * - each connection has a name: a chunk of uninterpreted text
 *   that is unique for each connection.
 * - security requirements (currently just the "policy" flags from
 *   the whack command to initiate the connection, but eventually
 *   much more.  Different for ISAKMP and IPsec connections.
 * - rekeying parameters:
 *   + time an SA may live
 *   + time before SA death that a rekeying should be attempted
 *     (only by the initiator)
 *   + number of times to attempt rekeying
 * - With the current KLIPS, we must route packets for a client
 *   subnet through the ipsec interface (ipsec0).  Only one
 *   gateway can get traffic for a specific (client) subnet.
 *   Furthermore, if the routing isn't in place, packets will
 *   be sent in the clear.
 *   "routed" indicates whether the routing has been done for
 *   this connection.
 * - With the current KLIPS, only one outbound IPsec SA bundle can be
 *   used for a particular client.  This is due to a limitation
 *   of using only routing for selection.  So only one IPsec state (SA)
 *   may "own" the eroute.  "eroute_owner" is the serial number of
 *   this state, or SOS_NOBODY if there is none.
 *
 * Operations on Connections:
 *
 * - add a new connection (with all details) [whack command]
 * - delete a connection (by name) [whack command]
 * - initiate a connection (by name) [whack command]
 * - find a connection (by IP addresses of hosts)
 *   [response to peer request; finding ISAKMP connection for IPsec connection]
 */

#define CONNECTION_NAME_LIMIT 32	/* limit on length of a connection name */

struct end {
    struct in_addr	/* network order */
	host, host_nexthop,
	client_net, client_mask;

    bool has_client;
    bool firewall;
    u_int16_t host_port;	/* host order */
};

#define inside_client(a, end) \
    (0 == (((a).s_addr ^ (end).client_net.s_addr) & (end).client_mask.s_addr))

#define same_client(a, b) \
    ((a).client_net.s_addr == (b).client_net.s_addr \
    && (a).client_mask.s_addr == (b).client_mask.s_addr)

struct connection {
    char name[CONNECTION_NAME_LIMIT];
    lset_t policy;
    time_t sa_ike_life_seconds;
    time_t sa_ipsec_life_seconds;
    time_t sa_rekey_window;
    unsigned long sa_keying_tries;
    struct end
	this,
	that;

    /* internal fields: */

    const struct iface *interface;	/* filled in iff oriented */
    bool routed;	/* is routing in place for peer's client? */
    so_serial_t	/* state object serial number */
	newest_isakmp_sa,
	newest_ipsec_sa,
	eroute_owner;
#ifdef ROAD_WARRIOR_FUDGE
    bool rw_instance;	/* to be deleted when no longer used */
#endif /* ROAD_WARRIOR_FUDGE */

    struct connection *next;
};

#define oriented(c) ((c).interface != NULL)
extern bool orient(struct connection *c, bool must);

struct whack_message;	/* forward declaration of tag whack_msg */
extern void add_connection(const struct whack_message *wm, int whack_fd);
extern void initiate_connection(const char *name, int whackfd);
extern void terminate_connection(const char *nm, int whackfd);
extern void delete_connection(struct connection *c);
extern void delete_every_connection(void);
extern void release_interface(struct iface *i);
extern struct connection *route_owner(struct connection *c, bool eroute);
#ifdef ROAD_WARRIOR_FUDGE
extern struct connection
    *rw_connection(struct connection *c, struct in_addr him);
#endif /* ROAD_WARRIOR_FUDGE */

extern struct connection
    *con_by_name(const char *nm, int wfd),
    *find_host_connection(struct in_addr me, struct in_addr him),
    *find_client_connection(struct connection *c,
	struct in_addr our_net, struct in_addr our_mask,
	struct in_addr peer_net, struct in_addr peer_mask);

extern void show_connections_status(int wfd);
