/*
 * IPSEC interface configuration
 * Copyright (C) 1996  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char tncfg_c_version[] = "RCSID $Id: tncfg.c,v 1.13 1999/04/11 00:12:09 henry Exp $";


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <linux/types.h>
#include <linux/if.h>
#include <sys/types.h>
#include <errno.h>
/* #include <sys/socket.h> */
#include <getopt.h>
#include "ipsec_tunnel.h"

#if 0
struct ipsectunnelconf
{
	__u32	cf_cmd;
	union
	{
		char 	cfu_name[12];
	} cf_u;
#define cf_name cf_u.cfu_name
};

#define IPSEC_SET_DEV	(SIOCDEVPRIVATE)
#define IPSEC_DEL_DEV	(SIOCDEVPRIVATE + 1)
#endif

void usage(char *name)
{	
	fprintf(stdout,"%s --attach --virtual <virtual-device> --physical <real-device>\n",
		name);
	fprintf(stdout,"%s --detach --virtual <virtual-device>\n",
		name);
	fprintf(stdout,"%s --clear\n",
		name);
	fprintf(stdout,"%s --help\n",
		name);
	fprintf(stdout,"%s --version\n",
		name);
	exit(1);
}

static struct option const longopts[] =
{
	{"virtual", 1, 0, 'V'},
	{"physical", 1, 0, 'P'},
	{"attach", 0, 0, 'a'},
	{"detach", 0, 0, 'd'},
	{"clear", 0, 0, 'c'},
	{"help", 0, 0, 'h'},
	{"version", 0, 0, 'v'},
	{"label", 1, 0, 'l'},
	{"optionsfrom", 1, 0, '+'},
	{0, 0, 0, 0}
};

/* void */
int
main(int argc, char *argv[])
{
	struct ifreq ifr;
	struct ipsectunnelconf *shc=(struct ipsectunnelconf *)&ifr.ifr_data;
	int s;
	int c, previous = -1;
	char *program_name;
     
	memset(&ifr, 0, sizeof(ifr));
	program_name = argv[0];

	while((c = getopt_long_only(argc, argv, "adchvV:P:l:+:", longopts, 0)) != EOF) {
		switch(c) {
		case 'a':
			if(shc->cf_cmd) {
				fprintf(stderr, "%s: exactly one of '--attach', '--detach' or '--clear' options must be specified.\n",	program_name);
				exit(1);
			}
			shc->cf_cmd = IPSEC_SET_DEV;
			break;
		case 'd':
			if(shc->cf_cmd) {
				fprintf(stderr, "%s: exactly one of '--attach', '--detach' or '--clear' options must be specified.\n",	program_name);
				exit(1);
			}
			shc->cf_cmd = IPSEC_DEL_DEV;
			break;
		case 'c':
			if(shc->cf_cmd) {
				fprintf(stderr, "%s: exactly one of '--attach', '--detach' or '--clear' options must be specified.\n",	program_name);
				exit(1);
			}
			shc->cf_cmd = IPSEC_CLR_DEV;
			break;
		case 'h':
			usage(program_name);
			break;
		case 'v':
			if(optarg) {
				fprintf(stderr, "%s: warning; '-v' and '--version' options don't expect arguments, arg '%s' found, perhaps unintended.\n",
					program_name, optarg);
			}
			printf("%s, %s\n", program_name, tncfg_c_version);
			exit(1);
			break;
		case 'V':
			strcpy(ifr.ifr_name, optarg);
			break;
		case 'P':
			strcpy(shc->cf_name, optarg);
			break;
		case 'l':
			program_name = optarg;
			break;
		case '+': /* optionsfrom */
			optionsfrom(optarg, &argc, &argv, optind, stderr);
			/* no return on error */
			break;
		default:
			usage(program_name);
			break;
		}
		previous = c;
	}

	switch(shc->cf_cmd) {
	case IPSEC_SET_DEV:
		if(!shc->cf_name) {
			fprintf(stderr, "%s: physical I/F parameter missing.\n",
				program_name);
			exit(1);
		}
	case IPSEC_DEL_DEV:
		if(!ifr.ifr_name) {
			fprintf(stderr, "%s: virtual I/F parameter missing.\n",
				program_name);
			exit(1);
		}
		break;
	case IPSEC_CLR_DEV:
		strcpy(ifr.ifr_name, "ipsec0");
		break;
	default:
		fprintf(stderr, "%s: exactly one of '--attach', '--detach' or '--clear' options must be specified.\n"
			"Try %s --help' for usage information.\n",
			program_name, program_name);
		exit(1);
	}

	s=socket(AF_INET, SOCK_DGRAM,0);
	if(s==-1)
	{
		printf("%s: Socket creation failed -- ", program_name);
		switch(errno)
		{
		case EACCES:
			if(getuid()==0)
				printf("Root denied permission!?!\n");
			else
				printf("Run as root user.\n");
			break;
		case EPROTONOSUPPORT:
			printf("Internet Protocol not enabled");
		case EMFILE:
		case ENFILE:
		case ENOBUFS:
			printf("Insufficient system resources.\n");
			break;
		default:
			printf("Unknown socket error %d.\n", errno);
		}
		exit(1);
	}
	if(ioctl(s, shc->cf_cmd, &ifr)==-1)
	{
		if(shc->cf_cmd == IPSEC_SET_DEV) {
			printf("%s: Socket ioctl failed on attach -- ", program_name);
			switch(errno)
			{
			case EINVAL:
				printf("Invalid argument, check kernel log messages for specifics.\n");
				break;
			case ENODEV:
				printf("No such device.  Is the virtual device valid?  Is the ipsec module linked into the kernel or loaded as a module?\n");
				break;
			case ENXIO:
				printf("No such device.  Is the physical device valid?\n");
				break;
			case EBUSY:
				printf("Device busy.  Virtual device %s is already attached to a physical device -- Use detach first.\n",
				       ifr.ifr_name);
				break;
			default:
				printf("Unknown socket error %d.\n", errno);
			}
			exit(1);
		}
		if(shc->cf_cmd == IPSEC_DEL_DEV) {
			printf("%s: Socket ioctl failed on detach -- ", program_name);
			switch(errno)
			{
			case EINVAL:
				printf("Invalid argument, check kernel log messages for specifics.\n");
				break;
			case ENODEV:
				printf("No such device.  Is the virtual device valid?  The ipsec module may not be linked into the kernel or loaded as a module.\n");
				break;
			case ENXIO:
				printf("Device requested is not linked to any physical device.\n");
				break;
			default:
				printf("Unknown socket error %d.\n", errno);
			}
			exit(1);
		}
		if(shc->cf_cmd == IPSEC_CLR_DEV) {
			printf("%s: Socket ioctl failed on clear -- ", program_name);
			switch(errno)
			{
			case EINVAL:
				printf("Invalid argument, check kernel log messages for specifics.\n");
				break;
			case ENODEV:
				printf("Failed.  Is the ipsec module linked into the kernel or loaded as a module?.\n");
				break;
			default:
				printf("Unknown socket error %d.\n", errno);
			}
			exit(1);
		}
	}
	exit(0);
}
	
/*
 * $Log: tncfg.c,v $
 * Revision 1.13  1999/04/11 00:12:09  henry
 * GPL boilerplate
 *
 * Revision 1.12  1999/04/06 04:54:39  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.11  1999/03/17 15:40:54  rgb
 * Make explicit main() return type of int.
 *
 * Revision 1.10  1998/11/12 21:08:04  rgb
 * Add --label option to identify caller from scripts.
 *
 * Revision 1.9  1998/10/09 18:47:30  rgb
 * Add 'optionfrom' to get more options from a named file.
 *
 * Revision 1.8  1998/10/09 04:36:55  rgb
 * Changed help output from stderr to stdout.
 * Deleted old commented out cruft.
 *
 * Revision 1.7  1998/08/28 03:15:14  rgb
 * Add some manual long options to the usage text.
 *
 * Revision 1.6  1998/08/05 22:29:00  rgb
 * Change includes to accomodate RH5.x.
 * Force long option names.
 * Add ENXIO error return code to narrow down error reporting.
 *
 * Revision 1.5  1998/07/29 21:45:28  rgb
 * Convert to long option names.
 *
 * Revision 1.4  1998/07/09 18:14:11  rgb
 * Added error checking to IP's and keys.
 * Made most error messages more specific rather than spamming usage text.
 * Added more descriptive kernel error return codes and messages.
 * Converted all spi translations to unsigned.
 * Removed all invocations of perror.
 *
 * Revision 1.3  1998/05/27 18:48:20  rgb
 * Adding --help and --version directives.
 *
 * Revision 1.2  1998/04/23 21:11:39  rgb
 * Fixed 0 argument usage case to prevent sigsegv.
 *
 * Revision 1.1.1.1  1998/04/08 05:35:09  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.5  1997/06/03 04:31:55  ji
 * New file.
 *
 */
