/*
 * manipulate eroutes
 * Copyright (C) 1996  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char eroute_c_version[] = "RCSID $Id: eroute.c,v 1.22 1999/04/11 00:12:08 henry Exp $";


#include <sys/types.h>
#include <linux/types.h> /* new */
#include <string.h>
#include <errno.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>


#include <unistd.h>
#include "../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_netlink.h"


#include <stdio.h>
#include <getopt.h>

/* XXX */
/* #include "atosubnet.c" */
#include <freeswan.h>

char *program_name;
extern char *optarg;
extern int optind, opterr, optopt;
char *edst_opt, *spi_opt, *proto_opt, *said_opt, *dst_opt, *src_opt;
		
void
usage(char* arg)
{
	fprintf(stdout, "usage: %s --{add,replace} --src <src>/<srcmaskbits>|<srcmask> --dst <dst>/<dstmaskbits>|<dstmask> <SA>\n", arg);
	fprintf(stdout, "            where <SA> is '--edst <edst> --spi <spi> --proto <proto>' OR '--said <said>' OR '--said %passthrough'.\n");
	fprintf(stdout, "       %s --del --src <src>/<srcmaskbits>|<srcmask> --dst <dst>/<dstmaskbits>|<dstmask>\n", arg);
	fprintf(stdout, "       %s --clear\n", arg);
	fprintf(stdout, "       %s --help\n", arg);
	fprintf(stdout, "       %s --version\n", arg);
exit(1);
}

int
resolve_ip(char *arg, struct in_addr *ip)
{
	struct hostent *hostp;
	struct netent *netp;
	
	if (!strcmp(arg, "default")) {/* default is really 0.0.0.0 */
		ip->s_addr = htonl(INADDR_ANY);
		return(1);
	}
	if (!(strcmp(arg, "255.255.255.255"))) {
		/* inet_addr() returns -1 for this, as it should, but that means an error */
		ip->s_addr = htonl(INADDR_BROADCAST);
		return (1);
	}
	if ((netp = getnetbyname(arg)) != (struct netent *)NULL) {
		ip->s_addr = htonl(netp->n_net);
		return(1);
	}
	if ((hostp = gethostbyname(arg)) == (struct hostent *)NULL) {
		errno = h_errno;
		return(-1);
	}
	memcpy((char *) ip, (char *) hostp->h_addr_list[0], hostp->h_length);
	return(0);
}

static struct option const longopts[] =
{
	{"dst", 1, 0, 'D'},
	{"src", 1, 0, 'S'},
	{"add", 0, 0, 'a'},
	{"replace", 0, 0, 'r'},
	{"clear", 0, 0, 'c'},
	{"del", 0, 0, 'd'},
	{"edst", 1, 0, 'e'},
	{"proto", 1, 0, 'p'},
	{"help", 0, 0, 'h'},
	{"spi", 1, 0, 's'},
	{"said", 1, 0, 'I'},
	{"version", 0, 0, 'v'},
	{"label", 1, 0, 'l'},
	{"optionsfrom", 1, 0, '+'},
	{0, 0, 0, 0}
};

int
main(argc, argv)
int argc;
char **argv;
{
	int fd;
	struct encap_msghdr emh;
	char *endptr;
	int ret;
	int c, previous = -1;
	const char* error_s;

	program_name = argv[0];
	memset(&emh, 0, sizeof(emh));

	while((c = getopt_long(argc, argv, "acdD:e:hprs:S:vl:+:", longopts, 0)) != EOF)
	{
		switch(c) {
		case 'a':
			if(emh.em_type) {
				fprintf(stderr, "%s: Only one of '--add', '--replace', '--clear', or '--del' options permitted.\n",
					program_name);
				exit(1);
			}
			emh.em_type = EMT_SETEROUTE;
			break;
		case 'r':
			if(emh.em_type) {
				fprintf(stderr, "%s: Only one of '--add', '--replace', '--clear', or '--del' options permitted.\n",
					program_name);
				exit(1);
			}
			emh.em_type = EMT_RPLACEROUTE;
			break;
		case 'c':
			if(emh.em_type) {
				fprintf(stderr, "%s: Only one of '--add', '--replace', '--clear', or '--del' options permitted.\n",
					program_name);
				exit(1);
			}
			emh.em_type = EMT_CLREROUTE;
			break;
		case 'd':
			if(emh.em_type) {
				fprintf(stderr, "%s: Only one of '--add', '--replace', '--clear', or '--del' options permitted.\n",
					program_name);
				exit(1);
			}
			emh.em_type = EMT_DELEROUTE;
			break;
		case 'e':
			if(said_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(edst_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, edst_opt);
				exit (1);
			}				
			if(ret = resolve_ip(optarg, &(emh.em_erdst)) != 0) {
				if(ret == 1) {
					fprintf(stderr, "%s: Destination address cannot be a network: %s\n",
						program_name, optarg);
					exit (1);
				} else {
					fprintf(stderr, "%s: Invalid formation of destination address: %s\n",
						program_name, optarg);
					exit (1);
				}
			}
			edst_opt = optarg;
			break;
		case 'h':
		case '?':
			usage(program_name);
			exit(1);
		case 's':
			if(said_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(spi_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, spi_opt);
				exit (1);
			}				
			emh.em_erspi = htonl(strtoul(optarg, &endptr, 0));
			if(!(endptr == optarg + strlen(optarg))) {
				fprintf(stderr, "%s: Invalid character in SPI parameter: %s\n",
					program_name, optarg);
				exit (1);
			}
			if(emh.em_erspi < 0x100) {
				fprintf(stderr, "%s: Illegal reserved spi: %s => 0x%x Must be larger than 0x100.\n",
					program_name, optarg, emh.em_erspi);
				exit(1);
			}
			spi_opt = optarg;
			break;
		case 'p':
			if(said_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(proto_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, proto_opt);
				exit (1);
			}
#if 0
			if(emh.em_erproto) {
				fprintf(stderr, "%s: Warning, PROTO parameter redefined:%s\n",
					program_name, optarg);
				exit (1);
			}
#endif
			if(!strcmp(optarg, "ah"))
				emh.em_erproto = SA_AH;
			if(!strcmp(optarg, "esp"))
				emh.em_erproto = SA_ESP;
			if(!strcmp(optarg, "tun"))
				emh.em_erproto = SA_IPIP;
			if(emh.em_erproto == 0) {
				fprintf(stderr, "%s: Invalid PROTO parameter: %s\n",
					program_name, optarg);
				exit (1);
			}
			proto_opt = optarg;
			break;
		case 'I':
			if(said_opt) {
				fprintf(stderr, "%s: Error, SAID parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(proto_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, proto_opt);
				exit (1);
			}
			if(edst_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, edst_opt);
				exit (1);
			}
			if(proto_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, spi_opt);
				exit (1);
			}
			if((error_s = atosa(optarg, 0, &emh.em_ersaid)) != NULL) {
				fprintf(stderr, "%s: Error, %s converting --sa argument:%s\n",
					program_name, error_s, optarg);
				exit (1);
			} else if((emh.em_erspi < 0x100) && (emh.em_erspi > 0x0)){
				fprintf(stderr, "%s: Illegal reserved spi: %s => 0x%x Must be larger than or equal to 0x100.\n",
					program_name, optarg, emh.em_erspi);
				exit(1);
			}
			said_opt = optarg;
			break;
		case 'v':
			printf("%s, %s\n", program_name, eroute_c_version);
			exit(1);
		case 'D':
			if(dst_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, dst_opt);
				exit (1);
			}				
			if (atosubnet(optarg, 0, &emh.em_eaddr.sen_ip_dst, &emh.em_emask.sen_ip_dst)) {
				fprintf(stderr, "%s: Invalid formation of destination network/mask: %s\n",
					program_name, optarg);
				exit (1);
			}
			dst_opt = optarg;
			break;
		case 'S':
			if(src_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, src_opt);
				exit (1);
			}				
			if (atosubnet(optarg, 0, &emh.em_eaddr.sen_ip_src, &emh.em_emask.sen_ip_src)) {
				fprintf(stderr, "%s: Invalid formation of source network/mask: %s\n",
					program_name, optarg);
				exit (1);
			}
			src_opt = optarg;
			break;
		case 'l':
			program_name = optarg;
			break;
		case '+': /* optionsfrom */
			optionsfrom(optarg, &argc, &argv, optind, stderr);
			/* no return on error */
			break;
		default:
		}
		previous = c;
	}

#ifdef DEBUG
	printf("%s: DEBUG: argc=%d\n", program_name, argc);
#endif /* DEBUG */

	/* Sanity checks */

	switch(emh.em_type) {
	case EMT_SETEROUTE:
	case EMT_RPLACEROUTE:
		if(!(edst_opt && spi_opt && proto_opt) && !(said_opt)) {
			fprintf(stderr, "%s: add option must have SA specified.\n",
				program_name);
			exit(1);
		}
	case EMT_DELEROUTE:
		if(!src_opt) {
			fprintf(stderr, "%s: Error -- %s option '--src' is required.\n",
				program_name, (emh.em_type == EMT_SETEROUTE) ? "add" : "del");
			exit(1);
		}
		if(!emh.em_eaddr.sen_ip_src.s_addr || !emh.em_emask.sen_ip_src.s_addr) {
			fprintf(stderr, "%s: warning -- %s option '--src' is 0.0.0.0.  If the <default address> was not intentional, then a name lookup failed.\n",
				program_name, (emh.em_type == EMT_SETEROUTE) ? "add" : "del");
/*			exit(1); */
		}
		if(!dst_opt) {
			fprintf(stderr, "%s: Error -- %s option '--dst' is required.\n",
				program_name, (emh.em_type == EMT_SETEROUTE) ? "add" : "del");
			exit(1);
		}
		if(!emh.em_eaddr.sen_ip_dst.s_addr || !emh.em_emask.sen_ip_dst.s_addr) {
			fprintf(stderr, "%s: warning -- %s option '--dst' is 0.0.0.0.  If the <default address> was not intentional, then a name lookup failed.\n",
				program_name, (emh.em_type == EMT_SETEROUTE) ? "add" : "del");
/*			exit(1); */
		}
	case EMT_CLREROUTE:
		emh.em_magic = EM_MAGIC;
		emh.em_msglen= sizeof emh;
		emh.em_version = 0;
		emh.em_eaddr.sen_len = emh.em_emask.sen_len = sizeof (struct sockaddr_encap);
		emh.em_eaddr.sen_family = emh.em_emask.sen_family = 26;
		emh.em_eaddr.sen_type = SENT_IP4;
		emh.em_emask.sen_type = 255;
		break;
	default:
		fprintf(stderr, "%s: exactly one of '--add', '--replace', '--del' or '--clear' options must be specified.\n"
			"Try %s --help' for usage information.\n",
			program_name, program_name);
		exit(1);
	}

	fd = open("/dev/ipsec", O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "%s: Could not open /dev/ipsec -- ", program_name);
		switch(errno) {
		case ENOENT:
			fprintf(stderr, "device does not exist.  See FreeS/WAN installation procedure.\n");
		case EACCES:
			fprintf(stderr, "access denied.  ");
			if(getuid() == 0) {
				fprintf(stderr, "Check permissions.  Should be 600.\n");
			} else {
				fprintf(stderr, "You must be root to open this file.\n");
			}
			break;
		case EUNATCH:
			fprintf(stderr, "Netlink not enabled OR KLIPS not loaded.\n");
			break;
		default:
			fprintf(stderr, "Unknown file open error %d\n", errno);
		}
		exit(1);
	}

#ifdef DEBUG
	printf("%s: DEBUG: open ok\n", program_name);
#endif /* DEBUG */

	if (write(fd, &emh, sizeof emh) != sizeof emh) {
		fprintf(stderr, "%s: Failed -- ", program_name);
		switch(errno) {
		case EINVAL:
			fprintf(stderr, "Invalid argument, check kernel log messages for specifics.\n");
			break;
		case ENXIO:
			if((emh.em_type == EMT_SETEROUTE) ||
			   (emh.em_type == EMT_RPLACEROUTE)) {
				fprintf(stderr, "Invalid mask.\n");
				break;
			}
			if(emh.em_type == EMT_DELEROUTE) {
				fprintf(stderr, "Mask not found.\n");
				break;
			}
		case EFAULT:
			if((emh.em_type == EMT_SETEROUTE) ||
			   (emh.em_type == EMT_RPLACEROUTE)) {
				fprintf(stderr, "Invalid address.\n");
				break;
			}
			if(emh.em_type == EMT_DELEROUTE) {
				fprintf(stderr, "Address not found.\n");
				break;
			}
		default:
			fprintf(stderr, "Unknown file write error %d\n", errno);
		}
		close(fd);
		exit(1);
	}
	close(fd);

#ifdef DEBUG
	printf("%s: DEBUG: write ok\n", program_name);
#endif /* DEBUG */

	exit(0);
}
/*
 * $Log: eroute.c,v $
 * Revision 1.22  1999/04/11 00:12:08  henry
 * GPL boilerplate
 *
 * Revision 1.21  1999/04/06 04:54:37  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.20  1999/03/17 15:40:54  rgb
 * Make explicit main() return type of int.
 *
 * Revision 1.19  1999/01/26 05:51:01  rgb
 * Updated to use %passthrough instead of bypass.
 *
 * Revision 1.18  1999/01/22 06:34:52  rgb
 * Update to include SAID command line parameter.
 * Add IPSEC 'bypass' switch.
 * Add error-checking.
 * Cruft clean-out.
 *
 * Revision 1.17  1998/11/29 00:52:26  rgb
 * Add explanation to warning about default source or destination.
 *
 * Revision 1.16  1998/11/12 21:08:03  rgb
 * Add --label option to identify caller from scripts.
 *
 * Revision 1.15  1998/10/27 00:33:27  rgb
 * Make output error text more fatal-sounding.
 *
 * Revision 1.14  1998/10/26 01:28:38  henry
 * use SA_* protocol names, not IPPROTO_*, to avoid compile problems
 *
 * Revision 1.13  1998/10/25 02:44:56  rgb
 * Institute more precise error return codes from eroute commands.
 *
 * Revision 1.12  1998/10/19 18:58:55  rgb
 * Added inclusion of freeswan.h.
 * a_id structure implemented and used: now includes protocol.
 *
 * Revision 1.11  1998/10/09 18:47:29  rgb
 * Add 'optionfrom' to get more options from a named file.
 *
 * Revision 1.10  1998/10/09 04:34:58  rgb
 * Changed help output from stderr to stdout.
 * Changed error messages from stdout to stderr.
 * Added '--replace' option.
 * Deleted old commented out cruft.
 *
 * Revision 1.9  1998/08/18 17:18:13  rgb
 * Delete old commented out cruft.
 * Reduce destination and source default subnet to warning, not fatal.
 *
 * Revision 1.8  1998/08/05 22:24:45  rgb
 * Change includes to accomodate RH5.x
 *
 * Revision 1.7  1998/07/29 20:49:08  rgb
 * Change to use 0x-prefixed hexadecimal for spi's.
 *
 * Revision 1.6  1998/07/28 00:14:24  rgb
 * Convert from positional parameters to long options.
 * Add --clean option.
 * Add hostname lookup support.
 *
 * Revision 1.5  1998/07/14 18:13:28  rgb
 * Restructured for better argument checking.
 * Added command to clear the eroute table.
 *
 * Revision 1.4  1998/07/09 18:14:10  rgb
 * Added error checking to IP's and keys.
 * Made most error messages more specific rather than spamming usage text.
 * Added more descriptive kernel error return codes and messages.
 * Converted all spi translations to unsigned.
 * Removed all invocations of perror.
 *
 * Revision 1.3  1998/05/27 18:48:19  rgb
 * Adding --help and --version directives.
 *
 * Revision 1.2  1998/04/13 03:15:29  rgb
 * Commands are now distinguishable from arguments when invoking usage.
 *
 * Revision 1.1.1.1  1998/04/08 05:35:10  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.3  1996/11/20 14:51:32  ji
 * Fixed problems with #include paths.
 * Changed (incorrect) references to ipsp into ipsec.
 *
 * Revision 0.2  1996/11/08 15:45:24  ji
 * First limited release.
 *
 *
 */
