/*
 * IPSEC Tunneling code. Heavily based on drivers/net/new_tunnel.c
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char ipsec_tunnel_c_version[] = "RCSID $Id: ipsec_tunnel.c,v 1.46 1999/04/11 00:29:00 henry Exp $";

/* Things I wish I had known when writing the tunnel driver:

	When the tunnel_xmit() function is called, the skb contains the
	packet to be sent (plus a great deal of extra info), and dev
	contains the tunnel device that _we_ are.

	When we are passed a packet, we are expected to fill in the
	source address with our source IP address.

	What is the proper way to allocate, copy and free a buffer?
	After you allocate it, it is a "0 length" chunk of memory
	starting at zero.  If you want to add headers to the buffer
	later, you'll have to call "skb_reserve(skb, amount)" with
	the amount of memory you want reserved.  Then, you call
	"skb_put(skb, amount)" with the amount of space you want in
	the buffer.  skb_put() returns a pointer to the top (#0) of
	that buffer.  skb->len is set to the amount of space you have
	"allocated" with skb_put().  You can then write up to skb->len
	bytes to that buffer.  If you need more, you can call skb_put()
	again with the additional amount of space you need.  You can
	find out how much more space you can allocate by calling 
	"skb_tailroom(skb)".
	Now, to add header space, call "skb_push(skb, header_len)".
	This creates space at the beginning of the buffer and returns
	a pointer to this new space.  If later you need to strip a
	header from a buffer, call "skb_pull(skb, header_len)".
	skb_headroom() will return how much space is left at the top
	of the buffer (before the main data).  Remember, this headroom
	space must be reserved before the skb_put() function is called.
*/

#define __NO_VERSION__
#include <linux/module.h>
#include <linux/config.h>	/* for CONFIG_IP_FORWARD */

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/udp.h>         /* struct udphdr */
#include <linux/skbuff.h>
#include <asm/checksum.h>
#include <net/icmp.h>		/* icmp_send() */
#include <net/ip.h>

#include "../../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_tunnel.h"
#include "ipsec_ipe4.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#include <net/ip.h>
#include <linux/if_arp.h>

extern void des_cbc_encrypt(caddr_t, caddr_t, int, caddr_t, caddr_t, int);
extern void des_ede3_cbc_encrypt(caddr_t, caddr_t, int, caddr_t, caddr_t, caddr_t, caddr_t, int);
static __u32 zeroes[64];

#ifdef DEBUG_IPSEC
int debug_tunnel = 0;
#endif /* DEBUG_IPSEC */

#ifdef DEBUG_IPSEC_
static void
dmp(char *s, caddr_t bb, int len)
{
	int i;
	unsigned char *b = bb;
  
	if (debug_ah & DB_AH_DMP) {
		printk("klips_debug:ipsec_tunnel_:at %s, len=%d:", s, len);
		for (i=0; i < len; i++) {
			if(!(i%16)){
				printk("\nklips_debug:  ");
			}
			printk(" %02x", *b++);
		}
		printk("\n");
	}
}
#else
#define dmp(_x, _y, _z) 
#endif

/*
 *	This is mostly skbuff.c:skb_copy().
 */
struct sk_buff *
skb_copy_expand(struct sk_buff *skb, int priority, int headroom, int tailroom)
{
	struct sk_buff *n;
	unsigned long offset;

	/*
	 *	Do sanity checking
	 */
	if((headroom < 0) || (tailroom < 0) || ((headroom+tailroom) < 0)) {
		printk("klips_error:skb_copy_expand: "
		       "Illegal negative head/tailroom %d, %d\n",
		       headroom, tailroom);
		return NULL;
	}
	/*
	 *	Allocate the copy buffer
	 */
	 
	IS_SKB(skb);
	
	n=alloc_skb(skb->truesize-sizeof(struct sk_buff) + headroom+tailroom, priority);
	if(n==NULL)
		return NULL;

	/*
	 *	Shift between the two data areas in bytes
	 */
	 
	/* offset=n->head-skb->head; */ /* moved down a few lines */

	/* Set the data pointer */
	skb_reserve(n,skb->data-skb->head+headroom);
	/* Set the tail pointer and length */
	if(skb_tailroom(n) < skb->len) {
		printk("klips_error:skb_copy_expand: "
		       "tried to skb_put %ld, %d available.  "
		       "This should never happen, please report.\n",
		       skb->len, skb_tailroom(n));
		dev_kfree_skb(n, FREE_WRITE);
		return NULL;
	}
	skb_put(n,skb->len);

	offset=n->head + headroom - skb->head;

	/* Copy the bytes */
	memcpy(n->head + headroom, skb->head,skb->end-skb->head);
	n->link3=NULL;
	n->list=NULL;
	n->sk=NULL;
	n->when=skb->when;
	n->dev=skb->dev;
	n->h.raw=skb->h.raw+offset;
	n->mac.raw=skb->mac.raw+offset;
	n->ip_hdr=(struct iphdr *)(((char *)skb->ip_hdr)+offset);
	n->saddr=skb->saddr;
	n->daddr=skb->daddr;
	n->raddr=skb->raddr;
	n->seq=skb->seq;
	n->end_seq=skb->end_seq;
	n->ack_seq=skb->ack_seq;
	n->acked=skb->acked;
	memcpy(n->proto_priv, skb->proto_priv, sizeof(skb->proto_priv));
	n->used=skb->used;
	n->free=1;
	n->arp=skb->arp;
	n->tries=0;
	n->lock=0;
	n->users=0;
	n->pkt_type=skb->pkt_type;
	n->stamp=skb->stamp;
	
	IS_SKB(n);
	return n;
}

#ifdef DEBUG_IPSEC
void
ipsec_print_ip(struct iphdr *ip)
{
	char buf[16];

	printk("klips_debug:   IP:");
	printk(" ihl:%d", ip->ihl*4);
	printk(" ver:%d", ip->version);
	printk(" tos:%d", ip->tos);
	printk(" tlen:%d", ntohs(ip->tot_len));
	printk(" id:%d", ip->id);
	printk(" frag_off:%d", ip->frag_off);
	printk(" ttl:%d", ip->ttl);
	printk(" proto:%d", ip->protocol);
	printk(" chk:%d", ip->check);
	addrtoa(*((struct in_addr*)(&ip->saddr)), 0, buf, sizeof(buf));
	printk(" saddr:%s", buf);
	addrtoa(*((struct in_addr*)(&ip->daddr)), 0, buf, sizeof(buf));
	printk(" daddr:%s", buf);
	printk("\n");

#if 0
	{
		__u8 *c;
		int i;
		
		c = ((__u8*)ip) + ip->ihl*4;
		for(i = 0; i < ntohs(ip->tot_len) - ip->ihl*4; i++, c++) {
			if(!(i % 16)) {
				printk("klips_debug:   contents:");
			}
			printk("%02x ", *c);
			if(!((i + 1) % 16)) {
				printk("\n");
			}
		}
		if(i % 16) {
			printk("\n");
		}
	}
#endif
}
#endif /* DEBUG_IPSEC */

#ifdef REAL_LOCKING_P
/*
 *	Locking
 */
 
static int
ipsec_tunnel_lock(struct ipsecpriv *prv)
{
	unsigned long flags;
	save_flags(flags);
	cli();
	/*
	 *	Lock in an interrupt may fail
	 */
	if(prv->locked && in_interrupt()) {
		restore_flags(flags);
		return 0;
	}
	while(prv->locked)
		sleep_on(&prv->wait_queue);
	prv->locked=1;
	restore_flags(flags);
	return 1;
}

static void
ipsec_tunnel_unlock(struct ipsecpriv *prv)
{
	prv->locked=0;
	wake_up(&prv->wait_queue);
}
#endif /* REAL_LOCKING_P */

static int
ipsec_tunnel_open(struct device *dev)
{
	struct ipsecpriv *prv = dev->priv;
	
	/*
	 * Can't open until attached.
	 */

	KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
		    "klips_debug:ipsec_tunnel_open: "
		    "dev = %s, prv->dev = %s\n",
		    dev->name, prv->dev?prv->dev->name:"NONE");

	if (prv->dev == NULL)
		return -ENODEV;
	
	MOD_INC_USE_COUNT;
	return 0;
}

static int
ipsec_tunnel_close(struct device *dev)
{
	MOD_DEC_USE_COUNT;
	return 0;
}

/*
 *	This function assumes it is being called from dev_queue_xmit()
 *	and that skb is filled properly by that function.
 */

int
ipsec_tunnel_start_xmit(struct sk_buff *skb, struct device *dev)
{
	struct ipsecpriv *prv;		/* Our device' private space */
	struct sk_buff *oskb = NULL;	/* Original skb pointer */
	struct enet_statistics *stats;	/* This device's statistics */
	struct iphdr  *iph;		/* Our new IP header */
	__u32   target;			/* The other host's IP address */
	__u32	orgdst;			/* Original IP destination address */
	int	iphlen;			/* IP header length */
	int	pyldsz;			/* upper protocol payload size */
	int	headroom;
	int	tailroom;
	int     max_headroom = 0;	/* The extra header space needed */
	int	max_tailroom = 0;	/* The extra stuffing needed */
	int     tot_headroom = 0;	/* The total header space needed */
	int	tot_tailroom = 0;	/* The totalstuffing needed */
	__u8	*saved_header = NULL;	/* saved copy of the hard header */
	int i;

	struct sockaddr_encap matcher;	/* eroute search key */
	struct eroute *er;
	struct tdb *tdbp, *tdbq;	/* Tunnel Descriptor Block pointers */
	char sa[SATOA_BUF];
	size_t sa_len;
	int hard_header_stripped = 0;	/* has the hard header been removed yet? */
	struct device *physdev;
	short physmtu;
	short mtudiff;

	/*
	 *	Return if there is nothing to do.  (Does this ever happen?) XXX
	 */
	if (skb == NULL) {
		KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
			    KERN_INFO "klips_error:tunnel: "
			    "Nothing to do!\n" );
		goto cleanup;
	}
	if (dev == NULL) {
		KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
			    KERN_INFO "klips_error:tunnel: "
			    "No device associated with skb!\n" );
		dev_kfree_skb(skb, FREE_WRITE);
		goto cleanup;
	}

	prv = dev->priv;
	if (prv == NULL) {
		KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
			    KERN_INFO "klips_error:tunnel: "
			    "Device has no private structure!\n" );
		dev_kfree_skb(skb, FREE_WRITE);
		goto cleanup;
	}

	physdev = prv->dev;
	if (physdev == NULL) {
		KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
			    KERN_INFO "klips_error:tunnel: "
			    "Device is not attached to physical device!\n" );
		dev_kfree_skb(skb, FREE_WRITE);
		goto cleanup;
	}

	physmtu = physdev->mtu;

	stats = (struct enet_statistics *) &(prv->mystats);

#ifdef DEBUG_IPSEC
	if (debug_tunnel & DB_TN_XMIT) {
		int i;
		char c;
		
		printk("klips_debug:ipsec_tunnel_start_xmit: "
		       "hard_header_len:dev:%d",
		       dev->hard_header_len);
		c = ' ';
		for (i=0; i < dev->hard_header_len; i++) {
			printk("%c%02x", c, skb->data[i]);
			c = ':';
		}
		printk("\n");
	}
#endif /* DEBUG_IPSEC */

	iph = (struct iphdr *) (skb->data + dev->hard_header_len);

	KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
		    "klips_debug:ipsec_tunnel_start_xmit: "
		    "skb->len=%ld, packet contents:\n", skb->len);
	KLIPS_IP_PRINT(debug_tunnel & DB_TN_XMIT, iph);

	/*
	 * Sanity checks
	 */
	if ((iph->ihl << 2) != sizeof (struct iphdr)) {
		printk("klips_debug:ipsec_tunnel_start_xmit: "
		       "cannot process IP header options yet\n"); /* XXX */
		stats->tx_dropped++;
		dev_kfree_skb(skb, FREE_WRITE);
		goto cleanup;
	}
	
	/* TTL decrement code borrowed from ip_forward.c */
	{
		unsigned long checksum = iph->check;
		iph->ttl--;
	/*
	 *	Re-compute the IP header checksum.
	 *	This is efficient. We know what has happened to the header
	 *	and can thus adjust the checksum as Phil Karn does in KA9Q
	 *	except we do this in "network byte order".
	 */
		checksum += htons(0x0100);
		/* carry overflow? */
		checksum += checksum >> 16;
		iph->check = checksum;
	}
	if (iph->ttl <= 0) {
		/* Tell the sender its packet died... */
		icmp_send(skb, ICMP_TIME_EXCEEDED, ICMP_EXC_TTL, 0, physdev);

		KLIPS_PRINT(debug_tunnel, "klips_debug:ipsec_tunnel_start_xmit: "
			    "TTL=0, too many hops!\n");
		stats->tx_dropped++;
		dev_kfree_skb(skb, FREE_WRITE);
		goto cleanup;
	}

	/*
	 * Quick cheat for now...are we udp/500? If so, let it through
	 * without interference since it is most likely an IKE packet.
	 */
	if(iph->protocol == IPPROTO_UDP) {
		struct udphdr *udph = (struct udphdr*)((caddr_t)iph + (iph->ihl << 2));
		if(ntohs(udph->dest) == 500) {
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "udp/500 IKE packet, transmit unprocessed, "
				    "calling dev_queue_xmit\n"); 
			dev_queue_xmit(skb, physdev, SOPRI_NORMAL);
				/* does this need to be freed? */
			goto cleanup;
		}
	}

	/*
	 * First things first -- look us up in the erouting tables.
	 */
	matcher.sen_len = sizeof (struct sockaddr_encap);
	matcher.sen_family = 26;
	matcher.sen_type = SENT_IP4;
	matcher.sen_ip_src.s_addr = iph->saddr;
	matcher.sen_ip_dst.s_addr = iph->daddr;

	er = ipsec_findroute(&matcher);

	KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
		    "klips_debug:ipsec_tunnel_start_xmit: "
		    "Original head/tailroom: %d, %d\n",
		    skb_headroom(skb), skb_tailroom(skb));

	/* start encapsulation loop here XXX */
	do {
		orgdst = iph->daddr;
		iphlen = iph->ihl << 2;
		pyldsz = ntohs(iph->tot_len) - iphlen;
		max_headroom = max_tailroom = 0;
		
		if (er == NULL)	{
#if ROUTE_WITH_NO_EROUTE
#ifdef DEBUG_IPSEC
			if (debug_tunnel & DB_TN_REVEC) {
				struct timeval tv;
				do_gettimeofday(&tv);
				printk("klips_debug:ipsec_tunnel_start_xmit: "
				       "no eroute!: ts=%02d.%04d calling dev_queue_xmit\n", 
				       tv.tv_sec % 60, tv.tv_usec / 100);
			}
#endif /* DEBUG_IPSEC */
			dev_queue_xmit(skb, physdev, SOPRI_NORMAL);
#else /* ROUTE_WITH_NO_EROUTE */
#ifdef DEBUG_IPSEC
			if (debug_tunnel & DB_TN_REVEC) {
				struct timeval tv;
				do_gettimeofday(&tv);
				printk("klips_debug:ipsec_tunnel_start_xmit: "
				       "no eroute!: ts=%02d.%04d, dropping.\n", 
				       tv.tv_sec % 60, tv.tv_usec / 100);
			}
#endif /* DEBUG_IPSEC */
			stats->tx_dropped++;
			dev_kfree_skb(skb, FREE_WRITE);
#endif /* ROUTE_WITH_NO_EROUTE */
			goto cleanup;
		}
		
		/*
		  If the packet matches an eroute with an SA.proto of IP tunnelling and 
		  an SA.spi of '0', then forward the packet unprotected.
		  XXX -- This should eventually go into an SPD. 
		*/
		if((er->er_proto == IPPROTO_IPIP) && (er->er_spi == 0)) {
			dev_queue_xmit(skb, physdev, SOPRI_NORMAL);
				/* does this need to be freed? */
			goto cleanup;
		}
		
		target = er->er_dst.s_addr;
		
		tdbp = gettdb(er->er_said);
		sa_len = satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
		if (tdbp == NULL) {
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "no Tunnel Descriptor Block for SA%s: "
				    "outgoing packet with no SA\n", sa);
			stats->tx_dropped++;
			dev_kfree_skb(skb, FREE_WRITE);
			goto cleanup;
		}
		
		KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
			    "klips_debug:ipsec_tunnel_start_xmit: "
			    "found Tunnel Descriptor Block -- SA:%s\n", sa);
		
		/*
		 * How much headroom do we need to be able to apply all the grouped transforms?
		 */
		tdbq = tdbp;			/* save the head of the tdb chain */
		while (tdbp && tdbp->tdb_xform)	{
			headroom = tailroom = 0;
			sa_len = satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "calling room for <%s>, SA:%s\n", 
				    tdbp->tdb_xform->xf_name, sa);
			switch(tdbp->tdb_proto) {
			case IPPROTO_AH:
				headroom += sizeof(struct ah);
				break;
			case IPPROTO_ESP:
				switch(tdbp->tdb_encalg) {
				case ESP_DES:
				case ESP_3DES:
					headroom += sizeof(struct esp);
					break;
				case ESP_NULL:
					headroom += offsetof(struct esp, esp_iv);
					break;
				}
				switch(tdbp->tdb_authalg) {
				case AH_MD5:
				case AH_SHA:
					tailroom += AHHMAC_HASHLEN;
					break;
				default:
				}			
				tailroom += ((8 - ((pyldsz + 2 * sizeof(unsigned char)) % 8)) % 8) + 2;
				break;
			case IPPROTO_IPIP:
				headroom += sizeof(struct iphdr);
				break;
			default:
			}
			tdbp = tdbp->tdb_onext;
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "Required head/tailroom: %d,%d\n", 
				    headroom, tailroom);
			max_headroom += headroom;
			max_tailroom += tailroom;
			pyldsz += (headroom + tailroom);
		}
		tdbp = tdbq;			/* restore the head of the tdb chain */
		
		KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
			    "klips_debug:ipsec_tunnel_start_xmit: "
			    "existing head/tailroom: %d, %d"
			    " before applying xforms with %d, %d head/tailroom.\n",
			    skb_headroom(skb), skb_tailroom(skb),
			    max_headroom, max_tailroom);
		
		tot_headroom += max_headroom;
		tot_tailroom += max_tailroom;
		
		mtudiff = prv->mtu + tot_headroom + tot_tailroom - physmtu;

		KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
			    "klips_debug:ipsec_tunnel_start_xmit: mtu:%d physmtu:%d "
			    "tothr:%d tottr:%d mtudiff:%d ippkttotlen:%d\n",
			    prv->mtu, physmtu,
			    tot_headroom, tot_tailroom, mtudiff, ntohs(iph->tot_len));
		if(mtudiff > 0) {
			KLIPS_PRINT(1 || debug_tunnel & DB_TN_CROUT,
				    "klips_info:ipsec_tunnel_start_xmit: "
				    "mtu of %d decreased by %d\n",
				    prv->mtu,
				    prv->mtu - (physmtu - tot_headroom - tot_tailroom));
			prv->mtu = physmtu - (tot_headroom + tot_tailroom + 7);	/* add some slop? */
		}
		
		if ((skb_headroom(skb) >= max_headroom +
		     (hard_header_stripped ? dev->hard_header_len : 0)) && 
		    (skb_tailroom(skb) >= max_tailroom) && skb->free) {
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "data fits in existing skb\n");
		} else {
			struct sk_buff* tskb = skb;

			if(!oskb) {
				oskb = skb;
			}

			tskb = skb_copy_expand(skb, GFP_ATOMIC,
					      max_headroom, max_tailroom);
			if(!(skb == oskb) ) {
				dev_kfree_skb(skb, FREE_WRITE);
			}
			skb = tskb;
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "head/tailroom: %d, %d after allocation\n",
				    skb_headroom(skb), skb_tailroom(skb));
			if (skb) {
				skb->dev = physdev;
				skb->arp = 1;
			} else {
				printk("klips_debug:ipsec_tunnel_start_xmit: Failed, "
				       "tried to allocate %d head and %d tailroom\n", 
				       max_headroom, max_tailroom);
				stats->tx_dropped++;
				goto cleanup;
			}
		}
		
		if(!hard_header_stripped) {
			if((saved_header = kmalloc(dev->hard_header_len, GFP_ATOMIC)) == NULL) {
				stats->tx_dropped++;
				dev_kfree_skb(skb, FREE_WRITE);
				goto cleanup;
			}
			for (i = 0; i < dev->hard_header_len; i++) {
				saved_header[i] = skb->data[i];
			}
			skb_pull(skb, dev->hard_header_len);
			hard_header_stripped = 1;
			
			iph = (struct iphdr *) (skb->data);
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "packet contents after pull:\n");
			KLIPS_IP_PRINT(debug_tunnel & DB_TN_XMIT, iph);
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "head/tailroom: %d, %d after hard_header stripped.\n",
				    skb_headroom(skb), skb_tailroom(skb));
			
		} else {
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "hard header already stripped.\n");
		}
		
		/*
		 * Apply grouped transforms to packet
		 */
		while (tdbp && tdbp->tdb_xform) {
			struct iphdr ipo;
			struct esp *espp;
			struct ah *ahp;
			int headroom = 0, tailroom = 0, authlen = 0, padlen = 0, ilen = 0, len = 0, i;
			unsigned char *dat, *idat, *pad;
			__u32 iv[2];
			__u8 hash[AH_AMAX];
			union {
				MD5_CTX md5;
				SHA1_CTX sha1;
			} tctx;
			
			iphlen = iph->ihl << 2;
			pyldsz = ntohs(iph->tot_len) - iphlen;
			sa_len = satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
			KLIPS_PRINT(debug_tunnel & DB_TN_OXFS,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "calling output for <%s>, SA:%s\n", 
				    tdbp->tdb_xform->xf_name, sa);
			
			switch(tdbp->tdb_proto) {
			case IPPROTO_AH:
				headroom += sizeof(struct ah);
				break;
			case IPPROTO_ESP:
				switch(tdbp->tdb_encalg) {
				case ESP_DES:
				case ESP_3DES:
					headroom += sizeof(struct esp);
					break;
				case ESP_NULL:
					headroom += offsetof(struct esp, esp_iv);
					break;
				}
				switch(tdbp->tdb_authalg) {
				case AH_MD5:
				case AH_SHA:
					authlen = AHHMAC_HASHLEN;
					break;
				default:
				}		
				tailroom += ((8 - ((pyldsz + 2 * sizeof(unsigned char)) % 8)) % 8) + 2;
				tailroom += authlen;
				break;
			case IPPROTO_IPIP:
				headroom += sizeof(struct iphdr);
				break;
			default:
			}
			
			KLIPS_PRINT(debug_tunnel & DB_TN_OXFS,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "pushing %d bytes, putting %d, proto %d.\n", 
				    headroom, tailroom, tdbp->tdb_proto);
			dat = skb_push(skb, headroom);
			ilen = skb->len - tailroom;
			if(skb_tailroom(skb) < tailroom) {
				printk("klips_error:ipsec_tunnel_start_xmit: "
				       "tried to skb_put %d, %d available.  "
				       "This should never happen, please report.\n",
				       tailroom, skb_tailroom(skb));
				stats->tx_errors++;
				dev_kfree_skb(skb, FREE_WRITE);
				goto cleanup;
			}
			skb_put(skb, tailroom);
			KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "head/tailroom: %d, %d before xform.\n",
				    skb_headroom(skb), skb_tailroom(skb));
			len = skb->len;
			memmove((void *)dat, (void *)(dat + headroom), iphlen);
			iph = (struct iphdr *)dat;
			iph->tot_len = htons(skb->len);
			
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "packet contents before xform:\n");
			KLIPS_IP_PRINT(debug_tunnel & DB_TN_XMIT, iph);
			
			switch(tdbp->tdb_proto) {
			case IPPROTO_ESP:
				espp = (struct esp *)(dat + iphlen);
				espp->esp_spi = tdbp->tdb_spi;
				espp->esp_rpl = htonl(++(tdbp->tdb_replaywin_lastseq));
				
				switch(tdbp->tdb_encalg) {
				case ESP_DES:
				case ESP_3DES:
					iv[0] = *((__u32*)&(espp->esp_iv)    ) =
						((__u32*)(tdbp->tdb_iv))[0];
					iv[1] = *((__u32*)&(espp->esp_iv) + 1) =
						((__u32*)(tdbp->tdb_iv))[1];
					break;
				case ESP_NULL:
					break;
				}
				
				idat = dat + iphlen + headroom;
				ilen = len - (iphlen + headroom + authlen);
				
				/* Self-describing padding */
				pad = &dat[len - tailroom];
				padlen = tailroom - 2 - authlen;
				for (i = 0; i < padlen; i++) {
					pad[i] = i + 1; 
				}
				dat[len - authlen - 2] = padlen;
				
				dat[len - authlen - 1] = iph->protocol;
				iph->protocol = IPPROTO_ESP;
				
				KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
					    "klips_debug:ipsec_tunnel_start_xmit: "
					    "packet contents before encryption:\n");
				KLIPS_IP_PRINT(debug_tunnel & DB_TN_XMIT, iph);
				
				switch(tdbp->tdb_encalg) {
				case ESP_3DES:
					des_ede3_cbc_encrypt(idat, idat, ilen,
							     (caddr_t)(&((struct des_eks*)(tdbp->tdb_key_e))[0]),
							     (caddr_t)(&((struct des_eks*)(tdbp->tdb_key_e))[1]),
							     (caddr_t)(&((struct des_eks*)(tdbp->tdb_key_e))[2]),
							     (caddr_t)iv, 1);
					break;
				case ESP_DES:
					des_cbc_encrypt(idat, idat, ilen,
							(caddr_t)tdbp->tdb_key_e,
							(caddr_t)iv, 1);
					break;
				}
				
				switch(tdbp->tdb_encalg) {
				case ESP_DES:
				case ESP_3DES:
					/* XXX update IV with the last 8 octets of the encryption */
					((__u32*)(tdbp->tdb_iv))[0] =
						((__u32 *)(idat))[(ilen >> 2) - 2];
					((__u32*)(tdbp->tdb_iv))[1] =
						((__u32 *)(idat))[(ilen >> 2) - 1];
					break;
				case ESP_NULL:
					break;
				}
				
				switch(tdbp->tdb_authalg) {
				case AH_MD5:
					dmp("espp", (char*)espp, len - iphlen - authlen);
					tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->ictx;
					dmp("ictx", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, (caddr_t)espp, len - iphlen - authlen);
					dmp("ictx+dat", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Final(hash, &tctx.md5);
					dmp("ictx hash", (char*)&hash, sizeof(hash));
					tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->octx;
					dmp("octx", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, hash, AHMD596_ALEN);
					dmp("octx+hash", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Final(hash, &tctx.md5);
					dmp("octx hash", (char*)&hash, sizeof(hash));
					memcpy(&(dat[len - authlen]), hash, authlen);

					/* paranoid */
					memset((caddr_t)&tctx.md5, 0, sizeof(tctx.md5));
					memset((caddr_t)hash, 0, sizeof(*hash));
					break;
				case AH_SHA:
					tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx;
					SHA1Update(&tctx.sha1, (caddr_t)espp, len - iphlen - authlen);
					SHA1Final(hash, &tctx.sha1);
					tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->octx;
					SHA1Update(&tctx.sha1, hash, AHSHA196_ALEN);
					SHA1Final(hash, &tctx.sha1);
					memcpy(&(dat[len - authlen]), hash, authlen);
					
					/* paranoid */
					memset((caddr_t)&tctx.sha1, 0, sizeof(tctx.sha1));
					memset((caddr_t)hash, 0, sizeof(*hash));
					break;
				}
				
				break;
			case IPPROTO_AH:
				ahp = (struct ah *)(dat + iphlen);
				ahp->ah_spi = tdbp->tdb_spi;
				ahp->ah_rpl = htonl(++(tdbp->tdb_replaywin_lastseq));
				ahp->ah_rv = 0;
				ahp->ah_nh = iph->protocol;
				ahp->ah_hl = (headroom >> 2) - sizeof(__u64)/sizeof(__u32);
				iph->protocol = IPPROTO_AH;
				dmp("ahp", (char*)ahp, sizeof(*ahp));
				
				ipo = *iph;
				ipo.tos = 0;
				ipo.frag_off = 0;
				ipo.ttl = 0;
				ipo.check = 0;
				dmp("ipo", (char*)&ipo, sizeof(ipo));
				
				switch(tdbp->tdb_authalg) {
				case AH_MD5:
					tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->ictx;
					dmp("ictx", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, (unsigned char *)&ipo, sizeof (struct iphdr));
					dmp("ictx+ipo", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, (unsigned char *)ahp, headroom - sizeof(ahp->ah_data));
					dmp("ictx+ahp", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, (unsigned char *)zeroes, AHHMAC_HASHLEN);
					dmp("ictx+zeroes", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5,  dat + iphlen + headroom, len - iphlen - headroom);
					dmp("ictx+dat", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Final(hash, &tctx.md5);
					dmp("ictx hash", (char*)&hash, sizeof(hash));
					tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->octx;
					dmp("octx", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Update(&tctx.md5, hash, AHMD596_ALEN);
					dmp("octx+hash", (char*)&tctx.md5, sizeof(tctx.md5));
					MD5Final(hash, &tctx.md5);
					dmp("octx hash", (char*)&hash, sizeof(hash));
					
					memcpy(ahp->ah_data, hash, AHHMAC_HASHLEN);
					
					/* paranoid */
					memset((caddr_t)&tctx.md5, 0, sizeof(tctx.md5));
					memset((caddr_t)hash, 0, sizeof(hash));
					break;
				case AH_SHA:
					tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx;
					SHA1Update(&tctx.sha1, (unsigned char *)&ipo, sizeof (struct iphdr));
					SHA1Update(&tctx.sha1, (unsigned char *)ahp, headroom - sizeof(ahp->ah_data));
					SHA1Update(&tctx.sha1, (unsigned char *)zeroes, AHHMAC_HASHLEN);
					SHA1Update(&tctx.sha1,  dat + iphlen + headroom, len - iphlen - headroom);
					SHA1Final(hash, &tctx.sha1);
					tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->octx;
					SHA1Update(&tctx.sha1, hash, AHSHA196_ALEN);
					SHA1Final(hash, &tctx.sha1);
					
					memcpy(ahp->ah_data, hash, AHHMAC_HASHLEN);
					
					/* paranoid */
					memset((caddr_t)&tctx.sha1, 0, sizeof(tctx.sha1));
					memset((caddr_t)hash, 0, sizeof(hash));
					break;
				default:
				}
				break;
			case IPPROTO_IPIP:
				iph->version  = 4; /* const should be used here */
				iph->tos      = skb->ip_hdr->tos; /* XXX is this right? */
				iph->ttl      = 64; /* ip_statistics.IpDefaultTTL; */
				iph->frag_off = 0;
				iph->saddr    = ((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr.s_addr;
				iph->daddr    = ((struct sockaddr_in*)(tdbp->tdb_addr_d))->sin_addr.s_addr;
				iph->protocol = IPPROTO_IPIP;
				iph->ihl      = sizeof(struct iphdr) >> 2 /* 5 */;
				iph->id       = htons(ip_id_count++);   /* Race condition here? */
				break;
			default:
			}
			
			skb->ip_hdr = skb->h.iph = (struct iphdr *) skb->data;
			iph->check = 0;
			iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);
			
			KLIPS_PRINT(debug_tunnel & DB_TN_XMIT,
				    "klips_debug:ipsec_tunnel_start_xmit: "
				    "packet contents after xform:\n");
			KLIPS_IP_PRINT(debug_tunnel & DB_TN_XMIT, iph);
			
			tdbp = tdbp->tdb_onext;
			
		}
		/* end encapsulation loop here XXX */
		matcher.sen_ip_src.s_addr = iph->saddr;
		matcher.sen_ip_dst.s_addr = iph->daddr;
		er = ipsec_findroute(&matcher);
		KLIPS_PRINT((orgdst != target) && er && (debug_tunnel & DB_TN_CROUT),
			    "klips_debug:ipsec_tunnel_start_xmit: We are recursing here.\n");
	} while((orgdst != target) && er);
	
	KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
		    "klips_debug:ipsec_tunnel_start_xmit: "
		    "After recursive xforms -- head/tailroom: %d, %d\n",
		    skb_headroom(skb), skb_tailroom(skb));

#if CONFIG_IPSEC_ICMP
	if(ntohs(iph->tot_len) > physmtu) {
		icmp_send(oskb ? oskb : skb,
			  ICMP_DEST_UNREACH,
			  ICMP_FRAG_NEEDED,
			  prv->mtu,
			  physdev);
		KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
			    "klips_debug:ipsec_tunnel_start_xmit: "
			    "IPSEC packet is larger than MTU, fragmented.\n");
	}
#endif /* CONFIG_IPSEC_ICMP */

	if(saved_header) {
		skb_push(skb, dev->hard_header_len);
		for (i = 0; i < dev->hard_header_len; i++) {
			skb->data[i] = saved_header[i];
		}
		kfree_s(saved_header, dev->hard_header_len);
	}
	KLIPS_PRINT(debug_tunnel & DB_TN_CROUT,
		    "klips_debug:ipsec_tunnel_start_xmit: "
		    "With hard_header, final head/tailroom: %d, %d\n",
		    skb_headroom(skb), skb_tailroom(skb));
#ifdef DEBUG_IPSEC
	if (debug_tunnel & DB_TN_REVEC)	{
		struct timeval tv;
		do_gettimeofday(&tv);
		printk("klips_debug:ipsec_tunnel_start_xmit: "
		       "ts=%02d.%04d calling dev_queue_xmit\n", 
		       tv.tv_sec % 60, tv.tv_usec / 100);
	}
#endif /* DEBUG_IPSEC */
	stats->tx_packets++;

	if(ntohs(iph->tot_len) > physmtu) {
		ip_fragment(NULL, skb, physdev, 0);
		dev_kfree_skb(skb, FREE_WRITE);
	} else {
		dev_queue_xmit(skb, physdev, SOPRI_NORMAL);
	}
 cleanup:
	dev->tbusy = 0;
	if(oskb)
		dev_kfree_skb(oskb, FREE_WRITE);
	return 0;
}

static struct enet_statistics *
ipsec_tunnel_get_stats(struct device *dev)
{
	return &(((struct ipsecpriv *)(dev->priv))->mystats);
}

/*
 * Revectored calls.
 * For each of these calls, a field exists in our private structure.
 */

static int
ipsec_tunnel_hard_header(struct sk_buff *skb, struct device *dev,
	unsigned short type, void *daddr, void *saddr, unsigned len)
{
	struct ipsecpriv *prv = dev->priv;
	
	KLIPS_PRINT(debug_tunnel & DB_TN_REVEC,
		    "klips_debug:ipsec_tunnel: "
		    "Revectored hard_header, 0x%p -> 0x%p, dev_addr=%x\n",
		    saddr, daddr, *((struct device *)(prv->dev))->dev_addr);
	return prv->hard_header(skb, prv->dev, type, (void *)daddr, (void *)saddr, len);
}

static int
ipsec_tunnel_rebuild_header(void *buff, struct device *dev,
			unsigned long raddr, struct sk_buff *skb)
{
	struct ipsecpriv *prv = skb->dev->priv;
	struct device *tmp;
	int ret;
	
	KLIPS_PRINT(debug_tunnel & DB_TN_REVEC,
		    "klips_debug:ipsec_tunnel: "
		    "Revectored rebuild_header\n");
	tmp = skb->dev;
	skb->dev = prv->dev;
	
	ret = prv->rebuild_header(buff, prv->dev, raddr, skb);
	skb->dev = tmp;
	return ret;
}

static int
ipsec_tunnel_set_mac_address(struct device *dev, void *addr)
{
	struct ipsecpriv *prv = dev->priv;
	
	KLIPS_PRINT(debug_tunnel & DB_TN_REVEC,
		    "klips_debug:ipsec_tunnel: "
		    "Revectored set_mac_address_\n");
	return prv->set_mac_address(prv->dev, addr);

}

static void
ipsec_tunnel_cache_bind(struct hh_cache **hhp, struct device *dev,
				 unsigned short htype, __u32 daddr)
{
	struct ipsecpriv *prv = dev->priv;
	
	KLIPS_PRINT(debug_tunnel & DB_TN_REVEC,
		    "klips_debug:ipsec_tunnel: "
		    "Revectored cache_bind\n");
	prv->header_cache_bind(hhp, dev, htype, daddr);
	return;
}

static void
ipsec_tunnel_cache_update(struct hh_cache *hh, struct device *dev, unsigned char *  haddr)
{
	struct ipsecpriv *prv = dev->priv;
	
	KLIPS_PRINT(debug_tunnel & DB_TN_REVEC,
		    "klips_debug:ipsec_tunnel: "
		    "Revectored cache_update\n");
	prv->header_cache_update(hh, prv->dev, haddr);
	return;
}

/*
 * We call the attach routine to attach another device.
 */


static int
ipsec_tunnel_attach(struct device *tndev, struct ipsecpriv *prv, struct device *dev)
{
        int i;

	prv->dev = dev;
	prv->hard_start_xmit = dev->hard_start_xmit;
	prv->get_stats = dev->get_stats;

	if (dev->hard_header) {
		prv->hard_header = dev->hard_header;
		tndev->hard_header = ipsec_tunnel_hard_header;
	} else
		tndev->hard_header = NULL;
	
	if (dev->rebuild_header) {
		prv->rebuild_header = dev->rebuild_header;
		tndev->rebuild_header = ipsec_tunnel_rebuild_header;
	} else
		tndev->rebuild_header = NULL;
	
	if (dev->set_mac_address) {
		prv->set_mac_address = dev->set_mac_address;
		tndev->set_mac_address = ipsec_tunnel_set_mac_address;
	} else
		tndev->set_mac_address = NULL;
	
	if (dev->header_cache_bind) {
		prv->header_cache_bind = dev->header_cache_bind;
		tndev->header_cache_bind = ipsec_tunnel_cache_bind;
	} else
		tndev->header_cache_bind = NULL;

	if (dev->header_cache_update) {
		prv->header_cache_update = dev->header_cache_update;
		tndev->header_cache_update = ipsec_tunnel_cache_update;
	} else
		tndev->header_cache_update = NULL;

	tndev->hard_header_len = dev->hard_header_len;
	tndev->mtu = dev->mtu;
	prv->mtu = tndev->mtu;
	tndev->addr_len = dev->addr_len;
	for (i=0; i<tndev->addr_len; i++) {
		tndev->dev_addr[i] = dev->dev_addr[i];
	}
#ifdef DEBUG_IPSEC
	if(debug_tunnel) {
		printk("klips_debug:ipsec_tunnel_attach: "
		       "physical device %s being attached has HW address: %2x",
		       dev->name, dev->dev_addr[0]);
		for (i=1; i < dev->addr_len; i++) {
			printk(":%02x", dev->dev_addr[i]);
		}
		printk("\n");
	}
#endif /* DEBUG_IPSEC */

	return 0;
}

/*
 * We call the detach routine to detach the ipsec tunnel from another device.
 */

static int
ipsec_tunnel_detach(struct device *dev, struct ipsecpriv *prv)
{
        int i;

	prv->dev = NULL;
	prv->hard_start_xmit = NULL;
	prv->get_stats = NULL;

	prv->hard_header = NULL;
	dev->hard_header = NULL;
	
	prv->rebuild_header = NULL;
	dev->rebuild_header = NULL;
	
	prv->set_mac_address = NULL;
	dev->set_mac_address = NULL;
	
	prv->header_cache_bind = NULL;
	dev->header_cache_bind = NULL;

	prv->header_cache_update = NULL;
	dev->header_cache_update = NULL;

	dev->hard_header_len = 0;
	dev->mtu = 0;
	prv->mtu = 0;
	for (i=0; i<MAX_ADDR_LEN; i++) {
		dev->dev_addr[i] = 0;
	}
	dev->addr_len = 0;
	
	return 0;
}

/*
 * We call the clear routine to detach all ipsec tunnels from other devices.
 */
static int
ipsec_tunnel_clear(void)
{
	int i;
	struct device *ipsecdev = NULL, *prvdev;
	struct ipsecpriv *prv;
	char name[9];
	int ret;

	KLIPS_PRINT(debug_tunnel,
		    "klips_debug:ipsec_tunnel_clear: called.\n");

	for(i = 0; i < IPSEC_NUM_IF; i++) {
		sprintf(name, "ipsec%d", i);
		if((ipsecdev = dev_get(name)) != NULL) {
			if((prv = (struct ipsecpriv *)(ipsecdev->priv))) {
				prvdev = (struct device *)(prv->dev);
				if(prvdev) {
					KLIPS_PRINT(debug_tunnel,
						    "klips_debug:ipsec_tunnel_clear: "
						    "physical device for device %s is %s\n",
						    name, prvdev->name);
					if((ret = ipsec_tunnel_detach(ipsecdev, prv))) {
						KLIPS_PRINT(debug_tunnel,
							    "klips_debug:ipsec_tunnel_clear: "
							    "error %d detatching device %s from device %s.\n",
							    ret, name, prvdev->name);
						return ret;
					}
				}
			}
		}
	}
	return 0;
}

static int
ipsec_tunnel_ioctl(struct device *dev, struct ifreq *ifr, int cmd)
{
	struct ipsectunnelconf *cf = (struct ipsectunnelconf *)&ifr->ifr_data;
	struct ipsecpriv *prv = dev->priv;
	struct device *them = dev_get(cf->cf_name); /* physical device */
	
	KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
		    "klips_debug:ipsec_tunnel_ioctl: "
		    "tncfg service call #%d\n", cmd);
	switch (cmd) {
	/* attatch a virtual ipsec? device to a physical device */
	case IPSEC_SET_DEV:
		if (them == NULL) {
			KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
				    "klips_debug:ipsec_tunnel_ioctl: "
				    "physical device requested is null\n");

			return -ENXIO;
		}
		
		if (prv->dev)
			return -EBUSY;
		return ipsec_tunnel_attach(dev, dev->priv, them);

	case IPSEC_DEL_DEV:
		if (! prv->dev)
			return -ENODEV;
		return ipsec_tunnel_detach(dev, dev->priv);
	       
	case IPSEC_CLR_DEV:
		KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
			    "klips_debug:ipsec_tunnel_ioctl: "
			    "calling ipsec_tunnel_clear.\n");
		return ipsec_tunnel_clear();

	default:
		return -EOPNOTSUPP;
	}
}

/*
 *	Called when an ipsec tunnel device is initialized.
 *	The ipsec tunnel device structure is passed to us.
 */
 
int
ipsec_tunnel_init(struct device *dev)
{
	int i;

	printk(KERN_INFO "klips_debug:ipsec_tunnel_init: "
	       "initialisation of device: %s\n",
	       dev->name ? dev->name : "NULL");

	/* Add our tunnel functions to the device */
	dev->open		= ipsec_tunnel_open;
	dev->stop		= ipsec_tunnel_close;
	dev->hard_start_xmit	= ipsec_tunnel_start_xmit;
	dev->get_stats		= ipsec_tunnel_get_stats;

	dev->priv = kmalloc(sizeof(struct ipsecpriv), GFP_KERNEL);
	if (dev->priv == NULL)
		return -ENOMEM;
	memset(dev->priv, 0, sizeof(struct ipsecpriv));

	for(i = 0; i < sizeof(zeroes); i++) {
		((__u8*)(zeroes))[i] = 0;
	}
	
	/* Initialize the tunnel device structure */
	for (i = 0; i < DEV_NUMBUFFS; i++)
		skb_queue_head_init(&dev->buffs[i]);

	dev->set_multicast_list = NULL;
	dev->do_ioctl		= ipsec_tunnel_ioctl;
	dev->hard_header	= NULL;
	dev->rebuild_header 	= NULL;
	dev->set_mac_address 	= NULL;
	dev->header_cache_bind 	= NULL;
	dev->header_cache_update= NULL;

	dev->hard_header_len 	= 0;
	dev->mtu		= 0;
	dev->addr_len		= 0;
	dev->type		= ARPHRD_TUNNEL;	/* initially */
	dev->tx_queue_len	= 10;		/* Small queue */
	memset(dev->broadcast,0xFF, ETH_ALEN);	/* what if this is not attached to ethernet? */

	/* New-style flags. */
	dev->flags		= IFF_NOARP /* 0 */ /* Petr Novak */;
	dev->family		= AF_INET;
	dev->pa_addr		= 0;
	dev->pa_brdaddr 	= 0;
	dev->pa_mask		= 0;
	dev->pa_alen		= 4;

	/* We're done.  Have I forgotten anything? */
	return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*  Module specific interface (but it links with the rest of IPSEC  */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int
ipsec_tunnel_probe(struct device *dev)
{
	ipsec_tunnel_init(dev); 
	return 0;
}

static struct device dev_ipsec1 = 
{
	"ipsec1\0   ",		/* name */
	0,			/* recv memory end */
	0,			/* recv memory start */
	0,			/* memory end */
	0,			/* memory start */
 	0x0,			/* base I/O address */
	0,			/* IRQ */
	0, 0, 0,		/* flags */
	NULL,			/* next device */
	ipsec_tunnel_probe	/* setup */
};

static struct device dev_ipsec0 = 
{
	"ipsec0\0   ",		/* name */
	0,			/* recv memory end */
	0,			/* recv memory start */
	0,			/* memory end */
	0,			/* memory start */
 	0x0,			/* base I/O address */
	0,			/* IRQ */
	0, 0, 0,		/* flags */
	NULL,			/* next device */
	ipsec_tunnel_probe	/* setup */
};

int 
ipsec_tunnel_init_module(void)
{
	KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
		    "klips_debug:ipsec_tunnel_init_module: "
		    "registering device %s\n",
		    dev_ipsec0.name);
	KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
		    "klips_debug:ipsec_tunnel_init_module: "
		    "registering device %s\n",
		    dev_ipsec1.name);
	if (register_netdev(&dev_ipsec0) != 0)
		return -EIO;
	if (register_netdev(&dev_ipsec1) != 0)
		return -EIO;
	return 0;
}

void 
ipsec_tunnel_cleanup_module(void)
{
	unregister_netdev(&dev_ipsec0);
	unregister_netdev(&dev_ipsec1);
	kfree_s(dev_ipsec0.priv,sizeof(struct ipsecpriv));
	kfree_s(dev_ipsec1.priv,sizeof(struct ipsecpriv));
	dev_ipsec0.priv=NULL;
	dev_ipsec1.priv=NULL;
}

/*
 * $Log: ipsec_tunnel.c,v $
 * Revision 1.46  1999/04/11 00:29:00  henry
 * GPL boilerplate
 *
 * Revision 1.45  1999/04/07 15:42:01  rgb
 * Fix mtu/ping bug AGAIN!
 *
 * Revision 1.44  1999/04/06 04:54:27  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.43  1999/04/04 03:57:07  rgb
 * ip_fragment() doesn't free the supplied skb.  Freed.
 *
 * Revision 1.42  1999/04/01 23:27:15  rgb
 * Preload size of virtual mtu.
 *
 * Revision 1.41  1999/04/01 09:31:23  rgb
 * Invert meaning of ICMP PMTUD config option and clarify.
 * Code clean-up.
 *
 * Revision 1.40  1999/04/01 04:37:17  rgb
 * SSH stalling bug fix.
 *
 * Revision 1.39  1999/03/31 23:44:28  rgb
 * Don't send ICMP on DF and frag_off.
 *
 * Revision 1.38  1999/03/31 15:20:10  rgb
 * Quiet down debugging.
 *
 * Revision 1.37  1999/03/31 08:30:31  rgb
 * Add switch to shut off ICMP PMTUD packets.
 *
 * Revision 1.36  1999/03/31 05:44:47  rgb
 * Keep PMTU reduction private.
 *
 * Revision 1.35  1999/03/27 15:13:02  rgb
 * PMTU/fragmentation bug fix.
 *
 * Revision 1.34  1999/03/17 21:19:26  rgb
 * Fix kmalloc nonatomic bug.
 *
 * Revision 1.33  1999/03/17 15:38:42  rgb
 * Code clean-up.
 * ESP_NULL IV bug fix.
 *
 * Revision 1.32  1999/03/01 20:44:25  rgb
 * Code clean-up.
 * Memory leak bug fix.
 *
 * Revision 1.31  1999/02/27 00:02:09  rgb
 * Tune to report the MTU reduction once, rather than after every recursion
 * through the encapsulating code, preventing tcp stream stalling.
 *
 * Revision 1.30  1999/02/24 20:21:01  rgb
 * Reformat debug printk's.
 * Fix recursive encapsulation, dynamic MTU bugs and add debugging code.
 * Clean-up.
 *
 * Revision 1.29  1999/02/22 17:08:14  rgb
 * Fix recursive encapsulation code.
 *
 * Revision 1.28  1999/02/19 18:27:02  rgb
 * Improve DF, fragmentation and PMTU behaviour and add dynamic MTU discovery.
 *
 * Revision 1.27  1999/02/17 16:51:37  rgb
 * Clean out unused cruft.
 * Temporarily tone down volume of debug output.
 * Temporarily shut off fragment rejection.
 * Disabled temporary failed recursive encapsulation loop.
 *
 * Revision 1.26  1999/02/12 21:21:26  rgb
 * Move KLIPS_PRINT to ipsec_netlink.h for accessibility.
 *
 * Revision 1.25  1999/02/11 19:38:27  rgb
 * More clean-up.
 * Add sanity checking for skb_copy_expand() to prevent kernel panics on
 * skb_put() values out of range.
 * Fix head/tailroom calculation causing skb_put() out-of-range values.
 * Fix return values to prevent 'nonatomic alloc_skb' warnings.
 * Allocate new skb iff needed.
 * Added more debug statements.
 * Make headroom depend on structure, not hard-coded values.
 *
 * Revision 1.24  1999/02/10 23:20:33  rgb
 * Shut up annoying 'statement has no effect' compiler warnings with
 * debugging compiled out.
 *
 * Revision 1.23  1999/02/10 22:36:30  rgb
 * Clean-up obsolete, unused and messy code.
 * Converted most IPSEC_DEBUG statements to KLIPS_PRINT macros.
 * Rename ipsec_tunnel_do_xmit to ipsec_tunnel_start_xmit and eliminated
 * original ipsec_tunnel_start_xmit.
 * Send all packet with different inner and outer destinations directly to
 * the attached physical device, rather than back through ip_forward,
 * preventing disappearing routes problems.
 * Do sanity checking before investing too much CPU in allocating new
 * structures.
 * Fail on IP header options: We cannot process them yet.
 * Add some helpful comments.
 * Use virtual device for parameters instead of physical device.
 *
 * Revision 1.22  1999/02/10 03:03:02  rgb
 * Duh.  Fixed the TTL bug: forgot to update the checksum.
 *
 * Revision 1.21  1999/02/09 23:17:53  rgb
 * Add structure members to ipsec_print_ip debug function.
 * Temporarily fix TTL bug preventing tunnel mode from functioning.
 *
 * Revision 1.20  1999/02/09 00:14:25  rgb
 * Add KLIPSPRINT macro.  (Not used yet, though.)
 * Delete old ip_tunnel code (BADCODE).
 * Decrement TTL in outgoing packet.
 * Set TTL on new IPIP_TUNNEL to default, not existing packet TTL.
 * Delete ethernet only feature and fix hard-coded hard_header_len.
 *
 * Revision 1.19  1999/01/29 17:56:22  rgb
 * 64-bit re-fix submitted by Peter Onion.
 *
 * Revision 1.18  1999/01/28 22:43:24  rgb
 * Fixed bug in ipsec_print_ip that caused an OOPS, found by P.Onion.
 *
 * Revision 1.17  1999/01/26 02:08:16  rgb
 * Removed CONFIG_IPSEC_ALGO_SWITCH macro.
 * Removed dead code.
 *
 * Revision 1.16  1999/01/22 06:25:26  rgb
 * Cruft clean-out.
 * Added algorithm switch code.
 * 64-bit clean-up.
 * Passthrough on IPIP protocol, spi 0x0 fix.
 * Enhanced debugging.
 *
 * Revision 1.15  1998/12/01 13:22:04  rgb
 * Added support for debug printing of version info.
 *
 * Revision 1.14  1998/11/30 13:22:55  rgb
 * Rationalised all the klips kernel file headers.  They are much shorter
 * now and won't conflict under RH5.2.
 *
 * Revision 1.13  1998/11/17 21:13:52  rgb
 * Put IKE port bypass debug output in user-switched debug statements.
 *
 * Revision 1.12  1998/11/13 13:20:25  rgb
 * Fixed ntohs bug in udp/500 hole for IKE.
 *
 * Revision 1.11  1998/11/10 08:01:19  rgb
 * Kill tcp/500 hole,  keep udp/500 hole.
 *
 * Revision 1.10  1998/11/09 21:29:26  rgb
 * If no eroute is found, discard packet and incr. tx_error.
 *
 * Revision 1.9  1998/10/31 06:50:00  rgb
 * Add tcp/udp/500 bypass.
 * Fixed up comments in #endif directives.
 *
 * Revision 1.8  1998/10/27 00:34:31  rgb
 * Reformat debug output of IP headers.
 * Newlines added before calls to ipsec_print_ip.
 *
 * Revision 1.7  1998/10/19 14:44:28  rgb
 * Added inclusion of freeswan.h.
 * sa_id structure implemented and used: now includes protocol.
 *
 * Revision 1.6  1998/10/09 04:31:35  rgb
 * Added 'klips_debug' prefix to all klips printk debug statements.
 *
 * Revision 1.5  1998/08/28 03:09:51  rgb
 * Prevent kernel log spam with default route through ipsec.
 *
 * Revision 1.4  1998/08/05 22:23:09  rgb
 * Change setdev return code to ENXIO for a non-existant physical device.
 *
 * Revision 1.3  1998/07/29 20:41:11  rgb
 * Add ipsec_tunnel_clear to clear all tunnel attachments.
 *
 * Revision 1.2  1998/06/25 20:00:33  rgb
 * Clean up #endif comments.
 * Rename dev_ipsec to dev_ipsec0 for consistency.
 * Document ipsec device fields.
 * Make ipsec_tunnel_probe visible from rest of kernel for static linking.
 * Get debugging report for *every* ipsec device initialisation.
 * Comment out redundant code.
 *
 * Revision 1.1  1998/06/18 21:27:50  henry
 * move sources from klips/src to klips/net/ipsec, to keep stupid
 * kernel-build scripts happier in the presence of symlinks
 *
 * Revision 1.8  1998/06/14 23:49:40  rgb
 * Clarify version reporting on module loading.
 *
 * Revision 1.7  1998/05/27 23:19:20  rgb
 * Added version reporting.
 *
 * Revision 1.6  1998/05/18 21:56:23  rgb
 * Clean up for numerical consistency of output and cleaning up debug code.
 *
 * Revision 1.5  1998/05/12 02:44:23  rgb
 * Clarifying 'no e-route to host' message.
 *
 * Revision 1.4  1998/04/30 15:34:35  rgb
 * Enclosed most remaining debugging statements in #ifdef's to make it quieter.
 *
 * Revision 1.3  1998/04/21 21:28:54  rgb
 * Rearrange debug switches to change on the fly debug output from user
 * space.  Only kernel changes checked in at this time.  radij.c was also
 * changed to temporarily remove buggy debugging code in rj_delete causing
 * an OOPS and hence, netlink device open errors.
 *
 * Revision 1.2  1998/04/12 22:03:24  rgb
 * Updated ESP-3DES-HMAC-MD5-96,
 * 	ESP-DES-HMAC-MD5-96,
 * 	AH-HMAC-MD5-96,
 * 	AH-HMAC-SHA1-96 since Henry started freeswan cvs repository
 * from old standards (RFC182[5-9] to new (as of March 1998) drafts.
 *
 * Fixed eroute references in /proc/net/ipsec*.
 *
 * Started to patch module unloading memory leaks in ipsec_netlink and
 * radij tree unloading.
 *
 * Revision 1.1  1998/04/09 03:06:12  henry
 * sources moved up from linux/net/ipsec
 *
 * Revision 1.1.1.1  1998/04/08 05:35:04  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.5  1997/06/03 04:24:48  ji
 * Added transport mode.
 * Changed the way routing is done.
 * Lots of bug fixes.
 *
 * Revision 0.4  1997/01/15 01:28:15  ji
 * No changes.
 *
 * Revision 0.3  1996/11/20 14:39:04  ji
 * Minor cleanups.
 * Rationalized debugging code.
 *
 * Revision 0.2  1996/11/02 00:18:33  ji
 * First limited release.
 *
 *
 */
