/*
 * receive code
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char ipsec_rcv_c_version[] = "RCSID $Id: ipsec_rcv.c,v 1.7 1999/04/11 00:28:59 henry Exp $";

#include <linux/config.h>

#define __NO_VERSION__
#include <linux/module.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <asm/checksum.h>
#include <net/ip.h>

#include "../../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_tunnel.h"
#include "ipsec_rcv.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#ifdef DEBUG_IPSEC
int debug_ah = 0;
int debug_esp = 0;
int debug_rcv = 0;
#endif

extern void des_cbc_encrypt(caddr_t, caddr_t, int, caddr_t, caddr_t, int);
extern void des_ede3_cbc_encrypt(caddr_t, caddr_t, int, caddr_t, caddr_t, caddr_t, caddr_t, int);
__u32 zeroes[AH_AMAX];

/*
 * Check-replay-window routine, adapted from the original 
 * by J. Hughes, from draft-ietf-ipsec-esp-des-md5-03.txt
 *
 *  This is a routine that implements a 64 packet window. This is intend-
 *  ed on being an implementation sample.
 */

static int
ipsec_checkreplaywindow(struct tdb*tdbp, __u32 seq)
{
	__u32 diff;
	
	if (tdbp->tdb_replaywin == 0)	/* replay shut off */
		return 1;
	if (seq == 0) 
		return 0;		/* first == 0 or wrapped */

	/* new larger sequence number */
	if (seq > tdbp->tdb_replaywin_lastseq) {
		diff = seq - tdbp->tdb_replaywin_lastseq;

		/* In win, set bit for this pkt */
		if (diff < tdbp->tdb_replaywin)
			tdbp->tdb_replaywin_bitmap =
				(tdbp->tdb_replaywin_bitmap << diff) | 1;
		else
			/* This packet has way larger seq num */
			tdbp->tdb_replaywin_bitmap = 1;
		
		tdbp->tdb_replaywin_lastseq = seq;
		return 1;		/* larger is good */
	}
	diff = tdbp->tdb_replaywin_lastseq - seq;

	/* too old or wrapped */ /* if wrapped, kill off SA? */
	if (diff >= tdbp->tdb_replaywin)
		return 0;
	
	/* this packet already seen */
	if (tdbp->tdb_replaywin_bitmap & (1 << diff))
		return 0;
	tdbp->tdb_replaywin_bitmap |= (1 << diff);	/* mark as seen */
	return 1;			/* out of order but good */
}

int
ipsec_rcv(struct sk_buff *skb, struct device *dev, struct options *opt, 
		__u32 daddr, unsigned short xlen, __u32 saddr,
                                   int redo, struct inet_protocol *protocol)
{
	struct iphdr *ipp;
	struct esp *espp = NULL;
	struct ah *ahp = NULL;
	int iphlen;
	unsigned char *dat;
	struct tdb *tdbp;
	struct sa_id said;
	struct enet_statistics *stats = NULL;		/* This device's statistics */
	struct device *ipsecdev = NULL, *prvdev;
	struct ipsecpriv *prv;
	char name[9];
	char sa[SATOA_BUF];
	size_t sa_len;
	char ipaddr_txt[16];
	int i;
	struct in_addr ipaddr;
	__u8 next_header = 0;
	__u8 proto;
	
	int esphlen = 0, ahhlen = 0, authlen = 0, pad = 0, ilen, padlen, len;
	union {
		MD5_CTX		md5;
		SHA1_CTX	sha1;
	} tctx;
	__u8 hash[AH_AMAX];
	__u8 *idat;
	__u32 iv[2];
	struct iphdr ipo;

	/* Don't unlink in the middle of a turnaround */
	MOD_INC_USE_COUNT;
	
	KLIPS_PRINT(debug_rcv, "klips_debug:ipsec_rcv: Info -- ");
	KLIPS_PRINT(debug_rcv  && skb->dev, "sdb->dev=%s ", skb->dev->name);
	KLIPS_PRINT(debug_rcv && dev, "dev=%s ", dev->name);
	KLIPS_PRINT(debug_rcv, "\n");

	KLIPS_PRINT(debug_rcv && !(skb->dev && dev && (skb->dev == dev)),
		    "klips_debug:ipsec_rcv: Informational -- "
		    "**if this happens, find out why** "
		    "skb->dev:%s is not equal to dev:%s\n",
		    skb->dev->name, dev->name);
	if( (protocol->protocol != IPPROTO_AH) &&
	    (protocol->protocol != IPPROTO_ESP) ) {
		KLIPS_PRINT(debug_rcv & DB_RX_TDB,
			    "klips_debug:ipsec_rcv: Why the hell is someone "
			    "passing me a non-ipsec packet? -- dropped.\n");
		goto rcvleave;
	}

	for(i = 0; i < IPSEC_NUM_IF; i++) {
		sprintf(name, "ipsec%d", i);
		if(!strcmp(name, skb->dev->name)) {
			prv = (struct ipsecpriv *)(skb->dev->priv);
			stats = (struct enet_statistics *) &(prv->mystats);
			ipsecdev = skb->dev;
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: Info -- pkt "
				    "already proc'ed an ipsec header, "
				    "proc'ing next ipsec header.\n");
			break;
		}
		if((ipsecdev = dev_get(name)) == NULL) {
			KLIPS_PRINT(debug_rcv,
				    "klips_error:ipsec_rcv: device %s does "
				    "not exist\n", name);
		}
		prv = ipsecdev ? (struct ipsecpriv *)(ipsecdev->priv) : NULL;
		prvdev = prv ? (struct device *)(prv->dev) : NULL;

		KLIPS_PRINT(debug_rcv && prvdev, 
			    "klips_debug:ipsec_rcv: physical device"
			    " for device %s is %s\n",
			    name, prvdev->name);
		KLIPS_PRINT(debug_rcv && !skb->dev, 
			    "klips_debug:ipsec_rcv: device supplied"
			    " with skb is NULL\n");

		if(prvdev && skb->dev &&
		   !strcmp(prvdev->name, skb->dev->name)) {
			stats = (struct enet_statistics *) &(prv->mystats);
			skb->dev = ipsecdev;
			stats->rx_packets++;
			break;
		}
	}
	KLIPS_PRINT(!stats,
		    "klips_error:ipsec_rcv: packet received from physical "
		    "I/F (%s) not connected to ipsec I/F.  Cannot record "
		    "stats.\n", skb->dev->name);
	KLIPS_PRINT(!stats,
		    "klips_error          May not have SA for decoding.  "
		    "Is IPSEC traffic expected on this I/F?  "
		    "Check routing.\n");

#ifdef DEBUG_IPSEC
	if (debug_tunnel & DB_TN_REVEC) {
		struct timeval tv;
		do_gettimeofday(&tv);
		printk("klips_debug:ipsec_rcv: ts=%02d.%04d %s packet received,"
		       " dev = %s\n", 
		       tv.tv_sec % 60, tv.tv_usec / 100,
		       protocol->protocol == IPPROTO_AH ? "AH" : "ESP",
		       ipsecdev ? ipsecdev->name : "NULL");
	}
#endif /* DEBUG_IPSEC */

	/* begin decapsulating loop here */
	do {
		espp = NULL;
		ahp = NULL;

		esphlen = 0;
		ahhlen = 0;
		authlen = 0;
		
		len = skb->len;
		dat = skb->data;
		ipp = (struct iphdr *)skb->data;
		proto = ipp->protocol;
		ipaddr.s_addr = ipp->saddr;
		addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
		
		KLIPS_PRINT(debug_rcv, "klips_debug:ipsec_rcv: old \n");
		KLIPS_IP_PRINT(debug_rcv, ipp);
		
		iphlen = ipp->ihl << 2;
		ipp->check = 0;			/* we know the sum is good */
		
		if ((len - iphlen) % 4) {
			printk("klips_error:ipsec_rcv: got packet with content length = %d from "
			       "%s -- should be on 4 octet boundary, packet dropped\n",
			       len - iphlen, ipaddr_txt);	/* XXX this will need to be 8 for IPv6 */
			goto rcvleave;
		}
		
		/*
		 * Find tunnel control block and (indirectly) call the appropriate
		 * tranform routine. The resulting sk_buf is a valid
		 * IP packet ready to go through input processing.
		 */
		
		said.dst.s_addr = ipp->daddr;
		if(proto == IPPROTO_ESP) {
			espp = (struct esp *)(skb->data + iphlen);
			said.spi = espp->esp_spi;
		} else {
			ahp = (struct ah *)(skb->data + iphlen);
			said.spi = ahp->ah_spi;
		}
		said.proto = proto;
		
		if(proto == IPPROTO_AH) {
			ahhlen = (ahp->ah_hl << 2) +
				((caddr_t)&(ahp->ah_rpl) - (caddr_t)ahp);
			next_header = ahp->ah_nh;
			if (ahhlen != sizeof(struct ah)) {
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: bad "
					    "authenticator length %d, expected "
					    "%d from %s\n",
					    ahhlen - ((caddr_t)(ahp->ah_data) -
						      (caddr_t)ahp),
					    AHHMAC_HASHLEN,
					    ipaddr_txt);
				goto rcvleave;
			}
			
		}
		
		sa_len = satoa(said, 0, sa, SATOA_BUF);
		tdbp = gettdb(said);
		if (tdbp == NULL) {
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: no Tunnel Descriptor "
				    "Block for SA:%s: incoming packet with no SA "
				    "dropped\n", sa);
			goto rcvleave;
		}
		
		/* authenticate, if required */
		idat = dat + iphlen;
		switch(tdbp->tdb_authalg) {
		case AH_MD5:
		case AH_SHA:
			authlen = AHHMAC_HASHLEN;
			break;
		default:
			authlen = 0;
		}
		ilen = len - iphlen - authlen;
		
		KLIPS_PRINT(debug_rcv, 
			    "klips_debug:ipsec_rcv: packet from %s received with"
			    " seq=%ld (iv)=0x%08lx%08lx iplen=%d esplen=%d sa=%s\n",
			    ipaddr_txt,
			    ntohl(espp->esp_rpl),
			    ntohl(*((__u32 *)(espp->esp_iv)    )),
			    ntohl(*((__u32 *)(espp->esp_iv) + 1)),
			    len, ilen, sa);
		
		if (!ipsec_checkreplaywindow(tdbp,
					     ntohl(proto == IPPROTO_ESP ?
						   espp->esp_rpl : ahp->ah_rpl))) {
			KLIPS_PRINT(debug_rcv & DB_RX_REPLAY,
				    "klips_debug:ipsec_rcv: duplicate frame from %s,"
				    " packet dropped\n",
				    ipaddr_txt);
			goto rcvleave;
		}
		
		/*
		 * verify authenticator
		 */
		
		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: encalg = %d, authalg = %d.\n",
			    tdbp->tdb_encalg, tdbp->tdb_authalg);
		
		if(tdbp->tdb_authalg) {
			switch(tdbp->tdb_authalg) {
			case AH_MD5:
				tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->ictx;
				if(proto == IPPROTO_ESP) {
					MD5Update(&tctx.md5, (caddr_t)espp, ilen);
				} else {
					ipo = *ipp;
					ipo.tos = 0;
					ipo.frag_off = 0;
					ipo.ttl = 0;
					ipo.check = 0;
					
					MD5Update(&tctx.md5, (caddr_t)&ipo,
						  sizeof(struct iphdr));
					MD5Update(&tctx.md5, (caddr_t)ahp,
						  ahhlen - AHHMAC_HASHLEN);
					MD5Update(&tctx.md5, (caddr_t)zeroes,
						  AHHMAC_HASHLEN);
					MD5Update(&tctx.md5,
						  (caddr_t)dat + iphlen + ahhlen,
						  len - iphlen - ahhlen);
				}
				MD5Final(hash, &tctx.md5);
				tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->octx;
				MD5Update(&tctx.md5, hash, AHMD596_ALEN);
				MD5Final(hash, &tctx.md5);
				break;
			case AH_SHA:
				tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx;
				if(proto == IPPROTO_ESP) {
					SHA1Update(&tctx.sha1, (caddr_t)espp, ilen);
				} else {
					ipo = *ipp;
					ipo.tos = 0;
					ipo.frag_off = 0;
					ipo.ttl = 0;
					ipo.check = 0;
					
					SHA1Update(&tctx.sha1, (caddr_t)&ipo,
						   sizeof(struct iphdr));
					SHA1Update(&tctx.sha1, (caddr_t)ahp,
						   ahhlen - AHHMAC_HASHLEN);
					SHA1Update(&tctx.sha1, (caddr_t)zeroes,
						   AHHMAC_HASHLEN);
					SHA1Update(&tctx.sha1,
						   (caddr_t)dat + iphlen + ahhlen,
						   len - iphlen - ahhlen);
				}
				SHA1Final(hash, &tctx.sha1);
				tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->octx;
				SHA1Update(&tctx.sha1, hash, AHSHA196_ALEN);
				SHA1Final(hash, &tctx.sha1);
				break;
			default:
			}
			
			if (memcmp(hash, proto == IPPROTO_ESP ?
				   &(dat[len - authlen]) : ahp->ah_data,
				   authlen)) {
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: auth failed on incoming "
					    "packet from %s: hash=%08x%08x%08x "
					    "auth=%08x%08x%08x, dropped\n", ipaddr_txt,
					    *(__u32*)&hash[0],
					    *(__u32*)&hash[4],
					    *(__u32*)&hash[8],
					    proto == IPPROTO_ESP ?
					    *(__u32*)&dat[len - 12] : *(__u32*)(ahp->ah_data),
					    proto == IPPROTO_ESP ?
					    *(__u32*)&dat[len - 8] : *&((__u32*)(ahp->ah_data))[4],
					    proto == IPPROTO_ESP ?
					    *(__u32*)&dat[len - 4] : *&((__u32*)(ahp->ah_data))[8]);
				goto rcvleave;
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: authentication successful.\n");
			}
			
			memset((caddr_t)&tctx, 0, sizeof(tctx));
			memset(hash, 0, sizeof(hash));
		}
		
		if(proto == IPPROTO_ESP) {
			switch(tdbp->tdb_encalg) {
			case ESP_DES:
			case ESP_3DES:
				iv[0] = *((__u32 *)(espp->esp_iv)    );
				iv[1] = *((__u32 *)(espp->esp_iv) + 1);
				esphlen = sizeof(struct esp);
				break;
			case ESP_NULL:
				esphlen = offsetof(struct esp, esp_iv);
				break;
			}
			idat += esphlen;
			ilen -= esphlen;
			
			switch(tdbp->tdb_encalg) {
			case ESP_DES:
				if ((ilen) % 8) {
					printk("klips_error:ipsec_rcv: got packet with esplen = %d from "
					       "%s -- should be on 8 octet boundary, packet dropped\n",
					       ilen, ipaddr_txt);
					goto rcvleave;
				}
				des_cbc_encrypt(idat, idat, ilen,
						tdbp->tdb_key_e,
						(caddr_t)iv, 0);
				break;
			case ESP_3DES:
				if ((ilen) % 8) {
					printk("klips_error:ipsec_rcv: got packet with esplen = %d from "
					       "%s -- should be on 8 octet boundary, packet dropped\n",
					       ilen, ipaddr_txt);
					goto rcvleave;
				}
				des_ede3_cbc_encrypt(idat, idat, ilen,
						     tdbp->tdb_key_e,
						     tdbp->tdb_key_e + sizeof(struct des_eks),
						     tdbp->tdb_key_e + 2 * sizeof(struct des_eks),
						     (caddr_t)iv, 0);
				break;
			case ESP_NULL:
				break;
			}
			next_header = idat[ilen - 1];
			padlen = idat[ilen - 2];
			pad = padlen + 2 + authlen;
#ifdef DEBUG_IPSEC
			if(debug_rcv & DB_RX_IPAD) {
				printk("klips_debug:ipsec_rcv: padlen=%d, contents: ",
				       padlen);
				
				for (i = 1; i <= padlen; i++) {
					printk(" %d:%d", i,
					       idat[ilen - 2 - padlen + i - 1]);
					if(i != idat[ilen - 2 - padlen + i - 1]) {
						printk("\nklips_debug:ipsec_rcv: warning, decrypted packet from %s has bad padding\n",
						       ipaddr_txt);
						printk("\nklips_debug:ipsec_rcv: ...may be bad decryption -- not dropped\n");
					} 
				}
				printk("\n");
				
				printk("klips_debug:ipsec_rcv: packet decrypted from "
				       "%s: next_header = %d, padding = %d\n",
				       ipaddr_txt,
				       next_header,
				       pad - 2 - authlen);
			}
#endif
			
		}
		
		/*
		 *	Discard the original IP header
		 */
		
		KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
			    "klips_debug:ipsec_rcv: proto = %d, protocol->protocol = %d,"
			    " tdbp->tdb_proto = %d, esphlen = %d, ahhlen = %d, len = %d.\n",
			    proto, protocol->protocol, tdbp->tdb_proto, esphlen, ahhlen, len);
		ipp->tot_len = htons(ntohs(ipp->tot_len) -
				     (proto == IPPROTO_ESP ?
				      (esphlen + pad) : ahhlen));
		ipp->protocol = next_header;
		
		memmove((void *)(skb->data + ((proto == IPPROTO_ESP) ? esphlen : ahhlen)),
			(void *)(skb->data), iphlen);
		
		skb_pull(skb, proto == IPPROTO_ESP ? esphlen : ahhlen);
		if(proto == IPPROTO_ESP) {
			KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
				    "klips_debug:ipsec_rcv: trimming to %d.\n",
				    len - esphlen - pad);
			skb_trim(skb, len - esphlen - pad);
		}
		
		/*
		 *	Adjust pointers
		 */
		
		len = skb->len;
		dat = skb->data;
		
		skb->h.iph=(struct iphdr *)skb->data;
		skb->ip_hdr=(struct iphdr *)skb->data;
		
		memset(skb->proto_priv, 0, sizeof(struct options));
		
		ipp = (struct iphdr *)dat;
		ipp->check = 0;
		ipp->check = ip_fast_csum((unsigned char *)dat, iphlen >> 2);
		
		if (skb == NULL) {
			goto rcvleave;
		}
		
		KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
			    "klips_debug:ipsec_rcv: new \n");
		KLIPS_IP_PRINT(debug_rcv & DB_RX_PKTRX, ipp);
		
		skb->protocol = htons(ETH_P_IP);
		skb->ip_summed = 0;
	} while((ipp->protocol == IPPROTO_ESP) ||
		(ipp->protocol == IPPROTO_AH));
	/* end decapsulation loop here */
	
	if(ipp->protocol == IPPROTO_IPIP) {
/*		skb->mac.raw = skb->data; */
		skb_pull(skb, iphlen);

		skb->h.iph=(struct iphdr *)skb->data;
		skb->ip_hdr=(struct iphdr *)skb->data;
		memset(skb->proto_priv, 0, sizeof(struct options));

		skb->protocol = htons(ETH_P_IP);
		skb->ip_summed = 0;
		KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
			    "klips_debug:ipsec_rcv: IPIP tunnel stripped.\n");
		KLIPS_IP_PRINT(debug_rcv & DB_RX_PKTRX, skb->ip_hdr);
	}

	netif_rx(skb);

	MOD_DEC_USE_COUNT;
	return(0);

     rcvleave:
	if(stats) {
		stats->rx_packets--;
		stats->rx_errors++;
	}
	
 	if(skb) {
                kfree_skb(skb, FREE_WRITE);
	}

	MOD_DEC_USE_COUNT;
	return(0);
}

#ifdef CONFIG_IPSEC_MODULE
struct inet_protocol ah_protocol =
{
	ipsec_rcv,				/* AH handler */
	NULL,				/* TUNNEL error control */
	0,				/* next */
	IPPROTO_AH,			/* protocol ID */
	0,				/* copy */
	NULL,				/* data */
	"AH"				/* name */
};

struct inet_protocol esp_protocol = 
{
	ipsec_rcv,			/* ESP handler          */
	NULL,				/* TUNNEL error control */
	0,				/* next */
	IPPROTO_ESP,			/* protocol ID */
	0,				/* copy */
	NULL,				/* data */
	"ESP"				/* name */
};
#endif /* CONFIG_IPSEC_MODULE */

/*
 * $Log: ipsec_rcv.c,v $
 * Revision 1.7  1999/04/11 00:28:59  henry
 * GPL boilerplate
 *
 * Revision 1.6  1999/04/06 04:54:27  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.5  1999/03/17 15:39:23  rgb
 * Code clean-up.
 * Bundling bug fix.
 * ESP_NULL esphlen and IV bug fix.
 *
 * Revision 1.4  1999/02/17 16:51:02  rgb
 * Ditch NET_IPIP dependancy.
 * Decapsulate recursively for an entire bundle.
 *
 * Revision 1.3  1999/02/12 21:22:47  rgb
 * Convert debugging printks to KLIPS_PRINT macro.
 * Clean-up cruft.
 * Process IPIP tunnels internally.
 *
 * Revision 1.2  1999/01/26 02:07:36  rgb
 * Clean up debug code when switched off.
 * Remove references to INET_GET_PROTOCOL.
 *
 * Revision 1.1  1999/01/21 20:29:11  rgb
 * Converted from transform switching to algorithm switching.
 *
 *
 * Id: ipsec_esp.c,v 1.16 1998/12/02 03:08:11 rgb Exp $
 *
 * Log: ipsec_esp.c,v $
 * Revision 1.16  1998/12/02 03:08:11  rgb
 * Fix incoming I/F bug in AH and clean up inconsistencies in the I/F
 * discovery routine in both AH and ESP.
 *
 * Revision 1.15  1998/11/30 13:22:51  rgb
 * Rationalised all the klips kernel file headers.  They are much shorter
 * now and won't conflict under RH5.2.
 *
 * Revision 1.14  1998/11/10 05:55:37  rgb
 * Add even more detail to 'wrong I/F' debug statement.
 *
 * Revision 1.13  1998/11/10 05:01:30  rgb
 * Clean up debug output to be quiet when disabled.
 * Add more detail to 'wrong I/F' debug statement.
 *
 * Revision 1.12  1998/10/31 06:39:32  rgb
 * Fixed up comments in #endif directives.
 * Tidied up debug printk output.
 * Convert to addrtoa and satoa where possible.
 *
 * Revision 1.11  1998/10/27 00:49:30  rgb
 * AH+ESP bundling bug has been squished.
 * Cosmetic brace fixing in code.
 * Newlines added before calls to ipsec_print_ip.
 * Fix debug output function ID's.
 *
 * Revision 1.10  1998/10/22 06:37:22  rgb
 * Fixed run-on error message to fit 80 columns.
 *
 * Revision 1.9  1998/10/20 02:41:04  rgb
 * Fixed a replay window size sanity test bug.
 *
 * Revision 1.8  1998/10/19 18:55:27  rgb
 * Added inclusion of freeswan.h.
 * sa_id structure implemented and used: now includes protocol.
 * \n bugfix to printk debug message.
 *
 * Revision 1.7  1998/10/09 04:23:03  rgb
 * Fixed possible DoS caused by invalid transform called from an ESP
 * packet.  This should not be a problem when protocol is added to the SA.
 * Sanity check added for null xf_input routine.  Sanity check added for null
 * socket buffer returned from xf_input routine.
 * Added 'klips_debug' prefix to all klips printk debug statements.
 *
 * Revision 1.6  1998/07/14 15:56:04  rgb
 * Set sdb->dev to virtual ipsec I/F.
 *
 * Revision 1.5  1998/06/30 18:07:46  rgb
 * Change for ah/esp_protocol stuct visible only if module.
 *
 * Revision 1.4  1998/06/30 00:12:46  rgb
 * Clean up a module compile error.
 *
 * Revision 1.3  1998/06/25 19:28:06  rgb
 * Readjust premature unloading of module on packet receipt.
 * Make protocol structure abailable to rest of kernel.
 * Use macro for protocol number.
 *
 * Revision 1.2  1998/06/23 02:49:34  rgb
 * Fix minor #include bug that prevented compiling without debugging.
 * Added code to check for presence of IPIP protocol if an incoming packet
 * is IPIP encapped.
 *
 * Revision 1.1  1998/06/18 21:27:44  henry
 * move sources from klips/src to klips/net/ipsec, to keep stupid
 * kernel-build scripts happier in the presence of symlinks
 *
 * Revision 1.9  1998/06/14 23:48:42  rgb
 * Fix I/F name comparison oops bug.
 *
 * Revision 1.8  1998/06/11 07:20:04  rgb
 * Stats fixed for rx_packets.
 *
 * Revision 1.7  1998/06/11 05:53:34  rgb
 * Added stats for rx error and good packet reporting.
 *
 * Revision 1.6  1998/06/05 02:27:28  rgb
 * Add rx_errors stats.
 * Fix DoS bug:  skb's not being freed on dropped packets.
 *
 * Revision 1.5  1998/05/27 21:21:29  rgb
 * Fix DoS potential bug.  skb was not being freed if the packet was bad.
 *
 * Revision 1.4  1998/05/18 22:31:37  rgb
 * Minor change in debug output and comments.
 *
 * Revision 1.3  1998/04/21 21:29:02  rgb
 * Rearrange debug switches to change on the fly debug output from user
 * space.  Only kernel changes checked in at this time.  radij.c was also
 * changed to temporarily remove buggy debugging code in rj_delete causing
 * an OOPS and hence, netlink device open errors.
 *
 * Revision 1.2  1998/04/12 22:03:19  rgb
 * Updated ESP-3DES-HMAC-MD5-96,
 * 	ESP-DES-HMAC-MD5-96,
 * 	AH-HMAC-MD5-96,
 * 	AH-HMAC-SHA1-96 since Henry started freeswan cvs repository
 * from old standards (RFC182[5-9] to new (as of March 1998) drafts.
 *
 * Fixed eroute references in /proc/net/ipsec*.
 *
 * Started to patch module unloading memory leaks in ipsec_netlink and
 * radij tree unloading.
 *
 * Revision 1.1  1998/04/09 03:05:59  henry
 * sources moved up from linux/net/ipsec
 *
 * Revision 1.1.1.1  1998/04/08 05:35:04  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.4  1997/01/15 01:28:15  ji
 * Minor cosmetic changes.
 *
 * Revision 0.3  1996/11/20 14:35:48  ji
 * Minor Cleanup.
 * Rationalized debugging code.
 *
 * Revision 0.2  1996/11/02 00:18:33  ji
 * First limited release.
 *
 *
 */
