/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */

#ifdef __KERNEL__

#include <linux/fs.h>
#include <linux/reiserfs_fs.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/smp_lock.h>

#else

#include "nokernel.h"

#endif


/*
** reiserfs_ioctl - handler for ioctl for inode
** supported commands:
**  1) REISERFS_IOC_UNPACK - try to unpack tail from direct item into indirect
**                           and prevent packing file (argument arg has to be non-zero)
**  2) That's all for a while ...
*/
int reiserfs_ioctl (struct inode * inode, struct file * filp, unsigned int cmd,
		unsigned long arg)
{
	switch (cmd) {
	    case REISERFS_IOC_UNPACK:
		if (arg)
		    return reiserfs_unpack (inode, filp);
			
	    default:
		return -ENOTTY;
	}
}


/*
** reiserfs_unpack
** Function try to convert tail from direct item into indirect.
** It set up nopack attribute in the inode.u.reiserfs_i.nopack
*/
int reiserfs_unpack (struct inode * inode, struct file * filp)
{
    int retval, windex, repeat, exitcode;
    struct path path;
    struct key key;
    struct buffer_head * bh, * unbh = 0;
    struct item_head * ih;
    int jbegin_count = JOURNAL_PER_BALANCE_CNT * 3 ;
    struct reiserfs_transaction_handle th ;

    lock_kernel();
    init_path (&path);
    unbh = NULL;
    exitcode = 0;
    
    down(&inode->i_sem) ;
    journal_begin(&th, inode->i_sb, jbegin_count) ;
    windex = push_journal_writer("reiserfs_ioctl_unpack") ;
    
search_again:
    /* find file's tail */    
    copy_key(&key, INODE_PKEY (inode));    
    key.k_offset =  (inode->i_size & (~(inode->i_sb->s_blocksize - 1))) + 1;
    key.k_uniqueness = TYPE_DIRECT;
    retval = search_by_key (inode->i_sb, &key, &path, &repeat,
			    DISK_LEAF_NODE_LEVEL, READ_BLOCKS);
		    
    if (retval == ITEM_NOT_FOUND) {
	if (unbh) {
	    /* something happened with file while allocating unformatted node */
	    reiserfs_free_block (&th, inode->i_sb, unbh->b_blocknr);
	    bforget (unbh);
	}
	/* don't pack tail until next open */
	inode->u.reiserfs_i.nopack = 1;
    } else {
	/* tail found */
	ih = B_N_PITEM_HEAD(bh = PATH_PLAST_BUFFER(&path),
			PATH_LAST_POSITION(&path));
	/* allocate new unformatted node to place tail */
	if (unbh == NULL) {
	    retval = get_new_buffer(&th, inode->i_sb, bh, &unbh, &path, inode, 
	                          (inode->i_size & (~(inode->i_sb->s_blocksize - 1))) + 1);
	    if (!unbh) {
		/* can't allocate block */
		pathrelse (&path);
		exitcode = -ENOSPC;
		goto finish;
	    }
	    if (retval & SCHEDULE_OCCURRED) {
		/* get_new_buffer was blocked and path had ability to change */
		pathrelse (&path);
		goto search_again;
	    }
	}
	mark_buffer_uptodate (unbh, 1);
	/* try to convert direct item into indirect */
	retval = direct_to_indirect (&th, inode->i_sb, inode, &path, 
				     0 /* n_item_zeros_to_add */,
				     NULL /* p_c_buf */, 
				     inode->i_size & (inode->i_sb->s_blocksize - 1) /* tail's size */,
				     unbh);
	if (retval < 0) {
	    /* direct_to_indirect() did not convert item */
	    reiserfs_free_block (&th, inode->i_sb, unbh->b_blocknr);
	    bforget (unbh);
	    inode->u.reiserfs_i.nopack = 0;
	    exitcode = -ENOSPC; /* FIXME: what error has to be returned here? -az */
	    goto finish;
	}
	/* direct_to_indirect brelses the unformatted node for us */
	pathrelse (&path);
	/* don't pack tail until next open */
	inode->u.reiserfs_i.nopack = 1;
    }
    
finish:
    pop_journal_writer(windex);
    pathrelse(&path) ;
    journal_end(&th, inode->i_sb, jbegin_count);
    up(&inode->i_sem) ;
    unlock_kernel();    
    return exitcode;
}
