/*
 * Copyright 2000 by Hans Reiser, licensing governed by reiserfs/README
 */
#ifdef __KERNEL__

#include <linux/module.h>
#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/reiserfs_fs.h>
#include <linux/locks.h>
#include <linux/init.h>

#else

#include "nokernel.h"
#include <stdlib.h> // for simple_strtoul

#endif

#define REISERFS_OLD_BLOCKSIZE 4096
#define REISERFS_SUPER_MAGIC_STRING_OFFSET_NJ 20


/* like fs.h:/mark_buffer_dirty but refile_buffer */
inline void reiserfs_mark_buffer_dirty (struct buffer_head * bh, int flag)
{
  if (!test_and_set_bit(BH_Dirty, &bh->b_state))
    set_writetime(bh, flag);
}


/* like fs.h:/mark_buffer_clean but refile_buffer */
inline void reiserfs_mark_buffer_clean (struct buffer_head * bh)
{
  test_and_clear_bit(BH_Dirty, &bh->b_state);
}


void reiserfs_write_super (struct super_block * s)
{

  int dirty = 0 ;
  if (!(s->s_flags & MS_RDONLY)) {
#if 0 /* journal victim */
    rs = SB_DISK_SUPER_BLOCK (s);
    /*
     * if reiserfs was mounted with read-write permissions make file
     * system state not valid so that if we crash without doing a
     * clean umount we know that we must run file system
     * checker. umount will mark it valid if it does a clean umount
     */
    if (le16_to_cpu (rs->s_state) == REISERFS_VALID_FS) {
      rs->s_state = cpu_to_le16 (REISERFS_ERROR_FS);
      /* mark_buffer_dirty (SB_BUFFER_WITH_SB (s), 1); */
      journal_begin(&th, s, 1) ;
      journal_mark_dirty(&th, s, SB_BUFFER_WITH_SB (s)) ;
      journal_end(&th, s, 1) ;
    }
#endif
    dirty = flush_old_commits(s, 1) ;
  }
  s->s_dirt = dirty;
}


void reiserfs_put_super (struct super_block * s)
{
  int i;
  kdev_t dev = s->s_dev;
  struct reiserfs_transaction_handle th ;

  unlock_super(s) ; 

  /* change file system state to current state if it was mounted with read-write permissions */
  if (!(s->s_flags & MS_RDONLY)) {
    journal_begin(&th, s, 10) ;
    SB_REISERFS_STATE (s) = le16_to_cpu (s->u.reiserfs_sb.s_mount_state);
    /* mark_buffer_dirty (SB_BUFFER_WITH_SB (s), 1); */
    journal_mark_dirty(&th, s, SB_BUFFER_WITH_SB (s));
  }

  /* note, journal_release can detect a readonly mount, and decide not to
  ** call journal_end
  */
  journal_release(&th, s) ;
  schedule() ; /* empty task queue one last time */
  lock_super(s) ;

  /* wait on write completion */
  for (i = 0; i < SB_BMAP_NR (s); i ++) {
    /* wait_on_buffer (SB_AP_CAUTIOUS_BITMAP (s)[i]); */
    /* brelse (SB_AP_CAUTIOUS_BITMAP (s)[i]); */
    brelse (SB_AP_BITMAP (s)[i]);
  }

  reiserfs_kfree (SB_AP_BITMAP (s), sizeof (struct buffer_head *) * SB_BMAP_NR (s), s);
  /* reiserfs_kfree (SB_AP_CAUTIOUS_BITMAP (s), sizeof (struct buffer_head *) * SB_BMAP_NR (s), s); */


  brelse (SB_BUFFER_WITH_SB (s));

  print_statistics (s);

  if (s->u.reiserfs_sb.s_kmallocs != 0) {
    reiserfs_warning ("vs-2004: reiserfs_put_super: aloocated memory left %d\n", s->u.reiserfs_sb.s_kmallocs);
  }

  s->s_dev = 0;

  fixup_reiserfs_buffers (dev);

  MOD_DEC_USE_COUNT;
  return;
}


/* super block operations are */
static struct super_operations reiserfs_sops = 
{
  reiserfs_read_inode,
  reiserfs_write_inode,
  NULL,				/* put_inode*/
  reiserfs_delete_inode,
  reiserfs_notify_change,
  reiserfs_put_super,
  reiserfs_write_super,
  reiserfs_statfs,
  reiserfs_remount,
  NULL, 				/* clear_inode */
  NULL				/* umount_begin */
};

/* this was (ext2)parse_options */
static int parse_options (char * options, unsigned long * mount_options, unsigned long * blocks)
{
    char * this_char;
    char * value;
  
    *blocks = 0;
    set_bit (GENERICREAD, mount_options);
    if (!options)
	/* use default configuration: complex read, create tails, preserve on */
	return 1;
    for (this_char = strtok (options, ","); this_char != NULL; this_char = strtok (NULL, ",")) {
	if ((value = strchr (this_char, '=')) != NULL)
	    *value++ = 0;
	if (!strcmp (this_char, "notail")) {
	    set_bit (NOTAIL, mount_options);
	} else if (!strcmp (this_char, "replayonly")) {
	    set_bit (REPLAYONLY, mount_options);
	} else if (!strcmp (this_char, "nolog")) {
	    set_bit (NOLOG, mount_options);
	} else if (!strcmp (this_char, "hash")) {
	    if (value && *value) {
		/* if they specify any hash option, we force detection
		** to make sure they aren't using the wrong hash
		*/
	        if (!strcmp(value, "rupasov")) {
		    set_bit (FORCE_RUPASOV_HASH, mount_options);
		    set_bit (FORCE_HASH_DETECT, mount_options);
		} else if (!strcmp(value, "tea")) {
		    set_bit (FORCE_TEA_HASH, mount_options);
		    set_bit (FORCE_HASH_DETECT, mount_options);
		} else if (!strcmp(value, "detect")) {
		    set_bit (FORCE_HASH_DETECT, mount_options);
		} else {
		    printk("reiserfs: invalid hash function specified\n") ;
		    return 0 ;
		}
	    } else {
	  	printk("reiserfs: hash option requires a value\n");
		return 0 ;
	    }
	} else if (!strcmp (this_char, "resize")) {
	    if (!value || !*value){
	  	printk("reiserfs: resize option requires a value\n");
		return 0 ;
	    }
	    *blocks = simple_strtoul (value, &value, 0);
	} else if (strcmp (this_char, "usrquota") && strcmp (this_char, "grpquota")) {
	    /* usrquota and grpquota are for quota tools only ? Kernel just
               ignores them */
	    printk ("reiserfs: Unrecognized mount option %s\n", this_char);
	    return 0;
	}
    }
    return 1;
}



int reiserfs_is_super(struct super_block *s) {
   return (s->s_dev != 0 && s->s_op == &reiserfs_sops) ;
}
int reiserfs_remount (struct super_block * s, int * flags, char * data)
{
  struct reiserfs_super_block * rs;
  struct reiserfs_transaction_handle th ;
  unsigned long blocks;
  unsigned long mount_options;

  rs = SB_DISK_SUPER_BLOCK (s);

  if (!parse_options(data, &mount_options, &blocks))
  	return 0;

  if(blocks && !(s->s_flags & MS_RDONLY)) 
  	reiserfs_resize(s, blocks);
	
  if ((unsigned long)(*flags & MS_RDONLY) == (s->s_flags & MS_RDONLY)) {
    /* there is nothing to do to remount read-only fs as read-only fs */
    return 0;
  }
  if (*flags & MS_RDONLY) {
    /* try to remount file system with read-only permissions */
    if (le16_to_cpu (rs->s_state) == REISERFS_VALID_FS ||
	s->u.reiserfs_sb.s_mount_state != REISERFS_VALID_FS) {
      return 0;
    }
    /* Mounting a rw partition read-only. */
    journal_begin(&th, s, 10) ;
    rs->s_state = cpu_to_le16 (s->u.reiserfs_sb.s_mount_state);
    /* mark_buffer_dirty (SB_BUFFER_WITH_SB (s), 1); journal victim */
    journal_mark_dirty(&th, s, SB_BUFFER_WITH_SB (s));
    s->s_dirt = 0;
  } else {
    /* Mount a partition which is read-only, read-write */
    s->u.reiserfs_sb.s_mount_state = le16_to_cpu (rs->s_state);
    s->s_flags &= ~MS_RDONLY;

    /* now the filesystem is no longer RDONLY, it is safe to call
    ** journal_begin
    */
    journal_begin(&th, s, 10) ;
    rs->s_state = cpu_to_le16 (REISERFS_ERROR_FS);
    /* mark_buffer_dirty (SB_BUFFER_WITH_SB (s), 1); */
    journal_mark_dirty(&th, s, SB_BUFFER_WITH_SB (s));
    s->s_dirt = 0;
    s->u.reiserfs_sb.s_mount_state = REISERFS_VALID_FS ;
    if (test_bit(NOTAIL, &mount_options)) {
      set_bit(NOTAIL, &(s->u.reiserfs_sb.s_mount_opt)) ;
    }
  }
  /* this will force a full flush of all journal lists */
  SB_JOURNAL(s)->j_must_wait = 1 ;
  journal_end(&th, s, 10) ;
  return 0;
}


struct key root_key = {REISERFS_ROOT_PARENT_OBJECTID, REISERFS_ROOT_OBJECTID, 0, 0};

static int read_bitmaps (struct super_block * s)
{
  int i, repeat, bmp, dl ;
  struct reiserfs_super_block * rs = SB_DISK_SUPER_BLOCK(s);

  repeat = 0 ;
  /* read true bitmap block */
  SB_AP_BITMAP (s) = reiserfs_kmalloc (sizeof (struct buffer_head *) * le16_to_cpu (rs->s_bmap_nr), GFP_KERNEL, s);
  if (SB_AP_BITMAP (s) == 0)
    return 1;

  memset (SB_AP_BITMAP (s), 0, sizeof (struct buffer_head *) * le16_to_cpu (rs->s_bmap_nr));

  /* read bitmap blocks */
				/* reiserfs leaves the first 64k unused
                                   so that any partition labeling scheme
                                   currently used will have enough
                                   space. Then we need one block for the
                                   super.  -Hans */
  bmp = (REISERFS_DISK_OFFSET_IN_BYTES / s->s_blocksize) + 1;	/* first of bitmap blocks */
  SB_AP_BITMAP (s)[0] = reiserfs_bread (s->s_dev, bmp, s->s_blocksize, &repeat);
  if(!SB_AP_BITMAP(s)[0])
	  return 1;
  for (i = 1, bmp = dl = rs->s_blocksize * 8; i < le16_to_cpu (rs->s_bmap_nr); i ++) {
    SB_AP_BITMAP (s)[i] = reiserfs_bread (s->s_dev, bmp, s->s_blocksize, &repeat);
    if (!SB_AP_BITMAP (s)[i])
      return 1;
	bmp += dl;
  }

  return 0;
}

static int read_old_bitmaps (struct super_block * s)
{
  int i, repeat ;
  struct reiserfs_super_block * rs = SB_DISK_SUPER_BLOCK(s);
  int bmp1 = (REISERFS_OLD_DISK_OFFSET_IN_BYTES / s->s_blocksize) + 1;  /* first of bitmap blocks */

  repeat = 0 ;
  /* read true bitmap */
  SB_AP_BITMAP (s) = reiserfs_kmalloc (sizeof (struct buffer_head *) * le16_to_cpu (rs->s_bmap_nr), GFP_KERNEL, s);
  if (SB_AP_BITMAP (s) == 0)
    return 1;

  memset (SB_AP_BITMAP (s), 0, sizeof (struct buffer_head *) * le16_to_cpu (rs->s_bmap_nr));

  for (i = 0; i < le16_to_cpu (rs->s_bmap_nr); i ++) {
    SB_AP_BITMAP (s)[i] = reiserfs_bread (s->s_dev, bmp1 + i, s->s_blocksize, &repeat);
    if (!SB_AP_BITMAP (s)[i])
      return 1;
  }
	
  return 0;
}


void check_bitmap (struct super_block * s)
{
  int i = 0;
  int free = 0;
  char * buf;

  while (i < SB_BLOCK_COUNT (s)) {
    buf = SB_AP_BITMAP (s)[i / (s->s_blocksize * 8)]->b_data;
    if (!test_bit (i % (s->s_blocksize * 8), buf))
      free ++;
    i ++;
  }

  if (free != SB_FREE_BLOCKS (s))
    reiserfs_warning ("vs-4000: check_bitmap: %d free blocks, must be %d\n",
		      free, SB_FREE_BLOCKS (s));
}


/* support old disk layout */
static int read_old_super_block (struct super_block * s, int size)
{
  struct buffer_head * bh;
  struct reiserfs_super_block * rs;
  int repeat ;

  printk("reiserfs_read_super: try to find super block in old location\n");
  repeat = 0 ;
  /* there are only 4k-sized blocks in v3.5.10 */
  if (size != REISERFS_OLD_BLOCKSIZE)
	  set_blocksize(s->s_dev, REISERFS_OLD_BLOCKSIZE);
  bh = bread (s->s_dev, 
  			  REISERFS_OLD_DISK_OFFSET_IN_BYTES / REISERFS_OLD_BLOCKSIZE, 
 		      REISERFS_OLD_BLOCKSIZE);
  if (!bh) {
    printk("reiserfs_read_super: unable to read superblock on dev %s\n", kdevname(s->s_dev));
    return 1;
  }

  rs = (struct reiserfs_super_block *)bh->b_data;
  if (strncmp (rs->s_magic,  REISERFS_SUPER_MAGIC_STRING, strlen ( REISERFS_SUPER_MAGIC_STRING))) {
	  /* pre-journaling version check */
	  if(!strncmp((char*)rs + REISERFS_SUPER_MAGIC_STRING_OFFSET_NJ,
				  REISERFS_SUPER_MAGIC_STRING, strlen(REISERFS_SUPER_MAGIC_STRING))) {
		  printk("reiserfs_read_super: a pre-journaling reiserfs filesystem isn't suitable there.\n");
		  brelse(bh);
		  return 1;
	  }
	  
    brelse (bh);
    printk ("reiserfs_read_super: can't find a reiserfs filesystem on dev %s.\n", kdevname(s->s_dev));
    return 1;
  }

  if(REISERFS_OLD_BLOCKSIZE != le16_to_cpu (rs->s_blocksize)) {
  	printk("reiserfs_read_super: blocksize mismatch, super block corrupted\n");
	brelse(bh);
	return 1;
  }	

  s->s_blocksize = REISERFS_OLD_BLOCKSIZE;
  s->s_blocksize_bits = 0;
  while ((1 << s->s_blocksize_bits) != s->s_blocksize)
    s->s_blocksize_bits ++;

  SB_BUFFER_WITH_SB (s) = bh;
  SB_DISK_SUPER_BLOCK (s) = rs;
  s->s_op = &reiserfs_sops;
  return 0;
}


static int read_super_block (struct super_block * s, int size)
{
  struct buffer_head * bh;
  struct reiserfs_super_block * rs;
  int repeat ;

  repeat = 0 ;
  bh = bread (s->s_dev, (REISERFS_DISK_OFFSET_IN_BYTES / size), size);
  if (!bh) {
    printk("reiserfs_read_super: unable to read superblock on dev %s\n", kdevname(s->s_dev));
    return 1;
  }

  rs = (struct reiserfs_super_block *)bh->b_data;
  if (strncmp (rs->s_magic,  REISERFS_SUPER_MAGIC_STRING, strlen ( REISERFS_SUPER_MAGIC_STRING))) {
    brelse (bh);
    printk ("reiserfs_read_super: can't find a reiserfs filesystem on dev %s.\n", kdevname(s->s_dev));
    return 1;
  }

  s->s_blocksize = le16_to_cpu (rs->s_blocksize);
  s->s_blocksize_bits = 0;
  while ((1 << s->s_blocksize_bits) != s->s_blocksize)
    s->s_blocksize_bits ++;

  if (size != rs->s_blocksize) {
    brelse (bh);
    set_blocksize (s->s_dev, s->s_blocksize);
    bh = reiserfs_bread (s->s_dev,  (REISERFS_DISK_OFFSET_IN_BYTES / s->s_blocksize), s->s_blocksize, &repeat);
    if (!bh) {
      printk("reiserfs_read_super: unable to read superblock on dev %s\n", kdevname(s->s_dev));
      return 1;
    }

    rs = (struct reiserfs_super_block *)bh->b_data;
    if (strncmp (rs->s_magic,  REISERFS_SUPER_MAGIC_STRING, strlen ( REISERFS_SUPER_MAGIC_STRING)) ||
	le16_to_cpu (rs->s_blocksize) != s->s_blocksize) {
      brelse (bh);
      printk ("reiserfs_read_super: can't find a reiserfs filesystem on dev %s.\n", kdevname(s->s_dev));
      return 1;
    }
  }
  /* must check to be sure we haven't pulled an old format super out of the
  ** old format's log.  This is a kludge of a check, but it will work.  
  ** If block we've just read in is inside the journal for that
  ** super, it can't be valid.
  */
  if (bh->b_blocknr >= rs->s_journal_block && 
      bh->b_blocknr < (rs->s_journal_block + JOURNAL_BLOCK_COUNT)) {
      brelse(bh) ;
      printk("super-459: reiserfs_read_super: super found at block %lu is within its own log.  It must not be of this format type.\n", bh->b_blocknr) ;
      return 1 ;
  }

  SB_BUFFER_WITH_SB (s) = bh;
  SB_DISK_SUPER_BLOCK (s) = rs;
  s->s_op = &reiserfs_sops;
  return 0;
}

/* after journal replay, reread all bitmap and super blocks */
static int reread_meta_blocks(struct super_block *s) {
  int i ;
  ll_rw_block(READ, 1, &(SB_BUFFER_WITH_SB(s))) ;
  wait_on_buffer(SB_BUFFER_WITH_SB(s)) ;
  if (!buffer_uptodate(SB_BUFFER_WITH_SB(s))) {
    printk("reread_meta_blocks, error reading the super\n") ;
    return 1 ;
  }

  for (i = 0; i < SB_BMAP_NR(s) ; i++) {
    ll_rw_block(READ, 1, &(SB_AP_BITMAP(s)[i])) ;
    wait_on_buffer(SB_AP_BITMAP(s)[i]) ;
    if (!buffer_uptodate(SB_AP_BITMAP(s)[i])) {
      printk("reread_meta_blocks, error reading bitmap block number %d at %ld\n", i, SB_AP_BITMAP(s)[i]->b_blocknr) ;
      return 1 ;
    }
  }
  return 0 ;

}


// if root directory is empty - we set default - Yura's - hash and
// warn about it
// FIXME: we look for only one name in a directory. If tea and yura
// bith have the same value - we ask user to send report to the
// mailing list
__u32 find_hash_out (struct super_block * s)
{
    int retval, repeat;
    struct inode * inode;
    struct key key;
    struct path path;
    struct reiserfs_dir_entry de;
    struct reiserfs_de_head * deh;
    __u32 hash = DEFAULT_HASH;

    init_path (&path);

    inode = s->s_root->d_inode;

    while (1) {
	copy_key (&key, INODE_PKEY (inode));
	key.k_offset = MAX_KEY_OFFSET;
	key.k_uniqueness = DIRENTRY_UNIQUENESS;
	retval = search_by_entry_key (s, &key, &path, &de.de_entry_num, &repeat);
	if (retval == IO_ERROR)
	    // FIXME: sigh, still not ready
	    reiserfs_panic (s, "reiserfs: find_hash: IO error are not handled properly yet");
	if (retval == POSITION_NOT_FOUND)
	    de.de_entry_num --;
	
	de.de_bh = PATH_PLAST_BUFFER (&path);
	de.de_ih = PATH_PITEM_HEAD (&path);
	de.de_deh = B_I_DEH (de.de_bh, de.de_ih);
	de.de_item_num = PATH_LAST_POSITION (&path);
	deh = de.de_deh + de.de_entry_num;
	de.de_entrylen = I_DEH_N_ENTRY_LENGTH (de.de_ih, deh, de.de_entry_num);
	de.de_namelen = de.de_entrylen - (de_with_sd (deh) ? SD_SIZE : 0);
	de.de_name = B_I_PITEM (de.de_bh, de.de_ih) + le16_to_cpu (deh->deh_location);
	if (de.de_name[de.de_namelen - 1] == 0)
	  de.de_namelen = strlen (de.de_name);
	//set_de_name_and_namelen (&de);
	if (le32_to_cpu (de.de_deh[de.de_entry_num].deh_offset) == DOT_DOT_OFFSET) {
	    /* allow override in this case */
	    if (reiserfs_rupasov_hash(s)) {
	        hash = YURA_HASH ;
	    }
	    reiserfs_warning("reiserfs: FS seems to be empty, "
	                     "autodetect is using default hash.\n") ;
	    break ;
	}
	if (GET_HASH_VALUE(yura_hash (de.de_name, de.de_namelen)) == 
	    GET_HASH_VALUE(keyed_hash (de.de_name, de.de_namelen))) {
	    reiserfs_warning("reiserfs: could not detect hash function, "
	                     "please mount with -o hash={tea,rupasov,r5}\n");
	    hash = UNSET_HASH ;
	    break;
	}
        if (GET_HASH_VALUE(le32_to_cpu(de.de_deh[de.de_entry_num].deh_offset))==
            GET_HASH_VALUE(r5_hash (de.de_name, de.de_namelen))) {
            hash = R5_HASH;
	} else if (GET_HASH_VALUE(le32_to_cpu(de.de_deh[de.de_entry_num].deh_offset))==
		   GET_HASH_VALUE(yura_hash (de.de_name, de.de_namelen))) {
	    hash = YURA_HASH;
	} else {
	    hash = TEA_HASH;
	}
	break;
    }

    pathrelse (&path);
    return hash;
}


// finds out which hash names are sorted with
static int what_hash (struct super_block * s)
{
    __u32 code;

    code = le32_to_cpu (s->u.reiserfs_sb.s_rs->s_hash_function_code);

    /* reiserfs_hash_detect() == true if any of the hash mount options
    ** were used.  We must check them to make sure the user isn't
    ** using a bad hash value
    */
    if (code == UNSET_HASH || reiserfs_hash_detect(s))
	code = find_hash_out (s);

    if (code != UNSET_HASH && reiserfs_hash_detect(s)) {
	/* detection has found the hash, and we must check against the 
	** mount options 
	*/
	if (reiserfs_rupasov_hash(s) && code != YURA_HASH) {
	    printk("REISERFS: Error, tea hash detected, "
		   "unable to force rupasov hash\n") ;
	    code = UNSET_HASH ;
	} else if (reiserfs_tea_hash(s) && code != TEA_HASH) {
	    printk("REISERFS: Error, rupasov hash detected, "
		   "unable to force tea hash\n") ;
	    code = UNSET_HASH ;
	} 
    } else { 
        /* find_hash_out was not called or could not determine the hash */
	if (reiserfs_rupasov_hash(s)) {
	    code = YURA_HASH ;
	} else if (reiserfs_tea_hash(s)) {
	    code = TEA_HASH ;
	} else if (reiserfs_r5_hash(s)) {
	    code = R5_HASH ;
	} 
    }

    /* if we are mounted RW, and we have a new valid hash code, update 
    ** the super
    */
    if (code != UNSET_HASH && 
	!(s->s_flags & MS_RDONLY) && 
        code != le32_to_cpu (s->u.reiserfs_sb.s_rs->s_hash_function_code)) {
        s->u.reiserfs_sb.s_rs->s_hash_function_code = cpu_to_le32(code) ;
    }
    return code;
}

// return pointer to appropriate function
static hashf_t hash_function (struct super_block * s)
{
    switch (what_hash (s)) {
    case TEA_HASH:
	reiserfs_warning ("Using tea hash to sort names\n");
	return keyed_hash;
    case YURA_HASH:
	reiserfs_warning ("Using rupasov hash to sort names\n");
	return yura_hash;
    case R5_HASH:
	reiserfs_warning ("Using r5 hash to sort names\n");
	return r5_hash;
    }
    return NULL;
}

// this is used to set up correct value for old partitions
int function2code (hashf_t func)
{
    if (func == keyed_hash)
	return TEA_HASH;
    if (func == yura_hash)
	return YURA_HASH;
    if (func == r5_hash)
	return R5_HASH;

    reiserfs_panic (0, "reiserfs: function2code: unknow hash function detected");
    return 0;
}


static void print_credits(void) {
  static int printed = 0 ;
  if (printed) 
    return ;
  printk("Primary Sponsor thresholdnetworks.com\n");
  printk("Raid Tuning sponsored by emusic.com\n");
  printk("HSM sponsored by bigstorage.com\n");
  printk("Alpha port and SMP sponsored by www.api-networks.com, alpha port by www.innovative-software.com and www.quant-x.com.\n");
  printed = 1 ;
}


struct super_block * reiserfs_read_super (struct super_block * s, void * data, int silent)
{
    int size;
    struct inode *root_inode;
    kdev_t dev = s->s_dev;
    int j;
    extern int *blksize_size[];
    struct reiserfs_transaction_handle th ;
    int old_format = 0;
    unsigned long blocks;
    int jinit_done = 0 ;

    memset (&s->u.reiserfs_sb, 0, sizeof (struct reiserfs_sb_info));

    if (parse_options ((char *) data, &(s->u.reiserfs_sb.s_mount_opt), &blocks) == 0) {
	s->s_dev = 0;
	return NULL;
    }

    if (blocks) {
  	printk("reserfs: resize option for remount only\n");
	return NULL;
    }	

    MOD_INC_USE_COUNT;
    lock_super (s);

    if (blksize_size[MAJOR(dev)] && blksize_size[MAJOR(dev)][MINOR(dev)] != 0) {
	/* as blocksize is set for partition we use it */
	size = blksize_size[MAJOR(dev)][MINOR(dev)];
    } else {
	size = BLOCK_SIZE;
	set_blocksize (s->s_dev, BLOCK_SIZE);
    }

    /* read block, containing reiserfs super block (it is stored at REISERFS_FIRST_BLOCK-th 1K block) */
    if (read_super_block (s, size)) {
	if(read_old_super_block(s,size)) 
	    goto error;
	else
	    old_format = 1;
    }

    s->u.reiserfs_sb.s_mount_state = le16_to_cpu (SB_DISK_SUPER_BLOCK (s)->s_state); /* journal victim */
    s->u.reiserfs_sb.s_mount_state = REISERFS_VALID_FS ;

    /* reiserfs can not be mounted when it propably contains errors */
#if 0 /* journal victim */
    if (le16_to_cpu (SB_DISK_SUPER_BLOCK (s)->s_state) != REISERFS_VALID_FS) {
	printk ("reiserfs_read_super:  mounting unchecked fs, run reiserfsck first\n");
	goto error;
    }
#endif
    if (old_format ? read_old_bitmaps(s) : read_bitmaps(s)) { 
	printk ("reiserfs_read_super: unable to read bitmap\n");
	goto error;
    }

    if (journal_init(s)) {
	printk("reiserfs_read_super: unable to initialize journal space\n") ;
	goto error ;
    } else {
	jinit_done = 1 ; /* once this is set, journal_release must be called
			 ** if we error out of the mount 
			 */
    }
    if (reread_meta_blocks(s)) {
	printk("reiserfs_read_super: unable to reread meta blocks after journal init\n") ;
	goto error ;
    }

    if (replay_only (s))
	goto error;

    /* VFS should do this for us, but we panic if the MS_RDONLY isn't
    ** set on a read only device, so I'm doing it again anyway. --clm
    */
    if (is_read_only(s->s_dev) && !(s->s_flags & MS_RDONLY)) {
      printk("clm-4000: Detected readonly device, marking FS readonly\n") ;
      s->s_flags |= MS_RDONLY ;
    }
    /*s->s_op = &reiserfs_sops;*/
   
    /* get root directory inode */
    store_key (s, &root_key);
    root_inode = iget (s, root_key.k_objectid);
    forget_key (s, &root_key);
    if (!root_inode) {
	printk ("reiserfs_read_super: get root inode failed\n");
	goto error;
    }

    s->s_root = d_alloc_root(root_inode, NULL);  
    if (!s->s_root) {
	iput(root_inode);
	goto error;
    }

    // define and initialize hash function
    s->u.reiserfs_sb.s_hash_function = hash_function (s);
    if (s->u.reiserfs_sb.s_hash_function == NULL) {
        dput(s->s_root) ;
	goto error ;
    }


    if (!(s->s_flags & MS_RDONLY)) {
	SB_DISK_SUPER_BLOCK (s)->s_state = cpu_to_le16 (REISERFS_ERROR_FS);
	/* mark_buffer_dirty (SB_BUFFER_WITH_SB (s), 1); */
	journal_begin(&th, s, 1) ;
	journal_mark_dirty(&th, s, SB_BUFFER_WITH_SB (s));
	journal_end(&th, s, 1) ;
	s->s_dirt = 0;
    }

    /*s->u.reiserfs_sb.unpreserve = dont_preserve (s) ? 0 : unpreserve;*/
    /* we have to do this to make journal writes work correctly */
    SB_BUFFER_WITH_SB(s)->b_end_io = reiserfs_end_buffer_io_sync ;

    unlock_super (s);
    print_credits() ;
    printk("%s\n", reiserfs_get_version_string()) ;
    return s;

 error:
    if (jinit_done) { /* kill the commit thread, free journal ram */
	journal_release_error(NULL, s) ;
    }
    if (SB_DISK_SUPER_BLOCK (s)) {
	for (j = 0; j < le16_to_cpu (SB_DISK_SUPER_BLOCK (s)->s_bmap_nr); j ++) {
	    if (SB_AP_BITMAP (s))
		brelse (SB_AP_BITMAP (s)[j]);
	}
	if (SB_AP_BITMAP (s))
	    reiserfs_kfree (SB_AP_BITMAP (s), sizeof (struct buffer_head *) * SB_BMAP_NR (s), s);
    }
    if (SB_BUFFER_WITH_SB (s))
	brelse(SB_BUFFER_WITH_SB (s));
    s->s_dev = 0;
    unlock_super(s);
    MOD_DEC_USE_COUNT;

    return NULL;
}


int reiserfs_statfs (struct super_block * s, struct statfs * buf, int bufsize)
{
  struct statfs tmp;
  struct reiserfs_super_block * rs = SB_DISK_SUPER_BLOCK (s);
  
				/* changed to accomodate gcc folks.*/
  tmp.f_type =  REISERFS_SUPER_MAGIC;
  tmp.f_bsize = le32_to_cpu (s->s_blocksize);
  tmp.f_blocks = le32_to_cpu (rs->s_block_count) - le16_to_cpu (rs->s_bmap_nr) - 1;
  tmp.f_bfree = le32_to_cpu (rs->s_free_blocks);
  tmp.f_bavail = tmp.f_bfree;
  tmp.f_files = -1;
  tmp.f_ffree = -1;
  tmp.f_namelen = (REISERFS_MAX_NAME_LEN (s->s_blocksize));
  return copy_to_user (buf, &tmp, bufsize) ? -EFAULT : 0;
}

#ifdef __KERNEL__

static struct file_system_type reiserfs_fs_type = {
  "reiserfs", FS_REQUIRES_DEV, reiserfs_read_super, NULL
};


__initfunc(int init_reiserfs_fs(void))
{
  return register_filesystem(&reiserfs_fs_type);
}

#endif

#ifdef MODULE
EXPORT_NO_SYMBOLS;

int init_module(void)
{
	return init_reiserfs_fs();
}

void cleanup_module(void)
{
        unregister_filesystem(&reiserfs_fs_type);
}

#endif




























