/*
-- ----------------------------------------------------------------------------
--
--  Object name : filer.c
--  Revision    : 1.0
--
--  Copyright (c) Inmos Ltd, 1987.
--  All Rights Reserved.
--
--  DESCRIPTION
--    TDS Server   Filer Main Routines
--
--  NOTES
--
--  HISTORY
--     ibm-pc   20-Oct-88  CREATION.
--     ibm-pc   17-Feb-88  RJO - Addition of read ahead/ write behind features.
--     ibm-pc   25-Feb-88  RJO - All open call are to have the "b" binary
--              option, except open_text_read/write.
-- ----------------------------------------------------------------------------
*/
#define LINT_ARGS 1;      /* Enable Compiler Parameter Checking */

#include "inmos.h"
#include "srvconst.h"
#include "filecoms.h"
#include "errno.h"
#include <stdio.h>
#include "filconst.h"
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "time.h"
#include "linux-dos.h"
/*
-- ----------------------------------------------------------------------------
-- External variables.
-- ----------------------------------------------------------------------------
*/

extern BYTE TopLevelFile[FILE_NAME_LENGTH];
/*
-- ----------------------------------------------------------------------------
-- Only those which do not return int or void are listed.
-- ----------------------------------------------------------------------------
*/

BYTE *FindExtension (BYTE*);
long LongWordFromLink ();
/*
-- ----------------------------------------------------------------------------
-- Filer static variables.
-- ----------------------------------------------------------------------------
*/

static struct FILER_SPACE_DEF filer_data_space[NUMBER_OF_FILERS];
static int FilerRunning;
static struct CHRC_TABLE_STRUCT ChrcTable[NUMBER_OF_CHRC+1] =
        { {"NULL               \0", F_CANNOT_CHANGE_CHRC},
          {"File creation date \0", F_CANNOT_CHANGE_CHRC},
          {"File size          \0", F_CANNOT_CHANGE_CHRC},
          {"Read only          \0", F_OK} };

/*
-- ----------------------------------------------------------------------------
-- void FilerInit - Initialise the driver of each filer channel.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FilerInit ()
{
  int i;
  for (i=0; i < NUMBER_OF_FILERS; i++)
    FilerInitFile (&filer_data_space[i]);
  FilerRunning = TRUE;
}
/*
-- ----------------------------------------------------------------------------
-- void FilerClose - Close any files open on the filer channels, and close
-- down the filers.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FilerClose ()
{
  int i;
  for (i=0; i < NUMBER_OF_FILERS; i++)
    FilerCloseFile (&filer_data_space[i]);
  FilerRunning = FALSE;
}

/*
-- ----------------------------------------------------------------------------
-- int FileLockRead.
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.lock.read protocol.
-- ----------------------------------------------------------------------------
*/
int FileLockRead (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  int Result = T_OK;
  BYTE ParentAbsIdStr [FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT FileId;

  StrFromLink (ParentAbsIdStr);
  FileIdFromLink (&FileId);

  if ((strlen(FileId.name) == 0) && (strlen(ParentAbsIdStr) == 0))
    Result = FixTopLevelFile (Channel, fptr, TopLevelFile);
  else
    {
      /* Otherwise we assume a good lock */
      ResultToLink (Channel, FKF_LOCK_READ, fptr->result);
      StrToLink (FileId.name);
    };
  return (Result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileLockWrite
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.lock.write protocol.
-- ----------------------------------------------------------------------------
*/
void FileLockWrite (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT file_id;
  StrFromLink (ParentAbsIdStr);
  FileIdFromLink (&file_id);
  if ((strlen(file_id.name) == 0) && (strlen(ParentAbsIdStr) == 0))
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_LOCK_WRITE, fptr->result);
  StrToLink (file_id.name);
}
/*
-- ----------------------------------------------------------------------------
-- void FileRelease
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.release protocol.
-- ----------------------------------------------------------------------------
*/
void FileRelease (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];

  StrFromLink (AbsIdStr);
  ResultToLink (Channel, FKF_RELEASE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileOpenRead
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.open.read protocol.
-- Now has ReadAhead feature.
-- ----------------------------------------------------------------------------
*/
void FileOpenRead (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE FileAbsIdStr[FILE_NAME_LENGTH];

  StrFromLink(FileAbsIdStr);
  ConvertToLinux(&FileAbsIdStr);

  if ((fptr->state) == FS_IDLE)
    {
      FILE *f_handle = fopen(FileAbsIdStr, "rb");
      if (f_handle != NULL)
        {
          fptr->handle = f_handle;
          strcpy (fptr->file_name, FileAbsIdStr);
          fptr->state = FS_OPEN_FOR_READING;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
          fptr->result = F_OK;
          ResultToLink(Channel, FKF_OPEN_READ, fptr->result);
          ReadAhead (fptr);
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;

  if (fptr->state != FS_OPEN_FOR_READING)
    ResultToLink(Channel, FKF_OPEN_READ, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileOpenWrite
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.open.write protocol.
-- ----------------------------------------------------------------------------
*/
void FileOpenWrite (Channel,fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE FileAbsIdStr[FILE_NAME_LENGTH];
  StrFromLink (FileAbsIdStr);
  ConvertToLinux(&FileAbsIdStr);
  if (fptr->state == FS_IDLE)
    {
      FILE *f_handle = fopen(FileAbsIdStr, "wb");
      if (f_handle != NULL)
        {
          fptr->handle = f_handle;
          strcpy(fptr->file_name, FileAbsIdStr);
          fptr->state = FS_OPEN_FOR_WRITING;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;

  fptr ->rwresult = F_OK;
  ResultToLink(Channel, FKF_OPEN_WRITE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileClose
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.close protocol.
-- ----------------------------------------------------------------------------
*/
void FileClose (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  switch (fptr->state)
    {
      case FS_OPEN_FOR_WRITING:
        FlushBuffer (fptr);
        FilerCloseFile (fptr);
        break;
      case FS_OPEN_FOR_READING:
      case FS_OPEN_FOR_READING_TEXT:
      case FS_OPEN_FOR_WRITING_TEXT:
      case FS_OPEN_BLOCK_READ:
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        FilerCloseFile(fptr);
        break;
      default:
        fptr->result = F_INVALID_OPERATION;
    };
  ResultToLink(Channel, FKF_CLOSE, fptr->result);
  FileNameToLink (fptr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileRead
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.read protocol.
-- ----------------------------------------------------------------------------
*/
void FileRead (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  switch (fptr->state)
    {
      case FS_OPEN_FOR_READING:
        {
          if (fptr->rwresult == F_OK)
            {
              ResultToLink (Channel, FKF_READ, fptr->result);
              BufferToLink (fptr);
              if (fptr -> buffer_count == 0)
                ReadAhead (fptr);                 /* Do a read ahead */
              return;                             /* lets get out of here... */
            }
          else
            fptr -> result = fptr -> rwresult;
          break;
        };
      case FS_OPEN_FOR_READING_TEXT:
        {
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;

          if (fgets(fptr->buffer, RECORD_LENGTH, fptr->handle) != NULL)
            fptr->buffer_count = strcspn (fptr->buffer, "\0\n");
          else
            {
              if (feof(fptr->handle))
                fptr->result = F_EOF;
              else
                fptr->result = F_CANNOT_READ;
            };
          break;
        };
      case FS_OPEN_BLOCK_READ:
      case FS_OPEN_BLOCK_UPDATE:
        {
          fptr -> buffer_count = 0;
          fptr -> buffer_ptr = 0;

          if (fptr->last_op != READ)
            /* We have to do a seek to ensure the file is updated, before reading */
            {
              fseek (fptr->handle, 0L, S_CURRENT);
              fptr->last_op = READ;
            };

          fptr->buffer_count = fread (fptr->buffer, sizeof(char),
                                      RECORD_LENGTH - (fptr->block_posn),
                                      fptr->handle);
          if (fptr->buffer_count > 0)
            fptr->block_posn = ((fptr->block_posn)+(fptr->buffer_count)) %
                                RECORD_LENGTH;
          else if (fptr -> buffer_count == 0)
            fptr->result = F_EOF;
          else
            fptr->result = F_CANNOT_READ;
          break;
        };
      default:
        {
          fptr->result = F_INVALID_OPERATION;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
          break;
        };
    };

  ResultToLink (Channel, FKF_READ, fptr->result);
  BufferToLink (fptr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileWrite
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.write protocol.
-- ----------------------------------------------------------------------------
*/
void FileWrite (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  switch (fptr->state)
    {
      case FS_OPEN_FOR_WRITING:
        {
          int ChCount;

          BufferFromLink (fptr);

          if (fptr->rwresult == F_OK)
            {
              ResultToLink (Channel, FKF_WRITE, fptr->result);
              if ((BUFFER_LENGTH - fptr->buffer_count) >= RECORD_LENGTH)
                return;        /* enuff room for another record */
              FlushBuffer (fptr);   /* Let flush do the work    */
              return;
            };
          break;
        };
      case FS_OPEN_FOR_WRITING_TEXT:
        {
          int ChCount;
          BufferFromLink (fptr);
          if (fptr->buffer_count > (RECORD_LENGTH - 2))
              fptr->buffer_count = RECORD_LENGTH - 2;
          fptr->buffer[fptr->buffer_count] = '\r';
          fptr->buffer[(fptr->buffer_count)+1] = '\n';
          fptr->buffer_count += 2;
          ChCount = fwrite (fptr->buffer, sizeof(char),
                             fptr->buffer_count, fptr->handle);

          if (ChCount != fptr->buffer_count)
            fptr->result = F_NOT_ALL_DATA_WRITTEN;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
          break;
        };
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        {
          int ChCount;
          BufferFromLink (fptr);
          if (fptr->last_op != WRITE)
            {
              fseek (fptr->handle, 0L, S_CURRENT);
              fptr->last_op = WRITE;
            };
          ChCount = fwrite (fptr->buffer, sizeof(char),
                             fptr->buffer_count, fptr->handle);
          fptr->block_posn = ((fptr->block_posn) + (fptr->buffer_count))
                               % RECORD_LENGTH;

          if (ChCount != fptr->buffer_count)
            fptr->result = F_NOT_ALL_DATA_WRITTEN;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
          break;
        };
      default:
        {
          BYTE DummySlice[1] ;
          TruncSliceFromLink ( 0, DummySlice);
          fptr->result = F_INVALID_OPERATION;
        };
    };

  ResultToLink (Channel, FKF_WRITE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileSuspend
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.suspend protocol.
-- ----------------------------------------------------------------------------
*/
void FileSuspend (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  if ((fptr->state == FS_OPEN_FOR_READING) ||
      (fptr->state == FS_OPEN_FOR_WRITING))
    {
      if (fptr->sr_stack_ptr < SR_STACK_DEPTH)
        {
          if (fptr->state == FS_OPEN_FOR_READING)
            {
              FlushBuffer (fptr);
              if (fptr->rwresult != F_OK)
                fptr->result = fptr->rwresult;
            };

          if (fptr->result == F_OK)
            {
              if (fclose (fptr->handle) != EOF)
                {
                  PushFile (fptr);
                  fptr->state = FS_IDLE;
                }
              else
                fptr->result = F_CANNOT_CLOSE_FILE;
            }
        }
      else
        fptr->result = F_TOO_MANY_SUSPENDS;
    }
  else
    fptr->result = F_INVALID_OPERATION;

  ResultToLink (Channel, FKF_SUSPEND, fptr->result);
  FileNameToLink (fptr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileResume
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.resume protocol.
-- ----------------------------------------------------------------------------
*/
void FileResume (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  if (fptr->state == FS_IDLE)
    {
      if (fptr->sr_stack_ptr > 0)
        {
          struct SR_STACK_ENTRY *stack_entry;
          FILE *file_ptr;
          int Opened = FALSE;

          /* Point stack_entry to top item on stack */
          stack_entry = &(fptr->sr_stack[(fptr->sr_stack_ptr - 1)]);

          switch (stack_entry->file_mode)
            {
              case FS_OPEN_FOR_READING:
		ConvertToLinux(stack_entry->file_str);
                Opened = ((file_ptr = fopen(stack_entry->file_str, "r"))
                            != NULL);
                break;
              case FS_OPEN_FOR_WRITING:
		ConvertToLinux(stack_entry->file_str);
                Opened = ((file_ptr = fopen(stack_entry->file_str, "a"))
                            != NULL);
                break;
              default:
                Opened = FALSE;
            };

          if (Opened)
            {
              fptr->handle = file_ptr;
              PullFile (fptr);
            }
          else
            fptr->result = F_CANNOT_OPEN_FILE;
        }
      else
        fptr->result = F_UNMATCHED_RESUME;
    }
  else
    fptr->result = F_INVALID_OPERATION;

  if (fptr->state == FS_OPEN_FOR_READING)
    fptr->buffer_count = 0;

  ResultToLink (Channel, FKF_RESUME, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileAsciiToId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.ascii.to.id protocol.
-- ----------------------------------------------------------------------------
*/
void FileAsciiToId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  BYTE TruncatedName[FILE_NAME_LENGTH];
  struct ASCII_ID_STRUCT ascii_id;
  struct FILE_ID_STRUCT file_id;
  BYTE *Ch;
  int NameLen;

  StrFromLink (ParentAbsIdStr);
  AsciiIdFromLink (&ascii_id);
  NameLen = strlen (ascii_id.name);

  if (NameLen == 0)
    MakeUniqueFileName (ascii_id.name);
  strcpy (file_id.name, ascii_id.name);

  if ((Ch = FindExtension (ascii_id.name)) != NULL)
    {
      /* We have an extension, so get the attributes from it */

      int NewType, NewContents;

      if (ValidExtension (Ch, &NewType, &NewContents))
        {
          file_id.type_attr = NewType;
          file_id.contents_attr = NewContents;
        }
      else
        fptr->result = F_ILLEGAL_EXTENSION;
    }
  else
    {
      /* Add an extension corresponding to the given attributes */

      file_id.type_attr = ascii_id.type_attr;
      file_id.contents_attr = ascii_id.contents_attr;
      Ch = strchr (file_id.name, '\0'); /* Point ch to end of string */

      if (DeriveExtension(Ch, file_id.type_attr, file_id.contents_attr) == FALSE)
        {
          BYTE StrippedName[FILE_NAME_LENGTH];
          /* Send back a real filename without the extension */
          StripExtension (file_id.name, StrippedName);
          strcpy (file_id.name, StrippedName);     /* Copy filename back */
          fptr->result = F_ILLEGAL_ATTRIBUTE;
        }
    };
  TruncateFileName (file_id.name, TruncatedName);
  strcpy (file_id.name, TruncatedName);     /* Copy filename back */
  ResultToLink (Channel, FKF_ASCII_TO_ID, fptr->result);
  FileIdToLink (&file_id);
}
/*
-- ----------------------------------------------------------------------------
-- void FileIdToAscii
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.id.to.ascii protocol.
-- ----------------------------------------------------------------------------
*/
void FileIdToAscii (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT file_id;

  StrFromLink (ParentAbsIdStr);
  FileIdFromLink (&file_id);
  ResultToLink (Channel, FKF_ID_TO_ASCII, fptr->result);
  StrToLink (file_id.name);
}
/*
-- ----------------------------------------------------------------------------
-- void FileDeriveId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.derive.id protocol.
-- ----------------------------------------------------------------------------
*/
void FileDeriveId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT source_file_id;
  BYTE NewParentAbsIdStr[FILE_NAME_LENGTH];
  struct ATTR_STRUCT attr;
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  FileIdFromLink (&source_file_id);
  StrFromLink (NewParentAbsIdStr);
  AttrFromLink (&attr);
  Ch = FindExtension (source_file_id.name);

  if (Ch == NULL)
    Ch = strchr (source_file_id.name, '\0');

  if (DeriveExtension (Ch, attr.type, attr.contents) == FALSE)
    fptr->result = F_ILLEGAL_ATTRIBUTE;

  ResultToLink (Channel, FKF_DERIVE_ID, fptr->result);
  StrToLink (source_file_id.name);
}
/*
-- ----------------------------------------------------------------------------
-- void FileGetVersion
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.get.version protocol.
-- ----------------------------------------------------------------------------
*/
void FileGetVersion (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE absid_str[FILE_NAME_LENGTH];
  struct stat file_stat;
  BYTE *P;

  StrFromLink (absid_str);
  if (stat(absid_str, &file_stat) == 0)
    P = &(file_stat.st_ctime);
  else
    fptr->result = F_NO_SUCH_FILE;
  ResultToLink (Channel, FKF_GET_VERSION, fptr->result);

  if (fptr->result == F_OK)
    {
      int i;
      WordToLink (sizeof(file_stat.st_ctime));
      for (i = 0; (i < sizeof(file_stat.st_ctime)); i++)
        ByteToLink (*(P++));
    }
  else
    NullSliceToLink();
}
/*
-- ----------------------------------------------------------------------------
-- void FileMoveFileId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.move.file.id protocol.
-- ----------------------------------------------------------------------------
*/
void FileMoveFileId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  strcpy (AbsIdStr, TopLevelFile);
  Ch = strchr (AbsIdStr, '.');

  if (Ch == NULL)
    Ch = strchr (AbsIdStr, '\0');
  strcpy (Ch, MOVE_FILE_EXTN);

  ResultToLink (Channel, FKF_MOVE_FILE_ID, fptr->result);
  StrToLink (AbsIdStr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileToolKitFileId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.move.file.id protocol.
-- ----------------------------------------------------------------------------
*/
void FileToolKitFileId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  strcpy (AbsIdStr, TOOLKIT_FILE_NAME);
  Ch = strchr (AbsIdStr, '.');

  if (Ch == NULL)
    Ch = strchr (AbsIdStr, '\0');
  strcpy (Ch, TOOLKIT_FILE_EXTN);

  ResultToLink (Channel, FKF_TOOLKIT_FILE_ID, fptr->result);
  StrToLink (AbsIdStr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileDeleteFileId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.move.file.id protocol.
-- ----------------------------------------------------------------------------
*/
void FileDeleteFileId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  strcpy (AbsIdStr, DELETE_FILE_NAME);
  Ch = strchr (AbsIdStr, '.');

  if (Ch == NULL)
    Ch = strchr (AbsIdStr, '\0');
  strcpy (Ch, DELETE_FILE_EXTN);

  ResultToLink (Channel, FKF_DELETE_FILE_ID, fptr->result);
  StrToLink (AbsIdStr);
}
/*
-- ----------------------------------------------------------------------------
-- void FilePickFileId
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.move.file.id protocol.
-- ----------------------------------------------------------------------------
*/
void FilePickFileId (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  strcpy (AbsIdStr, PICK_FILE_NAME);
  Ch = strchr (AbsIdStr, '.');

  if (Ch == NULL)
    Ch = strchr (AbsIdStr, '\0');
  strcpy (Ch, PICK_FILE_EXTN);

  ResultToLink (Channel, FKF_PICK_FILE_ID, fptr->result);
  StrToLink (AbsIdStr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileMoveFile
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.move.file protocol.
-- ----------------------------------------------------------------------------
*/
void FileMoveFile (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  BYTE NewParentAbsIdStr[FILE_NAME_LENGTH];

  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  StrFromLink (NewParentAbsIdStr);
  ConvertToLinux(&NewParentAbsIdStr);
  ResultToLink(Channel, FKF_MOVE_FILE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileDeleteFile
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.delete.file protocol.
-- ----------------------------------------------------------------------------
*/
void FileDeleteFile (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];

  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  if (unlink(AbsIdStr) != 0)
    fptr->result = F_CANNOT_DELETE;
  ResultToLink (Channel, FKF_DELETE_FILE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileCopyFile
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.copy.file protocol.
-- ----------------------------------------------------------------------------
*/
void FileCopyFile (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  int Length, ExtnStart, VariantStart, Exists, i;
  int FNameStart = 0;

  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  Length = ExtnStart = strlen(AbsIdStr);

  /* Set up pointers to start of filename and start of filename extension */
  for (i = Length; i >= 0; i--)
    {
      if (AbsIdStr[i] == '.')
        ExtnStart = i;
      else
        {
          if ((AbsIdStr[i] == '\\') || (AbsIdStr[i] == ':'))
            {
              FNameStart = i + 1;
              break;
            }
        };
    };

  /* Check we have a variant, and if not, create one */
  if ((ExtnStart - FNameStart) >= VARIANT_LENGTH)
    {
      if ((ExtnStart - FNameStart) <= F_LENGTH)
        VariantStart = ExtnStart - VARIANT_LENGTH;
      else
        VariantStart = FNameStart + F_LENGTH - VARIANT_LENGTH;

      if (VariantExists (AbsIdStr+VariantStart))
        IncVariant (AbsIdStr, VariantStart);
      else
        CreateVariant (AbsIdStr, FNameStart, ExtnStart);
    }
  else
    CreateVariant (AbsIdStr, FNameStart, ExtnStart);

  /* Test to see if the file exists, if so, modify the variant and try again */
  Exists = TRUE;
  for (i = 0; ((i < MAX_SEARCH_COUNT) && Exists); i++)
    {
      Exists = FileExists(AbsIdStr);
      if (Exists)
        IncVariant (AbsIdStr, VariantStart);
    };

  if (Exists)
    {
      BYTE *Extn;
      BYTE TempName[FILE_NAME_LENGTH];
      BYTE TempExtn[FILE_NAME_LENGTH];
      int *IgnoreAttr, *IgnoreContents;
      int TempLen, ExtnLen;

      Extn = FindExtension (AbsIdStr);
      strcpy (TempExtn, Extn);
      ExtnLen = strlen (Extn);

      if (ValidExtension (Extn, IgnoreAttr, IgnoreContents))
        {
          register int j = 0;

          MakeUniqueFileName (TempName);
          TempLen = strlen (TempName);
          strcpy (AbsIdStr, TempName);

          for (i=8; i < ExtnLen+8; i++)
            AbsIdStr[i] = TempExtn[j++];
        };
      AbsIdStr[i] = '\0';
    };
  ResultToLink (Channel, FKF_COPY_FILE, fptr->result);
  StrToLink (AbsIdStr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileExistsFile
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.exists.file protocol.
-- ----------------------------------------------------------------------------
*/
void FileExistsFile (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr[FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT file_id;

  StrFromLink(ParentAbsIdStr);
  ConvertToLinux(&ParentAbsIdStr);
  FileIdFromLink(&file_id);
  if ( ! FileExists(file_id.name))
    fptr->result = F_NO_SUCH_FILE;
  ResultToLink (Channel, FKF_EXISTS_FILE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileChrcCanChange
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.chrc.can.change protocol.
-- ----------------------------------------------------------------------------
*/
void FileChrcCanChange (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  int ChrcNumber;
  ChrcNumber = WordFromLink();
  if ((ChrcNumber >= 0) && (ChrcNumber <= NUMBER_OF_CHRC))
    fptr->result = ChrcTable[ChrcNumber].changeable;
  else
    fptr->result = F_ILLEGAL_CHRC;
  ResultToLink (Channel, FKF_CHRC_CAN_CHANGE, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileChrcName
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.chrc.name protocol.
-- ----------------------------------------------------------------------------
*/
void FileChrcName (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  int ChrcNumber;
  BYTE Ch = '\0';                                 /* Dummy string */
  BYTE *StrPtr = &Ch;
  ChrcNumber = WordFromLink();

  if ((ChrcNumber >= 0) && (ChrcNumber <= NUMBER_OF_CHRC))
    StrPtr = ChrcTable[ChrcNumber].name;
  else
    fptr->result = F_ILLEGAL_CHRC;

  ResultToLink (Channel, FKF_CHRC_NAME, fptr->result);
  StrToLink (StrPtr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileOpenReadChrc
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.open.read.chrc protocol.
-- ----------------------------------------------------------------------------
*/
void FileOpenReadChrc (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);

  if (fptr->state == FS_IDLE)
    {
      if (FileExists(AbsIdStr))
        {
          fptr->state = FS_OPEN_READ_CHRC;
          strcpy (fptr->file_name, AbsIdStr);
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;

  ResultToLink (Channel, FKF_OPEN_READ_CHRC, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileOpenWriteChrc
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.open.write.chrc protocol.
-- ----------------------------------------------------------------------------
*/
void FileOpenWriteChrc (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  if (fptr->state == FS_IDLE)
    {
      if (FileExists(AbsIdStr))
        {
          fptr->state = FS_OPEN_WRITE_CHRC;
          strcpy (fptr->file_name, AbsIdStr);
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_OPEN_WRITE_CHRC, fptr->result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileReadChrc
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.read.chrc protocol.
-- ----------------------------------------------------------------------------
*/
void FileReadChrc (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  int ChrcNumber;
  BYTE Ch = '\0';                          /* Dummy string */
  BYTE *StrPtr = &Ch;
  ChrcNumber = WordFromLink();
  if (fptr->state == FS_OPEN_READ_CHRC)
    {
      switch (ChrcNumber)
      /* Get the file characteristic and point str_ptr to it */
        {
          case FILETIMECHRC:
            {
              struct stat buffer;

              if (stat(fptr->file_name, &buffer) == 0)
                StrPtr = ctime (&(buffer.st_ctime));
              else
                fptr->result = F_CANNOT_READ_CHRC;
              break;
            };
          case FILESIZECHRC:
            {
              struct stat buffer;

              if (stat(fptr->file_name, &buffer) == 0)
                {
                  int length = sprintf (fptr->buffer, "%ld", buffer.st_size);
                  (fptr->buffer)[length] = '\0';
                  StrPtr = fptr->buffer;
                }
              else
                fptr->result = F_CANNOT_READ_CHRC;
              break;
            };
          case READONLYCHRC:
            {
              if (access(fptr->file_name, WRITE_PERMISSION) == 0)
                strcpy (fptr->buffer, "FALSE\0");
              else
                strcpy(fptr->buffer, "TRUE\0");
              StrPtr = fptr->buffer;
              break;
            };
          default: fptr->result = F_ILLEGAL_CHRC;
        }
    }
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_READ_CHRC, fptr->result);
  StrToLink (StrPtr);
}
/*
-- ----------------------------------------------------------------------------
-- void FileWriteChrc
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.write.chrc protocol.
-- ----------------------------------------------------------------------------
*/
void FileWriteChrc (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  int ChrcNumber;
  BYTE ChrcRecord[FILE_NAME_LENGTH];
  ChrcNumber = WordFromLink();
  StrFromLink (ChrcRecord);
  if (fptr->state == FS_OPEN_WRITE_CHRC)
    {
      switch (ChrcNumber)
      /* Decode chrc_record and assign it to file */
        {
          case READONLYCHRC:
            {
              if (strcmp(ChrcRecord, "TRUE\0") == 0)
              {
                  if (chmod(fptr->file_name, S_IREAD) != 0)
                      fptr->result = F_CANNOT_CHANGE_CHRC;
              }
              else
              {
                  if (strcmp(ChrcRecord, "FALSE\0") == 0)
                  {
                      if (chmod(fptr->file_name, S_IREAD | S_IWRITE) != 0)
                          fptr->result = F_CANNOT_CHANGE_CHRC;
                  } else
                      fptr->result = F_INVALID_OPERATION;
              }
              break;
            }
          default: fptr->result = F_ILLEGAL_CHRC;
        }
    }
  else
      fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_WRITE_CHRC, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileWriteChrc
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.close.chrc protocol.
-- ----------------------------------------------------------------------------
*/
void FileCloseChrc (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  if ((fptr->state == FS_OPEN_READ_CHRC) ||
      (fptr->state == FS_OPEN_WRITE_CHRC))
      fptr->state = FS_IDLE;
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_CLOSE_CHRC, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileSeek
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.seek protocol.
-- ----------------------------------------------------------------------------
*/
void FileSeek (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  long position = LongWordFromLink();
  switch (fptr->state)
    {
      case FS_OPEN_FOR_READING:
      case FS_OPEN_FOR_WRITING:
        {
          if (fptr->state == FS_OPEN_FOR_WRITING)
            FlushBuffer (fptr);

          if (fseek(fptr->handle, position, S_BEGINNING) != 0)
            fptr->result = F_CANNOT_SEEK;
          else
            {
              if (fptr -> state == FS_OPEN_FOR_READING)
                {
                  ResultToLink (Channel, FKF_SEEK, fptr->result);
                  ReadAhead (fptr);
                  return;                     /* Go back  */
                }
            }
          break;
        };
      case FS_OPEN_BLOCK_READ:
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        {
          if (fseek(fptr->handle, position, S_BEGINNING) != 0)
            fptr->result = F_CANNOT_SEEK;
          else
            fptr->block_posn = position % RECORD_LENGTH;
          break;
        };
      default:
        fptr->result = F_INVALID_OPERATION;
        break;
    };
  ResultToLink (Channel, FKF_SEEK, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileFreeSpace
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.free.space protocol.
-- ----------------------------------------------------------------------------
*/
void FileFreeSpace (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr[FILE_NAME_LENGTH];
  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  ResultToLink (Channel, FKF_FREE_SPACE, fptr->result);
  WordToLink (F_UNKNOWN_FREE_SPACE);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileOpenReadText
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.open.read.text protocol.
-- ----------------------------------------------------------------------------
*/
void FileOpenReadText (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr [FILE_NAME_LENGTH];

  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  if (fptr->state == FS_IDLE)
    {
      FILE *f_handle  = fopen(AbsIdStr, "r");
      if (f_handle != NULL)
        {
          fptr->handle = f_handle;
          strcpy (fptr->file_name, AbsIdStr);
          fptr->state = FS_OPEN_FOR_READING_TEXT;
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink(Channel, FKF_OPEN_READ_TEXT, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileOpenWriteText
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.open.write.text protocol.
-- ----------------------------------------------------------------------------
*/
void FileOpenWriteText (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE AbsIdStr [FILE_NAME_LENGTH];

  StrFromLink (AbsIdStr);
  ConvertToLinux(&AbsIdStr);
  if (fptr->state == FS_IDLE)
    {
      FILE *f_handle = fopen(AbsIdStr, "w");
      if (f_handle != NULL)
        {
          fptr->handle = f_handle;
          strcpy (fptr->file_name, AbsIdStr);
          fptr->state = FS_OPEN_FOR_WRITING_TEXT;
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, FKF_OPEN_WRITE_TEXT, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileResetChannel
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.reset.channel protocol.
-- ----------------------------------------------------------------------------
*/
void FileResetChannel(Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  FilerCloseFile (fptr); /* Ignore result */
  FilerInitFile (fptr);
  ResultToLink (Channel, FKF_RESET_CHANNEL, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileFileLength
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.file.length protocol.
-- ----------------------------------------------------------------------------
*/
void FileFileLength (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  long int FileLength = 0;

  switch (fptr->state)
    {
      case FS_OPEN_FOR_READING:
      case FS_OPEN_FOR_WRITING:
      case FS_OPEN_FOR_READING_TEXT:
      case FS_OPEN_FOR_WRITING_TEXT:
      case FS_OPEN_BLOCK_READ:
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        {
          fptr->result = SeekFileLength (fptr, &FileLength);
          break;
        };
      default:
        fptr->result = F_INVALID_OPERATION;
    };

  ResultToLink (Channel, FKF_FILE_LENGTH, fptr->result);
  LongWordToLink (FileLength);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileRename
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.file.rename protocol.
-- ----------------------------------------------------------------------------
*/
void FileRename (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  BYTE ParentAbsIdStr [FILE_NAME_LENGTH];
  struct FILE_ID_STRUCT file_id, new_file_id;
  BYTE *Ch;

  StrFromLink (ParentAbsIdStr);
  ConvertToLinux(&ParentAbsIdStr);
  FileIdFromLink(&file_id);
  FileIdFromLink(&new_file_id);

  if (DoRename (file_id.name, new_file_id.name) != 0)
    fptr->result = F_CANNOT_RENAME;
  ResultToLink (Channel, FKF_RENAME, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileFlush
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.flush protocol.
-- ----------------------------------------------------------------------------
*/
void FileFlush (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  switch (fptr->state)
    {
      case FS_OPEN_FOR_WRITING_TEXT:
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        {
          if (fflush (fptr->handle) != 0)
            fptr->result = F_CANNOT_FLUSH;
          break;
        }
      case FS_OPEN_FOR_WRITING:
        {
          if (fptr->rwresult == F_OK)
            {
              FlushBuffer (fptr);
              if (fflush (fptr->handle) != F_OK)
                fptr->result = F_CANNOT_FLUSH;
            }
          else
            fptr->result = fptr->rwresult;
          break;
        }
      default:
        fptr->result = F_INVALID_OPERATION;
    };
  ResultToLink (Channel, FKF_FLUSH, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileOpenBlock
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    None.
--
-- Notes:
-- Obey tkf.open.block.read, tkf.open.block.write, or tkf.open.block.update
-- protocol, depending upon the value of mode.
-- ----------------------------------------------------------------------------
*/
void FileOpenBlock (Channel, fptr, Mode)
int Channel;
struct FILER_SPACE_DEF *fptr;
int Mode;
{
  BYTE FileAbsIdStr [FILE_NAME_LENGTH];

  StrFromLink (FileAbsIdStr);
  ConvertToLinux(&FileAbsIdStr);
  if ((fptr->state) == FS_IDLE)
    {
      FILE *FHandle;
      switch (Mode)
        {
          case TKF_OPEN_BLOCK_READ:   FHandle = fopen(FileAbsIdStr, "rb");
                                      break;
          case TKF_OPEN_BLOCK_WRITE:  FHandle = fopen(FileAbsIdStr, "wb");
                                      break;
          case TKF_OPEN_BLOCK_UPDATE: FHandle = fopen(FileAbsIdStr, "rb+");
                                      break;
        };

      if (FHandle != NULL)
        {
          fptr->handle = FHandle;
          fptr->buffer_count = 0;
          fptr->buffer_ptr = 0;
          strcpy (fptr->file_name, FileAbsIdStr);
          switch (Mode)
            {
              case TKF_OPEN_BLOCK_READ:
                fptr->state = FS_OPEN_BLOCK_READ;
                fptr->last_op = READ;
                break;
              case TKF_OPEN_BLOCK_WRITE:
                fptr->state = FS_OPEN_BLOCK_WRITE;
                fptr->last_op = WRITE;
                break;
              case TKF_OPEN_BLOCK_UPDATE:
                fptr->state = FS_OPEN_BLOCK_UPDATE;
                fptr->last_op = READ;
                break;
            };
          fptr->block_posn = 0;
        }
      else
        fptr->result = F_NO_SUCH_FILE;
    }
  else
    fptr->result = F_INVALID_OPERATION;
  ResultToLink (Channel, Mode, fptr->result);
}
/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- void FileTruncate
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--    (struct) *fptr - Users filer data space.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.truncate protocol.
-- ----------------------------------------------------------------------------
*/
void FileTruncate (Channel, fptr)
int Channel;
struct FILER_SPACE_DEF *fptr;
{
  switch (fptr->state)                        /* Truncate not implemented */
    {
      default:
        fptr->result = F_INVALID_OPERATION;
        break;
    };
  ResultToLink (Channel, FKF_TRUNCATE, fptr->result);
}

/*
-- ---------------------------------------------------------------------------- void FileCloseChrc (Channel, fptr)
-- int DoFilerChan - Process the filer message which is coming down the link.
--
-- Input Parameters:
--    (int) Channel - Channel number of filer.
--
-- Output Parameters:
--    (int) Result.
--
-- Notes:
-- Obey tkf.truncate protocol.
-- ----------------------------------------------------------------------------
*/
int DoFilerChan(Channel)
int Channel;
{
  int Result = T_OK;          /* Result is a terminate code */
                              /* if we have a fatal error   */
  if (FilerRunning)           /* .. then chase it (teehee)  */
    {
      /* Read the filer command. */
      int Command = WordFromLink();
      /* Point fptr to the required filer's data space. */
      struct FILER_SPACE_DEF *fptr = &filer_data_space[Channel];
      fptr->result = F_OK;

      switch (Command)
        {
          case TKF_LOCK_READ:     Result = FileLockRead(Channel, fptr); break;
          case TKF_LOCK_WRITE:    FileLockWrite(Channel, fptr); break;
          case TKF_RELEASE:       FileRelease(Channel, fptr); break;
          case TKF_OPEN_READ:     FileOpenRead(Channel, fptr); break;
          case TKF_OPEN_WRITE:    FileOpenWrite(Channel, fptr); break;
          case TKF_CLOSE:         FileClose(Channel, fptr); break;
          case TKF_READ:          FileRead(Channel, fptr); break;
          case TKF_WRITE:         FileWrite(Channel, fptr); break;
          case TKF_SUSPEND:       FileSuspend(Channel, fptr); break;
          case TKF_RESUME:        FileResume(Channel, fptr); break;
          case TKF_NAMED_FILE_STORE:
                                  ResultToLink (Channel, FKF_NAMED_FILE_STORE,
                                      F_OK); break;
          case TKF_ASCII_TO_ID:   FileAsciiToId(Channel, fptr); break;
          case TKF_ID_TO_ASCII:   FileIdToAscii(Channel, fptr); break;
          case TKF_DERIVE_ID:     FileDeriveId(Channel, fptr); break;
          case TKF_GET_VERSION:   FileGetVersion(Channel, fptr); break;
          case TKF_MOVE_FILE_ID:  FileMoveFileId(Channel, fptr); break;
          case TKF_CAN_MOVE:      ResultToLink (Channel, FKF_CAN_MOVE, F_OK);
                                  break;
          case TKF_CAN_DELETE:    ResultToLink (Channel, FKF_CAN_DELETE, F_OK);
                                  break;
          case TKF_CAN_COPY:      ResultToLink (Channel, FKF_CAN_COPY, F_OK);
                                  break;
          case TKF_MOVE_FILE:     FileMoveFile(Channel, fptr); break;
          case TKF_DELETE_FILE:   FileDeleteFile(Channel, fptr); break;
          case TKF_COPY_FILE:     FileCopyFile(Channel, fptr); break;
          case TKF_EXISTS_FILE:   FileExistsFile(Channel, fptr); break;
          case TKF_NUMBER_OF_CHRC:
                                  ResultToLink (Channel, FKF_NUMBER_OF_CHRC,
                                      NUMBER_OF_CHRC); break;
          case TKF_CHRC_CAN_CHANGE:
                                  FileChrcCanChange(Channel, fptr); break;
          case TKF_CHRC_NAME:     FileChrcName(Channel, fptr); break;
          case TKF_OPEN_READ_CHRC:
                                  FileOpenReadChrc(Channel, fptr); break;
          case TKF_OPEN_WRITE_CHRC:
                                  FileOpenWriteChrc(Channel, fptr); break;
          case TKF_READ_CHRC:     FileReadChrc(Channel, fptr); break;
          case TKF_WRITE_CHRC:    FileWriteChrc(Channel, fptr); break;
          case TKF_CLOSE_CHRC:    FileCloseChrc(Channel, fptr); break;
          case TKF_GET_ERROR:     ResultToLink(Channel, FKF_GET_ERROR,
                                      fptr->host_error); break;
          case TKF_TERMINATE:     FilerRunning = FALSE;
                                  ResultToLink(Channel, FKF_TERMINATE,
                                      fptr->result); break;
          case TKF_SEEK:          FileSeek(Channel, fptr); break;
          case TKF_FREE_SPACE:    FileFreeSpace(Channel, fptr); break;
          case TKF_OPEN_READ_TEXT:
                                  FileOpenReadText (Channel, fptr); break;
          case TKF_OPEN_WRITE_TEXT:
                                  FileOpenWriteText(Channel, fptr); break;
          case TKF_RESET_CHANNEL: FileResetChannel (Channel, fptr); break;
          case TKF_FILE_LENGTH:   FileFileLength (Channel, fptr); break;
          case TKF_RENAME:        FileRename (Channel, fptr); break;
          case TKF_FLUSH:         FileFlush (Channel, fptr); break;
          case TKF_OPEN_BLOCK_READ:
          case TKF_OPEN_BLOCK_WRITE:
          case TKF_OPEN_BLOCK_UPDATE:
                                  FileOpenBlock (Channel, fptr, Command);
                                  break;
          case TKF_TRUNCATE:        FileTruncate (Channel, fptr); break;
          case TKF_TOOLKIT_FILE_ID: FileToolKitFileId (Channel, fptr); break;
          case TKF_DELETE_FILE_ID:  FileDeleteFileId (Channel, fptr); break;
          case TKF_PICK_FILE_ID:    FilePickFileId (Channel, fptr); break;
          default:
              Result = T_BAD_FILER_COMMAND;
        };
    }
  else
    Result = T_FILER_CLOSED;
  return(Result);
}

void ConvertToLinux(String)
char * String;
{
  /* Try to convert a DOS-Filename to Linux */
  char NewString[FILE_NAME_LENGTH];
  int ReadPointer = 0;
  int WritePointer = 0;
  int DosDrive, i;

  if ( *String == 'p' && *(String+1) == 'r' && *(String+2) == 'n' && *(String+3) == '\0' )
  {
    i = 0;
    ReadPointer = 3;
    while (Printer[i] != '\0')
     NewString[WritePointer++] = Printer[i++];
  }

  while ( *( String + ReadPointer ) != (char) 0 )
  {
    if ( ( ReadPointer == 0) && (*String == '\\') )
    {
      i = 0;
      while ( RootPrefix[i] != (char) 0 )
        NewString[WritePointer++] = RootPrefix[i++];
    }
    if ( *( String + ReadPointer + 1 ) == ':')
    {
      DosDrive = -1;
      if ( ( *( String + ReadPointer ) <= 'z' ) && 
	   ( *( String + ReadPointer ) >= 'a' ) )
	DosDrive = * (String + ReadPointer) - 'a';
      if ( ( *( String + ReadPointer ) <= 'Z' ) && 
	   ( *( String + ReadPointer ) >= 'A' ) )
	DosDrive = * (String + ReadPointer) - 'A';
      if (DosDrive >= DOS_DRIVES)
	DosDrive = -1;
      if (DosDrive >= 0)
      {
	i = 0;
	ReadPointer += 2;
	while ( DosMount[DosDrive][i] != (char) 0 )
          NewString[WritePointer++] = DosMount[DosDrive][i++];
      }
    }
    else if ( *( String + ReadPointer ) == '\\' )
    {
      NewString[WritePointer++] = '/';
      ReadPointer++;
    }
    else if ( ( *( String + ReadPointer ) >= 'A' ) && 
	      ( *( String + ReadPointer ) <= 'Z' ) )
      NewString[WritePointer++] = * (String + ReadPointer++) + ( 'a' - 'A' );
    else
      NewString[WritePointer++] = * (String + ReadPointer++);
  }
  NewString[WritePointer++] = (char) 0;

  for (i = 0; i < WritePointer; i++ ) * (String + i) = NewString[i];
}
