%{
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <netinet6/in6.h>
#include "radv.h"
#include "debug.h"

extern int yylex(void);

extern struct Interface *IfaceList;
struct Interface *iface = NULL;
struct AdvPrefix *prefix = NULL;

extern int iface_num;

extern int num_lines;
extern char *yytext;

void iface_init_defaults(struct Interface *);
void palloc_check(void);

void yyerror(char *msg);

%}

%token		T_INTERFACE
%token		T_PREFIX
%token		T_LEN

%token	<str>	STRING
%token	<num>	NUMBER
%token	<bool>	SWITCH
%token	<addr>	ADDR

%token		T_AdvSendAdvert
%token		T_MaxRtrAdvInterval
%token		T_MinRtrAdvInterval
%token		T_AdvManagedFlag
%token		T_AdvOtherConfigFlag
%token		T_AdvLinkMTU
%token		T_AdvReachableTime
%token		T_AdvRetransTimer
%token		T_AdvCurHopLimit
%token		T_AdvDefaultLifetime

%token		T_AdvOnLink
%token		T_AdvAutonomous
%token		T_AdvValidLifetime
%token		T_AdvPreferedLifetime

%type	<str>	name
%type	<pinfo> prefixdef prefixlist

%union {
	int			num;
	int			bool;
	struct in6_addr		*addr;
	char			*str;
	struct AdvPrefix	*pinfo;
};

%%

grammar		: grammar ifacedef
		| ifacedef
		;

ifacedef	: T_INTERFACE name '{' ifaceparams  '}' ';'
		{
			strcpy(iface->Name, $2);

			iface->next = IfaceList;
			IfaceList = iface;
			iface_num++;
			iface = NULL;

			dprintf(4, "interface definition ok\n");
		};
	
name		: STRING
		{
			/* check vality */
			$$ = $1;
		}
		;

ifaceparams	: iface_advt ifacevlist prefixlist
		{
			iface->AdvPrefixList = $3;
		}
		;

ifacevlist	: ifacevlist ifaceval
		| ifaceval
		|
		;

iface_advt	: T_AdvSendAdvert SWITCH ';'
		{
			iface = malloc(sizeof(struct Interface));
			iface_init_defaults(iface);
			iface->AdvSendAdvert = $2;
		}

ifaceval	: T_MinRtrAdvInterval NUMBER ';'
		{
			iface->MinRtrAdvInterval = $2;
		}
		| T_MaxRtrAdvInterval NUMBER ';'
		{
			iface->MaxRtrAdvInterval = $2;
		}
		| T_AdvManagedFlag SWITCH ';'
		{
			iface->AdvManagedFlag = $2;
		}
		| T_AdvOtherConfigFlag SWITCH ';'
		{
			iface->AdvOtherConfigFlag = $2;
		}
		| T_AdvLinkMTU NUMBER ';'
		{
			/* check vality */
			iface->AdvLinkMTU = $2;
		}
		| T_AdvReachableTime NUMBER ';'
		{
			iface->AdvReachableTime = $2;
		}
		| T_AdvRetransTimer NUMBER ';'
		{
			iface->AdvRetransTimer = $2;
		}
		| T_AdvDefaultLifetime NUMBER ';'
		{
			iface->AdvDefaultLifetime = $2;
		}
		| T_AdvCurHopLimit NUMBER ';'
		{
			if ($2 > 255)
			{
				fprintf(stderr, "Invalid HopLimit\n");
				exit(1);
			}
			iface->AdvCurHopLimit = $2;
		}
		;
		
prefixlist	: prefixdef
		{
			$$ = $1;
		}
		| prefixlist prefixdef
		{
			$2->next = $1;
			$$ = $2;
		}
		;

prefixdef	: T_PREFIX ADDR T_LEN NUMBER '{' prefixplist '}' ';'
		{
			palloc_check();

			if ($4 > 128)
			{
				fprintf(stderr, "invalid prefix length\n");
				exit(1);
			}
			
			if ($4 != 80)
			{
				fprintf(stderr, "warning: prefix length should"
					" be equal to 128 - link layer token"
					" length\n");
			}

			prefix->PrefixLen = $4;

			if (prefix->AdvPreferedLifetime >
			    prefix->AdvValidLifetime)
			{
				fprintf(stderr, "Valid lifetime must be "
					"superior to prefered lifetime\n");
				exit(1);
			}

			memcpy(&prefix->Prefix, $2, sizeof(struct in6_addr));
			
			$$ = prefix;
			prefix = NULL;
		}
		;

prefixplist	: prefixplist prefixparms
		| prefixparms
		;

prefixparms	: /* empty */
		| T_AdvOnLink SWITCH ';'
		{
			palloc_check();
			prefix->AdvOnLinkFlag = $2;
		}
		| T_AdvAutonomous SWITCH ';'
		{
			palloc_check();
			prefix->AdvAutonomousFlag = $2;
		}
		| T_AdvValidLifetime NUMBER ';'
		{
			palloc_check();
			prefix->AdvValidLifetime = $2;
		}
		| T_AdvPreferedLifetime NUMBER ';'
		{
			palloc_check();
			prefix->AdvPreferedLifetime = $2;
		}
		;

%%

void yyerror(char *msg)
{
	if (iface)
	{
		free(iface);
	}
	
	if (prefix)
	{
		free(prefix);
	}

	fprintf(stderr, "%s on line %d: %s\n", msg, num_lines, yytext);
	exit(1);
}

void iface_init_defaults(struct Interface *iface)
{
	memset(iface, 0, sizeof(struct Interface));
	iface->MaxRtrAdvInterval  = 600;
	iface->MinRtrAdvInterval  = 200;
	iface->AdvDefaultLifetime = 1800;
}

void palloc_check(void)
{
	if (prefix == NULL)
	{
		prefix = malloc(sizeof(struct AdvPrefix));
		
		if (prefix == NULL)
		{
			dprintf(0, "malloc failed\n");
			exit(1);
		}

		memset(prefix, 0, sizeof(struct AdvPrefix));
		
		prefix->AdvOnLinkFlag = 1;
		prefix->AdvAutonomousFlag = 1;
		prefix->AdvValidLifetime = ~0;
		prefix->AdvPreferedLifetime = 604800;
	}
}
