/* ------------------------------------------------------------------------- */
/* adap-bit.c i2c driver algorithms for bit-shift adapters		     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995-96 Simon G. Vogl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */
static char rcsid[] = "$Id: alg-bit.c,v 1.3 1996/11/18 22:24:45 i2c Exp $";
/*
 * $Log: alg-bit.c,v $
 * Revision 1.3  1996/11/18 22:24:45  i2c
 * fixed minor init bug: init now returns the value of bit_init.
 *
 * Revision 1.2  1996/11/17 22:00:04  i2c
 * removed intermediate comments
 *
 * Revision 1.1  1996/11/17 14:00:51  i2c
 * Initial revision
 *
 *
 */

#include <linux/kernel.h>

#include <linux/delay.h>

#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/sched.h>

#include "i2c.h"
#include "i2c_priv.h"

/* ----- global defines ---------------------------------------------------- */
#define DEB(x)		/* should be reasonable open, close &c. 	*/
#define DEB2(x) 	/* low level debugging - very slow 		*/
#define DEBE(x)	x	/* error messages 				*/
#define DEBI(x) 	/* ioctl and its arguments 			*/
#define DEBACK(x)	/* ack failed message				*/
#define DEBSTAT(x) x	/* print several statistical values		*/

/* debugging - slow down transfer to have a look at the data .. 	*/
/* I use this with two leds&resistors, each one connected to sda,scl 	*/
/* respectively. This makes sure that the algorithm works. Some chips   */
/* might not like this, as they have an internal timeout of some mils	*/
/*
#define SLO_IO      jif=jiffies;while(jiffies<=jif+i2c_table[minor].veryslow);
*/

/* ----- global variables ---------------------------------------------	*/

#ifdef SLO_IO
	int jif;
#endif


#if (BITADAPS > 1)
/*
 *  This array contains the hw-specific functions for
 *  each port (hardware) type.
 */
struct i2c_bit_opns *bit_adaps[I2C_HW_ADAPS];
#endif

#if (BITADAPS) > 1
/* --- dummy functions for empty entries ------------------------------	*/
static void __dummy_set_port(int minor,int i)
	{
	}
static int  __dummy_get_port(int minor) 
	{
		return 0;
	}
static int  __dummy_init(int minor) 
	{
		return -ENODEV;
	}
static void __dummy_exit(int minor) {}

static struct i2c_bit_opns dummy_bitops={
	__dummy_set_port,
	__dummy_set_port,
	__dummy_get_port,
	__dummy_get_port,
	__dummy_init,
	__dummy_exit,
};
#endif
/* --- setting states on the bus with the right timing: ---------------	*/

static void sdalo(int minor)
{
	setsda(minor,0);
	udelay(WLO);
}

static void sdahi(int minor)
{
	setsda(minor,1);
	udelay(WHI);
}

static void scllo(int minor)
{
	setscl(minor,0);
	udelay(WLO);
	udelay(i2c_table[minor].udelay);
}

/*
 * Raise scl line, and do checking for delays. This is necessary for slower
 * devices.
 */
static int sclhi(int minor)
{
	int start=jiffies;

	setscl(minor,1);
	udelay(WHI);				
	udelay(i2c_table[minor].udelay);
 	while (! getscl(minor) ) {		/* wait till high	*/
		setscl(minor,1);
		if (start+i2c_table[minor].timeout<=jiffies){
			DEBE(printk("sclhi failed\n"));
			return -ETIMEDOUT;
		}
		if (need_resched)
			schedule();
	}
/*	DEBSTAT(printk("needed %ld jiffies\n", jiffies-start));*/
	return 0;
} 



/* --- other auxiliary functions --------------------------------------	*/
static void i2c_start(int minor) 
{
	/* assert: scl, sda are high */
	DEB2(printk("i2c_start.\n"));
	sdalo(minor);
	udelay(WHI);   
	scllo(minor);
	udelay(WLO);   
}


static void i2c_stop(int minor) 
{
	DEB2(printk("i2c_stop.\n"));
	/* assert: scl is low */
	sdalo(minor);
	sclhi(minor); 
	udelay(WHI);    
	sdahi(minor);
	udelay(WHI);
}


static int i2c_outb(int minor, char c)
{
	int i;
	char ack;

	/* assert: scl is low */
	DEB2(printk("i2c_outb:%x\n",c));
	for ( i=7 ; i>=0 ; i-- ) {
		if ( c&(1<<i) ){
			sdahi(minor);
		} else {		
			sdalo(minor);
		}
		if (sclhi(minor)<0) { /* timed out */
			sdahi(minor); /* we don't want to block the net */
			return 0;
		};
		/* arbitrate here: */
		udelay(WHI);
		if ( (c&(1<<i)) && !getsda(minor) ){
			/* we lost the arbitration process -> get lost */
			DEBE(printk("i2c_write: arbitration bailout! \n"));
			return 0;
		}
		scllo(minor);	
	}
	sdahi(minor);
	if (sclhi(minor)<0){ /* timeout */
		return 0;
	};
	/* read ack */
	udelay(WHI);
	ack=!getsda(minor);	/* ack: sda is pulled low ->success.	 */
	DEB2(printk("sdain: 0x%x\n",sdain(minor)));
	scllo(minor);
	return ack;		/* return 1 if ok		 */
	/* assert: scl is low */
}



static int i2c_inb(int minor) 
{
	/* read byte via i2c port, without start/stop sequence	*/
	/* acknowledge is sent in i2c_read.			*/
	int i;
	char indata;

	/* assert: scl is low */
	DEB2(printk("i2c_inb.\n"));

	sdahi(minor);
	indata=0;
	for (i=0;i<8;i++){
		if (sclhi(minor)<0){ /* timeout */
			sdahi(minor); /* we don't want to block the net */
			return -ETIMEDOUT;
		};
		indata*=2;
		if ( getsda(minor) ) indata++;
		scllo(minor);
	}
	/* assert: scl is low */
	return indata;
}


/* ----- level 2: communication with the kernel  ----- */

static int i2c_open(struct inode * inode, struct file * file) 
{
	return 0;
}


static void i2c_release (struct inode * inode, struct file * file) 
{
}


static int i2c_write(struct inode * inode, struct file * file,
	const char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	char c,adr;
	const char *temp = buf;
	int retval,i;
	struct i2c_data *data;

	data=(struct i2c_data *)file->private_data;
	if (data && (data->magic==I2C_MAGIC))
		adr = data->address;
	else
		return -EINVAL;
	adr&=0xfe;
	i2c_start(minor);
	i=0;
	while ( ! i2c_outb(minor,adr) ) {
		i2c_stop(minor);
		i++;
		i2c_start(minor);
		if (i>=i2c_table[minor].retries ) {
			DEBACK(printk("i2c_write: i2c%d: address ack failed.\n",minor));
			i2c_stop(minor);
			return -EREMOTEIO;
		};
	}
	while (count > 0) {
		c = get_user(temp);
		DEB2(printk("i2c(%d)_write: writing %c\n",minor,c));
		retval = i2c_outb(minor,c);
		if (retval>0) { 
			count--; 
			temp++;
		} else { /* arbitration or no acknowledge */
			DEBE(printk("i2c(%d)_write: error - bailout.\n",minor));
			i2c_stop(minor);
			return -EREMOTEIO; /* got a better one ?? */
		}
		__delay(i2c_table[minor].mdelay * (loops_per_sec / 1000) );
	}
	i2c_stop(minor);
	DEB2(printk("i2c_write: wrote %d bytes.\n",temp-buf));
	return temp-buf ;
}


static int i2c_read(struct inode * inode, struct file * file,
	char * buf, int count) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	char *temp = buf;
	char adr;
	int i;
	struct i2c_data *data;

	data=(struct i2c_data *)file->private_data;
	if (data && (data->magic==I2C_MAGIC))
		adr = data->address;
	else
		return -EINVAL;
	adr |= 0x01;
	i2c_start(minor);
	i=0;
	while (!i2c_outb(minor,adr)) {
		i2c_stop(minor);
		i++;
		if (i>=i2c_table[minor].retries )  {
			DEBACK(printk("i2c_read: i2c%d: address ack failed.\n",minor));
			i2c_stop(minor);
			return -EREMOTEIO;
		};
		i2c_start(minor);
	}

	while (count >= 0) {
		put_user(i2c_inb(minor),temp);

		if (count==1){		/* send ack	*/
			sdahi(minor);
		}else{
			sdalo(minor);		/* reverse ack on last byte */
		}
		if (sclhi(minor)<0) { /* timeout */
			sdahi(minor);
			return -ETIMEDOUT;
		};
		scllo(minor);
		sdahi(minor);
		temp++;
		count--;
	}

	i2c_stop(minor);
	DEB(printk("i2c_read: %d bytes read.\n",temp-buf));
	return temp-buf ;
}


static int i2c_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	int retval = 0;

	switch ( cmd ) {
		case I2C_TIMEOUT:
			i2c_table[minor].timeout = arg;
			break;
		case I2C_UDELAY:
			i2c_table[minor].udelay = arg;
			case I2C_MDELAY:
	 			i2c_table[minor].mdelay = arg;
			break;
		case I2C_RETRIES:
			i2c_table[minor].retries = arg;
			break;
#ifdef SLO_IO
		case I2C_V_SLOW: 
			i2c_table[minor].veryslow = arg;
			break;
#endif
		case I2C_RESET:
			setsda(minor,1);
			setscl(minor,1);
			break;
		default:
			retval = -EINVAL;
	}
	return retval;
}


static int i2c_init(int minor)
{	
#if (BITADAPS > 1)
	bit_adaps[minor].init(minor);
#else	
	/* only one adapter type defined */
#endif
	return bit_init(minor);
}


static void i2c_exit(int minor)
{	
	bit_exit(minor);
}


/* -----exported file operations: -------------------------------------	*/
struct i2c_opns alg_bit_opns = {
	i2c_read,
	i2c_write,
	i2c_ioctl,
	i2c_open,
	i2c_release,
	i2c_init,
	i2c_exit
};


int alg_bit_init (void)
{
	printk("alg_bit_init\n");
#if (BITADAPS > 1)
	int i;
	for (i=0;i<I2C_HW_ADAPS;i++){	/* set dummy values first... */
		bit_adaps[i]=&dummy_bitops;
	}
#ifdef BIT_LP
	bit_adaps[IDX_B_LP]=&bit_lp_ops;
#endif
#ifdef BIT_SER
#endif
#else	
	/* only one adapter type defined -- nothing to do. */
#endif
	return 0;
}
