/* ----------------------------------------------------------------------- *
 * 
 * spawn.c - run programs synchronously
 *   
 *   Copyright 1997 Transmeta Corporation - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
 *   USA; either version 2 of the License, or (at your option) any later
 *   version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/wait.h>

#include "automount.h"

/* Used by subprocesses to avoid carrying over the main daemon's rather
   weird signalling environment */
void reset_signals(void)
{
  struct sigaction sa;
  sigset_t allsignals;
  int i;
  
  sigfillset(&allsignals);
  sigprocmask(SIG_BLOCK, &allsignals, NULL);
  
  sa.sa_handler = SIG_DFL;
  sigemptyset(&sa.sa_mask);
  sa.sa_flags = 0;
  for ( i = 1 ; i < NSIG ; i++ )
    sigaction(i, &sa, NULL);
  
  sigprocmask(SIG_UNBLOCK, &allsignals, NULL);
}

int spawnv(int logpri, char *prog, char **argv) {
  pid_t f;
  int status, nullfd, pipefd[2];
  FILE *errs;
  char errbuf[1024], *p;
  sigset_t allsignals, oldsig;
  
  sigfillset(&allsignals);
  sigprocmask(SIG_BLOCK, &allsignals, &oldsig);
  
  if ( pipe(pipefd) )
    return -1;

    f = fork();
  if ( f == 0 ) {
    reset_signals();
    if ( (nullfd = open("/dev/null", O_RDONLY)) ) {
      dup2(nullfd, STDIN_FILENO);
      close(nullfd);
    }
    close(pipefd[0]);
    dup2(pipefd[1], STDOUT_FILENO);
    dup2(pipefd[1], STDERR_FILENO);
    close(pipefd[1]);
    execv(prog, argv);
    _exit(255);	/* execv() failed */
  } else {
    sigprocmask(SIG_SETMASK, &oldsig, NULL);

    close(pipefd[1]);

    if ( f < 0 ) {
      close(pipefd[0]);
      return -1;
    }

    if ( (errs = fdopen(pipefd[0], "r")) ) {
      while ( fgets(errbuf, 1024, errs) ) {
	if ( (p = strchr(errbuf, '\n')) ) *p = '\0';
	if ( errbuf[0] )	/* Don't output empty lines */
	  syslog(logpri, ">> %s", errbuf);
      }
      fclose(errs);
    } else
      close(pipefd[0]);

    if ( waitpid(f, &status, 0) != f )
      return -1; /* waitpid() failed */
    else
      return status;
  }
}

int spawnl(int logpri, char *prog, ...)
{
  va_list arg;
  int argc;
  char **argv, **p;
  
  va_start(arg,prog);
  for ( argc = 1 ; va_arg(arg,char *) ; argc++ ); 
  va_end(arg);
  
  if ( !(argv = alloca(sizeof(char *) * argc)) )
    return -1;
  
  va_start(arg,prog);
  p = argv;
  while ((*p++ = va_arg(arg,char *)));
  va_end(arg);
  
  return spawnv(logpri, prog, argv);
}

