/* kernel.c - Processing of incoming kernel messages */
 
/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */
 

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <linux/atm.h>
#include <linux/atmsvc.h>
#include <linux/atmdev.h>

#include "atm.h"
#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "proto.h"
#include "io.h"
#include "timeout.h"


#define COMPONENT "KERNEL"


extern int assign_vci;

static struct sockaddr_atmsvc local; /* local address */


static int send_setup(SOCKET *sock)
{
    static unsigned long call_ref = 0;
    struct atm_bhli *bhli;
    struct atm_blli *blli;
    SOCKET *walk;
    Q_DSC dsc;
    int size;

    do {
	if (++call_ref == 0x800000) call_ref = 1;
	for (walk = sockets; walk; walk = walk->next)
	    if (walk->call_ref == call_ref) break;
    }
    while (walk);
    sock->call_ref = call_ref;
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_call_ref,call_ref);
    q_assign(&dsc,QF_msg_type,QMSG_SETUP);
    q_assign(&dsc,QF_aal_type,5); /* AAL 5 */
#ifdef UNI30
    q_assign(&dsc,QF_aal_mode,1); /* Message mode - LANE seems to really want
				     this */
#endif
    if (sock->remote->sas_txtp.max_sdu)
	q_assign(&dsc,QF_fw_max_sdu,sock->remote->sas_txtp.max_sdu);
    if (sock->remote->sas_rxtp.max_sdu)
	q_assign(&dsc,QF_bw_max_sdu,sock->remote->sas_rxtp.max_sdu);
    switch (sock->remote->sas_txtp.class) {
	case ATM_UBR:
    	    q_assign(&dsc,QF_fw_pcr_01,ATM_OC3_PCR);
    	    q_assign(&dsc,QF_bw_pcr_01,ATM_OC3_PCR);
   	    q_assign(&dsc,QF_best_effort,0);
	    break;
	case ATM_CBR:
    	    q_assign(&dsc,QF_fw_pcr_01,sock->remote->sas_txtp.min_pcr);
    	    q_assign(&dsc,QF_bw_pcr_01,sock->remote->sas_rxtp.min_pcr);
	    break;
	default:
	    return -EINVAL;
    }
    q_assign(&dsc,QF_bearer_class,16); /* BCOB-X */
    q_assign(&dsc,QF_traf_type,0); /* force presence - UNI 3.0 wants this */
    q_assign(&dsc,QF_upcc,0); /* p2p */
    q_assign(&dsc,QF_qos_fw,0); /* QOS 0 */
    q_assign(&dsc,QF_qos_bw,0);
    if ((sock->local && *sock->local->sas_addr.prv) || (local.sas_family &&
      *local.sas_addr.prv)) { /* handle E.164 too */
	q_assign(&dsc,QF_cgpn_plan,2);
	q_assign(&dsc,QF_cgpn_type,0);
	q_write(&dsc,QF_cgpn,sock->local && *sock->local->sas_addr.prv ?
	  (void *) sock->local->sas_addr.prv : (void *) local.sas_addr.prv,
	  ATM_ESA_LEN);
    }
    bhli = &sock->remote->sas_addr.bhli;
    if (bhli->hl_type != ATM_HL_NONE) {
	q_assign(&dsc,QF_hli_type,bhli->hl_type);
	switch (bhli->hl_type) {
	    case ATM_HL_ISO:
		q_write(&dsc,QF_iso_hli,bhli->hl_info,bhli->hl_length);
		break;
	    case ATM_HL_USER:
		q_write(&dsc,QF_iso_hli,bhli->hl_info,bhli->hl_length);
		break;
#ifdef UNI30
	    case ATM_HL_HLP:
		q_write(&dsc,QF_iso_hli,bhli->hl_info,4);
		break;
#endif
	    case ATM_HL_VENDOR:
		q_write(&dsc,QF_iso_hli,bhli->hl_info,7);
		break;
	    default:
		diag(COMPONENT,DIAG_FATAL,"bad hl_type (%d)",bhli->hl_type);
	}
    }
    blli = sock->remote->sas_addr.blli;
    if (blli) { /* @@@ loop */
	if (blli->l2_proto != ATM_L2_NONE) {
	    q_assign(&dsc,QF_uil2_proto,blli->l2_proto);
	    switch (blli->l2_proto) {
		case ATM_L2_X25_LL:
		case ATM_L2_X25_ML:
		case ATM_L2_HDLC_ARM:
		case ATM_L2_HDLC_NRM:
		case ATM_L2_HDLC_ABM:
		case ATM_L2_Q922:
		case ATM_L2_ISO7776:
		    if (blli->l2.itu.mode != ATM_IMD_NONE)
			q_assign(&dsc,QF_l2_mode,blli->l2.itu.mode);
		    if (blli->l2.itu.window)
			q_assign(&dsc,QF_window_size,blli->l2.itu.window);
		    break;
		case ATM_L2_USER:
		    q_assign(&dsc,QF_user_l2,blli->l2.user);
		    break;
		default:
		    break;
	    }
	}
	if (blli->l3_proto != ATM_L3_NONE) {
	    q_assign(&dsc,QF_uil3_proto,blli->l3_proto);
	    switch (blli->l3_proto) {
		case ATM_L3_X25:
		case ATM_L3_ISO8208:
		case ATM_L3_X223:
		    if (blli->l3.itu.mode != ATM_IMD_NONE)
			q_assign(&dsc,QF_l3_mode,blli->l3.itu.mode);
		    if (blli->l3.itu.def_size)
			q_assign(&dsc,QF_def_pck_size,blli->l3.itu.def_size);
		    if (blli->l3.itu.window)
			q_assign(&dsc,QF_pck_win_size,blli->l3.itu.window);
		    break;
		case ATM_L3_TR9577:
		    q_assign(&dsc,QF_ipi_high,blli->l3.tr9577.ipi >> 1);
		    q_assign(&dsc,QF_ipi_low,blli->l3.tr9577.ipi & 1);
		    if (blli->l3.tr9577.ipi == NLPID_IEEE802_1_SNAP) {
			q_write(&dsc,QF_oui,blli->l3.tr9577.snap,3);
			q_write(&dsc,QF_pid,blli->l3.tr9577.snap+3,2);
		    }
		    break;
		case ATM_L3_USER:
		    q_assign(&dsc,QF_user_l3,blli->l3.user);
		    break;
		default:
		    diag(COMPONENT,DIAG_FATAL,"bad l3_proto (%d)",
		      blli->l3_proto);
	    }
	}
    }
    if (*sock->remote->sas_addr.prv) /* improve @@@ */
	q_write(&dsc,QF_cdpn_esa,(void *) sock->remote->sas_addr.prv,
	  ATM_ESA_LEN);
    if (net) {
	q_assign(&dsc,QF_vpi,0);
	q_assign(&dsc,QF_vci,assign_vci); /* @@@ */
	sock->pvc.sap_addr.vpi = 0;
	sock->pvc.sap_addr.vci = assign_vci++;
    }
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
    return 0;
}


static int send_connect(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CONNECT);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    /* lots of data */
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
    return 0;
}


static void send_release_complete(SOCKET *sock,unsigned char reason)
{
    Q_DSC dsc;
    int size;
 
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_REL_COMP);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if (reason) q_assign(&dsc,QF_cause,reason); /* @@@ more data */
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static void dispatch(SOCKET *sock,struct atmsvc_msg *msg)
{
    SOCKET *next;
    int error;

    switch (msg->type) {
	case as_bind: /* only in NULL state */
	    if (sock) break;
	    send_kernel(msg->vcc,0,as_okay,0,NULL);
	    /* send_kernel(msg->vcc,0,as_close,-EINVAL,NULL); ... or */
	    return;
	case as_establish: /* NULL or LISTENING */
	    if (sock && sock->state != ss_listening) break;
	    if (!sock) {
		sock = new_sock(msg->vcc);
		sock->local = alloc_sap(&msg->local);
		sock->remote = alloc_sap(&msg->remote);
		if (!sock->remote) {
		    free_sock(sock);
		    send_kernel(msg->vcc,0,as_close,-EINVAL,NULL);
		    return;
		}
		sock->pvc.sap_txtp = sock->remote->sas_txtp;
		sock->pvc.sap_rxtp = sock->remote->sas_rxtp;
		sock->state = ss_connecting;
		error = send_setup(sock);
		if (error) {
		    send_kernel(msg->vcc,0,as_close,error,NULL);
		    free_sock(sock);
		    return;
		}
		START_TIMER(sock,T303);
		new_state(sock,ss_connecting);
		return;
	    }
	    /* becomes INDICATED or ZOMBIE */
	    if (!sock->listen) {
		diag(COMPONENT,DIAG_WARN,
		  "socket 0x%lx got accept with empty listen queue",msg->vcc);
		return;
	    }
	    next = sock->listen;
	    sock->listen = next->listen;
	    next->listen = NULL;
	    if (next->state == ss_zombie) {
		send_kernel(msg->vcc,0,as_close,-ECONNRESET,NULL);
			/* -ERESTARTSYS ? */
		free_sock(next);
		return;
	    }
	    if (next->state != ss_indicated) {
		sock = next; /* for error reporting */
		break;
	    }
	    next->id = msg->vcc;
	    error = send_connect(next);
	    if (!error) {
		START_TIMER(next,T313);
		new_state(next,ss_accepting);
		return;
	    }
	    send_kernel(next->id,0,as_close,error,NULL);
	    send_release(next,0); /* @@@ */
	    START_TIMER(next,T308_1);
	    new_state(next,ss_wait_rel);
	    return;
	case as_listen: /* NULL */
	    {
		struct sockaddr_atmsvc *sap;

		if (sock) break;
		sap = alloc_sap(&msg->local);
		if (lookup_sap(sap)) {
		    free_sap(sap);
		    send_kernel(msg->vcc,0,as_close,-EADDRINUSE,NULL);
		    return;
		}
		sock = new_sock(msg->vcc);
		sock->local = alloc_sap(&msg->local);
		send_kernel(sock->id,0,as_okay,0,NULL);
		sock->state = ss_listening;
		return;
	    }
	case as_close: /* all but ACCEPTING and WAIT_REL */
	    if (sock && (sock->state == ss_accepting || sock->state ==
	      ss_wait_rel)) break;
	    switch (sock ? sock->state : ss_null) {
		case ss_listening:
		case ss_zombie:
		    send_close(sock);
		    /* fall through */
		case ss_wait_close:
		    free_sock(sock);
		    /* fall through */
		case ss_null:
		case ss_rel_req:
		    return;
		case ss_connecting:
		    STOP_TIMER(sock);
		    /* fall through */
		case ss_connected:
		    if (sock->state == ss_connected)
			diag(COMPONENT,DIAG_INFO,"Active close (CR 0x%06X)",
			  sock->call_ref);
		    /* fall through */
		case ss_indicated:
		case ss_hold:
		    send_release(sock,
#ifdef UNI31
		      ATM_CV_NORMAL_CLEAR
#else
		      ATM_CV_NORMAL_UNSPEC
#endif
		      );
		    START_TIMER(sock,T308_1);
		    new_state(sock,sock->state == ss_connected ? ss_rel_req :
		      ss_wait_rel);
		    return;
		case ss_rel_ind:
		    send_release_complete(sock,0); /* @@@ */
		    free_sock(sock);
		    return;
		default:
		    break;
	    }
	    /* fall through */
	default:
    	    diag(COMPONENT,DIAG_WARN,"invalid message %d",(int) msg->type);
	    return;
    }
    diag(COMPONENT,DIAG_WARN,"message %s is incompatible with state %s (%d)",
      as_name[msg->type],state_name[sock ? sock->state : ss_null],
      (int) (sock ? sock->state : ss_null));
}


static void itf_notify(int itf)
{
    struct sockaddr_atmsvc addr;
    char buf[MAX_ATM_ADDR_LEN+1];

    if (!get_addr(itf,&addr)) {
	local.sas_family = 0;
	diag(COMPONENT,DIAG_INFO,"Removed local ATM address");
	return;
    }
    if (atm2text(buf,MAX_ATM_ADDR_LEN+1,(struct sockaddr *) &addr,A2T_PRETTY) <
      0) diag(COMPONENT,DIAG_ERROR,"bad address from kernel");
    else {
	diag(COMPONENT,DIAG_INFO,"Local ATM address set to %s",buf);
	local = addr;
    }
}


void from_kernel(struct atmsvc_msg *msg,int size)
{
    SOCKET *curr;
    struct atm_blli **this;
    int bllis,i;

    if (msg->type == as_itf_notify) {
	itf_notify(msg->pvc.sap_addr.itf);
	return;
    }
    for (curr = sockets; curr; curr = curr->next)
	if (msg->vcc == curr->id) break;
    if (!curr)
	for (curr = sockets; curr; curr = curr->next)
	    if (msg->listen_vcc == curr->id && curr->state == ss_listening)
		break;
    diag(COMPONENT,DIAG_DEBUG,"FROM KERNEL: %s for socket 0x%lx (0x%lx/0x%lx) "
      "in state %s",as_name[msg->type],(unsigned long) curr,msg->vcc,
      msg->listen_vcc,state_name[curr ? curr->state : ss_null]);
    msg->local.sas_addr.blli = NULL;
    msg->remote.sas_addr.blli = NULL;
    bllis = msg2bllis(size);
    if (msg->type == as_bind || msg->type == as_listen)
	this = &msg->local.sas_addr.blli;
    else this = &msg->remote.sas_addr.blli;
    for (i = 0; i < bllis; i++) {
	*this = &msg->blli[i];
	this = &msg->blli[i].next;
    }
    *this = NULL;
    dispatch(curr,msg);
}
