/* arp.h - ARP state machine */
 
/* Written 1995 by Werner Almesberger, EPFL-LRC */
 

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h> /* for linux/if_arp.h */
#include <netinet/in.h> /* for ntohs, etc. */
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#include <linux/atm.h>
#include <linux/atmarp.h>
#include <linux/atmclip.h>

#include "atm.h"
#include "atmd.h"

#include "table.h"
#include "itf.h"
#include "io.h"
#include "arp.h"


#define COMPONENT "ARP"


#define T_REPLY	 2000000  /* 2 sec */
#define R_REPLY	       5  /* 5 retries */
#define T_REVAL	30000000  /* 30 sec - make much larger (if srv ...) @@@ */


static void discard_vcc(VCC *vcc,int timeout)
{
    switch (vcc->state) {
	case vs_valid:
	    if (vcc->flags & ATF_PERM) break;
	    /* fall trough */
	case vs_discover:
	case vs_reval:
	    if (!timeout) stop_timer(vcc->timer);
	    break;
	default:
	    break;
    }
    if (do_close(vcc->fd))
	diag(COMPONENT,DIAG_ERROR,"close: %s",strerror(errno));
    Q_REMOVE(vcc->entry->vccs,vcc);
    free(vcc);
}


static void discard_entry(ENTRY *entry,int timeout)
{
    VCC *vcc,*next;

    if (!timeout && entry->state != as_res_wait && entry->state != as_unknown)
	if (entry->state != as_valid || !(entry->flags & ATF_PERM))
	    stop_timer(entry->timer);
    for (vcc = entry->vccs; vcc; vcc = next) {
	next = vcc->next;
	discard_vcc(vcc,0);
    }
    if (entry == entry->itf->arp_srv) entry->itf->arp_srv = NULL;
    Q_REMOVE(entry->itf->table,entry);
    if (entry->addr) free(entry->addr);
    free(entry);
}


static void vcc_detach(ENTRY *entry)
{
    if (entry->state != as_res_wait && entry->state != as_unknown) return;
    if (entry->flags & ATF_PERM) return;
    discard_entry(entry,0);
}


static void put_ip(unsigned char **here,unsigned long ip,unsigned char *len)
{
    if (!ip) {
	*len = 0;
	return;
    }
    ip = htonl(ip);
    memcpy(*here,&ip,4);
    *len = 4;
    (*here) += 4;
}


static void put_addr(unsigned char **here,struct sockaddr_atmsvc *addr,
  unsigned char *num_tl,unsigned char *sub_tl)
{
    if (!addr) return;
    if (!*addr->sas_addr.pub) *sub_tl = 0;
    else {
	*num_tl = strlen(addr->sas_addr.pub) | TL_E164;
	memcpy(*here,addr->sas_addr.pub,*num_tl & TL_LEN);
	(*here) += *num_tl & TL_LEN;
	num_tl = sub_tl;
    }
    if (!*addr->sas_addr.prv) *num_tl = 0;
    else {
	*num_tl = ATM_ESA_LEN;
	memcpy(*here,addr->sas_addr.prv,ATM_ESA_LEN);
	(*here) += ATM_ESA_LEN;
    }
}


static const unsigned char llc_oui_arp[] = {
	0xaa,	/* DSAP: non-ISO */
	0xaa,	/* SSAP: non-ISO */
	0x03,	/* Ctrl: Unnumbered Information Command PDU */
	0x00,	/* OUI: EtherType */
	0x00,
	0x00,
	0x08,	/* ARP protocol */
	0x06 };


static void send_arp(int fd,unsigned short op,unsigned long local_ip,
  struct sockaddr_atmsvc *local_addr,unsigned long remote_ip,
  struct sockaddr_atmsvc *remote_addr)
{
    struct atmarphdr *hdr;
    unsigned char *buffer,*here;

    buffer = alloc(MAX_ATMARP_SIZE+RFC1483LLC_LEN);
    memcpy(buffer,llc_oui_arp,RFC1483LLC_LEN);
    hdr = (struct atmarphdr *) (buffer+RFC1483LLC_LEN);
    memset(hdr,0,MAX_ATMARP_SIZE);
    hdr->ar_hrd = htons(ARPHRD_ATM);
    hdr->ar_pro = htons(ETH_P_IP);
    hdr->ar_op = htons(op);
    here = hdr->data;
    put_addr(&here,local_addr,&hdr->ar_shtl,&hdr->ar_sstl);
    put_ip(&here,local_ip,&hdr->ar_spln);
    put_addr(&here,remote_addr,&hdr->ar_thtl,&hdr->ar_tstl);
    put_ip(&here,remote_ip,&hdr->ar_tpln);
    send_packet(fd,buffer,here-buffer);
}


static void arp_request(ITF *itf,unsigned long ip)
{
    VCC *vcc;

    diag(COMPONENT,DIAG_DEBUG,"sending ARP request");
    if (!itf->arp_srv) {
	diag(COMPONENT,DIAG_ERROR,"no ARP server");
	return;
    }
    for (vcc = itf->arp_srv->vccs; vcc; vcc = vcc->next)
	if (vcc->state != vs_connecting) break;
    if (!vcc) {
	diag(COMPONENT,DIAG_ERROR,"ARP server has no usable VCC");
	return;
    }
    send_arp(vcc->fd,ARPOP_REQUEST,itf->local_ip,&itf->local_addr,ip,NULL);
}


static void inarp_requests(VCC *vcc)
{
    ITF *itf;

    diag(COMPONENT,DIAG_DEBUG,"sending InARP request");
    for (itf = itfs; itf; itf = itf->next)
	if (itf->local_ip)
	    send_arp(vcc->fd,ARPOP_InREQUEST,itf->local_ip,&itf->local_addr,0,
	      NULL);
}


static void arp_nak(VCC *vcc,unsigned long src_ip,unsigned long tgt_ip,
  struct sockaddr_atmsvc *tgt_addr)
{
    diag(COMPONENT,DIAG_DEBUG,"sending ARP NAK");
    send_arp(vcc->fd,ARPOP_NAK,tgt_ip,tgt_addr,src_ip,NULL);
}


static void arp_reply(VCC *vcc,unsigned long src_ip,
  struct sockaddr_atmsvc *src_addr,unsigned long tgt_ip,
  struct sockaddr_atmsvc *tgt_addr)
{
    if (!src_addr) {
	arp_nak(vcc,src_ip,tgt_ip,tgt_addr);
	return;
    }
    diag(COMPONENT,DIAG_DEBUG,"sending ARP reply");
    send_arp(vcc->fd,ARPOP_REPLY,src_ip,src_addr,tgt_ip,tgt_addr);
}


static void inarp_reply(VCC *vcc,unsigned long ip,struct sockaddr_atmsvc *addr)
{
    diag(COMPONENT,DIAG_DEBUG,"sending InARP reply");
    send_arp(vcc->fd,ARPOP_InREPLY,itfs->local_ip,&itfs->local_addr,ip,addr);
}


static void timeout_vcc(void *user)
{
    VCC *vcc;
    ENTRY *entry;

    diag(COMPONENT,DIAG_DEBUG,"VCC TIMEOUT");
    vcc = user;
    switch (vcc->state) {
	case vs_discover:
	    if (!(vcc->flags & ATF_PERM) && !(vcc->retries)) {
		discard_vcc(vcc,1);
		if (vcc->entry) vcc_detach(vcc->entry);
		break;
	    }
	    if (!(vcc->flags & ATF_PERM)) vcc->retries--;
	    inarp_requests(vcc);
	    vcc->timer = start_timer(T_REPLY,timeout_vcc,vcc);
	    break;
	case vs_valid:
	    vcc->state = vs_reval;
	    inarp_requests(vcc);
	    vcc->timer = start_timer(T_REPLY,timeout_vcc,vcc);
	    vcc->retries = R_REPLY;
	    break;
	case vs_reval:
	    if (!vcc->retries--) {
		vcc->entry->ip = 0;
		if (set_ip(vcc->fd,0) < 0) {
		    diag(COMPONENT,DIAG_ERROR,"can't set IP to 0");
		    entry = vcc->entry;
		    discard_vcc(vcc,1);
		    vcc_detach(entry);
		    break;
		}
		vcc->retries = R_REPLY;
		vcc->state = vs_discover;
	    }
	    inarp_requests(vcc);
	    vcc->timer = start_timer(T_REPLY,timeout_vcc,vcc);
	    break;
	default:
	    diag(COMPONENT,DIAG_FATAL,"invalid VCC state %d",vcc->state);
    }
}


/*
 * Returns:
 *  <0  there's no ARP server
 *   0  ARP server connection establishment in progress
 *  >0  ready to send ARP_REQUEST
 */


static int want_arp_srv(ITF *itf)
{
    VCC *vcc;
    int fd;

    if (!itf->arp_srv) return -1;
    for (vcc = itf->arp_srv->vccs; vcc; vcc = vcc->next)
	if (vcc->state != vs_connecting) return 1;
    if (itf->arp_srv->vccs) return 0;
    if ((fd = connect_vcc((struct sockaddr *) &itf->local_addr,
      (struct sockaddr *) itf->arp_srv->addr)) < 0)
	return 0;
    vcc = alloc_t(VCC);
    vcc->state = vs_connecting;
    vcc->fd = fd;
    vcc->flags = itf->arp_srv->flags;
    vcc->entry = itf->arp_srv;
    Q_INSERT_HEAD(itf->arp_srv->vccs,vcc);
    return 0;
}


static void timeout_entry(void *user);


static void validate_entry(ENTRY *entry)
{
    int result;

    if ((result = want_arp_srv(entry->itf)) < 0) {
	/* @@@ try to revalidate by sending InARP */
	return;
    }
    if (!result) {
	if (entry->state == as_unknown) entry->state = as_res_wait;
	return;
    }
    switch (entry->state) {
	case as_res_wait:
	    diag(COMPONENT,DIAG_FATAL,"as_res_wait with ARP server up");
	case as_resolv:
	    return; /* waiting for result ... */
	case as_unknown:
	    entry->state = as_resolv;
	    arp_request(entry->itf,entry->ip);
	    entry->timer = start_timer(T_REPLY,timeout_entry,entry);
	    entry->retries = R_REPLY;
	    return;
	case as_valid:
	    return; /* race ... */
	default:
	    diag(COMPONENT,DIAG_FATAL,"bad state %d",entry->state);
    }
}


static void timeout_entry(void *user)
{
    ENTRY *entry;
    unsigned char *ipp;

    entry = user;
    ipp = (unsigned char *) &entry->ip;
    diag(COMPONENT,DIAG_DEBUG,"TIMEOUT (entry %d.%d.%d.%d)",ipp[0],ipp[1],
      ipp[2],ipp[3]);
    switch (entry->state) {
	case as_resolv:
	    if (entry->retries) {
		entry->retries--;
		arp_request(entry->itf,entry->ip);
		entry->timer = start_timer(T_REPLY,timeout_entry,entry);
		break;
	    }
	    if (!entry->vccs && !(entry->flags & ATF_PERM))
		discard_entry(entry,1);
	    else entry->state = as_unknown;
	    break;
	case as_valid:
	    if (!entry->vccs && !(entry->flags & ATF_PERM)) {
		discard_entry(entry,1);
		break;
	    }
	    entry->state = as_unknown; /* avoid as_valid race */
	    validate_entry(entry);
	    break;
	default:
	    diag(COMPONENT,DIAG_FATAL,"timed out in state %s",
	      entry_state_name[entry->state]);
    }
}


static void learn_vcc(VCC *vcc,unsigned long ip)
{
    ITF *itf;
    ENTRY *entry;

    if (!ip) return;
    for (itf = itfs; itf; itf = itf->next) {
	entry = lookup_ip(itf,ip);
	if (entry) break;
    }
    if (!vcc->entry) {
	if (!entry) {
	    entry = alloc_t(ENTRY);
	    entry->state = as_unknown;
	    entry->ip = ip;
	    entry->addr = NULL;
	    entry->flags = 0;
	    entry->itf = itfs; /* lookup one ... @@@ */
	    Q_INSERT_HEAD(entry->itf->table,entry);
	}
	Q_INSERT_HEAD(entry->vccs,vcc);
	vcc->entry = entry;
    }
    else if (!entry) (entry = vcc->entry)->ip = ip;
	else if (entry != vcc->entry) {
		Q_REMOVE(vcc->entry->vccs,vcc);
		vcc_detach(vcc->entry);
		vcc->entry = entry;
		Q_INSERT_HEAD(entry->vccs,vcc);
	    }
    switch (vcc->state) {
	case vs_discover:
	case vs_svc:
	    if (set_ip(vcc->fd,ip) < 0) {
		diag(COMPONENT,DIAG_ERROR,"can't set IP");
		discard_vcc(vcc,0);
		vcc_detach(entry);
		break;
	    }
	    if (vcc->state == vs_svc) break;
	    /* fall through */
	case vs_valid:
	case vs_reval:
	    stop_timer(vcc->timer);
	    vcc->timer = start_timer(T_REVAL,timeout_vcc,vcc);
	    break;
	default:
	    diag(COMPONENT,DIAG_FATAL,"invalid VCC state %d",vcc->state);
    }
}


static void learn_addr(unsigned long ip,struct sockaddr_atmsvc *addr)
{
    ITF *itf;
    ENTRY *by_ip,*by_addr;
    VCC *vcc,*next,*last;
    int fd;

    if (!ip || !addr) return;
    by_ip = NULL;
    if (!by_ip) /* very very ugly */
	for (itf = itfs; itf; itf = itf->next) {
	    by_ip = lookup_ip(itf,ip);
	    if (by_ip) break;
	}
    if (by_ip) by_addr = lookup_addr(by_ip->itf,addr);
    else {
	for (itf = itfs; itf; itf = itf->next) {
	    by_ip = lookup_addr(itf,addr);
	    if (by_ip) break;
	}
	by_addr = by_ip;
    }
    if (by_ip && by_addr) {
	if (by_ip != by_addr) {
	    by_addr->ip = ip;
	    last = NULL;
	    for (vcc = by_addr->vccs; vcc; vcc = vcc->next) last = vcc;
	    if (!last) by_addr->vccs = by_ip->vccs;
	    else {
		last->next = by_ip->vccs;
		if (by_ip->vccs) by_ip->vccs->prev = last;
	    }
	    by_addr->flags |= by_ip->flags; /* keep ATF_ARPSRV and such */
	    by_ip->vccs = NULL;
	    discard_entry(by_ip,0);
	    by_ip = by_addr;
	}
    }
    else {
	if (!by_ip) {
	    by_ip = alloc_t(ENTRY);
	    by_ip->state = as_unknown; /* has no timer */
	    by_ip->flags = 0;
	    by_ip->vccs = NULL;
	    by_ip->itf = itfs; /* @@@ */
	}
	by_ip->ip = ip;
	by_ip->addr = alloc_t(struct sockaddr_atmsvc); /* @@@ fixme */
	*by_ip->addr = *addr;
    }
    if (by_ip->state == as_resolv || (by_ip->state == as_valid &&
      !(by_ip->flags & ATF_PERM))) stop_timer(by_ip->timer);
    by_ip->state = as_valid;
    if (ip)
	for (vcc = by_ip->vccs; vcc; vcc = next) {
	    next = vcc->next;
	    if (set_ip(vcc->fd,ip) < 0) {
		diag(COMPONENT,DIAG_ERROR,"set_ip failed");
		Q_REMOVE(by_ip->vccs,vcc);
	    }
	}
    if (!by_ip->vccs && (fd = connect_vcc(by_ip->itf ? (struct sockaddr *)
      &by_ip->itf->local_addr : NULL,(struct sockaddr *) addr)) >= 0) {
	vcc = alloc_t(VCC);
	vcc->state = vs_connecting;
	vcc->fd = fd;
	vcc->flags = by_ip->flags;
	vcc->entry = by_ip;
	Q_INSERT_HEAD(by_ip->vccs,vcc);
    }
    if (by_ip->state != as_valid || !(by_ip->flags & ATF_PERM))
	by_ip->timer = start_timer(T_REVAL,timeout_entry,by_ip);
}


static void learn_nak(unsigned long ip)
{
    ITF *itf;
    ENTRY *entry;

    if (!ip) return;
    for (itf = itfs; itf; itf = itf->next) { /* ugly ... */
	entry = lookup_ip(itf,ip);
	if (entry) break;
    }
    if (!entry) return;
    if (entry->flags & ATF_PERM) return;
    discard_entry(entry,0);
}


void need_ip(int itf_num,unsigned long ip)
{
    ITF *itf;
    ENTRY *entry;

    diag(COMPONENT,DIAG_DEBUG,"itf %d needs %d.%d.%d.%d",itf_num,
      ((unsigned char *) &ip)[0],((unsigned char *) &ip)[1],
      ((unsigned char *) &ip)[2],((unsigned char *) &ip)[3]);
    if (!(itf = lookup_itf(itf_num))) {
	diag(COMPONENT,DIAG_ERROR,"itf %d not found",itf_num);
	return;
    }
    if (!(entry = lookup_ip(itf,ip))) {
	entry = alloc_t(ENTRY);
	entry->state = as_unknown; /* force revalidation */
	entry->ip = ip;
	entry->addr = NULL;
	entry->flags = 0;
	entry->vccs = NULL;
	entry->itf = itf;
	Q_INSERT_HEAD(itf->table,entry);
    }
    validate_entry(entry);
}


static void *get_addr(unsigned char **here,int len)
{
    if (!len) return NULL;
    (*here) += len;
    return *here-len;
}


static void set_addr(struct sockaddr_atmsvc *addr,void *num,void *sub,
  int num_tl,int sub_tl)
{
    memset(addr,0,sizeof(struct sockaddr_atmsvc));
    addr->sas_family = AF_ATMSVC;
    if (num_tl & TL_E164)
	if ((num_tl & TL_LEN) > ATM_E164_LEN)
	    diag(COMPONENT,DIAG_ERROR,"bad E.164 length (%d)",num_tl &
	      TL_LEN);
	else {
	    memcpy(addr->sas_addr.pub,num,num_tl & TL_LEN);
	    if (sub)
		if (sub_tl != ATM_ESA_LEN) {
		    diag(COMPONENT,DIAG_ERROR,"bad ESA length (%d)",sub_tl);
		    *addr->sas_addr.pub = 0;
		}
		else memcpy(addr->sas_addr.prv,sub,ATM_ESA_LEN);
	}
    else if (num)
	    if (num_tl != ATM_ESA_LEN)
		diag(COMPONENT,DIAG_ERROR,"bad ESA length (%d)",num_tl);
	    else memcpy(addr->sas_addr.prv,num,ATM_ESA_LEN);
}


static unsigned long get_ip(unsigned char *ptr)
{
    if (!ptr) return 0;
    return (ptr[0] << 24) | (ptr[1] << 16) | (ptr[2] << 8) | ptr[3];
}


void incoming_arp(VCC *vcc,struct atmarphdr *hdr,int len)
{
    ITF *itf;
    ENTRY *entry;
    void *sha,*ssa,*spa,*tha,*tsa,*tpa;
    struct sockaddr_atmsvc source,target;
    unsigned long src_ip,tgt_ip;
    unsigned char *here;

    if (len < hdr->data-(unsigned char *) hdr) {
	diag(COMPONENT,DIAG_ERROR,"got truncated ARP packet (%d bytes)",len);
	return;
    }
    if (hdr->ar_hrd != htons(ARPHRD_ATM)) {
	diag(COMPONENT,DIAG_ERROR,"unknown hw protocol 0x%04x",
	  ntohs(hdr->ar_hrd));
	return;
    }
    if (hdr->ar_pro != htons(ETH_P_IP)) {
	diag(COMPONENT,DIAG_ERROR,"unknown upper protocol 0x%04x",
	  ntohs(hdr->ar_pro));
	return;
    }
    if (!(hdr->ar_shtl & TL_LEN)) hdr->ar_shtl = 0; /* paranoia */
    if (!(hdr->ar_thtl & TL_LEN)) hdr->ar_thtl = 0;
    here = hdr->data;
    sha = get_addr(&here,hdr->ar_shtl & TL_LEN);
    ssa = get_addr(&here,hdr->ar_sstl & TL_LEN);
    spa = get_addr(&here,hdr->ar_spln);
    tha = get_addr(&here,hdr->ar_thtl & TL_LEN);
    tsa = get_addr(&here,hdr->ar_tstl & TL_LEN);
    tpa = get_addr(&here,hdr->ar_tpln);
    if (here-(unsigned char *) hdr > len) {
	diag(COMPONENT,DIAG_ERROR,"message too short (got %d, need %d)",len,
	  here-(unsigned char *) hdr);
	return;
    }
    set_addr(&source,sha,ssa,hdr->ar_shtl,hdr->ar_sstl);
    set_addr(&target,tha,tsa,hdr->ar_thtl,hdr->ar_tstl);
    src_ip = get_ip(spa);
    tgt_ip = get_ip(tpa);
{
   unsigned char *ipp;
   char buffer[MAX_ATM_ADDR_LEN+1];

   ipp = (unsigned char *) &src_ip;
   diag(COMPONENT,DIAG_DEBUG,"  SRC IP: %d.%d.%d.%d",ipp[0],ipp[1],ipp[2],
     ipp[3]);
   if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,(struct sockaddr *) &source,
     A2T_PRETTY) >= 0)
	diag(COMPONENT,DIAG_DEBUG,"  SRC ATM: %s",buffer);
   ipp = (unsigned char *) &tgt_ip;
   diag(COMPONENT,DIAG_DEBUG,"  DST IP: %d.%d.%d.%d",ipp[0],ipp[1],ipp[2],
     ipp[3]);
   if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,(struct sockaddr *) &target,
     A2T_PRETTY) >= 0)
	diag(COMPONENT,DIAG_DEBUG,"  DST ATM: %s",buffer);
}
    switch (ntohs(hdr->ar_op)) {
	case ARPOP_REQUEST:
	    diag(COMPONENT,DIAG_DEBUG,"got ARP_REQ");
	    learn_vcc(vcc,src_ip);
	    learn_addr(src_ip,&source);
	    entry = NULL;
	    for (itf = itfs; itf; itf = itf->next) {
		entry = lookup_ip(itf,tgt_ip);
		if (entry) break;
	    }
	    if (!entry) arp_nak(vcc,tgt_ip,src_ip,&source);
	    else arp_reply(vcc,tgt_ip,entry->addr,src_ip,&source);
	    break;
	case ARPOP_REPLY:
	    diag(COMPONENT,DIAG_DEBUG,"got ARP_REP");
	    if (!vcc->entry || !(vcc->entry->flags & ATF_ARPSRV)) {
		diag(COMPONENT,DIAG_ERROR,"got ARP response from charlatan");
		break;
	    }
	    learn_addr(src_ip,&source); /* entry != NULL ? @@@ */
	    break;
	case ARPOP_InREQUEST:
	    diag(COMPONENT,DIAG_DEBUG,"got InARP_REQ");
	    learn_vcc(vcc,src_ip);
	    learn_addr(src_ip,&source);
	    inarp_reply(vcc,src_ip,&source);
	    break;
	case ARPOP_InREPLY:
	    diag(COMPONENT,DIAG_DEBUG,"got InARP_REP");
	    learn_vcc(vcc,src_ip);
	    learn_addr(src_ip,&source);
	    break;
	case ARPOP_NAK:
	    diag(COMPONENT,DIAG_DEBUG,"got ARP_NAK");
	    if (!vcc->entry || !(vcc->entry->flags & ATF_ARPSRV)) {
		diag(COMPONENT,DIAG_ERROR,"got ARP response from charlatan");
		break;
	    }
	    learn_nak(tgt_ip);
	    break;
	default:
	    diag(COMPONENT,DIAG_ERROR,"unrecognized ARP op 0x%x",
	      ntohs(hdr->ar_op));
    }
}


static int ioctl_set_pvc(ITF *itf,unsigned long ip,
  struct sockaddr_atmpvc *addr,int flags)
{
    ENTRY *entry;
    VCC *vcc;
    int fd,result;

    if (lookup_ip(itf,ip)) return -EEXIST; 
    if ((fd = connect_vcc(NULL,(struct sockaddr *) addr)) < 0) return fd;
    if (ip)
	if ((result = set_ip(fd,ip)) < 0) return result;
    entry = alloc_t(ENTRY);
    entry->state = as_unknown;
    entry->ip = ip;
    entry->addr = NULL;
    entry->vccs = NULL;
    entry->itf = itf;
    entry->vccs = NULL;
    vcc = alloc_t(VCC);
    vcc->state = vs_valid;
    vcc->fd = fd;
    vcc->entry = entry;
    entry->flags = vcc->flags = flags;
    if (ip) {
	if (!(flags & ATF_PERM))
	    vcc->timer = start_timer(T_REVAL,timeout_vcc,vcc);
    }
    else {
	vcc->state = vs_discover;
	inarp_requests(vcc);
	vcc->timer = start_timer(T_REPLY,timeout_vcc,vcc);
	vcc->retries = R_REPLY;
    }
    Q_INSERT_HEAD(entry->vccs,vcc);
    Q_INSERT_HEAD(itf->table,entry);
    return 0;
}


static void set_local(ITF *itf,unsigned long ip,struct sockaddr_atmsvc *addr)
{
    itf->local_ip = ip;
    itf->local_addr = *addr;
}


static int ioctl_set_svc(ITF *itf,unsigned long ip,
  struct sockaddr_atmsvc *addr,int flags)
{
    ENTRY *entry;

    if (flags & (ATF_LOCAL | ATF_ARPSRV)) flags |= ATF_PERM;
    if (flags & ATF_LOCAL) set_local(itf,ip,addr);
    if (lookup_ip(itf,ip)) return -EEXIST;
    entry = alloc_t(ENTRY);
    entry->state = as_valid;
    entry->ip = ip;
    entry->addr = alloc_t(struct sockaddr_atmsvc);
    *entry->addr = *addr;
    entry->flags = flags;
    if (!(flags & ATF_PERM))
	entry->timer = start_timer(T_REVAL,timeout_entry,entry);
    entry->vccs = NULL;
    entry->itf = itf;
    Q_INSERT_HEAD(itf->table,entry);
    if (flags & ATF_ARPSRV) {
	itf->arp_srv = entry;
	(void) want_arp_srv(itf);
    }
    return 0;
}


static int ioctl_delete(ITF *itf,unsigned long ip)
{
    ENTRY *entry;

    if (!(entry = lookup_ip(itf,ip))) {
	diag(COMPONENT,DIAG_WARN,"ioctl_delete didn't find entry");
	return -ENOENT;
    }
    discard_entry(entry,0);
    return 0;
}


int arp_ioctl(int itf_num,unsigned long cmd,struct atmarpreq *req)
{
    ITF *itf;
    char buffer[MAX_ATM_ADDR_LEN+1];
    unsigned long ip;
    unsigned char *ipp;

    if (req->arp_pa.sa_family != AF_INET) {
	diag(COMPONENT,DIAG_ERROR,"SIOCSARP: bad PA AF 0x%x",
	  req->arp_pa.sa_family);
	return -EPFNOSUPPORT;
    }
    ip = ((struct sockaddr_in *) &req->arp_pa)->sin_addr.s_addr;
    ipp = (unsigned char *) &ip;
    if (!(itf = lookup_itf(itf_num))) {
	diag(COMPONENT,DIAG_ERROR,"itf %d not found",itf_num);
	return -EPROTO;
    }
    switch (cmd) {
	case SIOCSARP:
	    if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,
	      (struct sockaddr *) &req->arp_ha,A2T_PRETTY) < 0) {
		diag(COMPONENT,DIAG_ERROR,"a2t fails on SIOCSARP");
		return -EINVAL;
	    }
	    diag(COMPONENT,DIAG_DEBUG,"got SIOCSARP for itf %d, IP %d.%d.%d.%d"
	      ", ATM %s, flags 0x%x",itf_num,ipp[0],ipp[1],ipp[2],ipp[3],buffer,
	      req->arp_flags);
	    switch (req->arp_ha.sas_family) {
		case AF_ATMPVC:	
		    return ioctl_set_pvc(itf,((struct sockaddr_in *)
		      &req->arp_pa)->sin_addr.s_addr,(struct sockaddr_atmpvc *)
		      &req->arp_ha,req->arp_flags);
		case AF_ATMSVC:
		    return ioctl_set_svc(itf,((struct sockaddr_in *)
		      &req->arp_pa)->sin_addr.s_addr,(struct sockaddr_atmsvc *)
		      &req->arp_ha,req->arp_flags);
		default: /* not reached - atm2text complains before */
		    diag(COMPONENT,DIAG_ERROR,"SIOCSARP: bad HA AF 0x%x",
		      req->arp_ha.sas_family);
		    return -EINVAL;
	    }
	case SIOCGARP:
	    diag(COMPONENT,DIAG_DEBUG,"got SIOCGARP for itf %d",itf_num);
	    return -ENOSYS;
	case SIOCDARP:
	    diag(COMPONENT,DIAG_DEBUG,"got SIOCDARP for itf %d, IP %d.%d.%d.%d"
	      ,itf_num,ipp[0],ipp[1],ipp[2],ipp[3]);
	    return ioctl_delete(itf,ip);
	default:
	    diag(COMPONENT,DIAG_ERROR,"unrecognized ioctl 0x%x",cmd);
	    return -EINVAL;
    }
}


void vcc_failed(VCC *vcc)
{
    Q_REMOVE(vcc->entry->vccs,vcc);
    vcc_detach(vcc->entry);
    free(vcc);
}


void vcc_connected(VCC *vcc)
{
    ENTRY *entry;

    diag(COMPONENT,DIAG_DEBUG,"connected entry 0x%08x",(unsigned long) entry);
    if (vcc->state != vs_connecting)
	diag(COMPONENT,DIAG_FATAL,"non-connecting VCC connected");
    vcc->state = vs_svc;
    if (set_ip(vcc->fd,vcc->entry->ip) < 0) {
	diag(COMPONENT,DIAG_ERROR,"can't set IP");
	vcc_failed(vcc);
	return;
    }
    if (!(vcc->entry->flags & ATF_ARPSRV)) return;
    for (entry = vcc->entry->itf->table; entry; entry = entry->next)
	if (entry->state == as_res_wait) {
	    entry->state = as_resolv;
	    entry->timer = start_timer(T_REPLY,timeout_entry,entry);
	    entry->retries = R_REPLY;
	    arp_request(entry->itf,entry->ip);
	}
}


void disconnect_vcc(VCC *vcc)
{
    ENTRY *entry;

    entry = vcc->entry;
    discard_vcc(vcc,0);
    if (entry) vcc_detach(entry);
}


void incoming_call(VCC *vcc)
{
    inarp_requests(vcc);
}
