/* proto.c - Common protocol functions and structures */
 
/* Written 1997-1998 by Roman Pletka, EPFL-SSC */
/* Modified 1998 by Werner Almesberger, EPFL ICA */


#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "atmd.h"
#include "io.h"
#include "proto.h"


#define COMPONENT "SWITCH"

static const char *as_msgs[] = {
		      "as_catch_null", "as_bind", "as_connect",
		      "as_accept", "as_reject","as_listen",
		      "as_okay", "as_error", "as_indicate",
		      "as_close", "as_itf_notify", "as_modify",
		      "as_identify"};
static const char *cs_states[]= {
		       "cs_invalid", "cs_null", "cs_listening",
		       "cs_connected", "cs_indicated", "cs_called_accepted",
		       "cs_rm_accepted", "cs_rm_accepted2", "cs_caller_error",
		       "cs_rejected", "cs_rejected2", "cs_caller_closing",
		       "cs_called_closed", "cs_called_closed2", "cs_free_rm",
		       "cs_rejecting", "cs_will_close", "cs_call_indicated",
		       "cs_caller_closed" };
static const char *sources[4] = {"CALLER","CALLED","RM"};


static CALL *calls = NULL;


CALL *new_call(unsigned long id)
{
    CALL *call;
 
    call = alloc_t(CALL);
    memset(call,0,sizeof(CALL));
    call->state = cs_invalid;
    call->in.id = id;
    call->out.id = 0;
    call->listen = NULL;
    call->next = calls;
    calls = call;
    return call;
}


void free_call(CALL *call)
{
  
  call->state = cs_invalid;
  free(call);
  printf("Call 0x%p killed\n",call);
  /* because call is not in a double linked chain, the next-pointer
     in the previous element will not point to the next element!
     But this chain is only for debbuging (except at startup) we neglect
     this fact...  */
}


void new_state(CALL *call,STATE state)
{
  call->state = state;
  print_state(call);
}


void send_sigd(unsigned long vcc,
	       unsigned long listen_vcc,
	       enum atmsvc_msg_type type,
	       int reply,
	       const struct sockaddr_atmpvc *pvc,
	       const struct sockaddr_atmsvc *svc,
	       const struct sockaddr_atmsvc *local,
	       const struct atm_sap *sap,
	       const struct atm_qos *qos)
{
    struct atmsvc_msg *msg;

    msg = alloc_t(struct atmsvc_msg);
    msg->vcc = vcc;
    msg->listen_vcc = listen_vcc;
    msg->type = type;
    msg->reply = reply;
    if (pvc) msg->pvc = *pvc;
    else memset(&msg->pvc,0,sizeof(msg->pvc));
    if (sap) msg->sap = *sap;
    else memset(&msg->sap,0,sizeof(msg->sap));
    if (qos) msg->qos = *qos;
    else memset(&msg->qos,0,sizeof(msg->qos));
    if (local) msg->local = *local;
    else memset(&msg->local,0,sizeof(msg->local));
    if (svc) msg->svc = *svc;
    else memset(&msg->svc,0,sizeof(msg->svc));
    to_sigd(vcc,msg);
    free(msg);
}

void send_listen(void)
{
    CALL *call;
    struct atmsvc_msg msg;

    for (call = calls; call; call = call->next) {
	memset(&msg,0,sizeof(msg));
	/* compose the message */
	msg.type = as_listen;
	msg.vcc = (unsigned long) call;
	msg.svc.sas_family = AF_ATMSVC;
	msg.qos.aal = ATM_AAL5;
	msg.qos.txtp.traffic_class = msg.qos.rxtp.traffic_class = ATM_ANYCLASS;
	/* msg.sap ; */
	call->listen = call;
	to_sigd(call->in.id,&msg);
	new_state(call,cs_null);
    }
}

void send_identify(CALL *call)
{
  struct atmsvc_msg msg;

  /*  this is always sent to caller */
  memset(&msg,0,sizeof(msg));
  
  /* compose the message */
  msg.type = as_identify;
  msg.vcc = (unsigned long) call;
  msg.listen_vcc = (unsigned long) call->listen;
  
  /* We have to complete the message (vci,vpi..) */
  msg.pvc = call->in.pvc;
  to_sigd(call->in.id,&msg);
}

void send_connect(CALL *call)
{
  struct atmsvc_msg msg;

  /*  this is always sent to called */
  memset(&msg,0,sizeof(msg));
  /* compose the message */
  msg.type = as_connect;
  msg.vcc = (unsigned long) call | CALLED; /* some kind of magic... */
  msg.local = call->in.svc;
  msg.qos = call->in.qos;
  msg.svc = call->out.svc;
  msg.sap = call->sap;
  /* we have to give VCI/VPI */
  msg.pvc = call->out.pvc;
  to_sigd(call->out.id,&msg);
}

void send_reject(CALL *call, int err_code)
{
  struct atmsvc_msg msg;

  /*  this is always sent to called */
  memset(&msg,0,sizeof(msg));
  msg.type = as_reject;
  msg.vcc = (unsigned long) call; 
  msg.reply = err_code;
  /* msg.listen_vcc = call->in.id; */
  to_sigd(call->in.id,&msg);
}

void send_reject_not_id(CALL *call, int err_code)
{
  struct atmsvc_msg msg;

  /*  this is always sent to called */
  memset(&msg,0,sizeof(msg));
  msg.type = as_reject;
  msg.vcc = (unsigned long) call->listen; 
  msg.reply = err_code;
  /* msg.listen_vcc = call->in.id; */
  to_sigd(call->in.id,&msg);
}

void send_close(CALL *call,int dest)
{
  unsigned long id;
  struct atmsvc_msg msg;

  memset(&msg,0,sizeof(msg));
  msg.type = as_close;
  msg.vcc = (unsigned long) call | dest; /* dest: CALLER or CALLED */
  id = dest == CALLER ? id = call->in.id : call->out.id;
  /* msg.reply = ??!! */  
  to_sigd(id,&msg);
}

void send_accept(CALL *call)
{
  struct atmsvc_msg msg;

  memset(&msg,0,sizeof(msg));
  msg.type = as_accept;
  msg.vcc = (unsigned long) call;
  /* msg.listen_vcc = call->in.id; */
  to_sigd(call->in.id,&msg);
}

void send_free_rm(long id, CALL *call)
{
  /* here we send an as_close msg */
  struct atmsvc_msg msg;

  memset(&msg,0,sizeof(msg));
  msg.type = as_close;
  msg.vcc = (unsigned long) call | RM;
  to_rm(id,&msg);
}

/*****************************************************************************/
/* Demultiplexing with magic number: caller - called - rm                    */
/*****************************************************************************/
CALL *demux_in(unsigned long *srce, struct atmsvc_msg *msg)
{
  /* The multiplexing informations are in the 3 least significant bits
     of the call pointer. We can do this, because the compiler aligns
     memory reservation to pointers with  3 ls-bits = 0. 
     */
  CALL *call;

  if ((msg->type == as_identify) || (msg->type == as_indicate)) {
    call = (struct _call *) (msg->listen_vcc);
    *srce = CALLER;
    return call;
  } /* all identify msgs come from a listening call */
  else {
    call = (struct _call *) (msg->vcc & ~3);
    *srce = (unsigned long) (msg->vcc & 3);
    return call;
  }
}

/*****************************************************************************/
/* Debugging functions                                                       */
/*****************************************************************************/
void print_msg(struct atmsvc_msg *msg, CALL *call,unsigned long source) {
  
  printf("Msg '%s' received from %s vcc=%ld for call 0x%p, listen: %ld\n",
	 as_msgs[msg->type], sources[source], msg->vcc, call,
	 msg->listen_vcc);
}
void print_state(CALL *call) {
  printf("    Call 0x%p entered state '%s'\n", 
	 call , cs_states[call->state]);
}
void print_call(CALL *call) {
  printf("    Call 0x%p in state %s, caller-id:%ld, called-id:%ld\n",
	 call, cs_states[call->state], 
	 call->in.id, call->out.id);
}
