/* text2qos.c - Converts textual representation of QOS parameters to binary
		encoding */

/* Written 1996 by Werner Almesberger, EPFL-LRC */


#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <linux/atm.h>

#include "atm.h"


#define RATE_ERROR -2


static int fetch(const char **pos,...)
{
    const char *value;
    int ref_len,best_len,len;
    int i,best;
    va_list ap;

    va_start(ap,pos);
    ref_len = strlen(*pos);
    best_len = 0;
    best = -1;
    for (i = 0; (value = va_arg(ap,const char *)); i++) {
	len = strlen(value);
	if (*value != '!' && len <= ref_len && len > best_len &&
	  !strncasecmp(*pos,value,len)) {
	    best = i;
	    best_len = len;
	}
    }
    va_end(ap);
    if (best > -1) (*pos) += best_len;
    return best;
}


static int get_rate(const char **text,int up)
{
    const char mult[] = "kKmMgGg";
    const char *multiplier;
    char *end;
    unsigned int rate;

    if (!strncmp(*text,"max",3)) {
	*text += 3;
	return ATM_MAX_PCR;
    }
    rate = strtoul(*text,&end,10);
    if (*end && (multiplier = strchr(mult,*end))) {
	while (multiplier >= mult) {
	    if (rate > UINT_MAX/1000) return RATE_ERROR;
	    rate *= 1000;
	    multiplier -= 2;
	}
	end++;
    }
    if (strlen(end) >= 3)
	if (!strncmp(end,"cps",3)) end += 3;
	else if (!strncmp(end,"bps",3)) {
		rate = (rate+(up ? 8*ATM_CELL_PAYLOAD-1 : 0))/8/
		  ATM_CELL_PAYLOAD;
		end += 3;
	    }
    if (rate > INT_MAX) return RATE_ERROR;
    *text = end;
    return rate;
}


static int params(const char **text,struct atm_trafprm *a,
  struct atm_trafprm *b)
{
    int value;
    char *end;

    if (*(*text)++ != ':') return -1;
    while (1) {
	if (!**text) return -1;
	switch (fetch(text,"max_pcr=","pcr=","min_pcr=","max_sdu=","sdu=",
	  NULL)) {
	    case 0:
	    case 1:
		if ((value = get_rate(text,0)) == RATE_ERROR) return -1;
		a->max_pcr = value;
		if (b) b->max_pcr = value;
		break;
	    case 2:
		if ((value = get_rate(text,1)) == RATE_ERROR) return -1;
		a->min_pcr = value;
		if (b) b->min_pcr = value;
		break;
	    case 3:
	    case 4:
		value = strtol(*text,&end,10);
		if (value < 0) return -1;
		*text = end;
		a->max_sdu = value;
		if (b) b->max_sdu = value;
		break;
	    default:
		return 0;
	}
	if (!**text) break;
	if (*(*text)++ != ',') return -1;
    }
    return 0;
}


int text2qos(const char *text,struct atm_qos *qos,int flags)
{
    int traffic_class;

    if ((traffic_class = fetch(&text,"!none","ubr","cbr",NULL)) < 0) return -1;
    if (!(flags & T2Q_DEFAULTS)) memset(qos,0,sizeof(*qos));
    qos->txtp.traffic_class = qos->rxtp.traffic_class = traffic_class;
    if (!*text) return 0;
    if (params(&text,&qos->txtp,&qos->rxtp)) return -1;
    if (!*text) return 0;
    switch (fetch(&text,"tx","rx",NULL)) {
	case 0:
	    if (!fetch(&text,":none",NULL)) {
		qos->txtp.traffic_class = ATM_NONE;
		if (*text == ',') text++;
		break;
	    }
	    if (params(&text,&qos->txtp,NULL)) return -1;
	    break;
	case 1:
	    text -= 2;
	    break;
	default:
	    return -1;
    }
    if (!*text) return 0;
    if (fetch(&text,"rx",NULL)) return -1;
    if (!fetch(&text,":none",NULL)) qos->rxtp.traffic_class = ATM_NONE;
    else if (params(&text,&qos->rxtp,NULL)) return -1;
    return *text ? -1 : 0;
}
