/* diag.c - Diagnostic messages */

/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */


#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>

#include "atmd.h"


#define MAX_DIAG_MSG 8200


typedef struct _component {
    const char *name;
    int verbosity;
    struct _component *next;
} COMPONENT;


static int sev2prio[] = { DIAG_DEBUG,LOG_DEBUG, DIAG_INFO,LOG_INFO,
  DIAG_WARN,LOG_WARNING, DIAG_ERROR,LOG_ERR, DIAG_FATAL,LOG_ALERT,
  -1,LOG_NOTICE };


static const char *app_name = NULL;
static COMPONENT *components = NULL;
static int default_verbosity = DIAG_INFO;
static FILE *log_to = stderr;


void set_application(const char *name)
{
    app_name = name;
}


void set_verbosity(const char *component,int level)
{
    COMPONENT *walk;

    if (!component) {
	default_verbosity = level;
	return;
    }
    for (walk = components; walk; walk = walk->next)
	if (!strcmp(walk->name,component)) break;
    if (!walk) {
	walk = alloc_t(COMPONENT);
	walk->name = component;
	walk->next = components;
	components = walk;
    }
    walk->verbosity = level;
}


void set_logfile(const char *name)
{
    if (log_to && log_to != stderr) {
	(void) fclose(log_to);
	log_to = stderr;
    }
    if (!name) {
	log_to = stderr;
	return;
    }
    if (!strcmp(name,"syslog")) {
	if (app_name) openlog(app_name,LOG_CONS,LOG_DAEMON);
	log_to = NULL;
    }
    else if (!(log_to = fopen(name,"w"))) {
	    perror(name);
	    log_to = stderr;
	}
}


void vdiag(const char *component,int severity,const char *fmt,va_list ap)
{
    COMPONENT *walk;
    char buffer[MAX_DIAG_MSG+1];
    int i;

    for (walk = components; walk; walk = walk->next)
	if (!strcmp(walk->name,component)) break;
    if (severity > (walk ? walk->verbosity : default_verbosity)) return;
    fflush(stdout);
    if (!log_to) {
	for (i = 0; sev2prio[i] == severity || sev2prio[i] == -1; i += 2);
	vsprintf(buffer,fmt,ap);
	syslog(sev2prio[i+1],"%s: %s",component,buffer);
    }
    else {
	if (app_name) fprintf(log_to,"%s:%s: ",app_name,component);
	else fprintf(log_to,"%s: ",component);
	vfprintf(log_to,fmt,ap);
	fputc('\n',log_to);
	fflush(log_to);
    }
    if (severity == DIAG_FATAL) {
	fprintf(stderr,"Fatal error - Terminating\n");
	exit(1);
    }
}


void diag(const char *component,int severity,const char *fmt,...)
{
    va_list ap;

    va_start(ap,fmt);
    vdiag(component,severity,fmt,ap);
    va_end(ap);
}
