/*
 * Lan Emulation Server
 *
 * $Id: lane.c,v 1.21 1995/11/15 08:27:11 carnil Exp $
 *
 */

/* System includes */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

/* Local includes */
#include "units.h"
#include "load.h"
#include "dump.h"
#include "mem.h"
#include "connect.h"
#include "events.h"
#include "timers.h"

/* Type definitions */

/* Local function prototypes */
static void main_init1(void);
static void parse_args(int argc, const char **argv);
static void usage(void);
static int dump_handler(const Event_t *event, void *funcdata);
static int exit_handler(const Event_t *event, void *funcdata);

/* Data */
static const char *rcsid = "$Id: lane.c,v 1.21 1995/11/15 08:27:11 carnil Exp $";
const Unit_t main_unit = {
  "main",
  NULL,
  main_init1,
  NULL,
  NULL
};

static const char *progname;

/* Functions */

/* Initialization for data that needs other units */
static void
main_init1(void)
{
  set_var_str(&main_unit, "version", rcsid);
  add_event_handler(CE_DUMP, &dump_handler, "dump_handler", NULL);
  add_event_handler(CE_EXIT, &exit_handler, "exit_handler", NULL);
  Debug_unit(&main_unit, "Initialized.");
}

/* Main loop */
int
main(int argc, const char **argv)
{
  short do_restart = 0;
  const Event_t *event;
  const Unit_t **units;

  /* Debugging facility */
  dump_type = DT_STDERR;

  while (1) {
    /* Call phase 0 initializers */
    Debug_unit(&main_unit, "Calling phase 0 initializers");
    FOR_ALL_UNITS(units) {
      if ((*units)->init0 != NULL) {
	Debug_unit(&main_unit, "Initializing %s", (*units)->name);
	(*((*units)->init0))();
      }
    }

    /* Get flags from command line */
    parse_args(argc, argv);

    /* Call phase 1 initializers */
    Debug_unit(&main_unit, "Calling phase 1 initializers");
    FOR_ALL_UNITS(units) {
      if ((*units)->init1 != NULL) {
	Debug_unit(&main_unit, "Initializing %s", (*units)->name);
	(*((*units)->init1))();
      }
    }

    do_restart = 0;

    while (!do_restart) {
      event = event_get_next();
      if (dispatch_handlers(event) == 1) {
	continue;
      }
      switch (event->type) {
      case CE_RESTART:
	do_restart = 1;
	break;
      case CE_EXIT:
	break;
      case CE_DUMP:
	break;
      case CE_SVC_OPEN:
	break;
      case CE_SVC_CLOSE:
	break;
      case CE_DATA:
	break;
      case CE_TIMER:
	break;
      }
      mem_free(&main_unit, event);
    }
    /* Restart */
    Debug_unit(&main_unit, "Releasing %d units", num_units);
    FOR_ALL_UNITS_REV(units) {
      if ((*units)->release != NULL) {
	Debug_unit(&main_unit, "Releasing %s", (*units)->name);
	(*((*units)->release))();
      }
    }
  }
  return 0;
}

/* Process CE_DUMP events */
static int
dump_handler(const Event_t *event, void *funcdata)
{
  const Unit_t **units;

  mem_free(&main_unit, event);
  FOR_ALL_UNITS(units) {
    if ((*units)->dump != NULL) {
      Debug_unit(&main_unit, "Dumping %s", (*units)->name);
      (*((*units)->dump))();
    }
  }
  return 1;
}

/* Process CE_EXIT events */
static int
exit_handler(const Event_t *event, void *funcdata)
{
  const Unit_t **units;

  mem_free(&main_unit, event);
  Debug_unit(&main_unit, "Releasing %d units", num_units);
  FOR_ALL_UNITS_REV(units) {
    if ((*units)->release != NULL) {
      Debug_unit(&main_unit, "Releasing %s", (*units)->name);
      (*((*units)->release))();
    }
  }
  exit(0);
}

/*
 * -dmodule sets debugging output of the module to true
 * -mmodule sets memory debugging of the module to true
 */
static void
parse_args(int argc, const char **argv)
{
  int i;
  const Unit_t *unit, **units;
  const char *arg;

  progname = argv[0];
  for (i = 1; i < argc; i++) {
    arg = argv[i];
    Debug_unit(&main_unit, "Arg %s", arg);
    if (arg[0] == '-') {
      switch (arg[1]) {
      case 'd':
	if (strcmp(&arg[2], "all") == 0) {
	  FOR_ALL_UNITS(units) {
	    set_var_bool(*units, "debug", BL_TRUE);
	  }
	}
	else {
	  unit = find_unit(&arg[2]);
	  if (unit != NULL) {
	    set_var_bool(unit, "debug", BL_TRUE);
	  }
	  else {
	    dump_printf(EL_ERROR, "Unknown module name: %s", &arg[2]);
	    usage();
	  }
	}
	break;
      case 'm':
	if (strcmp(&arg[2], "all") == 0) {
	  FOR_ALL_UNITS(units) {
	    set_var_bool(*units, "memdebug", BL_TRUE);
	  }
	}
	else {
	  unit = find_unit(&arg[2]);
	  if (unit != NULL) {
	    set_var_bool(unit, "memdebug", BL_TRUE);
	  }
	  else {
	    dump_printf(EL_ERROR, "Unknown module name: %s", &arg[2]);
	    usage();
	  }
	}
	break;
      default:
	usage();
	return;
      }
    }
  }
}

static void
usage(void)
{
    dump_printf(EL_ERROR, "Usage:");
    dump_printf(EL_ERROR, "%s [-dmodule]... [-mmodule]...", progname);
    exit(1);
}

