/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   lec.h
*   
* Overview:
*   This header file describes the interface to the LAN Emulation Client (LEC)
*   module.  It is through this interface that network protocols join
*   Emulated LANs (ELANs) and use their services.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   13-Dec-94  TLR   Created.
*    3-Jan-95  TLR   Updated based on changes to Connection Manager interface.
*   10-Jan-95  TLR   Updated based on team review.
*   10-May-95  TLR   Added Support for Event Logging.
*   27-Jun-95  TLR   Added LEC Management Features.
*
* Dependencies:
*   types.h    - Generally used type definitions.
*   codes.h    - Completion codes.
*   atm.h      - ATM specific definitions and types.
*
* Description:
*   This is the interface through which the LEC provides services to LAN
*   Emulation users (typically protocols).  It is a registration style
*   interface that allows multiple protocols to use the LEC simultaneously.
*   Each user becomes a member of a separate ELAN.
*
*   When a user registers with the LEC, the user provides its desired ELAN
*   characteristics (i.e. LAN type, maximum frame size, etc.).  When the
*   network is available, the LEC joins an ELAN on behalf of the user.  If the
*   network becomes unavailable, the ELAN membership is cancelled.  As long
*   as the user remains registered, the LEC waits for the network to
*   become available so it can rejoin on behalf of the user.
*
*   If possible, the LEC uses the services of a LAN Emulation Configuration
*   Server (LECS) to determine which ELAN each of its registered users should
*   join.  Each time a join is attempted, the LECS is contacted.  This is to
*   allow failover mechanisms built into the LECS to function correctly.  If
*   a LAN Emulation server fails, a backup server or backup ELAN can be
*   contacted by each of the affected LECs.
*
* Call Sequence Requirements:
*
*   The following diagram depicts the call sequence constraints:  Functions
*   with lesser indentation must be called before function with greater
*   indentation.
*
*   - lec_create
*     - lec_register
*       - lec_set_filters
*       - lec_get_filters
*       - lec_mcast_add
*       - lec_mcast_delete
*       - lec_mcast_get
*       - lec_xmt
*       - lec_event_mask_set
*       - lec_unregister
*     - lec_destroy
*
* Example Usage Scenario:
*
*   Callbacks are shown in parentheses.
*
*   lec_create..............Called at initialization time to create this
*                           instance of the LEC.
*
*   lec_register............A protocol is attempting to connect to the network
*                           and uses this call to register with the LEC.
*                           The "ELAN join" parameters are provided in the call.
*                           In this case, the network is available and the
*                           join process begins.
*
*                           Note that the registered user must assume that the
*                           initial status of the network is "UNAVAILABLE".
*                           Notification shall be provided when the join is
*                           complete.
*
*   lec_set_filters.........The protocol uses this call to set multicast,
*                           broadcast, and promiscuous filters.
*
*   lec_mcast_add...........The protocol uses this call to add multicast
*                           addresses to the multicast address table.  This
*                           is meaningful only if the multicast filter is
*                           enabled.
*
*   (lec_event_callback)....The join process has successfully completed.  This
*                           callback notifies the protocol that the network is
*                           now available for use.
*
*   lec_xmt.................The protocol calls this function to transmit a
*                           data packet on the network.  There are three
*                           possible status values that this function may
*                           return.
*
*                           If the function returns "SUCCESS", the protocol
*                           may assume that the packet was transmitted and
*                           that it is safe to deallocate or overwrite the
*                           packet buffer.
*
*                           If the function returns "PENDING", the packet was
*                           successfully queued for transmission but has not
*                           yet been copied from the buffer.  The protocol
*                           must not deallocate or overwrite the buffer until
*                           a transmit-done notification is returned for that
*                           buffer.
*
*                           If the transmit function returns "CONGESTED", the
*                           packet was neither transmitted nor buffered.  This
*                           indicates that there is congestion somewhere down
*                           the line which is preventing this packet from being
*                           transmitted.  The protocol has the option of
*                           queueing the packet for retransmission or
*                           discarding the packet.
*
*                           In the case of congestion, there is no notification
*                           that the congestion has cleared.  If the protocol
*                           chooses to queue the packet for retransmission, it
*                           must retry that transmission at a later time.
*
*   (lec_xmt_done_callback).This callback notifies the protocol that a packet
*                           that was queued for transmit has been successfully
*                           copied from the buffer and that the buffer may be
*                           deallocated or overwritten.  A pointer to the
*                           buffer is provided in the callback.
*
*   (lec_rcv_callback)......This callback is invoked when a packet is received
*                           from the network destined for this protocol.
*
*   (lec_event_callback)....If the network becomes unavailable for any reason,
*                           the event callback is used to notify the protocol
*                           that network services are no longer available.
*
*   (lec_event_callback)....The event callback is invoked when the network
*                           has become available again and the re-join to the
*                           ELAN has completed.
*
*   lec_unregister..........Once the registered protocol no longer needs to
*                           use the network it may unregister.  The membership
*                           with the ELAN is cancelled.
*
*   lec_destroy.............This call is used to destroy this instance of the
*                           LEC.  This is called when software is unloaded from
*                           a system or when a logical device is removed.
*/

/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* LEC_EVENT
 *
 *   This enumerated type describes the possible events that the LEC reports
 *   to the registered LEC user.
 *
 *   LEC_NETWORK_AVAILABLE......The LEC has just successfully completed the
 *                              join process and is ready to provide
 *                              communication services with the newly
 *                              connected Emulated LAN.
 *
 *   LEC_NETWORK_UNAVAILABLE....The LEC has just detected a network error
 *                              that is severe enough to cause membership in
 *                              the ELAN to be dropped.  This is either due to
 *                              loss of a critical control connection or loss
 *                              of the ATM link to the adjacent switch.  In
 *                              either case, the LEC has begun to attempt to
 *                              rejoin the ELAN with the parameters supplied by
 *                              the registered user during registration.
 */
#ifndef LEC_H
#define LEC_H
#include "atm.h"

typedef enum
   {
   LEC_NETWORK_AVAILABLE,
   LEC_NETWORK_UNAVAILABLE
   } LEC_EVENT;

/* LAN_TYPE
 *
 *   This type enumerates the supported Emulated LAN types.  With the
 *   exception of LAN_FDDI, these are derived directly from the ATM Forum
 *   specification.
 *
 *   LAN_802_3......ATM Forum standard 802.3/ethernet LAN type.
 *
 *   LAN_802_5......ATM Forum standard 802.5/token ring LAN type.
 *
 *   LAN_FDDI.......Non-ATM Forum standard FDDI LAN type.  This is a possible
 *                  Digital value added frame type
 */
typedef enum
   {
   LAN_802_3,
   LAN_802_5,
   LAN_FDDI
   } LAN_TYPE;

/* LAN_MTU
 *
 *   This enumerated type is used to request a maximum frame size when joining
 *   an emulated LAN.  These values are derived from the ATM Forum
 *   specification.
 *
 *   MTU_UNSPEC.....Unspecified maximum.  Will be assigned by the LES.
 *
 *   MTU_1516.......MTU for ethernet frame sizes.
 *
 *   MTU_4544.......MTU for FDDI frame sizes.
 *
 *   MTU_9234.......MTU for RFC1577 (Classical-IP) frame sizes.
 *
 *   MTU_18190......MTU for Token Ring frame sizes.
 */
typedef enum
   {
   MTU_UNSPEC,
   MTU_1516,
   MTU_4544,
   MTU_9234,
   MTU_18190
   } LAN_MTU;

/* INIT_METHOD
 *
 *   This type is used to indicate to an instance of the LEC how it is to
 *   join an ELAN.
 *
 *   INIT_NORMAL...........Normal mode of initialization.  In this case, the
 *                         LEC first attempts to determine the LECS ATM address
 *                         via ILMI.  If it is unable to do so, it will try the
 *                         well-known LECS ATM address, and then the LECS PVC.
 *
 *   INIT_MANUAL_LECS......Use a manually entered LECS address.  If
 *                         configuration fails using this address, the LEC will
 *                         next attempt to use the well-known LECS address.
 *
 *   INIT_WELL_KNOWN_LECS..Use the well-known LECS address.  If configuration
 *                         fails using this address, try the LECS PVC.
 *
 *   INIT_LECS_PVC.........Use the LECS PVC for configuration.
 *
 *   INIT_MANUAL_LES.......Skip the configuration phase altogether and directly
 *                         contact the LES using the manually entered ATM
 *                         address.
 */
typedef enum
   {
   INIT_NORMAL,
   INIT_MANUAL_LECS,
   INIT_WELL_KNOWN_LECS,
   INIT_LECS_PVC,
   INIT_MANUAL_LES
   } INIT_METHOD;

/*++
* ========================
* = (LEC_EVENT_CALLBACK) =
* ========================
*
* Overview:
*   Notifies the registered user of an important event that has taken place
*   within the LEC.
*
* Arguments:
*   context     - (IN) Context value provided at registration time.
*   event       - (IN) Indication of the event that took place.
*   mtu         - (IN) LAN MTU negotiated during the join.
*   p_elan_name - (IN) Name of ELAN that was joined (if event is
*                      "NETWORK_AVAILABLE").
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/

typedef void (*LEC_EVENT_CALLBACK) (HANDLE      context,
                                    LEC_EVENT   event,
                                    LAN_MTU     mtu,
                                    char       *p_elan_name);


/*++
* ======================
* = (LEC_RCV_CALLBACK) =
* ======================
*
* Overview:
*   Callback that is called when a packet is received on the emulated LAN.
*   A pointer to this routine is supplied at registration time by the user.
*   Note that this callback will not be invoked for packets that were received
*   but filtered by one of the receive packet filters.
*
* Arguments:
*   context     - (IN) Context value supplied at registration time.
*   length      - (IN) Length of the packet in bytes.
*   pp_pkt      - (IO) Pointer to a pointer to the received packet structure.
*                      The returned value of this pointer refers to a buffer
*                      that may be freed by the lower driver.  If the received
*                      packet was copied, the pointer may be returned
*                      unchanged.  If a fresh buffer is to replace the
*                      received one, a pointer to the fresh buffer must be
*                      returned.  If there are no free buffers, a NULL pointer
*                      must be returned.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
*
--*/
typedef void (*LEC_RCV_CALLBACK) (HANDLE  context,
                                  UINT32  length,
                                  void  **pp_pkt);


/*++
* ===========================
* = (LEC_XMT_DONE_CALLBACK) =
* ===========================
*
* Overview:
*   Callback for completed transmissions.  This call returns the context of
*   the registered user and the actual transmitted buffer that may now be
*   freed because the contents of the buffer were copied (via DMA or some
*   such mechanism).
*
* Arguments:
*   context     - (IN) Context value supplied at registration time.
*   p_pkt       - (IN) Pointer to the transmitted packet.
*              
* Returns:
*   None
*
* Preconditions:
*   The packet stored in the buffer is a packet that was previously
*   transmitted by this registered client.
*
* Postconditions:
*   None
*
--*/
typedef void (*LEC_XMT_DONE_CALLBACK) (HANDLE  context,
                                       void   *p_packet);


/*++
* ==============
* = lec_create =
* ==============
*
* Overview:
*   Creates a new instance of the LEC module.
*
* Arguments:
*   os_handle       - (IN)  Handle for OS utilities.
*   cm_handle       - (IN)  Handle for corresponding Connection Manager
*                           instance.
*   line_up_handle  - (IN)  Handle of associated line-up instance.
*   p_text          - (IN)  Pointer to a descriptive text string.
*   p_lec_handle    - (OUT) Handle for this LEC instance.
*
* Returns:
*   STATUS_K_SUCCESS    - Success: See Postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to complete registration.
*
* Preconditions:
*   None
*
* Postconditions:
*   The handle is valid.
*
--*/
STATUS lec_create (HANDLE   line_up_handle,
                   char    *p_text,
                   HANDLE  *p_lec_handle);

/*++
* ===============
* = lec_destroy =
* ===============
*
* Overview:
*   Destroys an instance of the LEC module.
*
* Arguments:
*   lec_handle   - (IN)  Handle returned by lec_create.
*
* Returns:
*   None
*
* Preconditions:
*   The handle is valid.
*
* Postconditions:
*   The handle is invalid.
*
--*/
void lec_destroy (HANDLE  lec_handle);

/*++
* ================
* = lec_register =
* ================
*
* Overview:
*   Registers a new LEC user and begins a join to an Emulated LAN.  Note that
*   before the join process can begin, the LEC must obtain a new ATM address
*   via Address Registration.
*
* Arguments:
*   lec_handle          - (IN)  Handle returned by lec_create.
*   context             - (IN)  Context handle to be passed back in callbacks.
*   addr_reg_handle     - (IN)  Handle of Address Registration module instance.
*   event_callback      - (IN)  Address of event callback function.
*   rcv_callback        - (IN)  Address of receive callback function.
*   xmt_done_callback   - (IN)  Address of transmit done callback function.
*   mac_addr            - (IN)  This station's MAC address.  If this address
*                               is 00-00-00-00-00-00, the LEC module will
*                               automatically generate a MAC address (from the
*                               ESI field of its ATM address).
*   lan_type            - (IN)  Requested LAN type.
*   max_frame_size      - (IN)  Requested MTU size.
*   p_elan_name         - (IN)  Name of preferred emulated LAN.
*   init_method         - (IN)  Selected method of initialization.  See the
*                               above description of INIT_METHOD.
*   manual_atm_addr     - (IN)  Manually entered ATM address.  This is required
*                               for some of the initialization methods that may
*                               be chosen using init_method.  If the
*                               init_method does not require a manual ATM
*                               address, this argument will be ignored.
*   proxy_flag          - (IN)  Proxy flag for this ELAN.
*   lport_handle        - (IN)  Logical Port Handle.  Used by proxy LEC's to
*                               identify the ELAN's logical port to the fast
*                               path.
*   p_text              - (IN)  Text constant which describes this instance.
*   p_elan_handle       - (OUT) Handle that is unique to the new registered
*                               ELAN.
*
* Returns:
*   STATUS_K_SUCCESS    - Success: See Postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to complete registration.
*
* Preconditions:
*   None
*
* Postconditions:
*   The handle is valid.
*
*   The status of the network must be assumed to be "NETWORK_UNAVAILABLE".
*
--*/
STATUS lec_register (HANDLE                 lec_handle,
                     HANDLE                 context,
                     HANDLE                 addr_reg_handle,
                     LEC_EVENT_CALLBACK     event_callback,
                     LEC_RCV_CALLBACK       rcv_callback,
                     LEC_XMT_DONE_CALLBACK  xmt_done_callback,
                     ESI                    mac_addr,
                     LAN_TYPE               lan_type,
                     LAN_MTU                max_frame_size,
                     char                  *p_elan_name,
                     INIT_METHOD            init_method,
                     ADDR_ATM               manual_atm_addr,
                     BOOLEAN                proxy_flag,
                     HANDLE                 lport_handle,
                     char                  *p_text,
                     HANDLE                *p_elan_handle);


/*++
* ==================
* = lec_deregister =
* ==================
*
* Overview:
*   Cancels registration with the LEC.  This registration client's Emulated
*   LAN session will be canceled (or will not retry if the network is down).
*
* Arguments:
*   elan_handle  - (IN) Handle returned by lec_register.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The handle is not valid.
*
--*/
void lec_deregister (HANDLE elan_handle);


/*++
* ======================
* = lec_event_mask_set =
* ======================
*
* Overview:
*   Sets the event mask for events logged from this module instance.
*
* Arguments:
*   elan_handle  - (IN) Handle returned by lec_register.
*   mask         - (IN) New mask value.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void lec_event_mask_set (HANDLE elan_handle,
                         UINT32 event_mask);


/*++
* ====================
* = lec_mac_addr_add =
* ====================
*
* Overview:
*   Adds an additional MAC address to be registered by the LEC.  This MAC
*   address will become a local MAC address.  Once the registration is
*   complete, all packets received with a Destination Address that macthes
*   the MAC address will be received and passed up to the LEC's user.
*
* Arguments:
*   elan_handle  - (IN) Handle returned by lec_register.
*   mac_addr     - (IN) New MAC address.
*
* Returns:
*   STATUS_K_SUCCESS    - Successful completion.  Address registration is in
*                         process.
*   STATUS_K_RESOURCES  - Insufficient resources to complete operation.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lec_mac_addr_add (HANDLE elan_handle,
                         ESI    mac_addr);


/*++
* ====================
* = lec_mac_addr_del =
* ====================
*
* Overview:
*   Deletes (unregisters) a previously registered MAC address.
*
* Arguments:
*   elan_handle  - (IN) Handle returned by lec_register.
*   mac_addr     - (IN) MAC address to be deleted.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void lec_mac_addr_del (HANDLE elan_handle,
                       ESI    mac_addr);


/*++
* ===================
* = lec_filters_set =
* ===================
*
* Overview:
*   Enable or disable reception of multicast and broadcast packets.
*
* Arguments:
*   elan_handle    - (IN)  Handle returned by lec_register.
*
*   mcast_enb      - (IN)  Multicast enable flag.  If this is set, all
*                          packets to multicast destinations shall be
*                          received.  If it is cleared, only packets with
*                          multicast destination addresses found in the
*                          multicast address table shall be recieved.
*
*   bcast_enb      - (IN)  Broadcast enable flag.  If this flag is set, all
*                          broadcast packets shall be received.  If it is
*                          cleared, no broadcast packets shall be received.
*
*   prom_enb       - (IN)  Promiscuous mode enable.  If this flag is set,
*                          all incoming packets shall be received.  This
*                          overrides the settings of the mcast_enb and
*                          bcast_enb flags.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The new filter settings immediately take effect.
*
--*/
void lec_filters_set (HANDLE  elan_handle,
                      BOOLEAN mcast_enb,
                      BOOLEAN bcast_enb,
                      BOOLEAN prom_enb);


/*++
* ===================
* = lec_filters_get =
* ===================
*
* Overview:
*   Gets the current settings of the receive packet filters.
*
* Arguments:
*   elan_handle    - (IN)  Handle returned by lec_register.
*   p_mcast_enb    - (OUT) Setting of multicast enable.
*   p_bcast_enb    - (OUT) Setting of broadcast enable.
*   p_prom_enb     - (OUT) Setting of promiscuous mode.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   Returned values reflect status of the filters that are in effect.
*
--*/
void lec_filters_get (HANDLE     elan_handle,
                      BOOLEAN   *p_mcast_enb,
                      BOOLEAN   *p_bcast_enb,
                      BOOLEAN   *p_prom_enb);


/*++
* =================
* = lec_mcast_add =
* =================
*
* Overview:
*   Adds a multicast 48-bit address to the multicast address table.  If this
*   address is already in the table, its instance count shall be incremented.
*
* Arguments:
*   elan_handle - (IN)  Handle returned by lec_register.
*   addr        - (IN)  48-bit multicast address to be added.    
*
* Returns:
*   STATUS_K_SUCCESS    - Success: See Postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to perform the operation.
*                         Note that this may occur because host memory could
*                         not be allocated or because an on-adapter multicast
*                         address table is full.
*
* Preconditions:
*   None
*
* Postconditions:
*   If the address was not previously in the table, it is added with an
*   instance count of 1.
*
*   If the address was previously in the table, its instance count is 1
*   greater than its previous value.
*   
--*/
STATUS lec_mcast_add (HANDLE  elan_handle,
                      ESI     addr);

/*++
* ====================
* = lec_mcast_delete =
* ====================
*
* Overview:
*   Deletes an instance of a multicast address from the multicast address
*   table.
*
* Arguments:
*   elan_handle  - (IN)  Handle returned by lec_register.
*   addr         - (IN)  Multicast address to be deleted.
*
* Returns:
*   STATUS_K_SUCCESS    - Success: See Postconditions.
*   STATUS_K_NOT_FOUND  - No such address was found.
*
* Preconditions:
*   None
*
* Postconditions:
*   If the address previously had an instance count of 2 or greater, the
*   instance count is decremented by one.
*
*   If the address previously has an instance count of 1, the address has
*   been deleted from the table.
*
--*/
STATUS lec_mcast_delete (HANDLE  elan_handle,
                         ESI     addr);


/*++
* ==================
* = lec_mcast_load =
* ==================
*
* Overview:
*   Clears and loads the multicast table in its entirety.
*
* Arguments:
*   elan_handle    - (IN)  Handle returned by lec_register.
*   table_size     - (IN)  Number of addresses in the table.
*   addr_array     - (IN)  Address array.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The multicast address table contains the same addresses as the address
*   array.  All instance counts are 1.
*
--*/
void lec_mcast_load (HANDLE   elan_handle,
                     UINT32   table_size,
                     ADDR_48  addr_array[]);


/*++
* =================
* = lec_mcast_get =
* =================
*
* Overview:
*   Gets the contents of the multicast address table.
*
* Arguments:
*   elan_handle    - (IN)  Handle returned by lec_register.
*   array_size     - (IN)  Number of elements in array buffers.
*   addr_array     - (IN)  Pointer to the address array.
*   count_array    - (IN)  Pointer to the instance count array.
*   p_addr_count   - (OUT) Number of addresses in multicast table.
*                          Note that if this number is greater than array_size,
*                          only array_size addresses have been copied into the
*                          arrays.  The remaining (*p_addr_count - array_size)
*                          entries are not copied.
*
* Returns:
*   None
*
* Preconditions:
*   If array_size is zero, the pointers to both arrays need not be valid.
*
* Postconditions:
*   MIN (array_size, *p_addr_count) entries have been copied into the two
*   arrays.
*
*   For i from 0 to MIN (array_size, *p_addr_count) : p_count_array[i] is the
*   instance count for the address in p_addr_array[i].
*
--*/
void lec_mcast_get (HANDLE   elan_handle,
                    UINT32   array_size,
                    ESI      addr_array[],
                    UINT32   count_array[],
                    UINT32  *p_addr_count);


/*++
* ===========
* = lec_xmt =
* ===========
*
* Overview:
*   Requests the transmission of a packet to the Emulated LAN network.
*
* Arguments:
*   elan_handle    - (IN)  Handle returned by lec_register.
*   da             - (IN)  Destination address of the packet.
*   p_pkt          - (IN)  Pointer to the packet to be transmitted.  The
*                          format of this packet is not known to the LEC.
*                          It simply sends the pointer down to the lower
*                          driver.
*   length         - (IN)  The length of the packet in bytes.
*
* Returns:
*   STATUS_K_SUCCESS      - The packet was successfully transmitted.  The
*                           buffer may be safely deallocated by the caller.
*
*   STATUS_K_PENDING      - The packet was successfully queued for
*                           transmission.  The buffer may not be deallocated
*                           until it is returned by the transmit-done callback.
*
*   STATUS_K_CONGESTED    - Packet was not transmitted due to congestion.
*                           The transmit packet buffer is still intact
*                           and may be buffered or dropped.
*
*   STATUS_K_FAILURE      - The network is not available.  The packet was not
*                           transmitted and the packet buffer is still intact.
*
* Preconditions:
*   None.
*
* Postconditions:
*   None.
*
--*/
STATUS lec_xmt (HANDLE   elan_handle,
                ESI      da,
                void    *p_pkt,
                UINT32   length);


/*++
* ================
* = lec_attr_get =
* ================
*
* Overview:
*   Get a management attribute for an ELAN instance.  Refer to lec_mgt.h for
*   a list of attribute IDs.
*
* Arguments:
*   elan_handle       - (IN)  Handle returned by lec_register.
*   attr_id           - (IN)  ID of attribute to be accessed.
*   p_out_buffer      - (IN)  Pointer to the output buffer to be filled with
*                             data.
*   out_buffer_length - (IN)  Number of bytes writable in the output buffer.
*   p_bytes_written   - (OUT) Number of bytes written.
*
* Returns:
*   STATUS_K_SUCCESS       - Successful Completion.
*   STATUS_K_ARG_2_BAD     - Invalid Attribute ID.
*   STATUS_K_ARG_4_BAD     - Not enough space in output buffer.
*   STATUS_K_NOT_FOUND     - Instance not found.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lec_attr_get (HANDLE     elan_handle,
                     UINT16     attr_id,
                     void      *p_out_buffer,
                     UINT16     out_buffer_length,
                     UINT16    *p_bytes_written);

/*++
* =====================
* = lec_attr_get_next =
* =====================
*
* Overview:
*   Get a management attribute for an ELAN instance.  Refer to lec_mgt.h for
*   a list of attribute IDs.  This function gets the next instance of the
*   attribute after the key supplied in the input buffer.
*
* Arguments:
*   elan_handle       - (IN)  Handle returned by lec_register.
*   attr_id           - (IN)  ID of attribute to be accessed.
*   p_in_buffer       - (IN)  Pointer to input buffer containing search key
*                             for get-next operation.
*   in_buffer_length  - (IN)  Number of bytes in the input buffer.
*   p_out_buffer      - (IN)  Pointer to the output buffer to be filled with
*                             data.
*   out_buffer_length - (IN)  Number of bytes writable in the output buffer.
*   p_bytes_written   - (OUT) Number of bytes written.
*
* Returns:
*   STATUS_K_SUCCESS       - Successful Completion.
*   STATUS_K_ARG_2_BAD     - Invalid Attribute ID.
*   STATUS_K_ARG_6_BAD     - Not enough space in output buffer.
*   STATUS_K_NOT_FOUND     - Instance not found.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lec_attr_get_next (HANDLE     elan_handle,
                          UINT16     attr_id,
                          void      *p_in_buffer,
                          UINT16     in_buffer_length,
                          void      *p_out_buffer,
                          UINT16     out_buffer_length,
                          UINT16    *p_bytes_written);

/*++
* ================
* = lec_attr_set =
* ================
*
* Overview:
*   Set a management attribute for an ELAN instance.  Refer to lec_mgt.h for
*   a list of attribute IDs.
*
* Arguments:
*   elan_handle       - (IN)  Handle returned by lec_register.
*   attr_id           - (IN)  ID of attribute to be accessed.
*   p_in_buffer       - (IN)  Pointer to the input buffer that contains data
*                             to be written.
*   in_buffer_length  - (IN)  Number of bytes to be written.
*
* Returns:
*   STATUS_K_SUCCESS       - Successful Completion.
*   STATUS_K_FAILURE       - Operation Failed.
*   STATUS_K_ARG_2_BAD     - Invalid Attribute ID.
*   STATUS_K_ARG_5_BAD     - Not enough space in output buffer.
*   STATUS_K_NOT_FOUND     - Instance not found.
*   STATUS_K_NO_PRIVELEGE  - Attempted to write a Read-Only attribute.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lec_attr_set (HANDLE     elan_handle,
                     UINT16     attr_id,
                     void      *p_in_buffer,
                     UINT16     in_buffer_length);

#endif /* LEC_H */
