/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/

/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */


/*++
* Module Name:
*   addr_reg.h
*
* Abstract:
*   This file is the interface to the ATM address registration.
*
* Authors:
*   DMW - Douglas M. Washabaugh
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description
*   06-Oct-94  DMW   Created.
*   07-Nov-94  DMW   Updated with review comments, reversed order of routines.
*   29-Mar-95  TLR   Changed callback typedef to a pointer.
*   16-May-95  TLR   Added support for event reporting.
*   06-Sep-95  TLR   Added special address allocation call.
*
* Description:   
*   OVERVIEW OF ADDRESS REGISTRATION:
*
*   Before reading this file, it is assumed that the reader has read section
*   5.8 of the UNI 3.0 specification (Address registration).  In this
*   description the term Network Interface Card (NIC) is used for the
*   "user-side" of the network interface.
*
*   Address registration is the process through which an end-node registers
*   ATM addresses with the switch.  An ATM address is the concatenation of a
*   network prefix, ESI (MAC address), and a selector.  Network prefixes are
*   provided by the switch, ESI values are typically in a ROM on the NIC, and 
*   the selector is an arbitrary value that can be chosen by the end-station.
*   In this implementation, the selector is used to enable 256 ATM addresses
*   to be derived from a single network prefix adn ESI concatenation.
*
*   All network communication between the address registration module and the 
*   switch is done through SNMP messages (get, get-next, set, and traps).
*
*   When this module detects that the link has just come up, it sends out a
*   "cold-start" trap to the switch. It then waits for up to 5 seconds for the
*   switch to set a network prefix.  If this does not occur, this module again
*   sends out a cold start trap. This process is repeated indefinitely.
*
*   While waiting for the switch to set the network prefix, address
*   registration verifies that the switch's address table is empty.  If the 
*   address table is not empty, then this module restarts and goes back and 
*   sends a cold start trap [the check for the empty address table ensures that
*   the switch processed the NIC's cold start trap].
*  
*   When the switch's address table is empty and the switch sets a network
*   prefix on the NIC, then this module can register ATM addresses.
*
*   If the NIC receives a cold start trap from the switch, then this
*   module must invalidate all network prefixes.  If the NIC receives an
*   SNMP message that invalidates a network prefix, then this module must
*   invalidate all registered ATM addresses that use that network prefix.
*   The address registration module may register new ATM addresses
*   immediately if there is a valid network prefix and an available ESI value.
*
*   HOW THE INTERFACE TO THIS MODULE WORKS:
*
*   The figure below shows a typical configuration in which this module is used.
*   It shows a configuration in which two clients have registered with the 
*   Address Registration module, and each client may be "called back" by the 
*   Address Registration module.  There is one instantiation of the Address
*   Registration module for each NIC in the system.
*
*   +-----------+    +------------+
*   | Classical |    |    LAN     |
*   |  TCP/IP   |    | Emulation  |
*   +-----------+    +------------+
*          |  ^          |  ^        ("|" means regular  call)
*          v  :          v  :        (":" means callback call)
*      +----------------------+
*      | Address Registration |
*      +----------------------+
*                |  ^
*                |  :
*                v  :
*              +-------+
*              | SNMP  |
*              | Agent |
*              +-------+
*
*   Initially, the Address Registration module is initalized with a set of MAC
*   addresses for the NIC.  Clients can subsequently register with the 
*   Address Registration module and specify a callback routine that is called
*   when ATM addresses are registered or deregistered.  After the client 
*   registeration is complete, the client can begin registering ATM
*   addresses with the switch.  If the registration of the ATM address
*   completes successfully, the client is called back with the ATM address that
*   was registered.  Note that the clients do not specify the ATM address
*   but instead are given one.
*
*   If a client attempts to register addresses when the Address Registration
*   module is not ready for this operation, the request is queued.  When the
*   switch is ready to register addresses, this module completes the
*   registration and executes a callback to the client.
*
*   If an address registration request is queued, there is no time limit until
*   the request is completed.  It is possible that client may wait for a 
*   long time before being given the requested address.  This is the mechanism
*   by which clients are held off when there is no switch to talk to.  For 
*   example, LAN Emulation will not attempt to communicate with a non-existent 
*   switch because it will be blocked waiting for an ATM address.  Likewise,
*   when a switch goes away, clients will have their ATM addresses invalidated
*   and will be forced to wait for new ones.
*
*   The switch may register an additional network prefix at any time.  When this
*   occurs, any future address registrations will use the newest network
*   prefix.  Address registrations that are in progress and previously
*   ATM addresses are not affected.
*
*   The switch may also deregister network prefixes at any time.  When this
*   occurs, then all clients which have registered addresses using the
*   deregistered network prefix are notified that they must re-register a 
*   network address.  This must be done before the client sends any more 
*   data using the invalidated ATM address.  If an ATM address using the
*   deleted network prefix is has started the registration process but not yet 
*   completed it, then the address registration fails and the client is
*   notified via the callback.
*
*   The switch may send a cold-start trap at any time.  When this occurs, this
*   module must invalidate all the network prefixes that the switch previously 
*   set. Clients are notified that their ATM address are now invalid via the 
*   callback.  If there are any address registrations in progress when the cold 
*   start trap is in progress, then will automatically be retried when a 
*   network prefix has been set.
*
*   This module must detect if the link goes down so that it can inform the
*   switch that a "cold start" event has occured.  The clients are notified 
*   that their ATM addresses are invalid via callback.
*
*   SNMP AGENT INTERFACE:
*
*   This module must interface with the SNMP agent to receive cold-start traps,
*   generate messages to send to the switch, and to receive SNMP messages from
*   the switch.
*
*   MANAGEMENT INTEFACE:
*   
*   The management interface does not perform any work, but enables applications
*   to have some visibility into this module.  Currently the management
*   interface exports functions to determine the state of address registration
*   and to get a list of the ATM addresses that have been registered.  Generally
*   these routines can be called at any time.
*
* Call Sequence:
*   addr_reg_create              - Creates an instance, inits the MAC addresses.
*   addr_reg_client_register     - Client registers
*   addr_reg_atm_addr_alloc      - Gets and registers an ATM address.
*   addr_reg_atm_addr_dealloc    - Deregister an ATM address.
*   addr_reg_destroy             - Returns any allocated resources.
*
* Issues:
*   - Do the clients need to be able to deregister themselves?  Currently this
*     interface assumes that they do not.
*
*   - Does the management interface require the state of each client?  Does
*     it require a list of clients?
--*/

/* Define the types of events for address status change callbacks. */

typedef enum
   {
   ADDR_REG_EVENT_ATM_ADDR_ALLOCATED,   /* An ATM address was registered.   */
   ADDR_REG_EVENT_ATM_ADDR_INVALIDATED  /* Registered addr was invalidated. */
   } ADDR_REG_ADDR_EVENT;

/* Define values for the addr_reg_special_alloc call. */

#define ADDR_REG_ESI_NEW  0x8001
#define ADDR_REG_ESI_ANY  0x8002
#define ADDR_REG_SEL_ANY  0x8003

/*++
* ---------------------
* - ADDR_REG_CALLBACK -
* ---------------------
*
* Overview:
*   This is the callback made whenever an address is invalidated or registered.
*   In the event an address is invalidated, the selector of the address is not
*   significant.  All addresses with the provided prefix and ESI are to be
*   considered invalidated by this single callback.
*
* Arguments:
*   context    - (IN)  The caller context (handle) that was specified when
*                      the address was registered.
*
*   p_atm_addr - (IN)  Pointer to the ATM address that was registered.  This
*                      pointer is only valid in the callback routine.  It 
*                      may not be saved away.  Instead, the ATM address must
*                      be copied.
*
*   event      - (IN)  The event associated with the ATM address.  The event 
*                      is one of: ADDR_REG_ADDR_EVENT_REGISTERED, 
*                      ADDR_REG_ADDR_EVENT_INVALIDATED.
*
* Returns:
*   none
*
* Preconditions:
*   The registration of an address was started and or completed. 
*
* Postconditions:
*   none
*
* Issues:
*   none
--*/                                              

typedef void (*ADDR_REG_CALLBACK) (HANDLE               context,
                                   ADDR_ATM            *p_atm_addr,
                                   ADDR_REG_ADDR_EVENT  event);   

/*++
* ============================
* = addr_reg_client_register =
* ============================
*
* Overview:
*   Registers a client.  After a client is registered, it may register and 
*   deregister ATM addresses.  A client can not be deregistered.  A client must
*   register so that it can be told about changes in ATM addresses.
*
* Arguments:
*   addr_reg_handle - (IN)  The handle that was returned from addr_reg_create.
*
*   p_callback      - (IN)  A pointer to a routine to call when an ATM address
*                           registration or deregistration has completed.
* 
*   callback_handle - (IN)  A value that is passed on a callback to the client.
*
*   p_text          - (IN)  A text string to describe this client.
*
*   p_client_handle - (OUT) Handle that may be required for subsequent calls
*                           to this module.
* Returns:
*   STATUS_K_SUCCESS   - Client is registered.
*   STATUS_K_RESOURCES - Client is NOT registered.
*
* Preconditions:
*   addr_reg_create returned a valid handle.
*
* Postconditions:
*   On success, the client handle is valid.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_client_register (HANDLE             addr_reg_handle,
                                 ADDR_REG_CALLBACK  callback,
                                 HANDLE             callback_handle,
                                 const char         *p_text,
                                 HANDLE            *p_client_handle); 

/*++
* ===========================
* = addr_reg_atm_addr_alloc =
* ===========================
*
* Overview:
*   Directs this module to start the process of registering an ATM address
*   with the switch.  When the registration completes, the client is called
*   back.
*  
*   This routine can be anytime after the client has registered.  Note the 
*   client can only have one outstanding registration request at a time.
*
* Arguments:
*   client_handle - (IN) The handle that was returned from
*                        addr_reg_client_register.
*
* Returns:
*   STATUS_K_PENDING - Operation successfully queued.
*   STATUS_K_BUSY    - Client already has a pending operation.
*
* Preconditions:
*   - The routine addr_reg_create was called. 
*   - The routine addr_reg_client_register was called.
*
* Postconditions:
*   The registration of the address will start at the next opportunity.  An
*   opportunity occurs when there is not another pending address registration
*   and there is at least one network prefix with an available MAC address.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_atm_addr_alloc (HANDLE client_handle);
 
/*++
* =============================
* = addr_reg_atm_addr_dealloc =
* =============================
*
* Overview:
*   The specified ATM address is deallocated and can be reused.  After this
*   call completes, no more packets destined to the ATM address should be sent.
*
* Arguments:
*   client_handle - (IN) The handle that was returned from
*                        addr_reg_client_register.
*
*   p_atm_addr    - (IN) Pointer to the ATM address to deregister.
*
* Returns:
*   STATUS_K_SUCCESS  - The the operation was successfully queued.
*   STATUS_K_ARG2_BAD - The MAC address is not registered.
*
* Preconditions:
*   - The routine addr_reg_create was called. 
*   - The routine addr_reg_client_register was called.
*
* Postconditions:
*   The specified ATM address available for reuse.
*
* Issues:
*   none
--*/                                              

STATUS addr_reg_atm_addr_dealloc (HANDLE    client_handle,
                                  ADDR_ATM *p_atm_addr); 

int address_convert(char *parsestring, ADDR_ATM *atm_addr);
int addr_getesi(unsigned char *mac_addr);




