/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   lec_ctrl.h
*   
* Overview:
*   The LEC_CTRL module is a subcomponent of the LAN Emulation client (LEC)
*   module.  It is responsible for all of the control aspects of LAN Emulation
*   clienthood.  It implements the LEC state machine and handles all LE control
*   packets.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*    4-Oct-94  TLR   Created.
*
* Description:
*
* Data SAP Requirements:
*   The LEC_DATA module is the user of this interface to the LEC_CTRL module.
*   Each instance of the LEC_DATA module joins an ELAN.  In the process, it
*   must first register with the LEC_CTRL module (via lc_register) to obtain
*   an lc_elan_handle.  This handle uniquely represents this ELAN's LEC_CTRL
*   instance.
*
*   Once LEC_CTRL registration is successful, the LEC_DATA instance must
*   register with the Connection Manager (CM) as a SAP client.  In so doing,
*   it MUST provide the following arguments:
*
*     - The 'sap_context' MUST be the lc_elan_handle returned by lc_register.
*
*     - The 'vc_notify_callback' MUST be the lc_sap_vc_notify function
*       described in this interface.
*
*     - The 'svc_connect_callback' MUST be the lc_sap_connect function
*       described in this interface.
*
*     - The 'link_status_callback' MAY be ignored by specifying it as NULL.
*
* Call Sequence Requirements:
*
*   - lc_create
*     - lc_register
*       - lc_join_start
*         - (ELAN_STATUS_CALLBACK)
*           - lc_addr_resolve
*       - lc_event_mask_set
*       - lc_deregister
*     - lc_destroy
*
* Example Usage Scenario  (callbacks are shown in parentheses):
*
*   lc_create...............At system initialization time, an instance of the
*                           LEC module is created.  
*   lc_register.............
*   lc_join_start...........
*   (elan_status_callback)..
*   lc_addr_resolve.........
*   lc_addr_resolve.........
*   (elan_status_callback)..
*   (elan_status_callback)..
*   lc_deregister...........
*   lc_destroy..............
*/
#ifndef LEC_CTRL_H
#define LEC_CTRL_H
#include "cm_sap.h"
/*************************************************************************
 * Callbacks
 *************************************************************************/

/*++
* ==========================
* = (ELAN_STATUS_CALLBACK) =
* ==========================
*
* Overview:
*   This callback is invoked from within the LEC_CTRL module and informs its
*   user (the LEC_DATA module) that the availability of a specific ELAN has
*   changed.  Either the ELAN has become available or it has become
*   unavailable.
*
*   ELAN availability differs from Link Status in that an ELAN is not available
*   until the link becomes available AND the join process is completed.
*
* Arguments:
*   ld_elan_handle  - (IN) Handle describing the ELAN instance of the data
*                          module.
*
*   elan_available  - (IN) Boolean value indicating whether the ELAN has become
*                          available or unavailable.
*
*   mac_addr        - (IN) The MAC address for this station in this ELAN.
*
*   lan_mtu         - (IN) MTU size of the ELAN.
*
*   p_elan_name     - (IN) Name of the ELAN.
*
*   lec_id          - (IN) LEC_ID of this LEC in the newly joined ELAN.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef void (*ELAN_STATUS_CALLBACK) (HANDLE     ld_elan_handle,
                                      BOOLEAN    elan_available,
                                      LAN_MTU    lan_mtu,
                                      char      *p_elan_name,
                                      UINT16     lec_id);

/*************************************************************************
 * Function Prototypes
 *************************************************************************/

/*++
* =============
* = lc_create =
* =============
*
* Overview:
*   Create an instance of the LEC_CTRL module.  This occurs during port
*   initialization.  This instance is not related to Emulated LANs.  A single
*   instance of this module must exist to support multiple ELANs on a single
*   physical interface.
*
* Arguments:
*   os_handle            - (IN)  The handle of the OS module.  This is used to
*                                allocate memory and timers.
*
*   cm_handle            - (IN)  Handle of the associated CM instance.
*
*   line_up_handle       - (IN)  Handle of the associated Line-Up instance.
*
*   elan_status_callback - (IN)  Address of ELAN status callback.
*
*   p_lc_handle          - (OUT) The returned handle for the newly created
*                                instance of the LEC_CTRL module.
*
* Returns:
*   STATUS_K_SUCCESS    - Success, see postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to complete the operation.
*
* Preconditions:
*   None
*
* Postconditions:
*   The new ctrl_handle is valid.
*
*   A le_cache module instance was also created.
--*/
STATUS lc_create (HANDLE                line_up_handle,
                  ELAN_STATUS_CALLBACK  elan_status_callback,
                  HANDLE               *p_lc_handle);


/*++
* ==============
* = lc_destroy =
* ==============
*
* Overview:
*   Destroys an instance of the LEC_CTRL module.  This is called when a
*   physical ATM port is disabled or removed.
*
* Arguments:
*   lc_handle     - (IN)  The handle returned by lc_create.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The handle is now invalid.
*
--*/
void lc_destroy (HANDLE  lc_handle);


/*++
* ===============
* = lc_register =
* ===============
*
* Overview:
*   Register a new ELAN with the LEC_CTRL module.  This is called by the
*   LEC_DATA module when a protocol registers with the LAN Emulation Client.
*
* Arguments:
*   lc_handle           - (IN)  Handle returned by lc_create.
*
*   ld_elan_handle      - (IN)  Handle of LEC_DATA ELAN instance.
*
*   addr_reg_handle     - (IN)  Handle of the associated Address Registration
*                               module instance.
*
*   mac_addr            - (IN)  This station's MAC address.  If this argument
*                               is not all zeros, this address shall override
*                               the internal address read from hardware.
*
*   lan_type            - (IN)  Requested LAN type.
*
*   max_frame_size      - (IN)  Requested MTU size.
*
*   p_elan_name         - (IN)  Name of preferred emulated LAN.
*
*   init_method         - (IN)  Specification of the initialization method to
*                               be used when joining the ELAN.
*
*   manual_atm_addr     - (IN)  Manually entered ATM address which may be needed
*                               for certain init methods that may be specified
*                               in init_method.
*
*   proxy_flag          - (IN)  Proxy flag for this ELAN join.
*
*   lport_handle        - (IN)  Logical Port Handle.  This is used when the
*                               LEC is a proxy.  It identifies the ELAN's
*                               logical port to the fast-path.
*
*   p_text              - (IN)  Descriptive text constant.
*
*   p_lc_elan_handle    - (OUT) Handle for this ELAN instance of the control
*                               module.
*
* Returns:
*   STATUS_K_SUCCESS    - Success, see postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to complete operation.
*
* Preconditions:
*   None
*
* Postconditions:
*   The new handle (*p_lc_elan_handle) is valid.
--*/
STATUS lc_register (HANDLE       lc_handle,
                    HANDLE       ld_elan_handle,
                    HANDLE       addr_reg_handle,
                    ESI          mac_addr,
                    LAN_TYPE     lan_type,
                    LAN_MTU      max_frame_size,
                    char        *p_elan_name,
                    INIT_METHOD  init_method,
                    ADDR_ATM     manual_atm_addr,
                    BOOLEAN      proxy_flag,
                    HANDLE       lport_handle,
                    char        *p_text,
                    HANDLE      *p_lc_elan_handle);


/*++
* =================
* = lc_deregister =
* =================
*
* Overview:
*   Deregisters an instance of an ELAN from the LEC_CONTROL module.  This is
*   called by the LEC_DATA module when a protocol deregisters with the LAN
*   Emulation Client.
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The lc_elan_handle is invalid.
--*/
void lc_deregister (HANDLE lc_elan_handle);


/*++
* =====================
* = lc_event_mask_set =
* =====================
*
* Overview:
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*   event_mask      - (IN) New value of log event mask.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void lc_event_mask_set (HANDLE lc_elan_handle,
                        UINT32 event_mask);


/*++
* ===================
* = lc_mac_addr_add =
* ===================
*
* Overview:
*   Registers a new destination address with the LES.  Note that MAC address
*   registration is asynchronous and does not take effect immediately.  This
*   routine begins the registration process.  There is no notification to the
*   caller that registration is complete.  It simply takes effect when the
*   registration process completes.
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*   mac_addr        - (IN) New LAN destination for this elan instance.
*
* Returns:
*   STATUS_K_SUCCESS - Successful Completion (registration is in process)
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lc_mac_addr_add (HANDLE lc_elan_handle,
                        ESI    mac_addr);

/*++
* ===================
* = lc_mac_addr_del =
* ===================
*
* Overview:
*   Deregisters a registered destination address with the LES.
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*   mac_addr        - (IN) LAN destination to be deregistered.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void lc_mac_addr_del (HANDLE lc_elan_handle,
                      ESI    mac_addr);

/*++
* =========================
* = lc_dest_is_registered =
* =========================
*
* Overview:
*   Checks to see if the specified LAN destination is a registered LAN
*   destination of this LEC
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*   p_lan_dest      - (IN) Pointer to LAN destination to be looked up.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
BOOLEAN lc_dest_is_registered (HANDLE  lc_elan_handle,
                               ESI     p_dst_addr);

/*++
* =================
* = lc_elan_reset =
* =================
*
* Overview:
*
* Arguments:
*   lc_elan_handle  - (IN) Handle returned by lc_register.
*
* Returns:
*   None
*
* Preconditions:
*
* Postconditions:
--*/
void lc_elan_reset (HANDLE lc_elan_handle);


/*++
* =================
* = lc_join_start =
* =================
*
* Overview:
*   Requests the initiation of the join process.  The LEC_CTRL module is
*   given the handle of the data SAP so it can request SVCs on its behalf.
*
* Arguments:
*   lc_elan_handle          - (IN) Handle returned by lc_register.
*   data_sap_handle         - (IN) Handle returned by cm_sap_register.
*   direct_conn_context     - (IN) Connection Context to be used in setting up
*                                  Data-Direct VCCs.
*   mcast_conn_context      - (IN) Connection Context to be used in setting up
*                                  Multicast VCCs.
*   ctrl_conn_context       - (IN) Connection Context to be used in setting up
*                                  Control VCCs.
*
* Returns:
*   None
*
* Preconditions:
*   The Data SAP for this instance must be registered according to the above
*   mentioned requirements.
*
* Postconditions:
*   None
--*/
void lc_join_start (HANDLE  lc_elan_handle,
                    HANDLE  data_sap_handle,
                    HANDLE  direct_conn_context,
                    HANDLE  mcast_conn_context,
                    HANDLE  ctrl_conn_context);


/*++
* ===================
* = lc_addr_resolve =
* ===================
*
* Overview:
*   Requests the resolution of a MAC address to a connection.  This lookup is
*   used to determine which connection a packet is to be transmitted on.  If
*   the destination address is multicast, the resolution is to the multicast-
*   send VCC.
*
* Arguments:
*   lc_elan_handle  - (IN)  Handle returned by lc_register.
*
*   dst_addr        - (IN)  MAC destination address to be looked up.
*
*   p_may_xmt       - (OUT) Flag indicating that the packet may be transmitted.
*                           If this flag is TRUE, the packet may be
*                           transmitted.  If it is FALSE, the packet must not
*                           be transmitted for reasons of "congestion".  This
*                           is normally due to flooding rate limits.
*
*   p_conn_handle   - (OUT) Connection Handle to be used for the transmit.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void lc_addr_resolve (HANDLE   lc_elan_handle,
                      ESI      dst_addr,
                      BOOLEAN *p_may_xmt,
                      HANDLE  *p_conn_handle);


/*++
* ====================
* = lc_sap_vc_notify =
* ====================
*
* Overview:
*   SAP VC notification callback to be used when registering with the CM as
*   a SAP.
*
*   Refer to cm_sap.h for the full description of this function.
*
--*/
void lc_sap_vc_notify (HANDLE           conn_context,
                       HANDLE           conn_handle,
                       NOTIFY_EVENT     event,
                       UINT16           reason,
                       UINT32           endpoint_ref,
                       BOOLEAN          calling_party_flag);

/*++
* ==================
* = lc_sap_connect =
* ==================
*
* Overview:
*   SAP connect notification callback to be used when registering with the
*   CM as a SAP.
*
*   Refer to cm_sap.h for the full description of this function.
*
--*/
BOOLEAN lc_sap_connect (HANDLE       sap_context,
                        CONN_INFO   *p_conn_info,
                        HANDLE       conn_handle,
                        HANDLE      *p_conn_context,
                        UINT32      *p_age_limit,
                        UINT32      *p_reject_reason);

/*++
* ==============
* = lc_sap_rcv =
* ==============
*
* Overview:
*   SAP receive notification callback to be used when registering with the
*   CM as a SAP.
*
*   Refer to cm_sap.h for the full description of this function.
*
--*/
void lc_sap_rcv (HANDLE     conn_context,
                 HANDLE     conn_handle,
                 UINT32     length,
                 UINT32     user_data,
                 AAL_DATA  *p_aal_data,
                 void     **pp_packet);

/*++
* ======================
* = lc_sap_link_status =
* ======================
*
* Overview:
*   SAP link status notification callback to be used when registering with the
*   CM as a SAP.
*
*   Refer to cm_sap.h for the full description of this function.
*
--*/
void lc_sap_link_status (HANDLE          lc_elan_handle,
                         ATM_LINK_STATE  link_state);

/*++
* ===============
* = lc_attr_get =
* ===============
*
* Overview:
*   Get a management attribute for an ELAN instance.  Refer to lec_mgt.h for
*   a list of attribute IDs.
*
* Arguments:
*   lc_elan_handle    - (IN)  Handle returned by lc_register.
*   attr_id           - (IN)  ID of attribute to be accessed.
*   p_out_buffer      - (IN)  Pointer to the output buffer to be filled with
*                             data.
*   out_buffer_length - (IN)  Number of bytes writable in the output buffer.
*   p_bytes_written   - (OUT) Number of bytes written.
*
* Returns:
*   STATUS_K_SUCCESS       - Successful Completion.
*   STATUS_K_ARG_2_BAD     - Invalid Attribute ID.
*   STATUS_K_ARG_4_BAD     - Not enough space in output buffer.
*   STATUS_K_NOT_FOUND     - Instance not found.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lc_attr_get (HANDLE     lc_elan_handle,
                    UINT16     attr_id,
                    void      *p_out_buffer,
                    UINT16     out_buffer_length,
                    UINT16    *p_bytes_written);

STATUS lc_attr_get_next (HANDLE     lc_elan_handle,
                         UINT16     attr_id,
                         void      *p_in_buffer,
                         UINT16     in_buffer_length,
                         void      *p_out_buffer,
                         UINT16     out_buffer_length,
                         UINT16    *p_bytes_written);

/*++
* ===============
* = lc_attr_set =
* ===============
*
* Overview:
*   Set a management attribute for an ELAN instance.  Refer to lec_mgt.h for
*   a list of attribute IDs.
*
* Arguments:
*   lc_elan_handle    - (IN)  Handle returned by lc_register.
*   attr_id           - (IN)  ID of attribute to be accessed.
*   p_in_buffer       - (IN)  Pointer to the input buffer that contains data
*                             to be written.
*   in_buffer_length  - (IN)  Number of bytes to be written.
*
* Returns:
*   STATUS_K_SUCCESS       - Successful Completion.
*   STATUS_K_ARG_2_BAD     - Invalid Attribute ID.
*   STATUS_K_ARG_5_BAD     - Not enough space in output buffer.
*   STATUS_K_NOT_FOUND     - Instance not found.
*   STATUS_K_NO_PRIVELEGE  - Attempted to write a Read-Only attribute.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS lc_attr_set (HANDLE     lc_elan_handle,
                    UINT16     attr_id,
                    void      *p_in_buffer,
                    UINT16     in_buffer_length);

#endif /* LEC_CTRL_H */
