/*
 * carray.c - encode a file into a C array
 *
 * Copyright (c) 2001 Andreas Ferber <af@devcon.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>

#define BUFSIZE 4096

char *me;			/* argv[0] */

/* program options */
char *output_filename = "-";	/* default: STDOUT */
char *input_filename = "-";	/* default: STDIN */
char *compress_program = NULL;	/* default: no compression */
char *array_name = "data";
char *qualifier = "const";

static void perror_fatal(const char *tag) __attribute__ ((noreturn));
static void usage(void) __attribute__ ((noreturn));
static void version(void) __attribute__ ((noreturn));

void perror_fatal(const char *tag)
{
	fprintf(stderr, "%s: ", me);
	perror(tag);
	exit(1);
}

FILE *run_compression_child(const char *filename)
{
	FILE *tmp;
	char *cmd;

	cmd = malloc(strlen(compress_program)+strlen(" -c ")+strlen(filename)+1);
	if (cmd == NULL) {
		fprintf(stderr, "malloc failed\n");
		exit(1);
	}

	/* malloc() above ensures that there is always enough space */
	strcpy(cmd, compress_program);
	strcat(cmd, " -c ");
	strcat(cmd, filename);

	tmp = popen(cmd, "r");
	if (tmp == NULL) {
		fprintf(stderr, "could not run gzip\n");
		exit(1);
	}

	return tmp;
}

void write_head(FILE * output)
{
	fprintf(output, "/* autogenerated by %s - do not edit */\n\n", me);
	if (strcmp(qualifier, "") != 0)
		fprintf(output, "%s ", qualifier);
	fprintf(output, "char %s[] = {\n", array_name);
}

void header_encode(FILE * input, FILE * output)
{
	size_t lw = 0;

	write_head(output);
	fprintf(output, "    ");

	while (!feof(input)) {
		static unsigned char buf[BUFSIZE];
		size_t r;
		size_t i;

		r = fread(buf, 1, BUFSIZE, input);
		for (i = 0; i < r; i++) {
			fprintf(output, "0x%02x", (int) (buf[i]));
			if (!(feof(input) && i == r - 1)) {
				if (lw++ == 7) {
					fprintf(output, ",\n    ");
					lw = 0;
				} else
					fprintf(output, ", ");
			}
		}
	}

	fprintf(output, "\n};\n");
}

void usage(void)
{
	fprintf(stderr,
		"Usage: carray [OPTION]... [SOURCE]\n"
		"Encodes data into a C array, with optional compression.\n"
		"Data is read from SOURCE or STDIN if SOURCE is omitted.\n"
		"\n"
		"  -a, --array-name=STRING      set the name of the generated array\n"
		"  -q, --qualifier=STRING       type qualifiers for the array\n"
		"  -o, --output=FILE            write output to FILE\n"
		"  -z, --compression=PATH       path to a compression program\n"
		"      --help                   display this help and exit\n"
		"      --version                output version information and exit\n");
	exit(0);
}

void version(void)
{
	fprintf(stderr,
		"carray 0.42\n" "Written by Andreas Ferber.\n");
	exit(0);
}

int main(int argc, char **argv)
{

	FILE *input;
	FILE *output = stdout;

	me = argv[0];

	/* read options */
	while (1) {
		int c;

		static struct option long_options[] = {
			{"array-name", 1, NULL, 'a'},
			{"qualifier", 1, NULL, 'q'},
			{"output", 1, NULL, 'o'},
			{"compression", 1, NULL, 'z'},
			{"help", 0, NULL, 'h'},
			{"version", 0, NULL, 'V'},
			{NULL, 0, NULL, 0}
		};

		c =
		    getopt_long(argc, argv, "a:q:o:z:", long_options,
				NULL);
		if (c == -1)	/* no more options */
			break;

		switch (c) {
		case 'a':
			array_name = optarg;
			break;
		case 'q':
			qualifier = optarg;
			break;
		case 'o':
			output_filename = optarg;
			break;
		case 'z':
			if (strcmp(optarg, "none") == 0)
				compress_program = NULL;
			else
				compress_program = optarg;
			break;
		case 'h':
			usage();
			break;
		case 'V':
			version();
			break;
		default:
			/* error message already printed by getopt_long. */
			fprintf(stderr,
				"Try `%s --help' for more information.\n",
				me);
			exit(1);
		}
	}

	/* get input filename if given */
	if (optind < argc) {
		if (optind < argc - 1) {
			fprintf(stderr,
				"%s: to many arguments\n"
				"Try `%s --help' for more information.\n",
				me, me);
			exit(1);
		}

		input_filename = argv[optind++];
	}

	/* check if input file exists before running gzip
	 * NOTE: this has a race condition, so we have to check gzip exit
	 *       code also. The check here is only for more graceful
	 *       "normal" error handling */
	if (strcmp(input_filename, "-") != 0) {
		struct stat s;
		if (stat(input_filename, &s) == -1)
			perror_fatal(input_filename);

		if (access(input_filename, R_OK) == -1)
			perror_fatal(input_filename);
	}

	/* open output file */
	if (strcmp(output_filename, "-") != 0) {
		output = fopen(output_filename, "w");
		if (output == NULL)
			perror_fatal(output_filename);
	}

	if (compress_program != NULL)
		input = run_compression_child(input_filename);
	else {
		if (strcmp(input_filename, "-") == 0)
			input = stdin;
		else
			input = fopen(input_filename, "r");
	}

	/* encode gzip output into a C header file */
	header_encode(input, output);

	fclose(output);

	if (compress_program != NULL) {
		if (pclose(input) != 0) {
			if (strcmp(output_filename, "-") != 0)
				(void) unlink(output_filename);
			fprintf(stderr, "%s: gzip returned an error\n", me);
			exit(1);
		}
	}
	else {
		fclose(input);
	}

	return 0;
}

/*
 * vi:ts=8 sw=8 noexpandtab
 */
