
/***************************************************************************
*
*   Copyright (c) 1998, 1999 Jeff V. Merkey
*   895 West Center Street
*   Orem, Utah  84057
*   jmerkey@utah-nac.org
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation, version 2, or any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You are free to modify and re-distribute this program in accordance
*   with the terms specified in the GNU Public License.  The copyright
*   contained in this code is required to be present in any derivative
*   works and you are required to provide the source code for this
*   program as part of any commercial or non-commercial distribution.
*   You are required to respect the rights of the Copyright holders
*   named within this code.
*
*   jmerkey@utah-nac.org is the official maintainer of
*   this code.  You are encouraged to report any bugs, problems, fixes,
*   suggestions, and comments about this software to jmerkey@utah-nac.org
*   or linux-kernel@vger.kernel.org.  New releases, patches, bug fixes, and
*   technical documentation can be found at www.kernel.org.  We will
*   periodically post new releases of this software to www.kernel.org
*   that contain bug fixes and enhanced capabilities.
*
*   Original Authorship      :
*      source code written by Jeff V. Merkey
*
*   Original Contributors    :
*      Jeff V. Merkey
*      Darren Major
*      
*
****************************************************************************
*
*
*   AUTHOR   :  Jeff V. Merkey (jmerkey@utah-nac.org)
*   FILE     :  SUPER.C
*   DESCRIP  :  NWFS VFS Super Block Module for Linux*   DATE     :  November 16, 1998
*
*
***************************************************************************/

#define NWFSMOD_VERSION_INFO 1
#include "globals.h"



int nwfs_remount(struct super_block *sb, int *flags, char *data)
{
    return 0;
}

struct super_block *nwfs_read_super(struct super_block *sb, void *data, int silent)
{
    register VOLUME *volume;
    BYTE name[64];
    register BYTE *p, *n;
    register ULONG count, AutoRepair = FALSE;
    kdev_t dev = sb->s_dev;

#if (VERBOSE)
    NWFSPrint("read super\n");
#endif
    
    if (!data)
    {
       NWFSPrint("nwfs:  missing VOLUME_NAME\n");
       NWFSPrint("usage: mount [name] [/mount_point] -t nwfs -o VOLUME_NAME(+)\n");
       NWFSPrint("       where VOLUME_NAME is the name of a NetWare file system Volume\n");
       NWFSPrint("       and '+' enables the driver to auto-repair volume errors during volume mount\n");
       NWFSPrint("       invoke NWVOL for a listing of detected NetWare Volumes in\n");
       NWFSPrint("       this system.\n\n");
       NWFSPrint("       NWFS Utilities for Linux are:\n");
       NWFSPrint("       NWVOL            - List detected NetWare volumes\n");
       NWFSPrint("       NWDISK/NWCONFIG  - File System/Partition Config Utility\n");
       sb->s_dev = 0;
       return NULL;
    }

    NWFSVolumeScan();

    count = 0; 
    p = (BYTE *) data;
    n = &name[0];
    while (*p && (*p != ' ') && (*p != '\r') && (*p != '\n') && (*p != '+') && 
	  (count < 16))
    {
       *n++ = *p++;
       count++;
    }
    *n = '\0';

    if (*p == '+')
    {
        NWFSPrint("nwfs:  volume auto-repair enabled\n");
        AutoRepair = TRUE;
    }
    
    volume = MountHandledVolume(name);
    if (!volume)
    {
       NWFSPrint("nwfs:  error mounting volume %s\n", name);
       sb->s_dev = 0;
       return NULL;
    }

    sb->s_blocksize = LogicalBlockSize;
    switch (sb->s_blocksize)
    {
       case 512:
          sb->s_blocksize_bits = 9;
	  break;
	  
       case 1024:
          sb->s_blocksize_bits = 10;
	  break;
	  
       case 2048:
          sb->s_blocksize_bits = 11;
	  break;
	  
       case 4096:
          sb->s_blocksize_bits = 12;
	  break;
	  
    }
    sb->s_magic = NWFS_LINUX_FS_ID;
    sb->s_dev = dev;
    sb->s_op = &nwfs_sops;
    sb->u.generic_sbp = (void *) volume;

    sb->s_root = d_alloc_root(iget(sb, 0));
    if (!sb->s_root)
    {
       NWFSPrint("nwfs:  get root inode failed\n");
       sb->s_dev = 0;
       return NULL;
    }
    return sb;
}

void nwfs_put_super(struct super_block *sb)
{
    register VOLUME *volume;
    register ULONG retCode;

#if (VERBOSE)
    NWFSPrint("put super\n");
#endif
    
    sb->s_dev = 0;
    volume = (VOLUME *) sb->u.generic_sbp;
    sb->u.generic_sbp = 0;

    retCode = DismountVolumeByHandle(volume);
    if (retCode)
       NWFSPrint("nwfs:  errors dismounting volume %s", volume->VolumeName);

    return;
}

int nwfs_statfs(struct super_block *sb, struct statfs *buf)
{
    register VOLUME *volume = (VOLUME *) sb->u.generic_sbp;
    register ULONG bpb = (IO_BLOCK_SIZE / LogicalBlockSize);

    buf->f_type = NWFS_LINUX_FS_ID;
    buf->f_bsize = LogicalBlockSize;
    buf->f_blocks = volume->VolumeClusters * volume->BlocksPerCluster * bpb;
    buf->f_bfree = volume->VolumeFreeClusters * volume->BlocksPerCluster * bpb;
    buf->f_bavail = volume->VolumeFreeClusters * volume->BlocksPerCluster * bpb;
    buf->f_namelen = 255;

    return 0;
}

