/*
 * dock.c -- docking station/port replicator support
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_dock;

int omnibook_get_dock(void)
{
	u8 dock;
	int retval;
	
	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_CSPR, &dock);
		if (retval)
			return retval;
		retval = (dock & XE3GF_CSPR_MASK) ? 1 : 0;
		break;
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
		retval = omnibook_ec_read(OB500_STA1, &dock);
		if (retval)
			return retval;
		retval = (dock & OB500_DCKS_MASK) ? 1 : 0;
		break;
	default:
		printk(KERN_INFO "%s: Docking station handling is unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV;
	}

	return retval;
}
	
static int omnibook_dock_status(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	char *str;

	str = (omnibook_get_dock()) ? "docked" : "undocked";

	b += sprintf(b, "Laptop is %s\n", str);

	len = b - buffer;
	if (len < off + count)
	*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
	len = count;
	if (len < 0)
	len = 0;
	return len;
}

int __init omnibook_dock_init(void)
{
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
		pmode = S_IFREG | S_IRUGO;
		proc_dock = create_proc_read_entry("dock", pmode, omnibook_proc_root, omnibook_dock_status, NULL);
		break;
	default:
		printk(KERN_INFO "%s: Docking station status monitoring is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	
	if (! proc_dock) {
		printk(KERN_ERR "%s: Unable to create /proc/%s/dock.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	printk(KERN_INFO "%s: Docking station status monitoring is enabled.\n", MODULE_NAME);
	return 0;
}

void __exit omnibook_dock_cleanup(void)
{
	if (proc_dock)
		remove_proc_entry("dock", omnibook_proc_root);
}

EXPORT_SYMBOL(omnibook_get_dock);

/* End of file */
