/*
 * High performance packet classification - optimized btree
 *
 * Authors: Michael Bellion and Thomas Heinz
 * (c) 2002-2003 by the hipac core team <nf@hipac.org>:
 *      +-----------------------+----------------------+
 *      |   Michael Bellion     |     Thomas Heinz     |
 *      | <mbellion@hipac.org>  |  <creatix@hipac.org> |
 *      +-----------------------+----------------------+
 * Licenced under the GNU General Public Licence, version >= 2.
 */


#include "global.h"
#include "btree.h"


static inline __u32
min_a_b(const __u32 a, const __u32 b)
{
	return a < b ? a : b;
}


static inline void 
insert_leaf_8(void *old_leaf, const __u32 old_kpn, const __u32 old_offset,
	      __u32 total_elem,
	      void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	      const __u8 ins_num, void *ins_pos[], const __u32 key[],
	      struct gen_spec *nextseg[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (1 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to insert either 1. or 2. key */
	__u8 insert_key1, insert_key2;
	/* maintains wether or not key1/key2 has been inserted */
	__u8 key_inserted[] = {0, 0};
	/* buckets left in the leaf in which one key will be inserted */
	__u32 ins_free = total_elem + 1;
	/* ins_id = {0 - first key to be inserted,
	             1 - second key to be inserted} */
	__u8 ins_id = 0;
	__u32 len;
	struct gen_spec **newn;	
	
	if (old_kpn == new_kpn && ins_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing 
		   changes here */
		len = (__u8 *) ins_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = ins_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		insert_key1 = (old_leaf >= ins_pos[0] && !key_inserted[0]);
		insert_key2 = (ins_num == 2 && old_leaf >= ins_pos[1] && 
			       key_inserted[0] && !key_inserted[1]);
		
		if (insert_key1 || insert_key2) {
			ins_id = insert_key1 ? 0 : 1;
			for (ins_free = 0; 
			     key[ins_id] > *((__u8 *) old_leaf + ins_free);
			     ins_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or
		   total_elem == 0 or insertion position is reached */
		len = min_a_b(ins_free, 
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));

		if (len > 0) {
			memcpy(new_leaf, old_leaf, len);
			memcpy((__u8 *) new_leaf + new_free + 
			       (new_kpn - new_free) * PTRSIZE,
			       (__u8 *) old_leaf + old_free + 
			       (old_kpn - old_free) * PTRSIZE, len * PTRSIZE);
		}
		
		old_free -= len;
		new_free -= len;
		total_elem -= len;
		ins_free -= len;
		
		old_leaf = (__u8 *) old_leaf + len;
		new_leaf = (__u8 *) new_leaf + len;
		
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u8 *) old_leaf + old_kpn * PTRSIZE + 
				old_offset;
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u8 *) new_leaf + new_kpn * PTRSIZE + 
				new_offset;
			new_free = new_kpn;
		}
		
		if (ins_free == 0) {
			/* insert new key in leaf */
			*(__u8 *) new_leaf = key[ins_id];
			newn = (struct gen_spec **)
				((__u8 *) new_leaf + new_free +
				 (new_kpn - new_free) * PTRSIZE);
			*newn = nextseg[ins_id];
			new_leaf = (__u8 *) new_leaf + 1;
			new_free--;
			ins_free = total_elem + 1;
			key_inserted[ins_id] = 1;
			
			if (new_free == 0) {
				/* goto next leaf in new leaf level */
				new_leaf = (__u8 *) new_leaf + 
					new_kpn * PTRSIZE + new_offset;
				new_free = new_kpn;
			}
		}
	}
}


static inline void 
insert_leaf_16(void *old_leaf, const __u32 old_kpn, const __u32 old_offset, 
	       __u32 total_elem,
	       void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	       const __u8 ins_num, void *ins_pos[], const __u32 key[],
	       struct gen_spec *nextseg[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (2 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to insert either 1. or 2. key */
	__u8 insert_key1, insert_key2;
	/* maintains wether or not key1/key2 has been inserted */
	__u8 key_inserted[] = {0, 0};
	/* buckets left in the leaf in which one key will be inserted */
	__u32 ins_free = total_elem + 1;
	/* ins_id = {0 - first key to be inserted,
	             1 - second key to be inserted} */
	__u8 ins_id = 0;
	__u32 len;
	struct gen_spec **newn;
	
	
	if (old_kpn == new_kpn && ins_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing 
		   changes here */
		len = (__u8 *) ins_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = ins_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		insert_key1 = (old_leaf >= ins_pos[0] && !key_inserted[0]);
		insert_key2 = (ins_num == 2 && old_leaf >= ins_pos[1] && 
			       key_inserted[0] && !key_inserted[1]);
		
		if (insert_key1 || insert_key2) {
			ins_id = insert_key1 ? 0 : 1;
			for (ins_free = 0;
			     key[ins_id] > *((__u16 *) old_leaf + ins_free);
			     ins_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or 
		   total_elem == 0 or insertion position is reached */
		len = min_a_b(ins_free, 
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));
		
		if (len > 0) {
			memcpy(new_leaf, old_leaf, 2 * len);
			memcpy((__u16 *) new_leaf + new_free + 
			       (new_kpn - new_free) * (PTRSIZE >> 1),
			       (__u16 *) old_leaf + old_free + 
			       (old_kpn - old_free) * (PTRSIZE >> 1), 
			       len * PTRSIZE);
		}

		old_free -= len;
		new_free -= len;
		total_elem -= len;
		ins_free -= len;
		
		old_leaf = (__u16 *) old_leaf + len;
		new_leaf = (__u16 *) new_leaf + len;
	
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u16 *) ((__u8 *) old_leaf + 
					      old_kpn * PTRSIZE + old_offset);
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u16 *) ((__u8 *) new_leaf + 
					      new_kpn * PTRSIZE + new_offset);
			new_free = new_kpn;
		}
		
		if (ins_free == 0) {
			/* insert new key in leaf */
			*(__u16 *) new_leaf = key[ins_id];
			newn = (struct gen_spec **) 
				((__u16 *) new_leaf + new_free + 
				 (new_kpn - new_free) * (PTRSIZE >> 1));
			*newn = nextseg[ins_id];
			new_leaf = (__u16 *) new_leaf + 1;
			new_free--;
			ins_free = total_elem + 1;
			key_inserted[ins_id] = 1;
			
			if (new_free == 0) {
				/* goto next leaf in new leaf level */
				new_leaf = (__u16 *) 
					((__u8 *) new_leaf 
					 + new_kpn * PTRSIZE + new_offset);
				new_free = new_kpn;
			}
		}
	}
}


static inline void 
insert_leaf_32(void *old_leaf, const __u32 old_kpn, const __u32 old_offset, 
	       __u32 total_elem,
	       void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	       const __u8 ins_num, void *ins_pos[], const __u32 key[],
	       struct gen_spec *nextseg[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (4 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to insert either 1. or 2. key */
	__u8 insert_key1, insert_key2;
	/* maintains wether or not key1/key2 has been inserted */
	__u8 key_inserted[] = {0, 0};
	/* buckets left in the leaf in which one key will be inserted */
	__u32 ins_free = total_elem + 1;
	/* ins_id = {0 - first key to be inserted,
	             1 - second key to be inserted} */
	__u8 ins_id = 0;
	__u32 len;
	struct gen_spec **newn;
	
	
	if (old_kpn == new_kpn && ins_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing 
		   changes here */
		len = (__u8 *) ins_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = ins_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		insert_key1 = (old_leaf >= ins_pos[0] && !key_inserted[0]);
		insert_key2 = (ins_num == 2 && old_leaf >= ins_pos[1] && 
			       key_inserted[0] && !key_inserted[1]);
		
		if (insert_key1 || insert_key2) {
			ins_id = insert_key1 ? 0 : 1;
			for (ins_free = 0;
			     key[ins_id] > *((__u32 *) old_leaf + ins_free);
			     ins_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or 
		   total_elem == 0 or insertion position is reached */
		len = min_a_b(ins_free, 
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));
		
		if (len > 0) {
			memcpy(new_leaf, old_leaf, 4 * len);
			memcpy((__u32 *) new_leaf + new_free +
			       (new_kpn - new_free) * (PTRSIZE >> 2),
			       (__u32 *) old_leaf + old_free +
			       (old_kpn - old_free) * (PTRSIZE >> 2),
			       len * PTRSIZE);
		}
		
		old_free -= len;
		new_free -= len;
		total_elem -= len;
		ins_free -= len;
		
		old_leaf = (__u32 *) old_leaf + len;
		new_leaf = (__u32 *) new_leaf + len;
		
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u32 *) ((__u8 *) old_leaf +
					    old_kpn * PTRSIZE + old_offset);
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u32 *) ((__u8 *) new_leaf +
					    new_kpn * PTRSIZE + new_offset);
			new_free = new_kpn;
		}
		
		if (ins_free == 0) {
			/* insert new key in leaf */
			*(__u32 *) new_leaf = key[ins_id];
			newn = (struct gen_spec **)
				((__u32 *) new_leaf + new_free + 
				 (new_kpn - new_free) * (PTRSIZE >> 2));
			*newn = nextseg[ins_id];
			new_leaf = (__u32 *) new_leaf + 1;
			new_free--;
			ins_free = total_elem + 1;
			key_inserted[ins_id] = 1;
			
			if (new_free == 0) {
				/* goto next leaf in new leaf level */
				new_leaf = (__u32 *) 
					((__u8 *) new_leaf + 
					 new_kpn * PTRSIZE + new_offset);
		new_free = new_kpn;
	    }
	}
    }
}


static inline void 
delete_leaf_8(void *old_leaf, const __u32 old_kpn, const __u32 old_offset,
	      __u32 total_elem,
	      void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	      const __u8 del_num, void *del_pos[], const __u32 key[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (1 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to delete either 1. or 2. key */
	__u8 delete_key1, delete_key2;
	/* maintains wether or not key1/key2 has been deleted */
	__u8 key_deleted[] = {0, 0};
	/* buckets left in the leaf in which one key will be deleted */
	__u32 del_free = total_elem + 1;
	/* del_id = {0 - first key to be deleted,
	             1 - second key to be deleted} */
	__u8 del_id = 0;
	__u32 len;

	
	if (old_kpn == new_kpn && del_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing
		   changes here */
		len = (__u8 *) del_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = del_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		delete_key1 = (old_leaf >= del_pos[0] && !key_deleted[0]);
		delete_key2 = (del_num == 2 && old_leaf >= del_pos[1] && 
			       key_deleted[0] && !key_deleted[1]);
		
		if (delete_key1 || delete_key2) {
			del_id = delete_key1 ? 0 : 1;
			for (del_free = 0; 
			     key[del_id] > *((__u8 *) old_leaf + del_free);
			     del_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or 
		   total_elem == 0 or deletion position is reached */
		len = min_a_b(del_free, 
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));
		
		if (len > 0) {
			memcpy(new_leaf, old_leaf, len);
			memcpy((__u8 *) new_leaf + new_free +
			       (new_kpn - new_free) * PTRSIZE,
			       (__u8 *) old_leaf + old_free +
			       (old_kpn - old_free) * PTRSIZE,
			       len * PTRSIZE);
		}			

		old_free -= len;
		new_free -= len;
		total_elem -= len;
		del_free -= len;
		
		old_leaf = (__u8 *) old_leaf + len;
		new_leaf = (__u8 *) new_leaf + len;
		
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u8 *) old_leaf + 
				old_kpn * PTRSIZE + old_offset;
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u8 *) new_leaf + 
				new_kpn * PTRSIZE + new_offset;
			new_free = new_kpn;
		}
		
		if (del_free == 0) {
			/* delete new key in leaf */
			old_leaf = (__u8 *) old_leaf + 1;
			old_free--;
			total_elem--;
			del_free = total_elem + 1;
			key_deleted[del_id] = 1;
			
			if (old_free == 0) {
				/* goto next leaf in new leaf level */
				old_leaf = (__u8 *) old_leaf + 
					old_kpn * PTRSIZE + old_offset;
				old_free = old_kpn;
			}
		}
	}
}


static inline void 
delete_leaf_16(void *old_leaf, const __u32 old_kpn, const __u32 old_offset, 
	       __u32 total_elem,
	       void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	       const __u8 del_num, void *del_pos[], const __u32 key[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (2 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to delete either 1. or 2. key */
	__u8 delete_key1, delete_key2;
	/* maintains wether or not key1/key2 has been deleted */
	__u8 key_deleted[] = {0, 0};
	/* buckets left in the leaf in which one key will be deleted */
	__u32 del_free = total_elem + 1;
	/* del_id = {0 - first key to be deleted,
	             1 - second key to be deleted} */
	__u8 del_id = 0;
	__u32 len;


	if (old_kpn == new_kpn && del_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing 
		   changes here */
		len = (__u8 *) del_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = del_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		delete_key1 = (old_leaf >= del_pos[0] && !key_deleted[0]);
		delete_key2 = (del_num == 2 && old_leaf >= del_pos[1] && 
			       key_deleted[0] && !key_deleted[1]);
		
		if (delete_key1 || delete_key2) {
			del_id = delete_key1 ? 0 : 1;
			for (del_free = 0;
			     key[del_id] > *((__u16 *) old_leaf + del_free);
			     del_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or 
		   total_elem == 0 or deletion position is reached */
		len = min_a_b(del_free, 
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));
		
		if (len > 0) {
			memcpy(new_leaf, old_leaf, 2 * len);
			memcpy((__u16 *) new_leaf + new_free + 
			       (new_kpn - new_free) * (PTRSIZE >> 1),
			       (__u16 *) old_leaf + old_free +
			       (old_kpn - old_free) * (PTRSIZE >> 1),
			       len * PTRSIZE);
		}

		old_free -= len;
		new_free -= len;
		total_elem -= len;
		del_free -= len;
		
		old_leaf = (__u16 *) old_leaf + len;
		new_leaf = (__u16 *) new_leaf + len;
		
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u16 *) ((__u8 *) old_leaf + 
					    old_kpn * PTRSIZE + old_offset);
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u16 *) ((__u8 *) new_leaf +
					    new_kpn * PTRSIZE + new_offset);
			new_free = new_kpn;
		}
		
		if (del_free == 0) {
			/* delete new key in leaf */
			old_leaf = (__u16 *) old_leaf + 1;
			old_free--;
			total_elem--;
			del_free = total_elem + 1;
			key_deleted[del_id] = 1;
			
			if (old_free == 0) {
				/* goto next leaf in new leaf level */
				old_leaf = (__u16 *) 
					((__u8 *) old_leaf + 
					 old_kpn * PTRSIZE +
					 old_offset);
				old_free = old_kpn;
			}
		}
	}
}


static inline void 
delete_leaf_32(void *old_leaf, const __u32 old_kpn, const __u32 old_offset, 
	       __u32 total_elem,
	       void *new_leaf, const __u32 new_kpn, const __u32 new_offset,
	       const __u8 del_num, void *del_pos[], const __u32 key[])
{
	/* size of one node in bytes */
	const __u32 nodesize = old_kpn * (4 + PTRSIZE) + old_offset;
	/* buckets left in current old_leaf */
	__u32 old_free = old_kpn;
	/* buckets left in current new_leaf */
	__u32 new_free = new_kpn;
	/* conditions to delete either 1. or 2. key */
	__u8 delete_key1, delete_key2;
	/* maintains wether or not key1/key2 has been deleted */
	__u8 key_deleted[] = {0, 0};
	/* buckets left in the leaf in which one key will be deleted */
	__u32 del_free = total_elem + 1;
	/* del_id = {0 - first key to be deleted,
	             1 - second key to be deleted} */
	__u8 del_id = 0;
	__u32 len;


	if (old_kpn == new_kpn && del_pos[0] > old_leaf) {
		/* first part of the leaf can be copied since nothing 
		   changes here */
		len = (__u8 *) del_pos[0] - (__u8 *) old_leaf;
		memcpy(new_leaf, old_leaf, len);
		old_leaf = del_pos[0];
		new_leaf = (__u8 *) new_leaf + len;
		total_elem -= old_kpn * len / nodesize;
	}
	
	while (total_elem > 0) {
		delete_key1 = (old_leaf >= del_pos[0] && !key_deleted[0]);
		delete_key2 = (del_num == 2 && old_leaf >= del_pos[1] && 
			       key_deleted[0] && !key_deleted[1]);
		
		if (delete_key1 || delete_key2) {
			del_id = delete_key1 ? 0 : 1;
			for (del_free = 0;
			     key[del_id] > *((__u32 *) old_leaf + del_free);
			     del_free++);
		}
		
		/* copy elements until either old_leaf or new_leaf is full or 
		   total_elem == 0 or deletion position is reached */
		len = min_a_b(del_free,
			      min_a_b(total_elem, 
				      min_a_b(old_free, new_free)));
		
		if (len > 0) {
			memcpy(new_leaf, old_leaf, 4 * len);
			memcpy((__u32 *) new_leaf + new_free + 
			       (new_kpn - new_free) * (PTRSIZE >> 2),
			       (__u32 *) old_leaf + old_free + 
			       (old_kpn - old_free) * (PTRSIZE >> 2),
			       len * PTRSIZE);
		}
			
		old_free -= len;
		new_free -= len;
		total_elem -= len;
		del_free -= len;
		
		old_leaf = (__u32 *) old_leaf + len;
		new_leaf = (__u32 *) new_leaf + len;
		
		if (old_free == 0) {
			/* goto next leaf in old leaf level */
			old_leaf = (__u32 *) ((__u8 *) old_leaf + 
					    old_kpn * PTRSIZE + old_offset);
			old_free = old_kpn;
		} 
		
		if (new_free == 0) {
			/* goto next leaf in new leaf level */
			new_leaf = (__u32 *) ((__u8 *) new_leaf + 
					    new_kpn * PTRSIZE + new_offset);
			new_free = new_kpn;
		}
		
		if (del_free == 0) {
			/* delete new key in leaf */
			old_leaf = (__u32 *) old_leaf + 1;
			old_free--;
			total_elem--;
			del_free = total_elem + 1;
			key_deleted[del_id] = 1;
			
			if (old_free == 0) {
				/* goto next leaf in new leaf level */
				old_leaf = (__u32 *) 
					((__u8 *) old_leaf + 
					 old_kpn * PTRSIZE +
					 old_offset);
				old_free = old_kpn;
			}
		}
	}
}


struct ptrblock **
termrule(const struct btree_spec *spec)
{
	if (unlikely(!spec)){
		ARG_MSG;
		return NULL;
	}

	if (spec->btreetype == BT0){
		__u8 wc = (spec->bittype == BIT_U32 ? PTRSIZE : 0);
		//FIXME_ALIGN
		return (struct ptrblock **)
			((__u8*)spec + sizeof(__u32) + wc + 
			 spec->num * (PTRSIZE + (1 << spec->bittype)));
	}
	return (struct ptrblock **)((__u8*)spec + spec->blocks * BLOCKSIZE 
			      - PTRSIZE);
}


hipac_error
btree_clone(const struct btree_spec *spec, struct btree_spec **clone)
{
	size_t size;
	hipac_error error;
		
	if (unlikely(!spec || !clone))
		ARG_ERR;

	size = btree_size(spec);

	if (!(*clone = hp_alloc(size, 1)))
		return HE_LOW_MEMORY;

	memcpy(*clone, spec, size);
	(*clone)->newspec = 0;

	if ((error = ptrblock_clone(*termrule(spec), termrule(*clone)))) {
		hp_free(*clone);
		return HE_LOW_MEMORY;
	}
	return HE_OK;
}


hipac_error
btree_locate(const struct btree_spec *spec, struct locate_inf *inf, __u32 key)
{
	__u32* start = ROOT(spec);
	
	if (spec == NULL || inf == NULL) {
		ARG_ERR;
	}
	
       	if (spec->btreetype == BT0){
		if (spec->bittype == BIT_U32){
			__u32* keyptr = findkey_32(key, start);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(start + spec->num +//FIXME_ALIGN
				 (PTRSIZE >> 2) * (keyptr - start));
		} 
		else if (spec->bittype == BIT_U16){
			__u16* keyptr = findkey_16(key, (__u16*)start);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				((__u16*)start + spec->num +//FIXME_ALIGN
				 (PTRSIZE >> 1) * (keyptr - (__u16*)start));
		} else {
			__u8* keyptr = findkey_8(key, (__u8*)start);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				((__u8*)start + spec->num +//FIXME_ALIGN
				 PTRSIZE  * (keyptr - (__u8*)start));
		}
	} 
	else if (spec->btreetype == BT1){
		if (spec->bittype == BIT_U32){
			__u32* leaf = next_node1_32(key, spec, start,
						  BT1_U32_LEAF_BLOCKS);
			__u32* keyptr = findkey_32(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT1_U32_LEAF +
				 (PTRSIZE >> 2) * (keyptr - leaf));
		}
		else if (spec->bittype == BIT_U16){
			__u16* leaf = next_node1_16(key, spec, (__u16*)start,
						  BT1_U16_LEAF_BLOCKS);
			__u16* keyptr = findkey_16(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT1_U16_LEAF +
				 (PTRSIZE >> 1) * (keyptr - leaf));
		} else {
			__u8* leaf = next_node1_8(key, spec, (__u8*)start,
						BT1_U8_LEAF_BLOCKS);
			__u8* keyptr = findkey_8(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT1_U8_LEAF +
				 PTRSIZE  * (keyptr - leaf));
		}
	}
	else if (spec->btreetype == BT2){
		if (spec->bittype == BIT_U32){
			__u32 root_num = DIV(spec->num, 
					   (BT2_U32_NODE * BT2_U32_LEAF));
			__u32* leaf = next_node2_32(key, LEAF1(spec), root_num,
						  next_node1_32(key, spec, 
								(__u32*)start,
							  BT2_U32_NODE_BLOCKS),
						  BT2_U32_NODE_BLOCKS,
						  BT2_U32_LEAF_BLOCKS);
			__u32* keyptr = findkey_32(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT2_U32_LEAF +
				 (PTRSIZE >> 2) * (keyptr - leaf));
		}
		else if (spec->bittype == BIT_U16){
			__u16* leaf = next_node1_16(key, spec, (__u16*)start, 
						   BT2_U16_LEAF_BLOCKS);
			__u16* keyptr = findkey_16(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT2_U16_LEAF +
				 (PTRSIZE >> 1) * (keyptr - leaf));
		}
	}
	else {
		if (spec->bittype == BIT_U32){
			__u32 root_num = DIV(spec->num, 
					   (BT3_U32_NODE * BT3_U32_LEAF));
			__u32* leaf = next_node2_32(key, LEAF1(spec), root_num,
						  next_node1_32(key, spec,
								(__u32*)start, 
							 BT3_U32_NODE_BLOCKS),
						  BT3_U32_NODE_BLOCKS,
						  BT3_U32_LEAF_BLOCKS);
			__u32* keyptr = findkey_32(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT3_U32_LEAF +
				 (PTRSIZE >> 2) * (keyptr - leaf));
		}
		else if (spec->bittype == BIT_U16){
			__u32 root_num = DIV(spec->num, 
					   (BT3_U16_NODE * BT3_U16_LEAF));
			__u16* leaf = next_node2_16(key, LEAF1(spec), root_num,
						  next_node1_16(key, spec,
								(__u16*)start,
							 BT3_U16_NODE_BLOCKS),
						  BT3_U16_NODE_BLOCKS,
						  BT3_U16_LEAF_BLOCKS);
			__u16* keyptr = findkey_16(key, leaf);
			inf->key = *keyptr;
			inf->nextspec = (struct gen_spec**)
				(leaf + BT3_U16_LEAF +
				 (PTRSIZE >> 1) * (keyptr - leaf));
		}
	}
	return HE_OK;
}


struct btree_spec *
new_segmentset_intern(__u32 num, __u8 bittype)
{
	struct btree_spec *result;
	__u8 btreetype, blocks, root_num;
	__u32 size;

	if (!(num < 242688))
		return NULL;
	
	if (bittype == BIT_U8){
		if (num <= BT0_U8_MAX){
			btreetype = BT0;
			blocks = 1;
			size = sizeof(__u32) + PTRSIZE 
				+ num * (sizeof(__u8) + PTRSIZE);//FIXME_ALIGN
			result = hp_alloc(size, 1);
		}
		else {
			btreetype = BT1;
			root_num = DIV(num, BT1_U8_LEAF);
			blocks = 1;
			size = (blocks + root_num * BT1_U8_LEAF_BLOCKS)
				* BLOCKSIZE
				- (root_num * BT1_U8_LEAF - num) * PTRSIZE
				- OFFSET(BT1_U8_LEAF, BIT_U8);
			result = hp_alloc(size, 1);
		}
	}
	else if (bittype == BIT_U16){
		if (num <= BT0_U16_MAX){
			btreetype = BT0;
			blocks = 0;
			size = sizeof(__u32) + PTRSIZE
				+ num * (sizeof(__u16) + PTRSIZE);//FIXME_ALIGN
			result = hp_alloc(size, 1);
		}
		else if (num <= BT1_U16_MAX){
			btreetype = BT1;
			root_num = DIV(num, BT1_U16_LEAF);
			blocks = 1;
			size = (blocks + root_num * BT1_U16_LEAF_BLOCKS)
				* BLOCKSIZE
				- (root_num * BT1_U16_LEAF - num) * PTRSIZE
				- OFFSET(BT1_U16_LEAF, BIT_U16);
			result = hp_alloc(size, 1);
		}
		else if (num <= BT2_U16_MAX){
			btreetype = BT2;
			root_num = DIV(num, BT2_U16_LEAF);
			blocks = BLOCKS_U16(root_num);
			size = (blocks + root_num * BT2_U16_LEAF_BLOCKS)
				* BLOCKSIZE
				- (root_num * BT2_U16_LEAF - num) * PTRSIZE;
			result = hp_alloc(size, 1);
		}
		else {
			btreetype = BT3;
			root_num = DIV(num, (BT3_U16_NODE * BT3_U16_LEAF));
			blocks = BLOCKS_U16(root_num);
			size = (blocks + root_num * BT3_U16_NODE_BLOCKS)
				* BLOCKSIZE + num * (2 + PTRSIZE) + 
				((BT3_U16_LEAF - (num % BT3_U16_LEAF)) 
				 % BT3_U16_LEAF) * 2;
			result = hp_alloc(size, 1);
		}
	}
	else {
		if (num <= BT0_U32_MAX){
			btreetype = BT0;
			blocks = 0;
			size = sizeof(__u32) + 2 * PTRSIZE
				+ num * (sizeof(__u32) + PTRSIZE);//FIXME_ALIGN
			result = hp_alloc(size, 1);
		}
		else if (num <= BT1_U32_MAX){
			btreetype = BT1;
			root_num = DIV(num, BT1_U32_LEAF);
			blocks =  BLOCKS_U32(root_num);
			size = (blocks + root_num * BT1_U32_LEAF_BLOCKS)
				* BLOCKSIZE 
				- (root_num * BT1_U32_LEAF - num) * PTRSIZE;
			result = hp_alloc(size, 1);
		}
		else if (num <= BT2_U32_MAX){
			btreetype = BT2;
			root_num = DIV(num, (BT2_U32_NODE * BT2_U32_LEAF));
			blocks = BLOCKS_U32(root_num);
			size = (blocks + root_num * BT2_U32_NODE_BLOCKS)
				* BLOCKSIZE + num * (4 + PTRSIZE) +
				((BT2_U32_LEAF - (num % BT2_U32_LEAF))
				 % BT2_U32_LEAF) * 4;
			result = hp_alloc(size, 1);
		}
		else {
			btreetype = BT3;
			root_num = DIV(num, (BT3_U32_NODE * BT3_U32_LEAF));
			blocks = BLOCKS_U32(root_num);
			size = (blocks + root_num * BT3_U32_NODE_BLOCKS)
				* BLOCKSIZE + num * (4 + PTRSIZE) +
				((BT3_U32_LEAF - (num % BT3_U32_LEAF))
				 % BT3_U32_LEAF) * 4;
			result = hp_alloc(size, 1);
		}
	}
	if (!(result))
		return NULL;
	
	result->bittype = bittype;
	result->btreetype = btreetype;
	result->blocks = blocks;
	result->dimid = 0;
	result->newspec = 0;
	result->num = num;
	//assert(size == btree_size(result));
	return result;
}


__u32 
btree_size(const struct btree_spec *spec)
{
	__u8 root_num;
	
	if (spec->num == 0){
		return (sizeof(__u32) + PTRSIZE);
	}

	
	if (spec->bittype == BIT_U8){
		if (spec->btreetype == BT0){
			return (sizeof(__u32) + PTRSIZE + 
				spec->num * (sizeof(__u8) + PTRSIZE));//FIXME_ALIGN
		}
		else {
			root_num = DIV(spec->num, BT1_U8_LEAF);
			return ((spec->blocks + 
				 root_num * BT1_U8_LEAF_BLOCKS) * BLOCKSIZE -
				(root_num * BT1_U8_LEAF - spec->num) * PTRSIZE
				- OFFSET(BT1_U8_LEAF, BIT_U8));
		}
	}
	else if (spec->bittype == BIT_U16){
		if (spec->btreetype == BT0){
			return (sizeof(__u32) + PTRSIZE +
				spec->num * (sizeof(__u16) + PTRSIZE));//FIXME_ALIGN
		}
		else if (spec->btreetype == BT1){
			root_num = DIV(spec->num, BT1_U16_LEAF);
			return ((spec->blocks + 
				 root_num * BT1_U16_LEAF_BLOCKS) * BLOCKSIZE -
				(root_num * BT1_U16_LEAF - spec->num) * PTRSIZE
				- OFFSET(BT1_U16_LEAF, BIT_U16));
		}
		else if (spec->btreetype == BT2){
			root_num = DIV(spec->num, BT2_U16_LEAF);
			return ((spec->blocks +
				 root_num * BT2_U16_LEAF_BLOCKS) * BLOCKSIZE -
				(root_num * BT2_U16_LEAF - spec->num) 
				* PTRSIZE);
		}
		else { 
			__u8 blocks;
			root_num = DIV(spec->num, 
				       (BT3_U16_NODE * BT3_U16_LEAF));
			blocks = BLOCKS_U16(root_num);
			return  ((blocks +  
				  root_num * BT3_U16_NODE_BLOCKS) * BLOCKSIZE +
				 spec->num * (2 + PTRSIZE) +  
				 ((BT3_U16_LEAF - (spec->num % BT3_U16_LEAF))
				  % BT3_U16_LEAF) * 2);
		}
	}
	else {
		if (spec->btreetype == BT0){
			return (sizeof(__u32) + 2 * PTRSIZE + 
				spec->num * (sizeof(__u32) + PTRSIZE));//FIXME_ALIGN
		}
		else if (spec->btreetype == BT1){
			root_num = DIV(spec->num, BT1_U32_LEAF);
			return ((spec->blocks + 
				 root_num * BT1_U32_LEAF_BLOCKS) * BLOCKSIZE -
				(root_num * BT1_U32_LEAF - spec->num) 
				* PTRSIZE);
		}
		else if (spec->btreetype == BT2){
			__u8 blocks;
			root_num = DIV(spec->num, 
				       (BT2_U32_NODE * BT2_U32_LEAF));
			blocks = BLOCKS_U32(root_num);
			return ((blocks + 
				 root_num * BT2_U32_NODE_BLOCKS) * BLOCKSIZE +
				spec->num * (4 + PTRSIZE) +
				((BT2_U32_LEAF - ( spec->num % BT2_U32_LEAF))
				 % BT2_U32_LEAF) * 4);
		}
		else {
			__u8 blocks;
			root_num = DIV(spec->num, 
				       (BT3_U32_NODE * BT3_U32_LEAF));
			blocks = BLOCKS_U32(root_num);
			return ((blocks + 
				 root_num * BT3_U32_NODE_BLOCKS) * BLOCKSIZE +
				spec->num * (4 + PTRSIZE) +
				((BT3_U32_LEAF - ( spec->num % BT3_U32_LEAF))
				 % BT3_U32_LEAF) * 4);
		}
	}
}


struct btree_spec *
btree_new(__u8 bittype, __u8 dimid, __u8 ins_num, const __u32 key[],
	  struct gen_spec *nextspec[])
{
	struct btree_spec *result;
	void* start;
	
	assert((ins_num > 0) && (ins_num) <= 2);

	result = new_segmentset_intern(ins_num, bittype);
	if (!result)
		return NULL;

	*termrule(result) = NULL;
	result->dimid = dimid;
	start = ROOT(result);
	
	if (bittype == BIT_U8){
		*(__u8*)start = key[0];
		*(void**)((__u8*)start + ins_num) = (void*)nextspec[0];//FIXME_ALIGN
		if (ins_num == 2){
			*((__u8*)start + 1) = key[1];
			*(void**)((__u8*)start + 2 + PTRSIZE) =
				(void*)nextspec[1];
		}	
	}
	else if (bittype == BIT_U16){
		*(__u16*)start = key[0];
		*(void**)((__u16*)start + ins_num) = (void*)nextspec[0];//FIXME_ALIGN
		if (ins_num == 2){
			*((__u16*)start + 1) = key[1];
			*(void**)((__u8*)start + 4 + PTRSIZE) =
				(void*)nextspec[1];
		}
	}
	else {
		*WILDCARD(result) = NULL;
		*(__u32*)start = key[0];
		*(void**)((__u32*)start + ins_num) = (void*)nextspec[0];//FIXME_ALIGN
		if (ins_num == 2){
			*((__u32*)start + 1) = key[1];
			*(void**)((__u8*)start + 8 + PTRSIZE) =
				(void*)nextspec[1];
		}
	}
	return result;
}


static inline void 
build_8(__u8* start, const __u32 num, const __u8* leaf_start,
	const __u8 leaf_size, const __u8 leaf_blocks)
{
	int i;
	for(i = 0; i < num - 1; i++){
		*(start + i) = *(leaf_start + 
				 i * leaf_blocks * BLOCKSIZE + leaf_size - 1);
	}
	*(start + i) = 255;
}


static inline void 
build_16(__u16* start, const __u32 num, const __u8* leaf_start,
	 const __u8 leaf_size, const __u8 leaf_blocks)
{
	int i;
	for(i = 0; i < num - 1; i++){
		*(start + i) = *((__u16*)(leaf_start + 
					i * leaf_blocks * BLOCKSIZE) +
				 leaf_size - 1);
	}
	*(start + i) = 65535;    
}


static inline void 
build_32(__u32* start, const __u32 num, const __u8* leaf_start,
	 const __u8 leaf_size, const __u8 leaf_blocks)
{
	int i;
	for(i = 0; i < num - 1; i++){
		*(start + i) = *((__u32*)(leaf_start + 
					i * leaf_blocks * BLOCKSIZE) +
				 leaf_size - 1);
	}
	*(start + i) = 4294967295U;
}


hipac_error
btree_insert(const struct btree_spec *spec, __u8 ins_num, const __u32 key[],
	     struct gen_spec *nextspec[], struct btree_spec **res)
{
	void* ins_pos[2];
	hipac_error error;
	struct btree_spec *result;

	if (unlikely(!spec || !key || !nextspec || !res ||
		     !(ins_num == 1 || ins_num == 2) ||
		     (ins_num == 2 && key[0] >= key[1])))
		ARG_ERR;
	
	result = new_segmentset_intern(spec->num + ins_num, spec->bittype);
	if (!(result))
		return HE_LOW_MEMORY;
	
	if ((error = ptrblock_clone(*termrule(spec), termrule(result)))){
		hp_free(result);
		return HE_LOW_MEMORY;
	}

	result->dimid = spec->dimid;
	
	if (spec->bittype == BIT_U8){
		if (spec->btreetype == BT0){
			ins_pos[0] = ROOT(spec);
			if (ins_num == 2) {
				ins_pos[1] = ROOT(spec);
			}
			if (result->btreetype == BT0){
				insert_leaf_8(ROOT(spec), spec->num, 0,
					      spec->num,
					      ROOT(result), result->num, 0,
					      ins_num, ins_pos, key, nextspec);
			}
			else {
				insert_leaf_8(ROOT(spec), spec->num, 0,//FIXME_ALIGN
					      spec->num,
					      LEAF1(result), BT1_U8_LEAF, 
					      OFFSET(BT1_U8_LEAF, BIT_U8),
					      ins_num, ins_pos, key, nextspec);
				build_8((__u8*)ROOT(result),
					DIV(result->num, BT1_U8_LEAF),
					LEAF1(result), BT1_U8_LEAF, 
					BT1_U8_LEAF_BLOCKS);
			}
		}
		else {
			ins_pos[0] = next_node1_8(key[0], spec, 
						  (__u8*)ROOT(spec),
						  BT1_U8_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node1_8(key[1], spec,
							  (__u8*)ROOT(spec),
							  BT1_U8_LEAF_BLOCKS);
			}
			insert_leaf_8(LEAF1(spec), BT1_U8_LEAF,
				      OFFSET(BT1_U8_LEAF, BIT_U8), spec->num,
				      LEAF1(result), BT1_U8_LEAF, 
				      OFFSET(BT1_U8_LEAF, BIT_U8),
				      ins_num, ins_pos, key, nextspec);
			build_8((__u8*)ROOT(result), 
				DIV(result->num, BT1_U8_LEAF),
				LEAF1(result), BT1_U8_LEAF, 
				BT1_U8_LEAF_BLOCKS);
		}
	}
	else if (spec->bittype == BIT_U16){
		if (spec->btreetype == BT0){
			ins_pos[0] = ROOT(spec);
			if (ins_num == 2) {
				ins_pos[1] = ROOT(spec);
			}
			if (result->btreetype == BT0){
				insert_leaf_16(ROOT(spec), spec->num, 0,//FIXME_ALIGN
					       spec->num,
					       ROOT(result), result->num, 0,
					       ins_num, ins_pos, 
					       key, nextspec);
			}
			else {
				insert_leaf_16(ROOT(spec), spec->num, 0,
					       spec->num,
					       LEAF1(result), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       ins_num, ins_pos,
					       key, nextspec);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT1_U16_LEAF),
					 LEAF1(result), BT1_U16_LEAF,
					 BT1_U16_LEAF_BLOCKS);
			}
		}
		else if (spec->btreetype == BT1){
			ins_pos[0] = next_node1_16(key[0], spec, 
						   (__u16*)ROOT(spec),
						   BT1_U16_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node1_16(key[1], spec,
							  (__u16*)ROOT(spec),
							  BT1_U16_LEAF_BLOCKS);
			}
			if (result->btreetype == BT1){
				insert_leaf_16(LEAF1(spec), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       ins_num, ins_pos,
					       key, nextspec);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT1_U16_LEAF),
					 LEAF1(result), BT1_U16_LEAF,
					 BT1_U16_LEAF_BLOCKS);
			}
			else {
				insert_leaf_16(LEAF1(spec), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       ins_num, ins_pos,
					       key, nextspec);
				build_16((__u16*)ROOT(result), 
					 DIV(result->num, BT2_U16_LEAF),
					 LEAF1(result), BT2_U16_LEAF,
					 BT2_U16_LEAF_BLOCKS);
			}
		}
		else if (spec->btreetype == BT2){
			ins_pos[0] = next_node1_16(key[0], spec, 
						   (__u16*)ROOT(spec),
						   BT2_U16_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node1_16(key[1], spec,
							  (__u16*)ROOT(spec),
							  BT2_U16_LEAF_BLOCKS);
			}
			if (result->btreetype == BT2){
				insert_leaf_16(LEAF1(spec), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16), 
					       spec->num,
					       LEAF1(result), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       ins_num, ins_pos,
					       key, nextspec);
				build_16((__u16*)ROOT(result), 
					 DIV(result->num, BT2_U16_LEAF),
					 LEAF1(result), BT2_U16_LEAF,
					 BT2_U16_LEAF_BLOCKS);
			}
			else {
				insert_leaf_16(LEAF1(spec), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF2(result, BT3_U16_NODE, 
						     BT3_U16_NODE_BLOCKS, 
						     BT3_U16_LEAF), 
					       BT3_U16_LEAF, 
					       OFFSET(BT3_U16_LEAF, BIT_U16),
					       ins_num, ins_pos,
					       key, nextspec);
				build_16((__u16*)LEAF1(result), 
					 DIV(result->num, BT3_U16_LEAF),
					 LEAF2(result, BT3_U16_NODE, 
					       BT3_U16_NODE_BLOCKS, 
					       BT3_U16_LEAF),
					 BT3_U16_LEAF, BT3_U16_LEAF_BLOCKS);
				build_16((__u16*)ROOT(result), 
					 DIV(result->num, 
					     (BT3_U16_NODE * BT3_U16_LEAF)),
					 LEAF1(result), BT3_U16_NODE, 
					 BT3_U16_NODE_BLOCKS);
			}
		}
		else { 
			__u32 root_num = DIV(spec->num, 
					   (BT3_U16_NODE * BT3_U16_LEAF));
			ins_pos[0] = next_node2_16(key[0], LEAF1(spec), 
						   root_num, 
						   next_node1_16(key[0], spec,
							  (__u16*)ROOT(spec),
							  BT3_U16_NODE_BLOCKS),
						   BT3_U16_NODE_BLOCKS,
						   BT3_U16_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node2_16(key[1], LEAF1(spec),
							  root_num, 
							  next_node1_16(key[1],
							   spec,
							   (__u16*)ROOT(spec),
							  BT3_U16_NODE_BLOCKS),
							   BT3_U16_NODE_BLOCKS,
							  BT3_U16_LEAF_BLOCKS);
			}
			insert_leaf_16(LEAF2(spec, BT3_U16_NODE, 
					     BT3_U16_NODE_BLOCKS,
					     BT3_U16_LEAF), 
				       BT3_U16_LEAF, 
				       OFFSET(BT3_U16_LEAF, BIT_U16), 
				       spec->num,
				       LEAF2(result, BT3_U16_NODE, 
					     BT3_U16_NODE_BLOCKS,
					     BT3_U16_LEAF), 
				       BT3_U16_LEAF, 
				       OFFSET(BT3_U16_LEAF, BIT_U16),
				       ins_num, ins_pos, key, nextspec);
			build_16((__u16*)LEAF1(result), 
				 DIV(result->num, BT3_U16_LEAF),
				 LEAF2(result, BT3_U16_NODE, 
				       BT3_U16_NODE_BLOCKS, BT3_U16_LEAF),
				 BT3_U16_LEAF, BT3_U16_LEAF_BLOCKS);
			build_16((__u16*)ROOT(result), 
				 DIV(result->num, 
				     (BT3_U16_NODE * BT3_U16_LEAF)),
				 LEAF1(result), BT3_U16_NODE, 
				 BT3_U16_NODE_BLOCKS);
		}
	}
	else {
		*WILDCARD(result) = *WILDCARD(spec);
		if (spec->btreetype == BT0){
			ins_pos[0] = ROOT(spec);
			if (ins_num == 2) {
				ins_pos[1] = ROOT(spec);
			}
			if (result->btreetype == BT0){
				insert_leaf_32(ROOT(spec), spec->num, 0,//FIXME_ALIGN
					       spec->num,
					       ROOT(result), result->num, 0,
					       ins_num, ins_pos,
					       key, nextspec);
			}
			else {
				insert_leaf_32(ROOT(spec), spec->num, 0,
					       spec->num,
					       LEAF1(result), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       ins_num, ins_pos,
					       key, nextspec);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, BT1_U32_LEAF),
					 LEAF1(result), BT1_U32_LEAF,
					 BT1_U32_LEAF_BLOCKS);
			}
		}
		else if (spec->btreetype == BT1){
			ins_pos[0] = next_node1_32(key[0], spec,
						   (__u32*)ROOT(spec), 
						   BT1_U32_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node1_32(key[1], spec,
							  (__u32*)ROOT(spec),
							  BT1_U32_LEAF_BLOCKS);
			}
			if (result->btreetype == BT1){
				insert_leaf_32(LEAF1(spec), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF1(result), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       ins_num, ins_pos,
					       key, nextspec);
				build_32((__u32*)ROOT(result),
					 DIV(result->num, BT1_U32_LEAF),
					 LEAF1(result), BT1_U32_LEAF, 
					 BT1_U32_LEAF_BLOCKS);
			}
			else {
				insert_leaf_32(LEAF1(spec), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF2(result, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF),
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       ins_num, ins_pos,
					       key, nextspec);
				build_32((__u32*)LEAF1(result),
					 DIV(result->num, BT2_U32_LEAF),
					 LEAF2(result, BT2_U32_NODE, 
					       BT2_U32_NODE_BLOCKS, 
					       BT2_U32_LEAF),
					 BT2_U32_LEAF, BT2_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result),
					 DIV(result->num, 
					     (BT2_U32_NODE * BT2_U32_LEAF)),
					 LEAF1(result), BT2_U32_NODE, 
					 BT2_U32_NODE_BLOCKS);
			}
		}
		else if (spec->btreetype == BT2){
			__u32 root_num = DIV(spec->num, 
					   (BT2_U32_NODE * BT2_U32_LEAF));
			ins_pos[0] = next_node2_32(key[0], LEAF1(spec),
						   root_num, 
						   next_node1_32(key[0],
								 spec,
								 ROOT(spec),
							  BT2_U32_NODE_BLOCKS),
						   BT2_U32_NODE_BLOCKS, 
						   BT2_U32_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node2_32(key[1], LEAF1(spec),
						      root_num, 
						      next_node1_32(key[1],
								    spec,
								    ROOT(spec),
							  BT2_U32_NODE_BLOCKS),
						      BT2_U32_NODE_BLOCKS,
						      BT2_U32_LEAF_BLOCKS);
			}
			if (result->btreetype == BT2){
				insert_leaf_32(LEAF2(spec, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF2(result, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF,
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       ins_num, ins_pos,
					       key, nextspec);
				build_32((__u32*)LEAF1(result),
					 DIV(result->num, BT2_U32_LEAF),
					 LEAF2(result, BT2_U32_NODE,
					       BT2_U32_NODE_BLOCKS,
					       BT2_U32_LEAF),
					 BT2_U32_LEAF, BT2_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result),
					 DIV(result->num, 
					     (BT2_U32_NODE * BT2_U32_LEAF)),
					 LEAF1(result), BT2_U32_NODE,
					 BT2_U32_NODE_BLOCKS);
			}
			else {
				insert_leaf_32(LEAF2(spec, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF),
					       BT2_U32_LEAF,
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF2(result, BT3_U32_NODE,
						     BT3_U32_NODE_BLOCKS,
						     BT3_U32_LEAF), 
					       BT3_U32_LEAF,
					       OFFSET(BT3_U32_LEAF, BIT_U32),
					       ins_num, ins_pos,
					       key, nextspec);
				build_32((__u32*)LEAF1(result),
					 DIV(result->num, BT3_U32_LEAF),
					 LEAF2(result, BT3_U32_NODE, 
					       BT3_U32_NODE_BLOCKS,
					       BT3_U32_LEAF),
					 BT3_U32_LEAF, BT3_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, 
					     (BT3_U32_NODE * BT3_U32_LEAF)),
					 LEAF1(result), BT3_U32_NODE,
					 BT3_U32_NODE_BLOCKS);
			}
		}
		else { 
			__u32 root_num = DIV(spec->num, 
					   (BT3_U32_NODE * BT3_U32_LEAF));
			ins_pos[0] = next_node2_32(key[0], LEAF1(spec),
						   root_num, 
						   next_node1_32(key[0], spec,
								 ROOT(spec),
							 BT3_U32_NODE_BLOCKS),
						   BT3_U32_NODE_BLOCKS,
						   BT3_U32_LEAF_BLOCKS);
			if (ins_num == 2) {
				ins_pos[1] = next_node2_32(key[1], LEAF1(spec),
						     root_num, 
						     next_node1_32(key[1],
								  spec,
								  ROOT(spec),
							 BT3_U32_NODE_BLOCKS),
						     BT3_U32_NODE_BLOCKS,
						     BT3_U32_LEAF_BLOCKS);
			}
			insert_leaf_32(LEAF2(spec, BT3_U32_NODE,
					     BT3_U32_NODE_BLOCKS,
					     BT3_U32_LEAF), 
				       BT3_U32_LEAF,
				       OFFSET(BT3_U32_LEAF, BIT_U32),
				       spec->num,
				       LEAF2(result, BT3_U32_NODE,
					     BT3_U32_NODE_BLOCKS,
					     BT3_U32_LEAF), 
				       BT3_U32_LEAF,
				       OFFSET(BT3_U32_LEAF, BIT_U32),
				       ins_num, ins_pos, key, nextspec);
			build_32((__u32*)LEAF1(result),
				 DIV(result->num, BT3_U32_LEAF),
				 LEAF2(result, BT3_U32_NODE,
				       BT3_U32_NODE_BLOCKS, BT3_U32_LEAF),
				 BT3_U32_LEAF, BT3_U32_LEAF_BLOCKS);
			build_32((__u32*)ROOT(result),
				 DIV(result->num, 
				     (BT3_U32_NODE * BT3_U32_LEAF)),
				 LEAF1(result), BT3_U32_NODE,
				 BT3_U32_NODE_BLOCKS);
		}
	}
	*res = result;
	return HE_OK;
}


hipac_error
btree_delete(const struct btree_spec *spec, __u8 del_num, const __u32 key[],
	     struct btree_spec **res)
{
	struct btree_spec *result;
	void* del_pos[2];
	hipac_error error;

	if (unlikely(!spec || !key || !res ||
		     !(del_num == 1 || del_num == 2) ||
		     (del_num == 2 && key[0] >= key[1])))
		ARG_ERR;
	
	result = new_segmentset_intern(spec->num - del_num, spec->bittype);
	if (!result)
		return HE_LOW_MEMORY;

	if ((error = ptrblock_clone(*termrule(spec), termrule(result)))) {
		hp_free(result);
		return HE_LOW_MEMORY;
	}
	
	result->dimid = spec->dimid;
	
	if (spec->bittype == BIT_U8){
		if (spec->btreetype == BT0){
			del_pos[0] = ROOT(spec);
			if (del_num == 2) {
				del_pos[1] = ROOT(spec);
			}
			//think harder
			delete_leaf_8(ROOT(spec), spec->num, 0, spec->num,//FIXME_ALIGN
				      ROOT(result), result->num, 0,
				      del_num, del_pos, key);
		}
		else {
			del_pos[0] = next_node1_8(key[0], spec,
						  (__u8*)ROOT(spec),
						  BT1_U8_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node1_8(key[1], spec,
							  (__u8*)ROOT(spec),
							  BT1_U8_LEAF_BLOCKS);
			}
			if (result->btreetype == BT1){
				delete_leaf_8(LEAF1(spec), BT1_U8_LEAF,
					      OFFSET(BT1_U8_LEAF, BIT_U8),
					      spec->num,
					      LEAF1(result), BT1_U8_LEAF,
					      OFFSET(BT1_U8_LEAF, BIT_U8),
					      del_num, del_pos, key);
				build_8((__u8*)ROOT(result),
					DIV(result->num, BT1_U8_LEAF),
					LEAF1(result), BT1_U8_LEAF,
					BT1_U8_LEAF_BLOCKS);
			}
			else {
				delete_leaf_8(LEAF1(spec), BT1_U8_LEAF,
					      OFFSET(BT1_U8_LEAF, BIT_U8),
					      spec->num,
					      ROOT(result), result->num, 0,//FIXME_ALIGN
					      del_num, del_pos, key);
			}
		}
	}
	else if (spec->bittype == BIT_U16){
		if (spec->btreetype == BT0){
			del_pos[0] = ROOT(spec);
			if (del_num == 2) {
				del_pos[1] = ROOT(spec);
			}
			//think harder
			delete_leaf_16(ROOT(spec), spec->num, 0, spec->num,//FIXME_ALIGN
				       ROOT(result), result->num, 0,
				       del_num, del_pos, key);
		}
		else if (spec->btreetype == BT1){
			del_pos[0] = next_node1_16(key[0], spec,
						   (__u16*)ROOT(spec),
						   BT1_U16_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node1_16(key[1], spec,
							 (__u16*)ROOT(spec),
							 BT1_U16_LEAF_BLOCKS);
			}
			if (result->btreetype == BT1){
				delete_leaf_16(LEAF1(spec), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       del_num, del_pos, key);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT1_U16_LEAF),
					 LEAF1(result), BT1_U16_LEAF,
					 BT1_U16_LEAF_BLOCKS);
			}
			else {
				delete_leaf_16(LEAF1(spec), BT1_U16_LEAF,//FIXME_ALIGN
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       spec->num,
					       ROOT(result), result->num, 0,
					       del_num, del_pos, key);
			}
		}
		else if (spec->btreetype == BT2){
			del_pos[0] = next_node1_16(key[0], spec,
						   (__u16*)ROOT(spec),
						   BT2_U16_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node1_16(key[1], spec, 
							  (__u16*)ROOT(spec),
							  BT2_U16_LEAF_BLOCKS);
			}
			if (result->btreetype == BT2){
				delete_leaf_16(LEAF1(spec), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       del_num, del_pos, key);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT2_U16_LEAF),
					 LEAF1(result), BT2_U16_LEAF,
					 BT2_U16_LEAF_BLOCKS);
			}
			else {
				delete_leaf_16(LEAF1(spec), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT1_U16_LEAF,
					       OFFSET(BT1_U16_LEAF, BIT_U16),
					       del_num, del_pos, key);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT1_U16_LEAF),
					 LEAF1(result), BT1_U16_LEAF, 
					 BT1_U16_LEAF_BLOCKS);
			}
		}
		else { 
			__u32 root_num = DIV(spec->num, 
					   (BT3_U16_NODE * BT3_U16_LEAF));
			del_pos[0] = next_node2_16(key[0], LEAF1(spec),
						   root_num, 
						   next_node1_16(key[0], spec,
							    (__u16*)ROOT(spec),
							 BT3_U16_NODE_BLOCKS),
						   BT3_U16_NODE_BLOCKS,
						   BT3_U16_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node2_16(key[1],
						      LEAF1(spec),
						      root_num, 
						      next_node1_16(key[1],
								      spec,
							    (__u16*)ROOT(spec),
							 BT3_U16_NODE_BLOCKS),
						      BT3_U16_NODE_BLOCKS,
						      BT3_U16_LEAF_BLOCKS);
			}
			if (result->btreetype == BT3){
				delete_leaf_16(LEAF2(spec, BT3_U16_NODE,
						     BT3_U16_NODE_BLOCKS,
						     BT3_U16_LEAF), 
					       BT3_U16_LEAF, 
					       OFFSET(BT3_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF2(result, BT3_U16_NODE,
						     BT3_U16_NODE_BLOCKS,
						     BT3_U16_LEAF), 
					       BT3_U16_LEAF, 
					       OFFSET(BT3_U16_LEAF, BIT_U16),
					       del_num, del_pos, key);
				build_16((__u16*)LEAF1(result),
					 DIV(result->num, BT3_U16_LEAF),
					 LEAF2(result, BT3_U16_NODE,
					       BT3_U16_NODE_BLOCKS,
					       BT3_U16_LEAF),
					 BT3_U16_LEAF, BT3_U16_LEAF_BLOCKS);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, 
					     (BT3_U16_NODE * BT3_U16_LEAF)),
					 LEAF1(result), BT3_U16_NODE,
					 BT3_U16_NODE_BLOCKS);
			}
			else {
				delete_leaf_16(LEAF2(spec, BT3_U16_NODE,
						     BT3_U16_NODE_BLOCKS,
						     BT3_U16_LEAF), 
					       BT3_U16_LEAF,
					       OFFSET(BT3_U16_LEAF, BIT_U16),
					       spec->num,
					       LEAF1(result), BT2_U16_LEAF,
					       OFFSET(BT2_U16_LEAF, BIT_U16),
					       del_num, del_pos, key);
				build_16((__u16*)ROOT(result),
					 DIV(result->num, BT2_U16_LEAF),
					 LEAF1(result), BT2_U16_LEAF,
					 BT2_U16_LEAF_BLOCKS);
			}
		}
	}
	else {
		*WILDCARD(result) = *WILDCARD(spec);
		if (spec->btreetype == BT0){
			del_pos[0] = ROOT(spec);
			if (del_num == 2) {
				del_pos[1] = ROOT(spec);
			}
			//think harder
			delete_leaf_32(ROOT(spec), spec->num, 0, spec->num,//FIXME_ALIGN
				       ROOT(result), result->num, 0,
				       del_num, del_pos, key);
			
		}
		else if (spec->btreetype == BT1){
			del_pos[0] = next_node1_32(key[0], spec, ROOT(spec),
						   BT1_U32_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node1_32(key[1], spec,
							 ROOT(spec),
							 BT1_U32_LEAF_BLOCKS);
			}
			if (result->btreetype == BT1){
				delete_leaf_32(LEAF1(spec), BT1_U32_LEAF, 
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF1(result), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       del_num, del_pos, key);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, BT1_U32_LEAF),
					 LEAF1(result), BT1_U32_LEAF,
					 BT1_U32_LEAF_BLOCKS);
			}
			else {
				delete_leaf_32(LEAF1(spec), BT1_U32_LEAF,//FIXME_ALIGN
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       spec->num,
					       ROOT(result), result->num, 0,
					       del_num, del_pos, key);
			}
		}
		else if (spec->btreetype == BT2){
			__u32 root_num = DIV(spec->num, 
					   (BT2_U32_NODE * BT2_U32_LEAF));
			del_pos[0] = next_node2_32(key[0], LEAF1(spec),
						   root_num, 
						   next_node1_32(key[0], spec,
							 ROOT(spec),
							 BT2_U32_NODE_BLOCKS),
						   BT2_U32_NODE_BLOCKS,
						   BT2_U32_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node2_32(key[1], 
						     LEAF1(spec),
						     root_num, 
						     next_node1_32(key[1],
								   spec,
								   ROOT(spec),
							 BT2_U32_NODE_BLOCKS),
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF_BLOCKS);
			}
			if (result->btreetype == BT2){
				delete_leaf_32(LEAF2(spec, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF2(result, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       del_num, del_pos, key);
				build_32((__u32*)LEAF1(result), 
					 DIV(result->num, BT2_U32_LEAF),
					 LEAF2(result, BT2_U32_NODE, 
					       BT2_U32_NODE_BLOCKS, 
					       BT2_U32_LEAF),
					 BT2_U32_LEAF, BT2_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, 
					     (BT2_U32_NODE * BT2_U32_LEAF)),
					 LEAF1(result), BT2_U32_NODE, 
					 BT2_U32_NODE_BLOCKS);
			}
			else {
				delete_leaf_32(LEAF2(spec, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF1(result), BT1_U32_LEAF,
					       OFFSET(BT1_U32_LEAF, BIT_U32),
					       del_num, del_pos, key);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, BT1_U32_LEAF),
					 LEAF1(result), BT1_U32_LEAF, 
					 BT1_U32_LEAF_BLOCKS);
			}
		}
		else { 
			__u32 root_num = DIV(spec->num, 
					   (BT3_U32_NODE * BT3_U32_LEAF));
			del_pos[0] = next_node2_32(key[0], LEAF1(spec), 
						   root_num, 
						   next_node1_32(key[0], spec,
								 ROOT(spec),
							 BT3_U32_NODE_BLOCKS),
						   BT3_U32_NODE_BLOCKS, 
						   BT3_U32_LEAF_BLOCKS);
			if (del_num == 2) {
				del_pos[1] = next_node2_32(key[1], 
						     LEAF1(spec), root_num,
						     next_node1_32(key[1], 
								   spec, 
								   ROOT(spec),
							 BT3_U32_NODE_BLOCKS),
						     BT3_U32_NODE_BLOCKS,
						     BT3_U32_LEAF_BLOCKS);
			}
			if (result->btreetype == BT3){
				delete_leaf_32(LEAF2(spec, BT3_U32_NODE, 
						     BT3_U32_NODE_BLOCKS, 
						     BT3_U32_LEAF), 
					       BT3_U32_LEAF, 
					       OFFSET(BT3_U32_LEAF, BIT_U32), 
					       spec->num,
					       LEAF2(result, BT3_U32_NODE, 
						     BT3_U32_NODE_BLOCKS, 
						     BT3_U32_LEAF), 
					       BT3_U32_LEAF, 
					       OFFSET(BT3_U32_LEAF, BIT_U32),
					       del_num, del_pos, key);
				build_32((__u32*)LEAF1(result), 
					 DIV(result->num, BT3_U32_LEAF),
					 LEAF2(result, BT3_U32_NODE, 
					       BT3_U32_NODE_BLOCKS, 
					       BT3_U32_LEAF),
					 BT3_U32_LEAF, BT3_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, 
					     (BT3_U32_NODE * BT3_U32_LEAF)),
					 LEAF1(result), BT3_U32_NODE, 
					 BT3_U32_NODE_BLOCKS);
			}
			else {
				delete_leaf_32(LEAF2(spec, BT3_U32_NODE, 
						     BT3_U32_NODE_BLOCKS, 
						     BT3_U32_LEAF), 
					       BT3_U32_LEAF, 
					       OFFSET(BT3_U32_LEAF, BIT_U32),
					       spec->num,
					       LEAF2(result, BT2_U32_NODE, 
						     BT2_U32_NODE_BLOCKS, 
						     BT2_U32_LEAF), 
					       BT2_U32_LEAF, 
					       OFFSET(BT2_U32_LEAF, BIT_U32),
					       del_num, del_pos, key);
				build_32((__u32*)LEAF1(result), 
					 DIV(result->num, BT2_U32_LEAF),
					 LEAF2(result, BT2_U32_NODE, 
					       BT2_U32_NODE_BLOCKS, 
					       BT2_U32_LEAF),
					 BT2_U32_LEAF, BT2_U32_LEAF_BLOCKS);
				build_32((__u32*)ROOT(result), 
					 DIV(result->num, 
					     (BT2_U32_NODE * BT2_U32_LEAF)),
					 LEAF1(result), BT2_U32_NODE, 
					 BT2_U32_NODE_BLOCKS);
			}
		}
	}
	*res = result;
	return HE_OK;
}
