/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

/* Routine Declaration - Local */
_local_  int   fc_binfo_init(fc_dev_ctl_t *p_dev_ctl);
_local_  int   fc_parse_vpd(fc_dev_ctl_t *p_dev_ctl, uchar *vpd);
_local_  int   fc_proc_ring_event( fc_dev_ctl_t *p_dev_ctl, RING *rp,
                           IOCBQ *saveq);
/* End Routine Declaration - Local */
extern uint32        fcPAGESIZE;
extern uint32 fc_diag_state;
extern int    fcinstance[];

int           fc_check_for_vpd = 1;
int           fc_reset_on_attach = 0;

extern int              fc_max_els_sent;

#define FC_MAX_VPD_SIZE   0x100
static uint32 fc_vpd_data[FC_MAX_VPD_SIZE];

static uint32 fc_run_biu_test[256] = {
   /* Walking ones */
   0x80000000, 0x40000000, 0x20000000, 0x10000000,
   0x08000000, 0x04000000, 0x02000000, 0x01000000,
   0x00800000, 0x00400000, 0x00200000, 0x00100000,
   0x00080000, 0x00040000, 0x00020000, 0x00010000,
   0x00008000, 0x00004000, 0x00002000, 0x00001000,
   0x00000800, 0x00000400, 0x00000200, 0x00000100,
   0x00000080, 0x00000040, 0x00000020, 0x00000010,
   0x00000008, 0x00000004, 0x00000002, 0x00000001,

   /* Walking zeros */
   0x7fffffff, 0xbfffffff, 0xdfffffff, 0xefffffff,
   0xf7ffffff, 0xfbffffff, 0xfdffffff, 0xfeffffff,
   0xff7fffff, 0xffbfffff, 0xffdfffff, 0xffefffff,
   0xfff7ffff, 0xfffbffff, 0xfffdffff, 0xfffeffff,
   0xffff7fff, 0xffffbfff, 0xffffdfff, 0xffffefff,
   0xfffff7ff, 0xfffffbff, 0xfffffdff, 0xfffffeff,
   0xffffff7f, 0xffffffbf, 0xffffffdf, 0xffffffef,
   0xfffffff7, 0xfffffffb, 0xfffffffd, 0xfffffffe,

   /* all zeros */
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,
   0x00000000, 0x00000000, 0x00000000, 0x00000000,

   /* all ones */
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
   0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,

   /* all 5's */
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,
   0x55555555, 0x55555555, 0x55555555, 0x55555555,

   /* all a's */
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,
   0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa,

   /* all 5a's */
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,
   0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a, 0x5a5a5a5a,

   /* all a5's */
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5,
   0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5, 0xa5a5a5a5
};

extern _static_ void fc_read_nv(FC_BRD_INFO *binfo, MAILBOX *mb);
#define S(N,V) (((V)<<(N))|((V)>>(32-(N))))
#define BYTESWAP(x) ((x<<24) | (x >> 24) | (0xFF00 & (x >> 8)) | (0xFF0000 & (x << 8)));

/************************************************************************/
/*                                                                      */
/*   fc_swap_bcopy                                                      */
/*                                                                      */
/************************************************************************/
_static_ void
fc_swap_bcopy(
uint32  *src,
uint32  *dest,
uint32 cnt)
{
   uint32 ldata;
   int  i;

   for (i = 0; i < (int)cnt; i += sizeof(uint32)) {
      ldata = *src++;
      ldata = cpu_to_be32(ldata);
      *dest++ = ldata;
   }
}       /* End fc_swap_bcopy */

/************************************************************************/
/*                                                                      */
/*   fc_init_hba                                                        */
/*                                                                      */
/************************************************************************/
uint32
fc_init_hba(
fc_dev_ctl_t  * p_dev_ctl,
MAILBOX       * mb,
uint32 * pwwnn)
{
   FC_BRD_INFO   * binfo;
   uint32 * pText;
   char licensed[56] = "key unlock for use with gnu public licensed code only\0";
   pText = (uint32 *) licensed;
   fc_swap_bcopy(pText, pText, 56);
   binfo = &BINFO;
   /* Setup and issue mailbox READ NVPARAMS command */
   binfo->fc_ffstate = FC_INIT_NVPARAMS;
   fc_read_nv(binfo, mb);
   memset((void*) mb->un.varRDnvp.rsvd3, 0, sizeof(mb->un.varRDnvp.rsvd3));
   memcpy((void*) mb->un.varRDnvp.rsvd3, licensed, sizeof(licensed));
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      /* Adapter initialization error, mbxCmd <cmd> READ_NVPARM, mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0303,                   /* ptr to msg structure */
              fc_mes0303,                      /* ptr to msg */
               fc_msgBlk0303.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      return(0);
   }
   fc_bcopy ((uchar*)mb->un.varRDnvp.nodename, (uchar*) pwwnn, sizeof(mb->un.varRDnvp.nodename));
   return(1);
}

/************************************************************************/
/*                                                                      */
/*   sha_initialize                                                     */
/*                                                                      */
/************************************************************************/
void 
sha_initialize(
uint32 *HashResultPointer)
{
  HashResultPointer[0] = 0x67452301;
  HashResultPointer[1] = 0xEFCDAB89;
  HashResultPointer[2] = 0x98BADCFE;
  HashResultPointer[3] = 0x10325476;
  HashResultPointer[4] = 0xC3D2E1F0;
}
/************************************************************************/
/*                                                                      */
/*   sha_iterate                                                        */
/*                                                                      */
/************************************************************************/
void 
sha_iterate(
uint32 *HashResultPointer, 
uint32 *HashWorkingPointer)
{
  int t;
  uint32 TEMP;
  uint32 A, B, C, D, E;
  t = 16;
  do
  {
    HashWorkingPointer[t] = S(1,HashWorkingPointer[t-3]^HashWorkingPointer[t-8]^HashWorkingPointer[t-14]^HashWorkingPointer[t-16]);
  } while (++t <= 79);
  t = 0;
  A = HashResultPointer[0];
  B = HashResultPointer[1];
  C = HashResultPointer[2];
  D = HashResultPointer[3];
  E = HashResultPointer[4];

  do
  {
    if (t < 20)
	{
      TEMP = ((B & C) | ((~B) & D)) + 0x5A827999;
    } else if (t < 40) {
      TEMP = (B ^ C ^ D) + 0x6ED9EBA1;
    } else if (t < 60) {
      TEMP = ((B & C) | (B & D) | (C & D)) + 0x8F1BBCDC;
    } else {
      TEMP = (B ^ C ^ D) + 0xCA62C1D6;
    }
    TEMP += S(5,A) + E + HashWorkingPointer[t];
    E = D;
    D = C;
    C = S(30,B);
    B = A;
    A = TEMP;
  } while (++t <= 79);
  
  HashResultPointer[0] += A;
  HashResultPointer[1] += B;
  HashResultPointer[2] += C;
  HashResultPointer[3] += D;
  HashResultPointer[4] += E;

}
/************************************************************************/
/*                                                                      */
/*   Challenge_XOR_KEY                                                  */
/*                                                                      */
/************************************************************************/
void 
Challenge_XOR_KEY 
(uint32 *RandomChallenge,
 uint32 *HashWorking)
{
   *HashWorking = (*RandomChallenge ^ *HashWorking);
}	

/************************************************************************/
/*                                                                      */
/*   fc_SHA1                                                            */
/*                                                                      */
/************************************************************************/
void  
fc_SHA1(
uint32 * pwwnn, 
uint32 * phbainitEx, 
uint32 * RandomData)
{
  int t;
  uint32  HashWorking[80];
  
  fc_bzero(HashWorking, sizeof(HashWorking));
  HashWorking[0]  = HashWorking[78] = *pwwnn++;
  HashWorking[1]  = HashWorking[79] = *pwwnn;
  for (t = 0; t < 7 ; t++) {
    Challenge_XOR_KEY(RandomData+t,HashWorking+t);
  }
  sha_initialize(phbainitEx);
  sha_iterate(phbainitEx, HashWorking);
}

/************************************************************************/
/*                                                                      */
/*   fc_ffinit                                                          */
/*                                                                      */
/************************************************************************/
_static_ int
fc_ffinit(
fc_dev_ctl_t    *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   fc_vpd_t      * vp;
   uint32          status, i, j;
   uint32          read_rev_reset, hbainit = 0;
   uint32          RandomData[7];
   uint32          hbainitEx[5];
   uint32          wwnn[2];
   struct pci_dev  *pdev;
   int             ipri, flogi_sent;
   MBUF_INFO       bufinfo;
   MBUF_INFO     * buf_info;
   void          * ioa;
   RING          * rp;
   MAILBOX       * mb;
   MATCHMAP      * mp, *mp1, *mp2;
   uchar         * inptr, *outptr;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   vp = &VPD;
   mb = 0;

   pdev = p_dev_ctl->pcidev ;
   /* Set board state to initialization started */
   binfo->fc_ffstate = FC_INIT_START;
   read_rev_reset = 0;

   if(fc_reset_on_attach) {
      binfo->fc_ffstate = 0; 
      fc_brdreset(p_dev_ctl);
      binfo->fc_ffstate = FC_INIT_START;
      DELAYMS(2500);
   }

top:
   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */

#if LITTLE_ENDIAN_HOST 
   /* For Little Endian, BIU_BSE is not supported */
#else
#ifdef BIU_BSE
   status = READ_CSR_REG(binfo, FC_BC_REG(binfo, ioa));
   WRITE_CSR_REG(binfo, FC_BC_REG(binfo, ioa), (BC_BSE_SWAP | status));
   i = READ_CSR_REG(binfo, FC_BC_REG(binfo, ioa));
#endif
#endif

   status = READ_CSR_REG(binfo, FC_STAT_REG(binfo, ioa));
   FC_UNMAP_MEMIO(ioa);

   i = 0;

   /* Check status register to see what current state is */
   while ((status & (HS_FFRDY | HS_MBRDY)) != (HS_FFRDY | HS_MBRDY)) {

      /* Check every 100ms for 5 retries, then every 500ms for 5, then
       * every 2.5 sec for 5, then reset board and every 2.5 sec for 4.
       */
      if (i++ >= 20) {
         /* Adapter failed to init, timeout, status reg <status> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0436,                   /* ptr to msg structure */
                 fc_mes0436,                      /* ptr to msg */
                  fc_msgBlk0436.msgPreambleStr,   /* begin varargs */
                   status);                       /* end varargs */
         binfo->fc_ffstate = FC_ERROR;
         return(EIO);
      }

      /* Check to see if any errors occurred during init */
      if (status & HS_FFERM) {
         /* ERROR: During chipset initialization */
         /* Adapter failed to init, chipset, status reg <status> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0437,                   /* ptr to msg structure */
                 fc_mes0437,                      /* ptr to msg */
                  fc_msgBlk0437.msgPreambleStr,   /* begin varargs */
                   status);                       /* end varargs */
         binfo->fc_ffstate = FC_ERROR;
         return(EIO);
      }

      if (i <= 5) {
         DELAYMS(100);
      }
      else if (i <= 10) {
         DELAYMS(500);
      }
      else {
         DELAYMS(2500);
      }

      if (i == 15) {
         /* Reset board and try one more time */
         binfo->fc_ffstate = 0; 
         fc_brdreset(p_dev_ctl);
         binfo->fc_ffstate = FC_INIT_START;
      }

      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      status = READ_CSR_REG(binfo, FC_STAT_REG(binfo, ioa));
      FC_UNMAP_MEMIO(ioa);
   }

   /* Check to see if any errors occurred during init */
   if (status & HS_FFERM) {
      /* ERROR: During chipset initialization */
      /* Adapter failed to init, chipset, status reg <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0438,                   /* ptr to msg structure */
              fc_mes0438,                      /* ptr to msg */
               fc_msgBlk0438.msgPreambleStr,   /* begin varargs */
                status);                       /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      return(EIO);
   }

   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */

   /* Clear all interrupt enable conditions */
   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), 0);

   /* setup host attn register */
   WRITE_CSR_REG(binfo, FC_HA_REG(binfo, ioa), 0xffffffff);

   FC_UNMAP_MEMIO(ioa);

   if(read_rev_reset)
      goto do_read_rev;

   fc_binfo_init(p_dev_ctl);
   
   /* Allocate some memory for buffers */
   if (fc_malloc_buffer(p_dev_ctl) == 0) {
      binfo->fc_ffstate = FC_ERROR;
      return(ENOMEM);
   }

   fc_get_dds_bind(p_dev_ctl);

   /* Get a buffer which will be used repeatedly for mailbox commands */
   if ((mb = (MAILBOX * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI)) == 0) {
      binfo->fc_ffstate = FC_ERROR;
      fc_free_buffer(p_dev_ctl);
      return(ENOMEM);
   }

do_read_rev:
   if((pdev->device == PCI_DEVICE_ID_TFLY)||
      (pdev->device == PCI_DEVICE_ID_PFLY))
      hbainit = fc_init_hba(p_dev_ctl, mb, wwnn);
   /* Setup and issue mailbox READ REV command */
   binfo->fc_ffstate = FC_INIT_REV;
   fc_read_rev(binfo, mb);
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      /* Adapter failed to init, mbxCmd <mbxCmd> READ_REV, mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0439,                   /* ptr to msg structure */
              fc_mes0439,                      /* ptr to msg */
               fc_msgBlk0439.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */

      /* If read_rev fails, give it one more chance */
      if(read_rev_reset == 0) {
         binfo->fc_ffstate = 0; 
         fc_brdreset(p_dev_ctl);
         binfo->fc_ffstate = FC_INIT_START;

         DELAYMS(2500); 
         DELAYMS(2500);

         read_rev_reset = 1;
         goto top;
      }
      binfo->fc_ffstate = FC_ERROR;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   if(mb->un.varRdRev.rr == 0) {

      if(read_rev_reset == 0) {
         binfo->fc_ffstate = 0; 
         fc_brdreset(p_dev_ctl);
         binfo->fc_ffstate = FC_INIT_START;

         DELAYMS(2500); 
         DELAYMS(2500);

         read_rev_reset = 1;
         goto top;
      }

      vp->rev.rBit = 0;
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0440,                   /* ptr to msg structure */
              fc_mes0440,                      /* ptr to msg */
               fc_msgBlk0440.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 read_rev_reset);              /* end varargs */
   }
   else {
      if(mb->un.varRdRev.un.b.ProgType != 2) {
         if(read_rev_reset == 0) {
            binfo->fc_ffstate = 0; 
            fc_brdreset(p_dev_ctl);
            binfo->fc_ffstate = FC_INIT_START;

            DELAYMS(2500);
            DELAYMS(2500);

            read_rev_reset = 1;
            goto top;
         }
      }
      vp->rev.rBit = 1;
      vp->rev.sli1FwRev = mb->un.varRdRev.sli1FwRev;
      fc_bcopy((uchar *)mb->un.varRdRev.sli1FwName, (uchar *)vp->rev.sli1FwName, 16);
      vp->rev.sli2FwRev = mb->un.varRdRev.sli2FwRev;
      fc_bcopy((uchar *)mb->un.varRdRev.sli2FwName, (uchar *)vp->rev.sli2FwName, 16);
   }

   /* Save information as VPD data */
   vp->rev.biuRev = mb->un.varRdRev.biuRev;
   vp->rev.smRev = mb->un.varRdRev.smRev;
   vp->rev.smFwRev = mb->un.varRdRev.un.smFwRev;
   vp->rev.endecRev = mb->un.varRdRev.endecRev;
   vp->rev.fcphHigh = mb->un.varRdRev.fcphHigh;
   vp->rev.fcphLow = mb->un.varRdRev.fcphLow;
   vp->rev.feaLevelHigh = mb->un.varRdRev.feaLevelHigh;
   vp->rev.feaLevelLow = mb->un.varRdRev.feaLevelLow;
   vp->rev.postKernRev = mb->un.varRdRev.postKernRev;
   vp->rev.opFwRev = mb->un.varRdRev.opFwRev;
   if((pdev->device == PCI_DEVICE_ID_TFLY)||
      (pdev->device == PCI_DEVICE_ID_PFLY))
   fc_bcopy((uchar *)&mb->un.varWords[24], (uchar *)RandomData, sizeof(RandomData));

   dfc_fmw_rev(p_dev_ctl); /* Save firmware rev for HBAAPI */

   if(fc_check_for_vpd) {
      /* Get adapter VPD information */
      fc_dump_mem(binfo, mb);
      if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
         /*
          * Let it go through even if failed.
          */
         /* Adapter failed to init, mbxCmd <cmd> DUMP VPD, mbxStatus <status> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0441,                   /* ptr to msg structure */
                 fc_mes0441,                      /* ptr to msg */
                  fc_msgBlk0441.msgPreambleStr,   /* begin varargs */
                   mb->mbxCommand,
                    mb->mbxStatus);               /* end varargs */

         /* If dump_mem times out, give it one more chance */
         if((read_rev_reset == 0) && (mb->mbxStatus == 0)) {
            binfo->fc_ffstate = 0; 
            fc_brdreset(p_dev_ctl);
            binfo->fc_ffstate = FC_INIT_START;

            DELAYMS(2500); 
            DELAYMS(2500);

            read_rev_reset = 1;
            goto top;
         }
      }
      else {
         if((mb->un.varDmp.ra == 1) &&
            (mb->un.varDmp.word_cnt <= FC_MAX_VPD_SIZE)) {
            uint32 *lp1, *lp2;
   
            lp1 = (uint32 * )&mb->un.varDmp.resp_offset;
            lp2 = (uint32 * )&fc_vpd_data[0];
            for(i=0;i<mb->un.varDmp.word_cnt;i++) {
               status = *lp1++;
               *lp2++ = SWAP_LONG(status);
            }
            fc_parse_vpd(p_dev_ctl, (uchar *)&fc_vpd_data[0]); 
         }
      }
   }

   /* Setup and issue mailbox CONFIG_PORT or PARTITION_SLIM command */
   binfo->fc_ffstate = FC_INIT_PARTSLIM;
   if (binfo->fc_sli == 2) {
   if((pdev->device == PCI_DEVICE_ID_TFLY)||
      (pdev->device == PCI_DEVICE_ID_PFLY)){
      fc_SHA1(wwnn, hbainitEx, RandomData);
      fc_config_port(binfo, mb, (uint32 *) hbainitEx);
   }
   else
      fc_config_port(binfo, mb, &hbainit);
      if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
         /* Adapter failed to init, mbxCmd <cmd> CONFIG_PORT, mbxStatus <status> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0442,                   /* ptr to msg structure */
                 fc_mes0442,                      /* ptr to msg */
                  fc_msgBlk0442.msgPreambleStr,   /* begin varargs */
                   mb->mbxCommand,
                    mb->mbxStatus,
                     0);                    /* end varargs */

         /* If config_port fails, give it one more chance */
         if(read_rev_reset == 0) {
            binfo->fc_ffstate = 0; 
            fc_brdreset(p_dev_ctl);
            binfo->fc_ffstate = FC_INIT_START;

            DELAYMS(2500); 
            DELAYMS(2500);

            read_rev_reset = 1;
            goto top;
         }

         binfo->fc_flag &= ~FC_SLI2;
         binfo->fc_mboxaddr = 0;
         if (binfo->fc_slim2.virt) {
            buf_info = &bufinfo;
            if (binfo->fc_slim2.phys) {
               buf_info->phys = (void * )binfo->fc_slim2.phys;
               buf_info->data_handle = binfo->fc_slim2.data_handle;
               buf_info->dma_handle  = binfo->fc_slim2.dma_handle;
               buf_info->flags = FC_MBUF_DMA;
            } else {
               buf_info->phys = 0;
               buf_info->data_handle = 0;
               buf_info->dma_handle = 0;
               buf_info->flags = 0;
            }
            buf_info->size = fcPAGESIZE;
            buf_info->virt = (void * )binfo->fc_slim2.virt;
            fc_free(p_dev_ctl, buf_info);
            binfo->fc_slim2.virt = 0;
            binfo->fc_slim2.phys = 0;
            binfo->fc_slim2.dma_handle = 0;
            binfo->fc_slim2.data_handle = 0;
         }
         binfo->fc_ffstate = FC_ERROR;
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         fc_free_buffer(p_dev_ctl);
         return(EIO);
      }
   } else {
      /* SLI1 not supported, mbxCmd <cmd>, mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0443,                   /* ptr to msg structure */
              fc_mes0443,                      /* ptr to msg */
               fc_msgBlk0443.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus,
                     0);                    /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   /* Initialize cmd/rsp ring pointers */
   for (i = 0; i < (uint32)binfo->fc_ffnumrings; i++) {
      rp = &binfo->fc_ring[i];

      rp->fc_ringno = (uchar)i;
      rp->fc_xmitstate = FC_LINK_UP;
      if ((i == FC_IP_RING) || (i == FC_FCP_RING))
         rp->fc_xmitstate = FC_READY;
      rp->fc_binfo = (uchar * )binfo;
      rp->fc_iocbhd = 0;
      rp->fc_iocbtl = 0;
      rp->fc_cmdidx = 0;
      rp->fc_rspidx = 0;
      rp->fc_iotag = 1;                 /* used to identify each I/O */
      if (i == FC_FCP_RING)
         rp->fc_bufcnt = MAX_FCP_CMDS;  /* Used for ABTS iotag */

      /* offsets are from the beginning of SLIM */
      if (!(binfo->fc_flag & FC_SLI2)) {
         /* offsets are from the beginning of SLIM */
         rp->fc_cmdringaddr = (void *)((ulong)(mb->un.varSlim.ringdef[i].offCiocb));
         rp->fc_rspringaddr = (void *)((ulong)(mb->un.varSlim.ringdef[i].offRiocb));

      }
   }

   mp1 = 0;
   mp2 = 0;
   /* Setup and issue mailbox RUN BIU DIAG command */
   /* setup test buffers */
   if (((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF | MEM_PRI)) == 0)  || 
       ((mp1 = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF | MEM_PRI)) == 0) ||
       ((mp2 = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF | MEM_PRI)) == 0)) {
      /* Adapter failed to init, no buffers for RUN_BIU_DIAG */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0444,                   /* ptr to msg structure */
              fc_mes0444,                      /* ptr to msg */
               fc_msgBlk0444.msgPreambleStr);  /* begin & end varargs */
      if (mp)
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      if (mp1)
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp1);
      binfo->fc_ffstate = FC_ERROR;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(ENOMEM);
   }

   fc_mpdata_incopy(p_dev_ctl, mp, (uchar * ) & fc_run_biu_test[0], FCELSSIZE);
   fc_mpdata_sync(mp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);
   inptr = mp->virt;
   /* Issue mailbox command */
   fc_runBIUdiag(binfo, mb, mp->phys, mp1->phys);
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      FC_UNMAP_MEMIO(ioa);
      /* Adapter failed init, mailbox cmd <mbxCmd> runBIUdiag mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0447,                   /* ptr to msg structure */
              fc_mes0447,                      /* ptr to msg */
               fc_msgBlk0447.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp1);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp2);
      binfo->fc_ffstate = FC_ERROR;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   fc_mpdata_sync(mp1->dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);
   fc_mpdata_outcopy(p_dev_ctl, mp1, (uchar * )mp2->virt, FCELSSIZE);
   outptr = (uchar * )mp2->virt;

   for (i = 0; i < FCELSSIZE; i++) {
      if (*outptr++ != *inptr++) {
         outptr--;
         inptr--;
         /* RUN_BIU_DIAG failed */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0445,                   /* ptr to msg structure */
                 fc_mes0445,                      /* ptr to msg */
                  fc_msgBlk0445.msgPreambleStr);  /* begin & end varargs */
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp1);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp2);
         binfo->fc_ffstate = FC_ERROR;
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         fc_free_buffer(p_dev_ctl);
         return(EIO);
      }
   }
   fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
   fc_mem_put(binfo, MEM_BUF, (uchar * )mp1);
   fc_mem_put(binfo, MEM_BUF, (uchar * )mp2);

   /* Setup and issue mailbox CONFIGURE RING command */
   for (i = 0; i < (uint32)binfo->fc_ffnumrings; i++) {
      binfo->fc_ffstate = FC_INIT_CFGRING;
      fc_config_ring(binfo, i, 0, mb);
      if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
         /* Adapter failed to init, mbxCmd <cmd> CFG_RING, mbxStatus <status>, ring <num> */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0446,                   /* ptr to msg structure */
                 fc_mes0446,                      /* ptr to msg */
                  fc_msgBlk0446.msgPreambleStr,   /* begin varargs */
                   mb->mbxCommand,
                    mb->mbxStatus,
                     i);                          /* ring num - end varargs */
         binfo->fc_ffstate = FC_ERROR;
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         fc_free_buffer(p_dev_ctl);
         return(EIO);
      }
   }

   /* Setup link timers */
   fc_config_link(p_dev_ctl, mb);
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      /* Adapter failed to init, mbxCmd <cmd> CONFIG_LINK mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0447,                   /* ptr to msg structure */
              fc_mes0447,                      /* ptr to msg */
               fc_msgBlk0447.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      fc_ffcleanup(p_dev_ctl);
      i_clear(&IHS);
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   /* We need to get login parameters for NID */
   fc_read_sparam(p_dev_ctl, mb);
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      /* Adapter failed to init, mbxCmd <cmd> READ_SPARM mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0448,                   /* ptr to msg structure */
              fc_mes0448,                      /* ptr to msg */
               fc_msgBlk0448.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   mp = (MATCHMAP * )binfo->fc_mbbp;
   fc_mpdata_sync(mp->dma_handle, 0, sizeof(SERV_PARM),
       DDI_DMA_SYNC_FORKERNEL);
   fc_mpdata_outcopy(p_dev_ctl, mp, (uchar * ) & binfo->fc_sparam,
       sizeof(SERV_PARM));

   fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
   binfo->fc_mbbp = 0;

   fc_bcopy((uchar * )&binfo->fc_sparam.nodeName, (uchar * )&binfo->fc_nodename,
       sizeof(NAME_TYPE));
   fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&binfo->fc_portname,
       sizeof(NAME_TYPE));
   fc_bcopy(binfo->fc_portname.IEEE, p_dev_ctl->phys_addr, 6);

   /* If no serial number in VPD data, use low 6 bytes of WWNN */
   if(binfo->fc_SerialNumber[0] == 0) {
      outptr = (uchar *) &binfo->fc_nodename.IEEE[0];
      for(i=0;i<12;i++) {
         status = *outptr++;
         j = ((status & 0xf0) >> 4);
         if(j <= 9)
            binfo->fc_SerialNumber[i] = (char)((uchar)0x30 + (uchar)j);
         else
            binfo->fc_SerialNumber[i] = (char)((uchar)0x61 + (uchar)(j-10));
         i++;
         j = (status & 0xf);
         if(j <= 9)
            binfo->fc_SerialNumber[i] = (char)((uchar)0x30 + (uchar)j);
         else
            binfo->fc_SerialNumber[i] = (char)((uchar)0x61 + (uchar)(j-10));
      }
   }

   if(clp[CFG_NETWORK_ON].a_current) {
      if ((binfo->fc_sparam.portName.nameType != NAME_IEEE) || 
          (binfo->fc_sparam.portName.IEEEextMsn != 0) || 
          (binfo->fc_sparam.portName.IEEEextLsb != 0)) {
         clp[CFG_NETWORK_ON].a_current = 0;
         /* WorldWide PortName Type <type> doesn't conform to IP Profile */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0449,                   /* ptr to msg structure */
                 fc_mes0449,                      /* ptr to msg */
                  fc_msgBlk0449.msgPreambleStr,   /* begin varargs */
                   binfo->fc_sparam.portName.nameType); /* end varargs */
      }

      fc_config_farp(binfo, mb);
      if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
         /*
          * Let it go through even if failed.
          */
         /* Adapter failed to init, mbxCmd <cmd> FARP, mbxStatus <status> */ 
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0450,                   /* ptr to msg structure */
                 fc_mes0450,                      /* ptr to msg */
                  fc_msgBlk0450.msgPreambleStr,   /* begin varargs */
                   mb->mbxCommand,
                    mb->mbxStatus);               /* end varargs */
      }
   }

   if (p_dev_ctl->intr_inited != 1) {
      /* Add our interrupt routine to kernel's interrupt chain & enable it */


      IHS.handler = fc_intr;
     
      if ((i_init((struct intr *) & IHS)) == INTR_FAIL) {
         /* Enable interrupt handler failed */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0451,                   /* ptr to msg structure */
                 fc_mes0451,                      /* ptr to msg */
                  fc_msgBlk0451.msgPreambleStr);  /* begin & end varargs */
         binfo->fc_ffstate = FC_ERROR;
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         fc_free_buffer(p_dev_ctl);
         return(EIO);
      }
      p_dev_ctl->intr_inited = 1;
   }

      fc_disable_tc(binfo, (MAILBOX * )mb);
      if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
         binfo->fc_ffstate = FC_ERROR;
         fc_ffcleanup(p_dev_ctl);
         i_clear(&IHS);
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         fc_free_buffer(p_dev_ctl);
         return(EIO);
   }

   fc_read_config(binfo, (MAILBOX * )mb);
   if (issue_mb_cmd(binfo, mb, MBX_POLL) != MBX_SUCCESS) {
      /* Adapter failed to init, mbxCmd <cmd> READ_CONFIG, mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0453,                   /* ptr to msg structure */
              fc_mes0453,                      /* ptr to msg */
               fc_msgBlk0453.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      fc_ffcleanup(p_dev_ctl);
      i_clear(&IHS);
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }

   if (mb->un.varRdConfig.lmt & LMT_2125_10bit)
      /* HBA is 2G capable */
      binfo->fc_flag |= FC_2G_CAPABLE;

   binfo->fc_ffstate = FC_LINK_DOWN;
   binfo->fc_flag |= FC_LNK_DOWN;

   /* Activate the adapter and allocate all the resources needed for ELS */
   fc_start(p_dev_ctl);

   /* Setup and issue mailbox INITIALIZE LINK command */
   fc_init_link(binfo, mb, clp[CFG_TOPOLOGY].a_current, 
      clp[CFG_LINK_SPEED].a_current);
   if (issue_mb_cmd(binfo, mb, MBX_NOWAIT) != MBX_SUCCESS) {
      /* Adapter failed to init, mbxCmd <cmd> INIT_LINK, mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0454,                   /* ptr to msg structure */
              fc_mes0454,                      /* ptr to msg */
               fc_msgBlk0454.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                 mb->mbxStatus);               /* end varargs */
      binfo->fc_ffstate = FC_ERROR;
      fc_ffcleanup(p_dev_ctl);
      i_clear(&IHS);
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      fc_free_buffer(p_dev_ctl);
      return(EIO);
   }


   /* Enable link attention interrupt */
   ipri = disable_lock(FC_LVL, &CMD_LOCK);
   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   status = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
   status = status | HC_LAINT_ENA;
   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), status);
   FC_UNMAP_MEMIO(ioa);
   binfo->fc_process_LA = 1;
   p_dev_ctl->fc_waitflogi = (FCCLOCK *)1;
   unlock_enable(ipri, &CMD_LOCK);

   fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);

   binfo->fc_prevDID = Mask_DID;
   /* If we are point to point, don't wait for link up */
   if ((clp[CFG_TOPOLOGY].a_current == FLAGS_TOPOLOGY_MODE_PT_PT) && 
       (clp[CFG_FCP_ON].a_current == 0)) {
      goto out;
   }

   flogi_sent = 0;
   i = 0;
   while (binfo->fc_ffstate != FC_READY) {
      /* Check every second for 20 retries. */
      if ((i++ > 20) ||
         ((i >= 10) && (binfo->fc_ffstate <= FC_LINK_DOWN))) {
         /* The link is down, so set linkdown timeout */
         rp = &binfo->fc_ring[FC_FCP_RING];
         RINGTMO = fc_clk_set(p_dev_ctl, rp->fc_ringtmo, fc_linkdown_timeout, 0, 0);
         break;
      }
      ipri = disable_lock(FC_LVL, &CMD_LOCK);
      if((i > 1) && (binfo->fc_ffstate == FC_FLOGI) &&
         (flogi_sent == 0) && (p_dev_ctl->power_up == 0)) {
	 if(p_dev_ctl->fc_waitflogi) {
            if (p_dev_ctl->fc_waitflogi != (FCCLOCK *)1)
              fc_clk_can(p_dev_ctl, p_dev_ctl->fc_waitflogi);
            p_dev_ctl->fc_waitflogi = 0;
         }
         fc_snd_flogi(p_dev_ctl, 0, 0);
         flogi_sent = 1;
         rp = &binfo->fc_ring[FC_ELS_RING];
         if(RINGTMO)
            fc_clk_res(p_dev_ctl, 20, RINGTMO);
      }
      unlock_enable(ipri, &CMD_LOCK);

      DELAYMS(1000);
   }

out:
   ipri = disable_lock(FC_LVL, &CMD_LOCK);
   if((binfo->fc_ffstate == FC_FLOGI) && (p_dev_ctl->power_up == 0)) {
      fc_snd_flogi(p_dev_ctl, 0, 0);
   }
   p_dev_ctl->power_up = 1;
   unlock_enable(ipri, &CMD_LOCK);

   return(0);
}       /* End fc_ffinit */

/************************************************************************/
/*                                                                      */
/*   fc_binfo_init                                                      */
/*   This routine will initialize the binfo structure                   */
/*                                                                      */
/************************************************************************/
_local_ int
fc_binfo_init(
fc_dev_ctl_t    *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   int             idx;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* Initialize configuration parameters */
   if(binfo->fc_flag & FC_ESTABLISH_LINK)
      binfo->fc_flag = FC_ESTABLISH_LINK;
   else
      binfo->fc_flag = 0;          /* don't change nvram or tov */

   binfo->fc_ffnumrings = MAX_CONFIGURED_RINGS - 1; /* number of rings */

   /* Ring 0 - ELS */
   binfo->fc_nummask[0] = 4;

   binfo->fc_rval[0] = FC_ELS_REQ;      /* ELS request */
   binfo->fc_tval[0] = FC_ELS_DATA;     /* ELS */
   binfo->fc_rval[1] = FC_ELS_RSP;      /* ELS response */
   binfo->fc_tval[1] = FC_ELS_DATA;     /* ELS */
   binfo->fc_rval[2] = FC_UNSOL_CTL;    /* NameServer Inquiries */
   binfo->fc_tval[2] = FC_COMMON_TRANSPORT_ULP; /* NameServer */
   binfo->fc_rval[3] = FC_SOL_CTL;              /* NameServer response */
   binfo->fc_tval[3] = FC_COMMON_TRANSPORT_ULP; /* NameServer */
   if (binfo->fc_sli == 2) {
      binfo->fc_ring[0].fc_numCiocb = SLI2_IOCB_CMD_R0_ENTRIES;
      binfo->fc_ring[0].fc_numRiocb = SLI2_IOCB_RSP_R0_ENTRIES;
   } else {
      binfo->fc_ring[0].fc_numCiocb = IOCB_CMD_R0_ENTRIES;
      binfo->fc_ring[0].fc_numRiocb = IOCB_RSP_R0_ENTRIES;
   }

   /* Ring 1 - IP */
   if(clp[CFG_NETWORK_ON].a_current) {
      binfo->fc_nummask[1] = 1;
      idx = 5;
   } else {
      binfo->fc_nummask[1] = 0;
      idx = 4;
   }
   binfo->fc_rval[4] = FC_UNSOL_DATA;   /* Unsolicited Data */
   binfo->fc_tval[4] = FC_LLC_SNAP;     /* LLC/SNAP */
   if (binfo->fc_sli == 2) {
      binfo->fc_ring[1].fc_numCiocb = SLI2_IOCB_CMD_R1_ENTRIES;
      binfo->fc_ring[1].fc_numRiocb = SLI2_IOCB_RSP_R1_ENTRIES;
      if(clp[CFG_NETWORK_ON].a_current == 0) {
         binfo->fc_ring[1].fc_numCiocb -= SLI2_IOCB_CMD_R1XTRA_ENTRIES;
         binfo->fc_ring[1].fc_numRiocb -= SLI2_IOCB_RSP_R1XTRA_ENTRIES;
      }
   } else {
      binfo->fc_ring[1].fc_numCiocb = IOCB_CMD_R1_ENTRIES;
      binfo->fc_ring[1].fc_numRiocb = IOCB_RSP_R1_ENTRIES;
   }

   /* Ring 2 - FCP */
   binfo->fc_nummask[2] = 0;
   if (binfo->fc_sli == 2) {
      binfo->fc_ring[2].fc_numCiocb = SLI2_IOCB_CMD_R2_ENTRIES;
      binfo->fc_ring[2].fc_numRiocb = SLI2_IOCB_RSP_R2_ENTRIES;
      if(clp[CFG_NETWORK_ON].a_current == 0) {
         binfo->fc_ring[2].fc_numCiocb += SLI2_IOCB_CMD_R2XTRA_ENTRIES;
         binfo->fc_ring[2].fc_numRiocb += SLI2_IOCB_RSP_R2XTRA_ENTRIES;
      }
   } else {
      binfo->fc_ring[2].fc_numCiocb = IOCB_CMD_R2_ENTRIES;
      binfo->fc_ring[2].fc_numRiocb = IOCB_RSP_R2_ENTRIES;
   }


   binfo->ipVersion = RNID_IPV4;
   return(0);
}       /* End fc_binfo_init */

/************************************************************************/
/*                                                                      */
/*   fc_parse_vpd                                                       */
/*   This routine will parse the VPD data                               */
/*                                                                      */
/************************************************************************/
_local_ int
fc_parse_vpd(
fc_dev_ctl_t    *p_dev_ctl,
uchar *vpd)
{
   FC_BRD_INFO   * binfo;
   int finished = 0;
   int index = 0;
   uchar lenlo, lenhi;
   unsigned char *Length;
   int i, j;        

   binfo = &BINFO;
   /* Vital Product */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0455,                   /* ptr to msg structure */
           fc_mes0455,                      /* ptr to msg */
            fc_msgBlk0455.msgPreambleStr,   /* begin varargs */
             (uint32)vpd[0],
              (uint32)vpd[1],
               (uint32)vpd[2],
                (uint32)vpd[3]);            /* end varargs */
   do {
      switch (vpd[index]) {
      case 0x82:
         index += 1;
         lenlo = vpd[index];                                
         index += 1;
         lenhi = vpd[index];                                
         index += 1;
         i = ((((unsigned short)lenhi) << 8) + lenlo);
         index += i;
         break;
      case 0x90:
         index += 1;
         lenlo = vpd[index];                                
         index += 1;
         lenhi = vpd[index];                                
         index += 1;
         i = ((((unsigned short)lenhi) << 8) + lenlo);
         do {
             /* Look for Serial Number */
             if ((vpd[index] == 'S') && (vpd[index+1] == 'N')) {
                 index += 2;
                 Length = &vpd[index];
                 index += 1;
                 i = *Length;
                 j = 0;
                 while(i--) {
                     binfo->fc_SerialNumber[j++] = vpd[index++];
                     if(j == 31)
                        break;
                 }
                 binfo->fc_SerialNumber[j] = 0;
                 return(1);
             }
             else {
                 index += 2;
                 Length = &vpd[index];
                 index += 1;
                 j = (int)(*Length);
                 index += j;
                 i -= (3 + j);
             }
         } while (i > 0);
         finished = 0;                
         break;
      case 0x78:
         finished = 1;
         break;
      default:
         return(0);
      }
   } while (!finished);
   return(1);
}

_static_ char     fwrevision[32];

/* TAPE */
_static_ char *
decode_firmware_rev(
FC_BRD_INFO *binfo,
fc_vpd_t  *vp)
{
   uint32       b1, b2, b3, b4, ldata;
   char         c;
   uint32   i, rev;
   uint32   *ptr, str[4];

   if ( vp->rev.rBit ) {
      if (binfo->fc_sli == 2) 
         rev = vp->rev.sli2FwRev;
      else
         rev = vp->rev.sli1FwRev;

      b1 = (rev & 0x0000f000) >> 12;
      b2 = (rev & 0x00000f00) >> 8;
      b3 = (rev & 0x000000c0) >> 6; 
      b4 = (rev & 0x00000030) >> 4;

      switch (b4) {
         case 0:
            c = 'N';
            break;
         case 1:
            c = 'A';
            break;
         case 2:
            c = 'B';
            break;
         case 3:
         default:
            c = 0;
            break;
      }
      b4 = (rev & 0x0000000f);

      if (binfo->fc_sli == 2) {
         for (i=0; i<16; i++) {
            if (vp->rev.sli2FwName[i] == 0x20) {
               vp->rev.sli2FwName[i] = 0;
            }
         }
         ptr = (uint32 *)vp->rev.sli2FwName;
      } else {
         for (i=0; i<16; i++) {
            if (vp->rev.sli1FwName[i] == 0x20) {
               vp->rev.sli1FwName[i] = 0;
            }
         }
         ptr = (uint32 *)vp->rev.sli1FwName;
      }
      for (i=0; i<3; i++) {
         ldata = *ptr++;
         ldata = SWAP_DATA(ldata); 
         str[i] = ldata;
      }

      fwrevision[0] = (char)((int)'0' + b1);
      fwrevision[1] = '.';
      fwrevision[2] = (char)((int)'0' + b2);
      fwrevision[3] = (char)((int)'0' + b3);
      if(c) {
         fwrevision[4] = c;
         fwrevision[5] = (char)((int)'0' + b4);
         fwrevision[6] = 0;
      }
      else {
         fwrevision[4] = 0;
      }
   } else {
      rev = vp->rev.smFwRev;

      b1 = (rev & 0xff000000) >> 24;
      b2 = (rev & 0x00f00000) >> 20;
      b3 = (rev & 0x000f0000) >> 16;
      c =  (char)((rev & 0x0000ff00) >> 8);
      b4 = (rev & 0x000000ff);

      fwrevision[0] = (char)((int)'0' + b1);
      fwrevision[1] = '.';
      fwrevision[2] = (char)((int)'0' + b2);
      fwrevision[3] = (char)((int)'0' + b3);
      fwrevision[4] = c;
      fwrevision[5] = (char)((int)'0' + b4);
      fwrevision[6] = 0;
   }
   return(fwrevision);
}       /* End decode_firmware_rev */


/*****************************************************************************/
/*
 * NAME:     fc_intr
 *
 * FUNCTION: Fibre Channel driver interrupt routine.
 *
 * EXECUTION ENVIRONMENT: interrupt only
 *
 * CALLED FROM:
 *      The FLIH
 *
 * INPUT:
 *      p_ihs           - point to the interrupt structure.
 *
 * RETURNS:  
 *      INTR_SUCC - our interrupt
 *      INTR_FAIL - not our interrupt
 */
/*****************************************************************************/
_static_ int
fc_intr(
struct intr *p_ihs)     /* This also points to device control area */
{
   fc_dev_ctl_t * p_dev_ctl = (fc_dev_ctl_t * )p_ihs;
   volatile uint32      ha_copy;
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   fcipbuf_t     * mbp;
   MAILBOXQ      * mb;
   IOCBQ         * delayiocb;
   IOCBQ         * temp;
   IOCBQ         * processiocb;
   IOCBQ         * endiocb;
   void          * ioa;
   int  ipri, rc;

   binfo = &BINFO;

   ipri = disable_lock(FC_LVL, &CMD_LOCK);
   binfo->fc_flag |= FC_INTR_THREAD;

   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */

   /* Read host attention register to determine interrupt source */
   ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));

   /* Clear Attention Sources, except ERROR (to preserve status) & LATT */
   WRITE_CSR_REG(binfo, FC_HA_REG(binfo, ioa),
       (ha_copy & ~HA_ERATT & ~HA_LATT));

   FC_UNMAP_MEMIO(ioa);


   if (ha_copy) {
      rc = INTR_SUCC;
      binfo->fc_flag |= FC_INTR_WORK;
   } else {
      clp = DD_CTL.p_config[binfo->fc_brd_no];
      if (clp[CFG_INTR_ACK].a_current && (binfo->fc_flag&FC_INTR_WORK)) {
         rc = INTR_SUCC;  /* Just claim the first non-working interrupt */
         binfo->fc_flag &= ~FC_INTR_WORK;
      } else {
         if (clp[CFG_INTR_ACK].a_current == 2)
            rc = INTR_SUCC;  /* Always claim the interrupt */
         else
            rc = INTR_FAIL;
      }
   }

   if (binfo->fc_flag & FC_OFFLINE_MODE) {
      binfo->fc_flag &= ~FC_INTR_THREAD;
      unlock_enable(ipri, &CMD_LOCK);
      return(INTR_FAIL);
   }

   processiocb = 0;
   if(binfo->fc_delayxmit) {
      delayiocb = binfo->fc_delayxmit;
      binfo->fc_delayxmit = 0;
      endiocb = 0;
      while(delayiocb) {
         temp = delayiocb;
         delayiocb = (IOCBQ *)temp->q;
         temp->rsvd2--;
         /* If retry == 0, process IOCB */
         if(temp->rsvd2 == 0) {
            if(processiocb == 0) {
               processiocb = temp;
            }
            else {
               endiocb->q = (uchar *)temp;
            }
            endiocb = temp;
            temp->q = 0;
         }
         else {
            /* Make delayxmit point to first non-zero retry */
            if(binfo->fc_delayxmit == 0)
               binfo->fc_delayxmit = temp;
         }
      }
      if(processiocb) {
         /* Handle any delayed IOCBs */
         endiocb = processiocb;
         while(endiocb) {
            temp = endiocb;
            endiocb = (IOCBQ *)temp->q;
            temp->q = 0;
            issue_iocb_cmd(binfo, &binfo->fc_ring[FC_ELS_RING], temp);
         }
      }
   }


   if (ha_copy & HA_ERATT) {     /* Link / board error */
      unlock_enable(ipri, &CMD_LOCK);
      handle_ff_error(p_dev_ctl);
      return (rc);
   } else {
      if (ha_copy & HA_MBATT) { /* Mailbox interrupt */
         handle_mb_event(p_dev_ctl);
         if(binfo->fc_flag & FC_PENDING_RING0) {
            binfo->fc_flag &= ~FC_PENDING_RING0;
            ha_copy |= HA_R0ATT; /* event on ring 0 */
         }
      }

      if (ha_copy & HA_LATT) {   /* Link Attention interrupt */
         if (binfo->fc_process_LA) {
            handle_link_event(p_dev_ctl);
         }
      }

      if (ha_copy & HA_R0ATT) { /* event on ring 0 */
         if(binfo->fc_mbox_active == 0)
            handle_ring_event(p_dev_ctl, 0, (ha_copy & 0x0000000F));
         else
            binfo->fc_flag |= FC_PENDING_RING0;
      }

      if (ha_copy & HA_R1ATT) { /* event on ring 1 */
         /* This ring handles IP. Defer processing anything on this ring
        * till all FCP ELS traffic settles down.
        */
         if (binfo->fc_ffstate <= FC_NODE_DISC)
            binfo->fc_deferip |= (uchar)((ha_copy >> 4) & 0x0000000F);
         else
            handle_ring_event(p_dev_ctl, 1, ((ha_copy >> 4) & 0x0000000F));
      }

      if (ha_copy & HA_R2ATT) { /* event on ring 2 */
         handle_ring_event(p_dev_ctl, 2, ((ha_copy >> 8) & 0x0000000F));
      }

      if (ha_copy & HA_R3ATT) { /* event on ring 3 */
         handle_ring_event(p_dev_ctl, 3, ((ha_copy >> 12) & 0x0000000F));
      }
   }

   if((processiocb == 0) && (binfo->fc_delayxmit) &&
      (binfo->fc_mbox_active == 0)) {
      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
         fc_read_rpi(binfo, (uint32)1, (MAILBOX * )mb, (uint32)0);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }
   }

   binfo->fc_flag &= ~FC_INTR_THREAD;

   while (p_dev_ctl->mbufl_head != 0) {
      binfo->fc_flag |= FC_INTR_WORK;
      mbp = (fcipbuf_t * )p_dev_ctl->mbufl_head;
      p_dev_ctl->mbufl_head = (uchar * )fcnextpkt(mbp);
      fcnextpkt(mbp) = 0;
      fc_xmit(p_dev_ctl, mbp);
   }
   p_dev_ctl->mbufl_tail = 0;


   unlock_enable(ipri, &CMD_LOCK);
   return(rc);
}       /* End fc_intr */



/**************************************************/
/**  handle_ff_error                             **/
/**                                              **/
/**    Runs at Interrupt level                   **/
/**                                              **/
/**************************************************/
_static_ void
handle_ff_error(
fc_dev_ctl_t *p_dev_ctl)
{
   volatile uint32 status, status1, status2;
   void *ioa;
   FC_BRD_INFO * binfo;
   iCfgParam     * clp;
   int  ipri;

   ipri = disable_lock(FC_LVL, &CMD_LOCK);

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   status =  p_dev_ctl->dpc_hstatus;
   p_dev_ctl->dpc_hstatus = 0;

   ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
   status1 = READ_SLIM_ADDR(binfo, ((volatile uchar * )ioa + 0xa8));
   status2 = READ_SLIM_ADDR(binfo, ((volatile uchar * )ioa + 0xac));
   FC_UNMAP_MEMIO(ioa);


   if (status & HS_FFER6) {


       /* Re-establishing Link */
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk1301,                   /* ptr to msg structure */
               fc_mes1301,                      /* ptr to msg */
                fc_msgBlk1301.msgPreambleStr,   /* begin varargs */
                 status,
                  status1,
                   status2);                    /* end varargs */
      binfo->fc_flag |= FC_ESTABLISH_LINK;
      fc_cfg_remove(p_dev_ctl);

      binfo->fc_flag |= FC_OFFLINE_MODE;

      lpfc_cfg_init(p_dev_ctl);

      unlock_enable(ipri, &CMD_LOCK);
   } else {
     /* Adapter Hardware Error */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0457,                   /* ptr to msg structure */
             fc_mes0457,                      /* ptr to msg */
              fc_msgBlk0457.msgPreambleStr,   /* begin varargs */
               status,
                status1,
                 status2);                    /* end varargs */
     if (status & HS_FFER8) {             /* Chipset error 8 */
     } else if (status & HS_FFER7) {      /* Chipset error 7 */
     } else if (status & HS_FFER5) {      /* Chipset error 5 */
     } else if (status & HS_FFER4) {      /* Chipset error 4 */
     } else if (status & HS_FFER3) {      /* Chipset error 3 */
     } else if (status & HS_FFER2) {      /* Chipset error 2 */
     } else if (status & HS_FFER1) {      /* Chipset error 1 */
     }

     fc_free_rpilist(p_dev_ctl, 0);

     p_dev_ctl->device_state = DEAD;
     binfo->fc_ffstate = FC_ERROR;
     unlock_enable(ipri, &CMD_LOCK);
   }

}       /* End handle_ff_error */


/**************************************************/
/**  handle_link_event                           **/
/**                                              **/
/**    Description: Process a Link Attention.    **/
/**                                              **/
/**************************************************/
_static_ void
handle_link_event(
fc_dev_ctl_t *p_dev_ctl)
{
   /* called from host_interrupt, to process LATT */
   MAILBOX       * mb;
   FC_BRD_INFO * binfo;
   void *ioa;
   volatile uint32 control;

   binfo = &BINFO;
   FCSTATCTR.linkEvent++;

   /* Get a buffer which will be used for mailbox commands */
   if ((mb = (MAILBOX * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
      if (fc_read_la(p_dev_ctl, mb) == 0) {
         if (issue_mb_cmd(binfo, mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
         /* Turn off Link Attention interrupts until CLEAR_LA done */
         binfo->fc_process_LA = 0;
         ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
         control = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
         control &= ~HC_LAINT_ENA;
         WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), control);
         /* Clear Link Attention in HA REG */
         WRITE_CSR_REG(binfo, FC_HA_REG(binfo, ioa),
             (volatile uint32)(HA_LATT));
         FC_UNMAP_MEMIO(ioa);
      }
      else {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }
}       /* End handle_link_event */


/**************************************************/
/**  handle_ring_event                           **/
/**                                              **/
/**    Description: Process a Ring Attention.    **/
/**                                              **/
/**************************************************/
_static_ void
handle_ring_event(
fc_dev_ctl_t    *p_dev_ctl,
int ring_no,
uint32          reg_mask)
{
   FC_BRD_INFO   * binfo;
   RING          * rp;
   IOCB          * entry;
   IOCBQ         * saveq;
   IOCBQ         * temp;
   void          * ioa;
   int  fcpfound = 0;
   uint32        * xx;
   uint32          portGet;
   volatile uint32 chipatt;
   uint32       portRspPut;

   binfo = &BINFO;
   /* called from host_interrupt() to process RxATT */

   rp = &binfo->fc_ring[ring_no];
   temp = NULL;
   fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

   /* Gather iocb entries off response ring.
    * Ensure entry is owned by the host.
    */
   entry = (IOCB * )IOCB_ENTRY(rp->fc_rspringaddr, rp->fc_rspidx);
   portRspPut = PCIMEM_LONG(((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.port[ring_no].rspPutInx);
   if (portRspPut >= rp->fc_numRiocb) {
      return;
   }

   while (rp->fc_rspidx != portRspPut) {
      if((ring_no == 0) && (binfo->fc_mbox_active)) {
         binfo->fc_flag |= FC_PENDING_RING0;
         break;
      }
      /* get an iocb buffer to copy entry into */
      if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) == NULL) {
         break;
      }


      fc_pcimem_bcopy((uint32 * )entry, (uint32 * ) & temp->iocb, sizeof(IOCB));
      temp->q = NULL;

      /* bump iocb available response index */
      if (++rp->fc_rspidx >= rp->fc_numRiocb) {
         rp->fc_rspidx = 0;
      }

      /* SLIM POINTER */
   if (binfo->fc_busflag & FC_HOSTPTR) {
        ((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.host[ring_no].rspGetInx = 
          PCIMEM_LONG(rp->fc_rspidx);
      } else {
        void          * ioa2;

        ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
        ioa2 = (void *)((char *)ioa + ((SLIMOFF+(ring_no*2)+1)*4));
        WRITE_SLIM_ADDR(binfo, (volatile uint32 *)ioa2, rp->fc_rspidx);
        FC_UNMAP_MEMIO(ioa);
      }

      /* chain all iocb entries until LE is set */
      if (rp->fc_iocbhd == NULL) {
         rp->fc_iocbhd = temp;
         rp->fc_iocbtl = temp;
      } else {
         rp->fc_iocbtl->q = (uchar * )temp;
         rp->fc_iocbtl = temp;
      }

      /* when LE is set, entire Command has been received */
      if (temp->iocb.ulpLe) {
         saveq = rp->fc_iocbhd;

         rp->fc_iocbhd = NULL;
         rp->fc_iocbtl = NULL;

         /* get a ptr to first iocb entry in chain and process it */
         xx = (uint32 * ) & saveq->iocb;
         fcpfound = fc_proc_ring_event(p_dev_ctl, rp, saveq);

         /* Free up iocb buffer chain for command just processed */
         while (saveq) {
            temp = saveq;
            saveq = (IOCBQ * )temp->q;
            fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         }

      } /* Entire Command has been received */

      entry = (IOCB * )IOCB_ENTRY(rp->fc_rspringaddr, rp->fc_rspidx);

   } /* While(entry->ulpOwner == 0) */

   if ((temp != NULL) && (reg_mask & HA_R0RE_REQ)) {
      /* At least one response entry has been freed */
      FCSTATCTR.chipRingFree++;
      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      /* SET R0RE_RSP in Chip Att register */
      chipatt = ((CA_R0ATT | CA_R0RE_RSP) << (ring_no * 4));
      WRITE_CSR_REG(binfo, FC_FF_REG(binfo, ioa), chipatt);
      FC_UNMAP_MEMIO(ioa);
   }

   if (reg_mask != 0xffffffff) {
      if (fcpfound) {
         fc_issue_cmd(p_dev_ctl);
      } else if (reg_mask & HA_R0CE_RSP) {
         FCSTATCTR.hostRingFree++;
         /* Cmd ring is available, queue any available cmds */
         portGet = issue_iocb_cmd(binfo, rp, 0);
         if(portGet != PCIMEM_LONG(((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.port[rp->fc_ringno].cmdGetInx)) {
            issue_iocb_cmd(binfo, rp, 0);
         }
      }
      FCSTATCTR.ringEvent++;
   }

   return;
}       /* End handle_ring_event */

_static_ int
fc_proc_ring_event(
fc_dev_ctl_t    *p_dev_ctl,
RING            *rp,
IOCBQ           *saveq)
{
   FC_BRD_INFO   * binfo;
   NODELIST      * ndlp;
   IOCB          * cmd;
   int             rc;

   binfo = &BINFO;
   cmd = &saveq->iocb;
   rc = 0;
   FCSTATCTR.iocbRsp++;

   switch (cmd->ulpCommand) {
   case CMD_FCP_ICMND_CR:
   case CMD_FCP_ICMND_CX:
   case CMD_FCP_IREAD_CR:
   case CMD_FCP_IREAD_CX:
   case CMD_FCP_IWRITE_CR:
   case CMD_FCP_IWRITE_CX:
   case CMD_FCP_ICMND64_CR:
   case CMD_FCP_ICMND64_CX:
   case CMD_FCP_IREAD64_CR:
   case CMD_FCP_IREAD64_CX:
   case CMD_FCP_IWRITE64_CR:
   case CMD_FCP_IWRITE64_CX:
      handle_fcp_event(p_dev_ctl, rp, saveq);
      rc = 1;
      break;

   case CMD_RCV_SEQUENCE_CX:            /* received incoming frame */
   case CMD_RCV_SEQUENCE64_CX:          /* received incoming frame */
      switch(rp->fc_ringno) {
      case FC_ELS_RING: 
         handle_elsrcv_seq(p_dev_ctl, rp, saveq);
         break;
      case FC_IP_RING: 
         handle_iprcv_seq(p_dev_ctl, rp, saveq);
         break;
      }
      break;

   case CMD_XMIT_BCAST_CN:              /* process xmit completion */
   case CMD_XMIT_BCAST_CX:
   case CMD_XMIT_SEQUENCE_CX:
   case CMD_XMIT_SEQUENCE_CR:
   case CMD_XMIT_BCAST64_CN:            /* process xmit completion */
   case CMD_XMIT_BCAST64_CX:
   case CMD_XMIT_SEQUENCE64_CX:
   case CMD_XMIT_SEQUENCE64_CR:
      handle_xmit_cmpl(p_dev_ctl, rp, saveq);
      break;

   case CMD_RCV_ELS_REQ_CX:             /* received an els frame */
   case CMD_RCV_ELS_REQ64_CX:           /* received an els frame */
      handle_rcv_els_req(p_dev_ctl, rp, saveq);
      break;

   case CMD_CREATE_XRI_CR:
   case CMD_CREATE_XRI_CX:
      handle_create_xri(p_dev_ctl, rp, saveq);
      break;

   case CMD_ELS_REQUEST_CR:             /* xmit els frame completion */
   case CMD_ELS_REQUEST_CX:
   case CMD_XMIT_ELS_RSP_CX:
   case CMD_ELS_REQUEST64_CR:
   case CMD_ELS_REQUEST64_CX:
   case CMD_XMIT_ELS_RSP64_CX:
   case CMD_GEN_REQUEST64_CR:
   case CMD_GEN_REQUEST64_CX:
      handle_els_event(p_dev_ctl, rp, saveq);
      break;

   case CMD_ABORT_XRI_CN:               /* Abort fcp command */
      break;

   case CMD_ABORT_XRI_CX:               /* Abort command */
      break;

   case CMD_XRI_ABORTED_CX:            /* Handle ABORT condition */
      /*
       * If we find an NODELIST entry that matches the aborted
       * XRI, clear out the Xri field.
       */
      if (((ndlp = fc_findnode_oxri(binfo, NLP_SEARCH_UNMAPPED | NLP_SEARCH_MAPPED,
         cmd->ulpContext)) != NULL) && !(ndlp->nlp_flag & NLP_RPI_XRI)) {
         ndlp->nlp_Xri = 0;  /* xri */
         /* establish a new exchange */
         if ((ndlp->nlp_Rpi) && 
             ((ndlp->nlp_DID & CT_DID_MASK) != CT_DID_MASK) && 
             (binfo->fc_ffstate == FC_READY)) {
            ndlp->nlp_flag |= NLP_RPI_XRI;
            fc_create_xri(binfo, &binfo->fc_ring[FC_ELS_RING], ndlp);
         }
      }
      break;

   case CMD_ADAPTER_MSG:
      if ((binfo->fc_msgidx + MAX_MSG_DATA) <= FC_MAX_ADPTMSG) {
         fc_bcopy((uchar * )cmd, &binfo->fc_adaptermsg[binfo->fc_msgidx],
             MAX_MSG_DATA);
         binfo->fc_msgidx += MAX_MSG_DATA;
         con_print("lpfc%d: %s", binfo->fc_brd_no, binfo->fc_adaptermsg);
         fc_bzero((void *)binfo->fc_adaptermsg, FC_MAX_ADPTMSG);
         binfo->fc_msgidx = 0;
      } else {
         con_print("lpfc%d: %s\n", binfo->fc_brd_no, binfo->fc_adaptermsg);
         fc_bzero(binfo->fc_adaptermsg, FC_MAX_ADPTMSG);
         binfo->fc_msgidx = 0;
      }
      break;


   default:
      /* Unknown IOCB command */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1400,                   /* ptr to msg structure */
              fc_mes1400,                      /* ptr to msg */
               fc_msgBlk1400.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand,
                 cmd->ulpStatus,
                  cmd->ulpIoTag,
                   cmd->ulpContext);           /* end varargs */
      break;
   } /* switch(cmd->ulpCommand) */

   return(rc);
}       /* End fc_proc_ring_event */


/**************************************************/
/**  handle_mb_event                             **/
/**                                              **/
/**  Description: Process a Mailbox Attention.   **/
/**  Called from host_interrupt to process MBATT **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
handle_mb_event(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   MAILBOX * mb;
   MAILBOX       * swpmb;
   MAILBOXQ      * mbox;
   IOCBQ         * iocbq;
   NODELIST      * ndlp;
   void *ioa;
   uint32            control;
   volatile uint32   word0;
   volatile uint32   ldata;
   volatile uint32   ldid;
   volatile uint32   lrpi;
   iCfgParam     * clp;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   if (binfo->fc_flag & FC_SLI2) {
      fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);
      /* First copy command data */
      mb = FC_SLI2_MAILBOX(binfo);
      word0 = *((volatile uint32 * )mb);
      word0 = PCIMEM_LONG(word0);
   } else {
      /* First copy command data */
      ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
      mb = FC_MAILBOX(binfo, ioa);
      word0 = READ_SLIM_ADDR(binfo, ((volatile uint32 * )mb));
      FC_UNMAP_MEMIO(ioa);
   }

   swpmb = (MAILBOX * ) & word0;

   FCSTATCTR.mboxEvent++;

   /* Sanity check to ensure the host owns the mailbox */
   if (swpmb->mbxOwner != OWN_HOST) {
      int i;

      for(i=0; i<10240;i++) {
         if (binfo->fc_flag & FC_SLI2) {
            fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);
            /* First copy command data */
            mb = FC_SLI2_MAILBOX(binfo);
            word0 = *((volatile uint32 * )mb);
            word0 = PCIMEM_LONG(word0);
         } else {
            /* First copy command data */
            ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
            mb = FC_MAILBOX(binfo, ioa);
            word0 = READ_SLIM_ADDR(binfo, ((volatile uint32 * )mb));
            FC_UNMAP_MEMIO(ioa);
         }
      
         swpmb = (MAILBOX * ) & word0;
         if (swpmb->mbxOwner == OWN_HOST)
            goto out;
      }
      /* Stray Mailbox Interrupt, mbxCommand <cmd> mbxStatus <status> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0304,                   /* ptr to msg structure */
              fc_mes0304,                      /* ptr to msg */
               fc_msgBlk0304.msgPreambleStr,   /* begin varargs */
                swpmb->mbxCommand,
                 swpmb->mbxStatus);            /* end varargs */
      return(1);
   }

out:

   /* stop watchdog timer */
   if(MBOXTMO) {
      fc_clk_can(p_dev_ctl, MBOXTMO);
      MBOXTMO = 0;
   }

   if (swpmb->mbxStatus) {
      if (swpmb->mbxStatus == MBXERR_NO_RESOURCES) {
         FCSTATCTR.mboxStatErr++;
         /* Mbox cmd cmpl error - RETRYing */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0305,                   /* ptr to msg structure */
                 fc_mes0305,                      /* ptr to msg */
                  fc_msgBlk0305.msgPreambleStr,   /* begin varargs */
                   swpmb->mbxCommand,
                    word0,
                     binfo->fc_ffstate,
                      binfo->fc_flag);            /* end varargs */
         if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
            if (binfo->fc_flag & FC_SLI2) {
               /* First copy mbox command data */
               mb = FC_SLI2_MAILBOX(binfo);
               fc_pcimem_bcopy((uint32 * )mb, (uint32 * )mbox,
                   (sizeof(uint32) * (MAILBOX_CMD_WSIZE)));
            } else {
               /* First copy mbox command data */
               ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
               mb = FC_MAILBOX(binfo, ioa);
               READ_SLIM_COPY(binfo, (uint32 *)mbox, (uint32 *)mb,
                  MAILBOX_CMD_WSIZE);
               FC_UNMAP_MEMIO(ioa);
            }
            switch(((MAILBOX *)mbox)->mbxCommand) {
            case MBX_READ_SPARM:
               control = ((MAILBOX *)mbox)->un.varRdSparm.un.sp.bdeSize;
               if(control == 0) {
                  fc_read_sparam(p_dev_ctl, (MAILBOX *)mbox);
               }
            case MBX_READ_SPARM64:
               control = ((MAILBOX *)mbox)->un.varRdSparm.un.sp64.tus.f.bdeSize;
               if(control == 0) {
                  fc_read_sparam(p_dev_ctl, (MAILBOX *)mbox);
               }
            case MBX_REG_LOGIN:
               control = ((MAILBOX *)mbox)->un.varRegLogin.un.sp.bdeSize;
               if(control == 0) {
                  goto mbout;
               }
            case MBX_REG_LOGIN64:
               control = ((MAILBOX *)mbox)->un.varRegLogin.un.sp64.tus.f.bdeSize;
               if(control == 0) {
                  goto mbout;
               }
            case MBX_READ_LA:
               control = ((MAILBOX *)mbox)->un.varReadLA.un.lilpBde.bdeSize;
               if(control == 0) {
                  fc_read_la(p_dev_ctl, (MAILBOX *)mbox);
               }
            case MBX_READ_LA64:
               control = ((MAILBOX *)mbox)->un.varReadLA.un.lilpBde64.tus.f.bdeSize;
               if(control == 0) {
                  fc_read_la(p_dev_ctl, (MAILBOX *)mbox);
               }
            }
            ((MAILBOX *)mbox)->mbxOwner = OWN_HOST;
            ((MAILBOX *)mbox)->mbxStatus = 0;
            mbox->bp = (uchar * )binfo->fc_mbbp;
            binfo->fc_mbox_active = 0;
            if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
            }
            return(0);
         }
      }
      if (!((swpmb->mbxCommand == MBX_CLEAR_LA) &&
           (swpmb->mbxStatus == 0x1601))) {
         /* Mbox cmd cmpl error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0306,                   /* ptr to msg structure */
                 fc_mes0306,                      /* ptr to msg */
                  fc_msgBlk0306.msgPreambleStr,   /* begin varargs */
                   swpmb->mbxCommand,
                    word0,
                     binfo->fc_ffstate,
                      binfo->fc_flag);            /* end varargs */
         FCSTATCTR.mboxStatErr++;
         switch (swpmb->mbxCommand) {
         case MBX_REG_LOGIN:
         case MBX_REG_LOGIN64:
            if (binfo->fc_flag & FC_SLI2) {
               /* First copy command data */
               mb = FC_SLI2_MAILBOX(binfo);
               ldata = mb->un.varWords[1]; /* get did */
               ldata = PCIMEM_LONG(ldata);
            } else {
               /* First copy command data */
               ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
               mb = FC_MAILBOX(binfo, ioa);
               ldata = READ_SLIM_ADDR(binfo, &mb->un.varWords[1]);
               FC_UNMAP_MEMIO(ioa);
            }
   
            ldid = ldata & Mask_DID;
            if ((ndlp=fc_findnode_odid(binfo,(NLP_SEARCH_MAPPED | NLP_SEARCH_UNMAPPED), ldid))) {
               if (ndlp->nlp_action & NLP_DO_DISC_START) {
                  /* Goto next entry */
                  fc_nextnode(p_dev_ctl, ndlp);
               }
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
            break;
   
         case MBX_UNREG_LOGIN:
            if (binfo->fc_flag & FC_SLI2) {
               /* First copy command data */
               mb = FC_SLI2_MAILBOX(binfo);
               ldata = mb->un.varWords[0]; /* get rpi */
               ldata = PCIMEM_LONG(ldata);
            } else {
               /* First copy command data */
               ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
               mb = FC_MAILBOX(binfo, ioa);
               ldata = READ_SLIM_ADDR(binfo, &mb->un.varWords[0]);
               FC_UNMAP_MEMIO(ioa);
            }
   
            lrpi = ldata & 0xffff;

            if ((ndlp = fc_findnode_rpi(binfo, lrpi)) == 0)
               break;
            binfo->fc_nlplookup[ndlp->nlp_Rpi] = 0;
            ndlp->nlp_Rpi = 0;
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;

         case MBX_READ_LA:
         case MBX_READ_LA64:
         case MBX_CLEAR_LA:
            /* Turn on Link Attention interrupts */
            binfo->fc_process_LA = 1;

            ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
            control = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
            control |= HC_LAINT_ENA;
            WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), control);
            FC_UNMAP_MEMIO(ioa);
            break;
         
         case MBX_INIT_LINK:
            if (binfo->fc_flag & FC_SLI2) {
               if ((clp[CFG_LINK_SPEED].a_current > 0) && 
                  ((swpmb->mbxStatus == 0x0011) || (swpmb->mbxStatus == 0x0500))) {
                  /* Reset link speed to auto. 1G node detected in loop. */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk1302,                     /* ptr to msg structure */
                          fc_mes1302,                        /* ptr to msg */
                           fc_msgBlk1302.msgPreambleStr);    /* begin & end varargs */
                  clp[CFG_LINK_SPEED].a_current = LINK_SPEED_AUTO;
                  if ((mbox = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
                     /* First copy mbox command data */
                     mb = FC_SLI2_MAILBOX(binfo);
                     fc_pcimem_bcopy((uint32 * )mb, (uint32 * )mbox,
                        (sizeof(uint32) * (MAILBOX_CMD_WSIZE)));
                     ((MAILBOX *)mbox)->un.varInitLnk.link_flags &= ~FLAGS_LINK_SPEED;
                     ((MAILBOX *)mbox)->un.varInitLnk.link_speed = 0; /* LINK_SPEED_AUTO */
                     ((MAILBOX *)mbox)->mbxOwner = OWN_HOST;
                     ((MAILBOX *)mbox)->mbxStatus = 0;
                     mbox->bp = (uchar * )binfo->fc_mbbp;
                     binfo->fc_mbox_active = 0;
                     if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
                     }
                     return(0);
                  }
               }
            }
            break;
         }
         if (binfo->fc_mbbp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )binfo->fc_mbbp);
            binfo->fc_mbbp = 0;
         }
         goto mbout;
      }
   }

   /* Mbox cmd cmpl */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0307,                   /* ptr to msg structure */
           fc_mes0307,                      /* ptr to msg */
            fc_msgBlk0307.msgPreambleStr,   /* begin varargs */
             swpmb->mbxCommand,
              word0,
               binfo->fc_ffstate,
                binfo->fc_flag);            /* end varargs */

   if(binfo->fc_mbox_active == 2) {
      MAILBOX *mbslim;

      /* command was issued by dfc layer, so save mbox cmpl */
      if ((binfo->fc_flag & FC_SLI2) && (!(binfo->fc_flag & FC_OFFLINE_MODE))) {
         /* First copy command data */
         mbslim = FC_SLI2_MAILBOX(binfo);
         /* copy results back to user */
         fc_pcimem_bcopy((uint32 * )mbslim, (uint32 * )&p_dev_ctl->dfcmb,
             (sizeof(uint32) * MAILBOX_CMD_WSIZE));
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mbslim = FC_MAILBOX(binfo, ioa);
         /* copy results back to user */
         READ_SLIM_COPY(binfo, (uint32 * )&p_dev_ctl->dfcmb, (uint32 * )mbslim,
            MAILBOX_CMD_WSIZE);
         FC_UNMAP_MEMIO(ioa);
      }
   }
   else {
        handle_mb_cmd(p_dev_ctl, swpmb, (uint32)swpmb->mbxCommand);
   }


mbout:
   /* Process next mailbox command if there is one */
   binfo->fc_mbox_active = 0;
   if ((mbox = fc_mbox_get(binfo))) {
      if (issue_mb_cmd(binfo, (MAILBOX * )mbox, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mbox);
      }
   } else {
      if (binfo->fc_flag & FC_DELAY_PLOGI) {
         binfo->fc_flag &= ~FC_DELAY_PLOGI;
         if((binfo->fc_flag & FC_RSCN_MODE) && (binfo->fc_ffstate == FC_READY))
            fc_nextrscn(p_dev_ctl, fc_max_els_sent);
         else
            fc_nextdisc(p_dev_ctl, fc_max_els_sent);
      }
      if (binfo->fc_flag & FC_DELAY_NSLOGI) {
         if ((iocbq = fc_plogi_get(binfo))) {
            fc_els_cmd(binfo, ELS_CMD_PLOGI,
             (void *)((ulong)iocbq->iocb.un.elsreq.remoteID),
             (uint32)0, (ushort)0, (NODELIST *)0);
            fc_mem_put(binfo, MEM_IOCB, (uchar * )iocbq);
         }
         else {
            binfo->fc_flag &= ~FC_DELAY_NSLOGI;
         }
      }
      if (binfo->fc_flag & FC_DELAY_RSCN) {
         IOCBQ *temp;
         IOCB *iocb;
         MATCHMAP *mp;
         RING     *rp;
         int  i;

         rp = &binfo->fc_ring[FC_ELS_RING];
         binfo->fc_flag &= ~FC_DELAY_RSCN;
         while (binfo->fc_rscn.q_first) {
            temp = (IOCBQ * )binfo->fc_rscn.q_first;
            if ((binfo->fc_rscn.q_first = temp->q) == 0) {
               binfo->fc_rscn.q_last = 0;
            }
            binfo->fc_rscn.q_cnt--;
            iocb = &temp->iocb;
            mp = *((MATCHMAP **)iocb);
            *((MATCHMAP **)iocb) = 0;
            temp->q = NULL;
            fc_process_rscn(p_dev_ctl, temp, mp);

            fc_mem_put(binfo, MEM_BUF, (uchar * )mp);

            i = 1;
            /* free resources associated with this iocb and repost the ring buffers */
            if (!(binfo->fc_flag & FC_SLI2)) {
               for (i = 1; i < (int)iocb->ulpBdeCount; i++) {
                  mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[i].bdeAddress));
                  if (mp) {
                     fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
                  }
               }
            }
            fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         }
      }
   }
   return(0);
}       /* End handle_mb_event */


/**********************************************************/
/** issue_mb_cmd  Issue a mailbox command.               **/
/**               If the mailbox is currently busy,      **/
/**               queue command to mbox queue.           **/
/**********************************************************/
_static_ int
issue_mb_cmd(
FC_BRD_INFO *binfo,
MAILBOX     *mb,
int flag)
{
   MAILBOX       * mbox;
   MAILBOXQ      * mbq;
   int  i;
   void *ioa;
   uint32       status, evtctr;
   uint32       ha_copy;
   fc_dev_ctl_t    *p_dev_ctl;
   volatile uint32 word0, ldata;

   mbq = (MAILBOXQ * )mb;
   status = MBX_SUCCESS;

   if (binfo->fc_mbox_active) {
      /* Another mailbox command is still being processed, queue this
       * command to be processed later.
       */
      fc_mbox_put(binfo, mbq);

      /* Mbox cmd issue - BUSY */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0308,                   /* ptr to msg structure */
              fc_mes0308,                      /* ptr to msg */
               fc_msgBlk0308.msgPreambleStr,   /* begin varargs */
                mb->mbxCommand,
                  binfo->fc_ffstate,
                   binfo->fc_flag,
                    flag);                     /* end varargs */
      FCSTATCTR.mboxCmdBusy++;

      return(MBX_BUSY);
   }

   binfo->fc_mbox_active = 1;
   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);

   /* Mailbox cmd <cmd> issue */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0309,                   /* ptr to msg structure */
           fc_mes0309,                      /* ptr to msg */
            fc_msgBlk0309.msgPreambleStr,   /* begin varargs */
             mb->mbxCommand,
               binfo->fc_ffstate,
                binfo->fc_flag,
                 flag);                     /* end varargs */
   /* If we are not polling, turn on watchdog timer */
   if (flag != MBX_POLL) {
      MBOXTMO = fc_clk_set(p_dev_ctl, MBOX_TMO_DFT, fc_mbox_timeout, 0, 0);
   }

   FCSTATCTR.issueMboxCmd++;
   evtctr = FCSTATCTR.mboxEvent;

   /* if there is one, save buffer to release in completion */
   if (mbq->bp) {
      binfo->fc_mbbp = mbq->bp;
      mbq->bp = 0;
   }

   /* next set own bit for the adapter and copy over command word */
   mb->mbxOwner = OWN_CHIP;

   if (binfo->fc_flag & FC_SLI2) {
      /* First copy command data */
      mbox = FC_SLI2_MAILBOX(binfo);
      fc_pcimem_bcopy((uint32 * )mb, (uint32 * )mbox,
          (sizeof(uint32) * (MAILBOX_CMD_WSIZE)));
      fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);
   } else {
      if (mb->mbxCommand == MBX_CONFIG_PORT) {
         /* copy command data into host mbox for cmpl */
         fc_pcimem_bcopy((uint32 * )mb,
            (uint32 * ) & ((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx,
            (sizeof(uint32) * (MAILBOX_CMD_WSIZE)));
      }

      /* First copy command data */
      ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */

      mbox = FC_MAILBOX(binfo, ioa);
      WRITE_SLIM_COPY(binfo, &mb->un.varWords, &mbox->un.varWords,
          (MAILBOX_CMD_WSIZE - 1));


      /* copy over last word, with mbxOwner set */
      ldata = *((volatile uint32 * )mb);

      WRITE_SLIM_ADDR(binfo, ((volatile uint32 * )mbox), ldata);
      FC_UNMAP_MEMIO(ioa);

      if (mb->mbxCommand == MBX_CONFIG_PORT) {
         /* switch over to host mailbox */
         binfo->fc_mboxaddr = (uint32 *)&((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx;
         binfo->fc_flag |= FC_SLI2;
      }
   }


   /* interrupt board to doit right away */
   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   WRITE_CSR_REG(binfo, FC_FF_REG(binfo, ioa), CA_MBATT);
   FC_UNMAP_MEMIO(ioa);

   switch (flag) {
   case MBX_SLEEP:
   case MBX_NOWAIT:
      break;

   case MBX_POLL:
      i = 0;
      if (binfo->fc_flag & FC_SLI2) {
         fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0,
             DDI_DMA_SYNC_FORKERNEL);

         /* First copy command data */
         mbox = FC_SLI2_MAILBOX(binfo);
         word0 = *((volatile uint32 * )mbox);
         word0 = PCIMEM_LONG(word0);
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mbox = FC_MAILBOX(binfo, ioa);
         word0 = READ_SLIM_ADDR(binfo, ((volatile uint32 * )mbox));
         FC_UNMAP_MEMIO(ioa);
      }

      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));
      FC_UNMAP_MEMIO(ioa);

      /* Wait for command to complete */
      while (((word0 & OWN_CHIP) == OWN_CHIP) || !(ha_copy & HA_MBATT)) {
         if (i++ >= 100) {
            binfo->fc_mbox_active = 0;
            return(MBX_NOT_FINISHED);
         }

         /* Check if we took a mbox interrupt while we were polling */
         if(((word0 & OWN_CHIP) != OWN_CHIP) && (evtctr != FCSTATCTR.mboxEvent))
            break;

         DELAYMS(i);  

         if (binfo->fc_flag & FC_SLI2) {
            fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0,
                DDI_DMA_SYNC_FORKERNEL);

            /* First copy command data */
            mbox = FC_SLI2_MAILBOX(binfo);
            word0 = *((volatile uint32 * )mbox);
            word0 = PCIMEM_LONG(word0);
         } else {
            /* First copy command data */
            ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
            mbox = FC_MAILBOX(binfo, ioa);
            word0 = READ_SLIM_ADDR(binfo, ((volatile uint32 * )mbox));
            FC_UNMAP_MEMIO(ioa);
         }
         ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
         ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));
         FC_UNMAP_MEMIO(ioa);
      }

      if (binfo->fc_flag & FC_SLI2) {
         fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0,
             DDI_DMA_SYNC_FORKERNEL);

         /* First copy command data */
         mbox = FC_SLI2_MAILBOX(binfo);
         /* copy results back to user */
         fc_pcimem_bcopy((uint32 * )mbox, (uint32 * )mb,
             (sizeof(uint32) * MAILBOX_CMD_WSIZE));
      } else {
         /* First copy command data */
         ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
         mbox = FC_MAILBOX(binfo, ioa);
         /* copy results back to user */
         READ_SLIM_COPY(binfo, (uint32 * )mb, (uint32 * )mbox, MAILBOX_CMD_WSIZE);
         FC_UNMAP_MEMIO(ioa);
      }

      ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
      WRITE_CSR_REG(binfo, FC_HA_REG(binfo, ioa), HA_MBATT);
      FC_UNMAP_MEMIO(ioa);

      binfo->fc_mbox_active = 0;
      status = mb->mbxStatus;
      break;
   }
   return(status);
}       /* End issue_mb_cmd */


/*
 * This routine will issue as many iocb commands from the
 * ring's xmit queue to the adapter as it can.
 * If iocb_cmd is specified it will be queued to the xmit queue.
 */
_static_ uint32
issue_iocb_cmd(
FC_BRD_INFO *binfo,
RING        *rp,
IOCBQ       *iocb_cmd)
{
   IOCB * iocb;
   IOCB * icmd;
   void * ioa;
   uint32 status;
   uint32 * xx;
   int  onetime;
   uint32 portCmdGet, rc;
   fc_dev_ctl_t    *p_dev_ctl;

   rc = PCIMEM_LONG(((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.port[rp->fc_ringno].cmdGetInx);
   onetime = 0;
   if ((binfo->fc_flag & FC_LNK_DOWN) || 
       (binfo->fc_ffstate < rp->fc_xmitstate)) {
      if (iocb_cmd) {
         icmd = &iocb_cmd->iocb;
         if ((icmd->ulpCommand != CMD_QUE_RING_BUF_CN) && 
             (icmd->ulpCommand != CMD_QUE_RING_BUF64_CN) && 
             (icmd->ulpCommand != CMD_CREATE_XRI_CR)) {
            fc_ringtx_put(rp, iocb_cmd);

            FCSTATCTR.NoIssueIocb++;
            /* If link is down, just return */
            return(rc);
         }
         onetime = 1;
      } else {
     /* If link is down, just return */
         return(rc);
      }
   } else {
      if (iocb_cmd) {
         /* Queue command to ring xmit queue */
         fc_ringtx_put(rp, iocb_cmd);
      }
      if((binfo->fc_process_LA == 0) &&
         (rp->fc_ringno == FC_FCP_RING)) {
         return(rc);
      }
   }

   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);
   fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

   /* onetime should only be set for QUE_RING_BUF or CREATE_XRI
    * iocbs sent with link down.
    */

   /* get the next available command iocb */
   iocb = (IOCB * )IOCB_ENTRY(rp->fc_cmdringaddr, rp->fc_cmdidx);

   portCmdGet = rc;

   if (portCmdGet >= rp->fc_numCiocb) {
      if (iocb_cmd) {
         /* Queue command to ring xmit queue */
         fc_ringtx_put(rp, iocb_cmd);
      }
      return(rc);
   }

   /* bump iocb available command index */
   if (++rp->fc_cmdidx >= rp->fc_numCiocb) {
      rp->fc_cmdidx = 0;
   }

   /* While IOCB entries are available */
   while (rp->fc_cmdidx != portCmdGet) {
      /* get next command from ring xmit queue */
      if ((onetime == 0) && ((iocb_cmd = fc_ringtx_get(rp)) == NULL)) {
out:
         fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, 
             DDI_DMA_SYNC_FORKERNEL);
         fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, 
             DDI_DMA_SYNC_FORDEV);

         /* SLIM POINTER */
         if (binfo->fc_busflag & FC_HOSTPTR) {
           rp->fc_cmdidx = 
             (uchar)PCIMEM_LONG(((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.host[rp->fc_ringno].cmdPutInx);
         } else {
           void  *ioa2;

           ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
           ioa2 = (void *)((char *)ioa +((SLIMOFF+(rp->fc_ringno*2))*4));
           rp->fc_cmdidx = (uchar)READ_SLIM_ADDR(binfo, (volatile uint32 *)ioa2);
           FC_UNMAP_MEMIO(ioa);
         }

         ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
         status = ((CA_R0ATT) << (rp->fc_ringno * 4));
         WRITE_CSR_REG(binfo, FC_FF_REG(binfo, ioa), (volatile uint32)status);
         FC_UNMAP_MEMIO(ioa);
         return(rc);
      }
      icmd = &iocb_cmd->iocb;

      xx = (uint32 * ) icmd;
      /* issue iocb command to adapter */
      fc_pcimem_bcopy((uint32 * )icmd, (uint32 * )iocb, sizeof(IOCB));
      FCSTATCTR.IssueIocb++;

      if ((icmd->ulpCommand == CMD_QUE_RING_BUF_CN) || 
          (icmd->ulpCommand == CMD_QUE_RING_BUF64_CN) || 
          (rp->fc_ringno == FC_FCP_RING) || 
          (icmd->ulpCommand == CMD_ABORT_XRI_CX) || 
          (icmd->ulpCommand == CMD_ABORT_XRI_CN)) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )iocb_cmd);
      } else {
         fc_ringtxp_put(rp, iocb_cmd);
      }

      /* SLIM POINTER */
      if (binfo->fc_busflag & FC_HOSTPTR) {
        ((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.host[rp->fc_ringno].cmdPutInx = PCIMEM_LONG(rp->fc_cmdidx);
      } else {
        void  *ioa2;

        ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
        ioa2 = (void *)((char *)ioa +((SLIMOFF+(rp->fc_ringno*2))*4));
        WRITE_SLIM_ADDR(binfo, (volatile uint32 *)ioa2, rp->fc_cmdidx);
        FC_UNMAP_MEMIO(ioa);
      }

      if (onetime) {
         goto out;
      }

      /* get the next available command iocb */
      iocb = (IOCB * )IOCB_ENTRY(rp->fc_cmdringaddr, rp->fc_cmdidx);

      /* bump iocb available command index */
      if (++rp->fc_cmdidx >= rp->fc_numCiocb) {
         rp->fc_cmdidx = 0;
      }
   }

   fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, 
       DDI_DMA_SYNC_FORKERNEL);
   fc_mpdata_sync(binfo->fc_slim2.dma_handle, 0, 0, 
       DDI_DMA_SYNC_FORDEV);

   /* SLIM POINTER */
   if (binfo->fc_busflag & FC_HOSTPTR) {
     rp->fc_cmdidx = 
       (uchar)PCIMEM_LONG(((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.host[rp->fc_ringno].cmdPutInx);
   } else {
     void  *ioa2;

     ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */
     ioa2 = (void *)((char *)ioa +((SLIMOFF+(rp->fc_ringno*2))*4));
     rp->fc_cmdidx = (uchar)READ_SLIM_ADDR(binfo, (volatile uint32 *)ioa2);
     FC_UNMAP_MEMIO(ioa);
   }


   /* If we get here, iocb list is full */
   /* 
    * Set ring 'x' to SET R0CE_REQ in Chip Att register.
    * Chip will tell us when an entry is freed.
    */
   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
   status = ((CA_R0ATT | CA_R0CE_REQ) << (rp->fc_ringno * 4));
   WRITE_CSR_REG(binfo, FC_FF_REG(binfo, ioa), (volatile uint32)status);
   FC_UNMAP_MEMIO(ioa);

   FCSTATCTR.iocbRingBusy++;

   if (onetime) {
      /* Queue command to ring xmit queue */
      fc_ringtx_put(rp, iocb_cmd);
   }
   return(rc);
}       /* End issue_iocb_cmd */




/*****************************************************************************/
/*
 * NAME:     fc_brdreset
 *
 * FUNCTION: hardware reset of adapter is performed
 *
 * EXECUTION ENVIRONMENT: process only
 *
 * NOTES:
 *
 * CALLED FROM:
 *      fc_cfg_init
 *
 * INPUT:
 *      p_dev_ctl       - point to the dev_ctl area
 *
 */
/*****************************************************************************/
_static_ void
fc_brdreset (
fc_dev_ctl_t    *p_dev_ctl)     /* point to the dev_ctl area */
{
   uint32 word0;
   ushort cfg_value, skip_post;
   void *ioa;
   FC_BRD_INFO * binfo;
   MAILBOX * swpmb;
   MAILBOX * mb;

   binfo = &BINFO;
   ioa = (void *)FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in SLIM */


   /* use REAL SLIM !!! */
   binfo->fc_mboxaddr = 0;
   binfo->fc_flag &= ~FC_SLI2;

   /* Reset the board - First put restart command in mailbox */
   mb = FC_MAILBOX(binfo, ioa);
   word0 = 0;
   swpmb = (MAILBOX * ) & word0;
   swpmb->mbxCommand = MBX_RESTART;
   swpmb->mbxHc = 1;
   WRITE_SLIM_ADDR(binfo, ((volatile uint32 * )mb), word0);
   /* Only skip post after fc_ffinit is completed */
   if (binfo->fc_ffstate) {
      skip_post = 1;
      WRITE_SLIM_ADDR(binfo, (((volatile uint32 * )mb) + 1), 1); /* Skip post */
   }
   else {
      skip_post = 0;
   }
   FC_UNMAP_MEMIO(ioa);

   /* Turn off SERR, PERR in PCI cmd register */
   binfo->fc_ffstate = FC_INIT_START;

   cfg_value = fc_rdpci_cmd(p_dev_ctl);
   fc_wrpci_cmd(p_dev_ctl, (ushort)(cfg_value & ~(CMD_PARITY_CHK | CMD_SERR_ENBL)));

   ioa = (void *)FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */

   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), (volatile uint32)HC_INITFF);
   DELAYMS(1);

   WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), (volatile uint32)0);

   FC_UNMAP_MEMIO(ioa);

   /* Restore PCI cmd register */
   fc_wrpci_cmd(p_dev_ctl, cfg_value);

   if(skip_post) {
      DELAYMS(100);
   }
   else {
      DELAYMS(2000);
   }

   binfo->fc_ffstate = FC_INIT_START;
   binfo->fc_eventTag = 0;
   binfo->fc_myDID = 0;
   binfo->fc_prevDID = 0;
   p_dev_ctl->power_up = 0; 
   return;
}
