/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include <gtk/gtkradiomenuitem.h>
#include <gtk/gtktearoffmenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkimage.h>
#include "galeon-css-menu.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "eel-gconf-extensions.h"
#include "gul-string.h"

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GaleonCssMenuPrivate {
	GaleonEmbed *embed;
	GtkMenuShell *ms;
	guint rebuild_timeout;

	GSList *menuitems;
	GList *stylesheets;

	gint ignore_menu_callbacks;

	GHashTable *menuitem_to_stylesheet;

	GtkWidget *parent_widget;
};

#define REBUILD_TIMEOUT 700

/**
 * Private functions, only availble from this file
 */
static void		galeon_css_menu_class_init		(GaleonCssMenuClass *klass);
static void		galeon_css_menu_init			(GaleonCssMenu *e);
static void		galeon_css_menu_finalize_impl		(GObject *o);
static void		galeon_css_menu_build_submenu		(GaleonCssMenu *gm);
static void		galeon_css_menu_build_item		(GaleonCssMenu *gm, EmbedStyleSheet *item);

static void		galeon_css_menu_rebuild			(GaleonCssMenu *gm);
static void 		galeon_css_menu_rebuild_with_timeout	(GaleonCssMenu *gm);
static gboolean		galeon_css_menu_rebuild_timeout_cb	(gpointer data);
static void		galeon_css_menu_activate_cb		(GtkMenuItem *mi, GaleonCssMenu *cm);

static gpointer g_object_class;

/**
 * GtkMenu object
 */

MAKE_GET_TYPE (galeon_css_menu, "GaleonCssMenu", GaleonCssMenu, galeon_css_menu_class_init,
	       galeon_css_menu_init, G_TYPE_OBJECT);

static void
galeon_css_menu_class_init (GaleonCssMenuClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_css_menu_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

}

static void 
galeon_css_menu_init (GaleonCssMenu *m)
{
	GaleonCssMenuPrivate *p = g_new0 (GaleonCssMenuPrivate, 1);
	m->priv = p;
	
	p->menuitem_to_stylesheet = g_hash_table_new (NULL, NULL);
	p->ignore_menu_callbacks = 0;
	p->parent_widget = NULL;
}

static void
galeon_css_menu_finalize_impl (GObject *o)
{
	GaleonCssMenu *gm = GALEON_CSS_MENU (o);
	GaleonCssMenuPrivate *p = gm->priv;
	GSList *li;

	DEBUG_MSG (("Finalizing GaleonCssMenu\n"));

	galeon_css_menu_set_parent_widget (gm, NULL);
	
	if (p->embed)
	{
		g_signal_handlers_disconnect_matched (p->embed, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, gm);
		g_object_unref (p->embed);
	}

	if (p->rebuild_timeout)
	{
		g_source_remove (p->rebuild_timeout);
	}

	for (li = p->menuitems; li; li = li->next)
	{
		g_object_unref (li->data);
	}
	g_slist_free (p->menuitems);

	galeon_embed_stylesheet_list_free (p->stylesheets);
	p->stylesheets = NULL;

	if (p->ms)
	{
		g_object_unref (p->ms);
	}

	g_hash_table_destroy (p->menuitem_to_stylesheet);

	g_free (p);

	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GaleonCssMenu *
galeon_css_menu_new (GtkMenuShell *ms)
{
	GaleonCssMenu *ret = g_object_new (GALEON_TYPE_CSS_MENU, NULL);
	GaleonCssMenuPrivate *p = ret->priv;

	p->embed = NULL;

	p->ms = g_object_ref (ms);

	p->menuitems = NULL;
	p->stylesheets = NULL;
	galeon_css_menu_rebuild_with_timeout (ret);

	return ret;
}

static void
galeon_css_menu_activate_current_item (GaleonCssMenu *gm)
{
	GaleonCssMenuPrivate *p = gm->priv;
	gresult result;
	EmbedStyleSheet *current = NULL;

	p->ignore_menu_callbacks++;

	result = galeon_embed_get_selected_stylesheet (p->embed, &current);
	if ((result == G_OK) && current)
	{
		GSList *sli;
		for (sli = p->menuitems; sli; sli = sli->next)
		{
			EmbedStyleSheet *it = g_hash_table_lookup (p->menuitem_to_stylesheet, sli->data);
			if (it && galeon_embed_stylesheet_compare (it, current))
			{
				gtk_menu_item_activate (sli->data);
				break;
			}
		}
		galeon_embed_stylesheet_free (current);
	}

	p->ignore_menu_callbacks--;
}

static gboolean 
menu_shell_has_tearoff (GtkMenuShell *ms)
{
	gboolean ret = FALSE;
	GList *children;
	GList *li;

	children = gtk_container_get_children (GTK_CONTAINER (ms));
	for (li = children; li; li = li->next)
	{
		if (GTK_IS_TEAROFF_MENU_ITEM (li->data))
		{
			ret = TRUE;
			break;
		}
	}
	g_list_free (children);

	return ret;
}

static void
galeon_css_menu_build_submenu (GaleonCssMenu *gm)
{
	GaleonCssMenuPrivate *p = gm->priv;
	GList *li;
	GSList *group;
	GSList *sli;
	gboolean has_tearoff = eel_gconf_get_boolean ("/desktop/gnome/interface/menus_have_tearoff");
	gresult result;

	p->ignore_menu_callbacks++;

	if (has_tearoff && !menu_shell_has_tearoff (p->ms))
	{
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_widget_show (tearoff);
		gtk_menu_shell_prepend (p->ms, tearoff);
	}

	result = galeon_embed_get_stylesheets (p->embed, &p->stylesheets);

	if (result == G_OK)
	{
		for (li = p->stylesheets; li; li = li->next)
		{
			galeon_css_menu_build_item (gm, li->data);
		}
		
		group = NULL;
		for (sli = p->menuitems; sli; sli = sli->next)
		{
			gtk_radio_menu_item_set_group (sli->data, group);
			group = gtk_radio_menu_item_get_group (sli->data);
		}
		
		/* GtkRadioMenuItem sucks, or i can't figure how to use them right */
		for (sli = p->menuitems; sli; sli = sli->next)
		{
			if (gtk_check_menu_item_get_active (sli->data))
			{
				gtk_menu_item_activate (sli->data);
			}
		}

		galeon_css_menu_activate_current_item (gm);
	}
	else
	{
		p->stylesheets = NULL;
	}

	if (GTK_IS_WIDGET (p->parent_widget))
	{
		if (p->stylesheets)
		{
			gtk_widget_set_sensitive (p->parent_widget, TRUE);
		}
		else
		{
			gtk_widget_set_sensitive (p->parent_widget, FALSE);
		}
	}
	p->ignore_menu_callbacks--;
}

static void
galeon_css_menu_build_item (GaleonCssMenu *gm, EmbedStyleSheet *item)
{
	GaleonCssMenuPrivate *p = gm->priv;
	GtkWidget *w;

	p->ignore_menu_callbacks++;

	w = gtk_radio_menu_item_new_with_label (NULL, item->name);

	g_signal_connect (w, "activate", 
			  G_CALLBACK (galeon_css_menu_activate_cb), gm);

	p->menuitems = g_slist_prepend (p->menuitems, g_object_ref (w));

	g_hash_table_insert (p->menuitem_to_stylesheet, w, item);

	gtk_widget_show (w);
	gtk_container_add (GTK_CONTAINER (p->ms), w);

	p->ignore_menu_callbacks--;
}

static void
galeon_css_menu_activate_cb (GtkMenuItem *mi, GaleonCssMenu *cm)
{
	GaleonCssMenuPrivate *p = cm->priv;
	
	if (p->ignore_menu_callbacks > 0)
	{
		return;
	}

	DEBUG_MSG (("galeon_css_menu_activated_cb\n"));

	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (mi)))
	{
		EmbedStyleSheet *item = g_hash_table_lookup (p->menuitem_to_stylesheet, mi);
		if (item)
		{
			DEBUG_MSG (("Setting stylesheet '%s'\n", item->name));
			
			g_return_if_fail (IS_GALEON_EMBED (p->embed));
			galeon_embed_set_stylesheet (p->embed, item);
		}
		else
		{
			DEBUG_MSG (("NULL item assoicated to menu item\n"));
		}
	}
}

static gboolean
galeon_css_menu_rebuild_timeout_cb (gpointer data)
{
	GaleonCssMenu *gm = data;
	GaleonCssMenuPrivate *p = gm->priv;
	DEBUG_MSG (("GaleonCssMenu rebuild timeout\n"));

	p->rebuild_timeout = 0;

	galeon_css_menu_rebuild (data);
	return FALSE;
}

static void 
galeon_css_menu_rebuild_with_timeout (GaleonCssMenu *gm) 
{
	GaleonCssMenuPrivate *p = gm->priv;

	DEBUG_MSG (("galeon_css_menu_rebuild_with_timeout\n"));

	if (p->rebuild_timeout)
	{
		g_source_remove (p->rebuild_timeout);
	}
	
	p->rebuild_timeout = g_timeout_add (REBUILD_TIMEOUT, 
					    galeon_css_menu_rebuild_timeout_cb, gm);
}

static void
galeon_css_menu_reset (GaleonCssMenu *gm)
{
	GaleonCssMenuPrivate *p = gm->priv;
	GList *children;
	GList *li;
	GSList *sli;

	p->ignore_menu_callbacks++;

	/* free the list of menuitems */
	for (sli = p->menuitems; sli; sli = sli->next)
	{
		g_signal_handlers_disconnect_matched (sli->data, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, gm);
		g_object_unref (sli->data);
	}
	g_slist_free (p->menuitems);
	p->menuitems = NULL;

	galeon_embed_stylesheet_list_free (p->stylesheets);
	p->stylesheets = NULL;

	if (g_hash_table_size (p->menuitem_to_stylesheet) > 0)
	{
		g_hash_table_destroy (p->menuitem_to_stylesheet);
		p->menuitem_to_stylesheet = g_hash_table_new (NULL, NULL);
	}

	children = gtk_container_get_children (GTK_CONTAINER (p->ms));
	for (li = children; li; li = li->next)
	{
		if (!GTK_IS_TEAROFF_MENU_ITEM (li->data))
		{
			gtk_container_remove (GTK_CONTAINER (p->ms), li->data);
		}
	}
	g_list_free (children);

	p->ignore_menu_callbacks--;
}

static void
galeon_css_menu_rebuild (GaleonCssMenu *gm)
{
	GaleonCssMenuPrivate *p = gm->priv;

	DEBUG_MSG (("galeon_css_menu_rebuild called\n"));

	p->ignore_menu_callbacks++;

	galeon_css_menu_reset (gm);
	
	galeon_css_menu_build_submenu (gm);

	p->ignore_menu_callbacks--;
}


static void
galeon_css_menu_ge_net_state_cb (GaleonEmbed *embed, const char *uri,
				 EmbedState state, GaleonCssMenu *cm)
{
	GaleonCssMenuPrivate *p = cm->priv;

	g_return_if_fail (embed == p->embed);

	if (state & EMBED_STATE_STOP)
	{			
		galeon_css_menu_rebuild_with_timeout (cm);
	}	
}

void 
galeon_css_menu_set_embed (GaleonCssMenu *cm, GaleonEmbed *e)
{
	GaleonCssMenuPrivate *p;

	g_return_if_fail (GALEON_IS_CSS_MENU (cm));
	g_return_if_fail (!e || IS_GALEON_EMBED (e));

	DEBUG_MSG (("galeon_css_menu_set_embed\n"));

	p = cm->priv;

	galeon_css_menu_reset (cm);

	if (p->embed)
	{
		g_signal_handlers_disconnect_matched (p->embed, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, cm);
		g_object_unref (p->embed);
		p->embed = NULL;
	}

	if (e)
	{
		p->embed = g_object_ref (e);
		g_signal_connect (p->embed, "ge_net_state",
				  G_CALLBACK (galeon_css_menu_ge_net_state_cb), cm);
	}

	galeon_css_menu_rebuild_with_timeout (cm);
}

void
galeon_css_menu_set_parent_widget (GaleonCssMenu *cm, GtkWidget *w)
{
	GaleonCssMenuPrivate *p = cm->priv;
	
	if (p->parent_widget)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->parent_widget),
					      (gpointer *) &p->parent_widget);
	}

	p->parent_widget = w;

	if (p->parent_widget)
	{
		g_object_add_weak_pointer (G_OBJECT (p->parent_widget), 
					   (gpointer *) &p->parent_widget);
	}
}
