/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%              EEEEE  N   N  H   H   AAA   N   N   CCCC  EEEEE                %
%              E      NN  N  H   H  A   A  NN  N  C      E                    %
%              EEE    N N N  HHHHH  AAAAA  N N N  C      EEE                  %
%              E      N  NN  H   H  A   A  N  NN  C      E                    %
%              EEEEE  N   N  H   H  A   A  N   N   CCCC  EEEEE                %
%                                                                             %
%                                                                             %
%                    ImageMagick Image Enhancement Methods                    %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 July 1992                                   %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/error.h"
#include "magick/gem.h"
#include "magick/monitor.h"
#include "magick/utility.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     C o n t r a s t I m a g e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ContrastImage() enhances the intensity differences between the lighter and
%  darker elements of the image.  Set sharpen to a value other than 0 to
%  increase the image contrast otherwise the contrast is reduced.
%
%  The format of the ContrastImage method is:
%
%      unsigned int ContrastImage(Image *image,const unsigned int sharpen)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o sharpen: Increase or decrease image contrast.
%
%
*/
MagickExport unsigned int ContrastImage(Image *image,const unsigned int sharpen)
{
#define DullContrastImageTag  "DullContrast/Image"
#define SharpenContrastImageTag  "SharpenContrast/Image"

  int
    sign;

  long
    y;

  register long
    i,
    x;

  register PixelPacket
    *q;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  sign=sharpen ? 1 : -1;
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      unsigned int
        status;

      /*
        Contrast enhance DirectClass image.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          Contrast(sign,&q->red,&q->green,&q->blue);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          {
            if (sharpen)
              status=MagickMonitor(SharpenContrastImageTag,y,image->rows,
                &image->exception);
            else
              status=MagickMonitor(DullContrastImageTag,y,image->rows,
                &image->exception);
            if (status == False)
              break;
          }
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Contrast enhance PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
        Contrast(sign,&image->colormap[i].red,&image->colormap[i].green,
          &image->colormap[i].blue);
      SyncImage(image);
      break;
    }
  }
  return(False);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     E q u a l i z e I m a g e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  EqualizeImage() applies a histogram equalization to the image.
%
%  The format of the EqualizeImage method is:
%
%      unsigned int EqualizeImage(Image *image)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
*/
MagickExport unsigned int EqualizeImage(Image *image)
{
#define EqualizeImageTag  "Equalize/Image"

  DoublePixelPacket
    high,
    *histogram,
    intensity,
    low,
    *map;

  long
    y;

  PixelPacket
    *equalize_map;

  register const PixelPacket
    *p;

  register long
    i,
    x;

  register PixelPacket
    *q;

  /*
    Allocate and initialize histogram arrays.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  histogram=(DoublePixelPacket *)
    AcquireMemory((MaxMap+1)*sizeof(DoublePixelPacket));
  map=(DoublePixelPacket *) AcquireMemory((MaxMap+1)*sizeof(DoublePixelPacket));
  equalize_map=(PixelPacket *) AcquireMemory((MaxMap+1)*sizeof(PixelPacket));
  if ((histogram == (DoublePixelPacket *) NULL) ||
      (map == (DoublePixelPacket *) NULL) ||
      (equalize_map == (PixelPacket *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToEqualizeImage");
  /*
    Form histogram.
  */
  memset(histogram,0,(MaxMap+1)*sizeof(DoublePixelPacket));
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    for (x=(long) image->columns-1; x >= 0; x--)
    {
      histogram[ScaleQuantumToMap(p->red)].red++;
      histogram[ScaleQuantumToMap(p->green)].green++;
      histogram[ScaleQuantumToMap(p->blue)].blue++;
      histogram[ScaleQuantumToMap(p->opacity)].opacity++;
      p++;
    }
  }
  /*
    Integrate the histogram to get the equalization map.
  */
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (i=0; i <= (long) MaxMap; i++)
  {
    intensity.red+=histogram[i].red;
    intensity.green+=histogram[i].green;
    intensity.blue+=histogram[i].blue;
    intensity.opacity+=histogram[i].opacity;
    map[i]=intensity;
  }
  low=map[0];
  high=map[MaxMap];
  memset(equalize_map,0,(MaxMap+1)*sizeof(PixelPacket));
  for (i=0; i <= (long) MaxMap; i++)
  {
    if (high.red != low.red)
      equalize_map[i].red=ScaleMapToQuantum(
        (MaxMap*(map[i].red-low.red))/(high.red-low.red));
    if (high.green != low.green)
      equalize_map[i].green=ScaleMapToQuantum(
        (MaxMap*(map[i].green-low.green))/(high.green-low.green));
    if (high.blue != low.blue)
      equalize_map[i].blue=ScaleMapToQuantum(
        (MaxMap*(map[i].blue-low.blue))/(high.blue-low.blue));
    if (high.opacity != low.opacity)
      equalize_map[i].opacity=ScaleMapToQuantum(
        (MaxMap*(map[i].opacity-low.opacity))/(high.opacity-low.opacity));
  }
  LiberateMemory((void **) &histogram);
  LiberateMemory((void **) &map);
  /*
    Stretch the histogram.
  */
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Equalize DirectClass packets.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          if (low.red != high.red)
            q->red=equalize_map[ScaleQuantumToMap(q->red)].red;
          if (low.green != high.green)
            q->green=equalize_map[ScaleQuantumToMap(q->green)].green;
          if (low.blue != high.blue)
            q->blue=equalize_map[ScaleQuantumToMap(q->blue)].blue;
          if (low.opacity != high.opacity)
            q->opacity=equalize_map[ScaleQuantumToMap(q->opacity)].opacity;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(EqualizeImageTag,y,image->rows,&image->exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Equalize PseudoClass packets.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        if (low.red != high.red)
          image->colormap[i].red=
            equalize_map[ScaleQuantumToMap(image->colormap[i].red)].red;
        if (low.green != high.green)
          image->colormap[i].green=
            equalize_map[ScaleQuantumToMap(image->colormap[i].green)].green;
        if (low.blue != high.blue)
          image->colormap[i].blue=
            equalize_map[ScaleQuantumToMap(image->colormap[i].blue)].blue;
      }
      SyncImage(image);
      break;
    }
  }
  LiberateMemory((void **) &equalize_map);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     G a m m a I m a g e                                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Use GammaImage() to gamma-correct an image.  The same image viewed on
%  different devices will have perceptual differences in the way the
%  image's intensities are represented on the screen.  Specify individual
%  gamma levels for the red, green, and blue channels, or adjust all three
%  with the gamma parameter.  Values typically range from 0.8 to 2.3.
%
%  You can also reduce the influence of a particular channel with a gamma
%  value of 0.
%
%  The format of the GammaImage method is:
%
%      unsigned int GammaImage(Image *image,const char *level)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o level: Define the level of gamma correction.
%
%
*/
MagickExport unsigned int GammaImage(Image *image,const char *level)
{
#define GammaCorrectImageTag  "GammaCorrect/Image"

  DoublePixelPacket
    gamma;

  GeometryInfo
    geometry_info;

  long
    y;

  PixelPacket
    *gamma_map;

  register long
    i,
    x;

  register PixelPacket
    *q;

  unsigned int
    flags,
    status;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (level == (char *) NULL)
    return(False);
  flags=ParseGeometry(level,&geometry_info);
  gamma.red=geometry_info.rho;
  gamma.green=geometry_info.sigma;
  if (!(flags & SigmaValue))
    gamma.green=gamma.red;
  gamma.blue=geometry_info.xi;
  if (!(flags & XiValue))
    gamma.blue=gamma.red;
  if ((gamma.red == 1.0) && (gamma.green == 1.0) && (gamma.blue == 1.0))
    return(True);
  /*
    Allocate and initialize gamma maps.
  */
  gamma_map=(PixelPacket *) AcquireMemory((MaxMap+1)*sizeof(PixelPacket));
  if (gamma_map == (PixelPacket *) NULL)
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToGammaCorrectImage");
  (void) memset(gamma_map,0,(MaxMap+1)*sizeof(PixelPacket));
  for (i=0; i <= (long) MaxMap; i++)
  {
    if (gamma.red != 0.0)
      gamma_map[i].red=
        ScaleMapToQuantum(MaxMap*pow((double) i/MaxMap,1.0/gamma.red));
    if (gamma.green != 0.0)
      gamma_map[i].green=
       ScaleMapToQuantum(MaxMap*pow((double) i/MaxMap,1.0/gamma.green));
    if (gamma.blue != 0.0)
      gamma_map[i].blue=
        ScaleMapToQuantum(MaxMap*pow((double) i/MaxMap,1.0/gamma.blue));
  }
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Gamma-correct DirectClass image.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          q->red=gamma_map[ScaleQuantumToMap(q->red)].red;
          q->green=gamma_map[ScaleQuantumToMap(q->green)].green;
          q->blue=gamma_map[ScaleQuantumToMap(q->blue)].blue;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          {
            status=MagickMonitor(GammaCorrectImageTag,y,image->rows,
              &image->exception);
            if (status == False)
              break;
          }
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Gamma-correct PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        image->colormap[i].red=
          gamma_map[ScaleQuantumToMap(image->colormap[i].red)].red;
        image->colormap[i].green=
          gamma_map[ScaleQuantumToMap(image->colormap[i].green)].green;
        image->colormap[i].blue=
          gamma_map[ScaleQuantumToMap(image->colormap[i].blue)].blue;
      }
      SyncImage(image);
      break;
    }
  }
  if (image->gamma != 0.0)
    image->gamma*=(gamma.red+gamma.green+gamma.blue)/3.0;
  LiberateMemory((void **) &gamma_map);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     L e v e l I m a g e                                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LevelImage() adjusts the levels of an image by scaling the colors falling
%  between specified white and black points to the full available quantum
%  range. The parameters provided represent the black, mid, and white points.
%  The black point specifies the darkest color in the image. Colors darker than
%  the black point are set to zero. Mid point specifies a gamma correction to
%  apply to the image.  White point specifies the lightest color in the image.
%  Colors brighter than the white point are set to the maximum quantum value.
%
%  The format of the LevelImage method is:
%
%      unsigned int LevelImage(Image *image,const char *level)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o levels: Specify the levels as a string of the form "black/white/gamma"
%      (e.g. "10,65000,1.0" or "10%,98%,0.5") where black and white have the
%      range of 0-MaxRGB or 0-100%, and gamma has the range 0.1 to 10.0.
%
%
*/
MagickExport unsigned int LevelImage(Image *image,const char *levels)
{
#define LevelImageTag  "Level/Image"

  double
    black_point,
    gamma,
    *levels_map,
    white_point;

  GeometryInfo
    geometry_info;

  int
    flags;

  long
    y;

  register long
    i,
    x;

  register PixelPacket
    *q;

  /*
    Allocate and initialize levels map.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (levels == (char *) NULL)
    return(False);
  flags=ParseGeometry(levels,&geometry_info);
  black_point=geometry_info.rho;
  if (flags & SigmaValue)
    white_point=geometry_info.sigma;
  else
    white_point=MaxRGB;
  if (flags & XiValue)
    gamma=geometry_info.xi;
  else
    gamma=1.0;
  if ((white_point <= 10.0) && (gamma > 10.0))
    {
      double
        swap;

      swap=gamma;
      gamma=white_point;
      white_point=swap;
    }
  if (flags & PercentValue)
    {
      black_point*=MaxRGB/100.0;
      white_point*=MaxRGB/100.0;
    }
  if (!(flags & SigmaValue))
    white_point=MaxRGB-black_point;
  black_point=ScaleQuantumToMap(black_point);
  white_point=ScaleQuantumToMap(white_point);
  levels_map=(double *) AcquireMemory((MaxMap+1)*sizeof(double));
  if (levels_map == (double *) NULL)
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToLevelImage");
  for (i=0; i < black_point; i++)
    levels_map[i]=0;
  if (gamma == 1.0)
    {
      for ( ; i < white_point; i++)
        levels_map[i]=MaxMap*(((double) i-black_point)/
          (white_point-black_point));
    }
  else
    for ( ; i < white_point; i++)
      levels_map[i]=MaxMap*(pow(((double) i-black_point)/
        (white_point-black_point),1.0/gamma));
  for ( ; i <= (long) MaxMap; i++)
    levels_map[i]=MaxMap;
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Level DirectClass image.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          q->red=ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->red)]);
          q->green=ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->green)]);
          q->blue=ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->blue)]);
          q->opacity=
            ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->opacity)]);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(LevelImageTag,y,image->rows,&image->exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Level PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        image->colormap[i].red=ScaleMapToQuantum(
          levels_map[ScaleQuantumToMap(image->colormap[i].red)]);
        image->colormap[i].green=ScaleMapToQuantum(
          levels_map[ScaleQuantumToMap(image->colormap[i].green)]);
        image->colormap[i].blue=ScaleMapToQuantum(
          levels_map[ScaleQuantumToMap(image->colormap[i].blue)]);
      }
      SyncImage(image);
      break;
    }
  }
  LiberateMemory((void **) &levels_map);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     L e v e l I m a g e C h a n n e l                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LevelImageChannel() adjusts the levels of a particular image channel by
%  scaling the colors falling between specified white and black points to
%  the full available quantum range. The parameters provided represent the
%  black, mid, and white points.  The black point specifies the darkest
%  color in the image. Colors darker than the black point are set to
%  zero. Gamma specifies a gamma correction to apply to the image.
%  White point specifies the lightest color in the image.  Colors brighter
%  than the white point are set to the maximum quantum value.
%
%  The format of the LevelImage method is:
%
%      unsigned int LevelImageChannel(Image *image,const char *level,
%        const ChannelType channel,const double black_point,
%        const double white_point,const double gamma)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o channel: Identify which channel to extract: Red, Cyan, Green, Magenta,
%      Blue, Yellow, or Opacity.
%
%    o black_point, white_point, gamma: Specify the levels where the black
%      and white points have the range of 0-MaxRGB, and gamma has the range
%      0-10.
%
%
*/
MagickExport unsigned int LevelImageChannel(Image *image,
  const ChannelType channel,const double black_point,const double white_point,
  const double gamma)
{
  double
    black,
    *levels_map,
    white;

  long
    y;

  register long
    i,
    x;

  register PixelPacket
    *q;

  unsigned int
    status;

  /*
    Allocate and initialize levels map.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  levels_map=(double *) AcquireMemory((MaxMap+1)*sizeof(double));
  if (levels_map == (double *) NULL)
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToLevelImage");
  black=ScaleQuantumToMap(black_point);
  white=ScaleQuantumToMap(white_point);
  for (i=0; i < black; i++)
    levels_map[i]=0;
  if (gamma == 1.0)
    {
      for ( ; i < white; i++)
        levels_map[i]=MaxMap*(((double) i-black)/(white-black));
    }
  else
    for ( ; i < white; i++)
      levels_map[i]=MaxMap*(pow(((double) i-black)/(white-black),1.0/gamma));
  for ( ; i <= (long) MaxMap; i++)
    levels_map[i]=MaxMap;
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Level DirectClass image.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          switch (channel)
          {
            case RedChannel:
            case CyanChannel:
            {
              q->red=ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->red)]);
              break;
            }
            case GreenChannel:
            case MagentaChannel:
            {
              q->green=ScaleMapToQuantum(
                levels_map[ScaleQuantumToMap(q->green)]);
              break;
            }
            case BlueChannel:
            case YellowChannel:
            {
              q->blue=ScaleMapToQuantum(levels_map[ScaleQuantumToMap(q->blue)]);
              break;
            }
            case OpacityChannel:
            case BlackChannel:
            {
              q->opacity=ScaleMapToQuantum(
                levels_map[ScaleQuantumToMap(q->opacity)]);
              break;
            }
            default:
              break;
          }
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          {
            status=MagickMonitor(GammaCorrectImageTag,y,image->rows,
              &image->exception);
            if (status == False)
              break;
          }
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Level PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        switch (channel)
        {
          case RedChannel:
          case CyanChannel:
          {
            image->colormap[i].red=ScaleMapToQuantum(
              levels_map[ScaleQuantumToMap(image->colormap[i].red)]);
            break;
          }
          case GreenChannel:
          case MagentaChannel:
          {
            image->colormap[i].green=ScaleMapToQuantum(
              levels_map[ScaleQuantumToMap(image->colormap[i].green)]);
            break;
          }
          case BlueChannel:
          case YellowChannel:
          {
            image->colormap[i].blue=ScaleMapToQuantum(
              levels_map[ScaleQuantumToMap(image->colormap[i].blue)]);
            break;
          }
          default:
            break;
        }
      }
      SyncImage(image);
      break;
    }
  }
  LiberateMemory((void **) &levels_map);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     M o d u l a t e I m a g e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ModulateImage() lets you control the brightness, saturation, and hue
%  of an image.  Modulate represents the brightness, saturation, and hue
%  as one parameter (e.g. 90,150,100).
%
%  The format of the ModulateImage method is:
%
%      unsigned int ModulateImage(Image *image,const char *modulate)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o modulate: Define the percent change in brightness, saturation, and
%      hue.
%
%
*/
MagickExport unsigned int ModulateImage(Image *image,const char *modulate)
{
#define ModulateImageTag  "Modulate/Image"

  double
    percent_brightness,
    percent_hue,
    percent_saturation;

  GeometryInfo
    geometry_info;

  int
    flags;

  long
    y;

  register long
    i,
    x;

  register PixelPacket
    *q;

  /*
    Initialize gamma table.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (modulate == (char *) NULL)
    return(False);
  flags=ParseGeometry(modulate,&geometry_info);
  percent_brightness=geometry_info.rho;
  percent_saturation=geometry_info.sigma;
  if (!(flags & SigmaValue))
    percent_saturation=100.0;
  percent_hue=geometry_info.xi;
  if (!(flags & XiValue))
    percent_hue=100.0;
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Modulate the color for a DirectClass image.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          Modulate(percent_hue,percent_saturation,percent_brightness,
            &q->red,&q->green,&q->blue);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(ModulateImageTag,y,image->rows,&image->exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Modulate the color for a PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
        Modulate(percent_hue,percent_saturation,percent_brightness,
          &image->colormap[i].red,&image->colormap[i].green,
          &image->colormap[i].blue);
      SyncImage(image);
      break;
    }
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     N e g a t e I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method NegateImage negates the colors in the reference image.  The
%  Grayscale option means that only grayscale values within the image are
%  negated.
%
%  The format of the NegateImage method is:
%
%      unsigned int NegateImage(Image *image,const unsigned int grayscale)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%
*/
MagickExport unsigned int NegateImage(Image *image,const unsigned int grayscale)
{
#define NegateImageTag  "Negate/Image"

  long
    y;

  register long
    x;

  register PixelPacket
    *q;

  register long
    i;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Negate DirectClass packets.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          if (grayscale)
            if ((q->red != q->green) || (q->green != q->blue))
              {
                q++;
                continue;
              }
          q->red=(~q->red);
          q->green=(~q->green);
          q->blue=(~q->blue);
          if (image->colorspace == CMYKColorspace)
            q->opacity=(~q->opacity);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(NegateImageTag,y,image->rows,&image->exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Negate PseudoClass packets.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        if (grayscale)
          if ((image->colormap[i].red != image->colormap[i].green) ||
              (image->colormap[i].green != image->colormap[i].blue))
            continue;
        image->colormap[i].red=(~image->colormap[i].red);
        image->colormap[i].green=(~image->colormap[i].green);
        image->colormap[i].blue=(~image->colormap[i].blue);
      }
      SyncImage(image);
      break;
    }
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     N o r m a l i z e I m a g e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  The NormalizeImage() method enhances the contrast of a color image by
%  adjusting the pixels color to span the entire range of colors available.
%
%  The format of the NormalizeImage method is:
%
%      unsigned int NormalizeImage(Image *image)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%
*/
MagickExport unsigned int NormalizeImage(Image *image)
{
#define MaxRange(color)  ScaleQuantumToMap(color)
#define NormalizeImageTag  "Normalize/Image"

  DoublePixelPacket
    high,
    *histogram,
    intensity,
    low;

  ExtendedSignedIntegralType
    number_pixels;

  long
    y;

  PixelPacket
    *normalize_map;

  register const PixelPacket
    *p;

  register long
    x;

  register PixelPacket
    *q;

  register long
    i;

  unsigned long
    threshold_intensity;

  /*
    Allocate histogram and normalize map.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  histogram=(DoublePixelPacket *)
    AcquireMemory((MaxMap+1)*sizeof(DoublePixelPacket));
  normalize_map=(PixelPacket *) AcquireMemory((MaxMap+1)*sizeof(PixelPacket));
  if ((histogram == (DoublePixelPacket *) NULL) ||
      (normalize_map == (PixelPacket *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      "UnableToNormalizeImage");
  /*
    Form histogram.
  */
  memset(histogram,0,(MaxMap+1)*sizeof(DoublePixelPacket));
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    for (x=(long) image->columns-1; x >= 0; x--)
    {
      histogram[ScaleQuantumToMap(p->red)].red++;
      histogram[ScaleQuantumToMap(p->green)].green++;
      histogram[ScaleQuantumToMap(p->blue)].blue++;
      histogram[ScaleQuantumToMap(p->opacity)].opacity++;
      p++;
    }
  }
  /*
    Find the histogram boundaries by locating the 0.1 percent levels.
  */
  number_pixels=(ExtendedSignedIntegralType) image->columns*image->rows;
  threshold_intensity=number_pixels/1000;
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (low.red=0; low.red < MaxRange(MaxRGB); low.red++)
  {
    intensity.red+=histogram[(long) low.red].red;
    if (intensity.red > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (high.red=MaxRange(MaxRGB); high.red != 0; high.red--)
  {
    intensity.red+=histogram[(long) high.red].red;
    if (intensity.red > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (low.green=low.red; low.green < high.red; low.green++)
  {
    intensity.green+=histogram[(long) low.green].green;
    if (intensity.green > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (high.green=high.red; high.green != low.red; high.green--)
  {
    intensity.green+=histogram[(long) high.green].green;
    if (intensity.green > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (low.blue=low.green; low.blue < high.green; low.blue++)
  {
    intensity.blue+=histogram[(long) low.blue].blue;
    if (intensity.blue > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (high.blue=high.green; high.blue != low.green; high.blue--)
  {
    intensity.blue+=histogram[(long) high.blue].blue;
    if (intensity.blue > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (low.opacity=0; low.opacity < MaxRange(MaxRGB); low.opacity++)
  {
    intensity.opacity+=histogram[(long) low.opacity].opacity;
    if (intensity.opacity > threshold_intensity)
      break;
  }
  memset(&intensity,0,sizeof(DoublePixelPacket));
  for (high.opacity=MaxRange(MaxRGB); high.opacity != 0; high.opacity--)
  {
    intensity.opacity+=histogram[(long) high.opacity].opacity;
    if (intensity.opacity > threshold_intensity)
      break;
  }
  LiberateMemory((void **) &histogram);
  /*
    Stretch the histogram to create the normalized image mapping.
  */
  memset(normalize_map,0,(MaxMap+1)*sizeof(PixelPacket));
  for (i=0; i <= (long) MaxMap; i++)
  {
    if (i < (long) low.red)
      normalize_map[i].red=0;
    else
      if (i > (long) high.red)
        normalize_map[i].red=MaxMap;
      else
        if (low.red != high.red)
          normalize_map[i].red=
            ScaleMapToQuantum((MaxMap*(i-low.red))/(high.red-low.red));
    if (i < (long) low.green)
      normalize_map[i].green=0;
    else
      if (i > (long) high.green)
        normalize_map[i].green=MaxMap;
      else
        if (low.green != high.green)
          normalize_map[i].green=ScaleMapToQuantum((MaxMap*(i-low.green))/
            (high.green-low.green));
    if (i < (long) low.blue)
      normalize_map[i].blue=0;
    else
      if (i > (long) high.blue)
        normalize_map[i].blue=MaxMap;
      else
        if (low.blue != high.blue)
          normalize_map[i].blue=
            ScaleMapToQuantum((MaxMap*(i-low.blue))/(high.blue-low.blue));
    if (i < (long) low.opacity)
      normalize_map[i].opacity=0;
    else
      if (i > (long) high.opacity)
        normalize_map[i].opacity=MaxMap;
      else
        if (low.opacity != high.opacity)
          normalize_map[i].opacity=ScaleMapToQuantum(
            (MaxMap*(i-low.opacity))/(high.opacity-low.opacity));
  }
  /*
    Normalize the image.
  */
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      ExceptionInfo
        *exception;

      /*
        Normalize DirectClass image.
      */
      exception=(&image->exception);
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=(long) image->columns-1; x >= 0; x--)
        {
          if (low.red != high.red)
            q->red=normalize_map[ScaleQuantumToMap(q->red)].red;
          if (low.green != high.green)
            q->green=normalize_map[ScaleQuantumToMap(q->green)].green;
          if (low.blue != high.blue)
            q->blue=normalize_map[ScaleQuantumToMap(q->blue)].blue;
          if (low.opacity != high.opacity)
            q->opacity=normalize_map[ScaleQuantumToMap(q->opacity)].opacity;
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (QuantumTick(y,image->rows))
          if (!MagickMonitor(NormalizeImageTag,y,image->rows,exception))
            break;
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Normalize PseudoClass image.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        if (low.red != high.red)
          image->colormap[i].red=
            normalize_map[ScaleQuantumToMap(image->colormap[i].red)].red;
        if (low.green != high.green)
          image->colormap[i].green=
            normalize_map[ScaleQuantumToMap(image->colormap[i].green)].green;
        if (low.blue != high.blue)
          image->colormap[i].blue=
            normalize_map[ScaleQuantumToMap(image->colormap[i].blue)].blue;
      }
      SyncImage(image);
      break;
    }
  }
  LiberateMemory((void **) &normalize_map);
  return(True);
}
