/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * twlog.c - support for WA0EIR Ted's twlog file format, as of v1.3
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <glib.h>

#include "logfile.h"
#include "types.h"


/*
 * file fields
 */
const gint twlog_fields[] = { DATE, GMT, CALL, BAND, MODE, POWER,
  RST, MYRST, GMTEND, REMARKS
};
const gint twlog_widths[] = { 11, 11, 11, 7, 7, 7, 7, 7, 11, 80 };
const gint twlog_field_nr = 10;

static gint twlog_open (LOGDB *);
static void twlog_close (LOGDB *);
static gint twlog_create (LOGDB *);
static gint twlog_qso_append (LOGDB *, const qso_t *);
static gint twlog_qso_foreach (LOGDB *,
			       gint (*fn) (LOGDB *, qso_t *, gpointer arg),
			       gpointer arg);

const struct log_ops twlog_ops = {
  open:twlog_open,
  close:twlog_close,
  create:twlog_create,
  qso_append:twlog_qso_append,
  qso_foreach:twlog_qso_foreach,
  type:TYPE_TWLOG,
  name:"Twlog",
  extension:"",
};

/*
 * open for read
 */
gint
twlog_open (LOGDB * handle)
{
  FILE *fp;

  fp = fopen (handle->path, "r");
  if (!fp)
    return -1;
  handle->priv = (gpointer) fp;

  handle->column_nr = twlog_field_nr;
  memcpy (handle->column_fields, twlog_fields, sizeof (twlog_fields));
  memcpy (handle->column_widths, twlog_widths, sizeof (twlog_widths));

  return 0;
}

/*
 * open for write
 */
gint
twlog_create (LOGDB * handle)
{
  FILE *fp;

  fp = fopen (handle->path, "w");
  if (!fp)
    return -1;
  handle->priv = (gpointer) fp;
  return 0;
}

void
twlog_close (LOGDB * handle)
{
  FILE *fp = (FILE *) handle->priv;
  fclose (fp);
}

/* 
 * each field 15 positions, remarks on a separate line, 80 wide
 */
gint
twlog_qso_append (LOGDB * handle, const qso_t * q)
{
  FILE *fp = (FILE *) handle->priv;

  fprintf (fp, "%-11s%-11s%-11s%-7s%-7s%-7s%-7s%-7s%-11s\n%-80s\n\n",
	   q[DATE], q[GMT], q[CALL], q[BAND], q[MODE], q[POWER],
	   q[RST], q[MYRST], q[GMTEND], q[REMARKS]);
  return 0;
}


static gchar *
find_field_end (gchar * p)
{
  while (*p && *p != ' ')
    p++;
  if (*p == ' ')
    while (*(p + 1) && *(p + 1) == ' ')
      p++;

  return p;
}

gint
twlog_qso_foreach (LOGDB * handle,
		   gint (*fn) (LOGDB *, qso_t *, gpointer arg), gpointer arg)
{
  FILE *fp = (FILE *) handle->priv;
  gint i, ret, rlen;
  qso_t q[QSO_FIELDS];
  gchar buffer[81], *field, *end;

  while (!feof (fp))
    {
      fgets (buffer, 13, fp);
      if (strlen (buffer) != 12 || feof (fp))
	{
	  return 0;
	}
      memset (q, 0, sizeof (q));

      q[DATE] = g_strdup (g_strstrip (buffer));

      fgets (buffer, 80, fp);
      field = buffer;
      end = find_field_end (field);
      if (*end)
	*end++ = '\0';
      q[GMT] = g_strdup (g_strstrip (field));
      field = end;

      /* skip useless (optional) GMT keyword */
      if (!strncmp (field, "GMT", 3))
	{
	  end = find_field_end (field);
	  if (*end)
	    *end++ = '\0';
	  field = end;
	}

      /* from CALL to GMTEND */
      for (i = 2; i < 9; i++)
	{
	  end = find_field_end (field);
	  if (*end)
	    *end++ = '\0';
	  q[twlog_fields[i]] = g_strdup (g_strstrip (field));
	  field = end;
	}
      /* ignore "GMT" keyword on last column */

      /* does not exist in twlog. TODO: select a default? */
      q[QSLOUT] = g_strdup ("");
      q[QSLIN] = g_strdup ("");

      /* line with remarks */
      fgets (buffer, 81, fp);
      rlen = strlen (buffer);
      q[REMARKS] = g_strdup (g_strstrip (buffer));

      if (rlen == 80 && buffer[79] != '\n')
	{
	  /* old twlog format, with space padded remarks */
	  fgets (buffer, 3, fp);
	/*EOL*/}

      /* (supposed)empty line */
      fgets (buffer, 80, fp);

      for (i = 0; i < twlog_field_nr; i++)
	if (!q[twlog_fields[i]])
	  q[twlog_fields[i]] = g_strdup ("");

      ret = (*fn) (handle, q, arg);
      if (ret)
	return ret;
    }
  return 0;
}
