/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "themes.h"
#include "eel-gconf-extensions.h"

#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <libgnome/gnome-util.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* cache of theme pixmaps */
static GHashTable *pixmaps_cache = NULL;
static GHashTable *prelight_pixmaps_cache = NULL;

/* theme directory, cached */
static gchar *themedir = NULL;

/* Local prototypes */
static gchar *themes_get_nautilus_theme_path (void);
static gboolean themes_pixmap_free_cb (gpointer key, PixmapData *value,
				      gpointer user_data);
static const gchar *themes_get_theme_dir (void);

/**
 * themes_get_nautilus_theme_path: get the path of the 
 * current nautilus theme
 */
static gchar *
themes_get_nautilus_theme_path (void)
{
	gchar *theme, *icon, *tmp, *result = NULL;
	gint i;

	static gchar *path[] =
	{
		GNOME_PIXMAPS_DIR "/nautilus/%s",
		GNOME_PIXMAPS_DIR "/nautilus/%s/fake-aa-navibar",
		NULL,
		NULL,
		GNOME_PIXMAPS_DIR "/nautilus/",
		NULL
	};

	theme = eel_gconf_get_string (CONF_NAUTILUS_THEME);

	if (!theme || strcmp (theme, "default") == 0)
	{
		g_free (theme);
		theme = g_strdup ("");
	}
	else if (strcmp (theme, "gnome") == 0) /* use standard icons */
	{
		g_free (theme);
		return NULL;
	}

	path[2] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/%s", NULL);
	path[3] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/fake-aa-navibar/%s",
			       NULL);

	for (i = 0; path[i] != NULL && !result; i++)
	{
		tmp = g_strdup_printf (path[i], theme);
		icon = g_strconcat (tmp, "/Refresh.png", NULL);

		if (g_file_exists (icon))
		{
			result = g_strdup (tmp);
		}
		g_free (tmp);
		g_free (icon);
	}

	g_free (path [2]);
	g_free (path [3]);
	g_free (theme);

	return result;
}

/**
 * themes_pixmap_free_cb: free one theme pixmap
 */
static gboolean
themes_pixmap_free_cb (gpointer key, PixmapData *value, gpointer user_data)
{
	g_free (key);
	misc_gui_pixmap_data_free (value);
	return TRUE;
}

/**
 * themes_reset_cache: Reset the theme pixmap cache, used when changing
 * themes
 */
void
themes_reset_cache (void)
{
	if (prelight_pixmaps_cache != NULL)
	{
		g_hash_table_freeze (prelight_pixmaps_cache);
		g_hash_table_foreach_remove (prelight_pixmaps_cache,
			(GHRFunc) themes_pixmap_free_cb, NULL);
		g_hash_table_thaw (prelight_pixmaps_cache);
	}

	if (pixmaps_cache != NULL)
	{
		g_hash_table_freeze (pixmaps_cache);
		g_hash_table_foreach_remove (pixmaps_cache,
			(GHRFunc) themes_pixmap_free_cb, NULL);
		g_hash_table_thaw (pixmaps_cache);
	}

	g_free (themedir);
	themedir = NULL;
}

/**
 * themes_get_pixmap: get the PixmapData of a given file, taking themes and
 * caching into account.
 */
PixmapData *
themes_get_pixmap (const gchar *filename, gboolean prelight)
{
	PixmapData *pixmap = NULL;
	const gchar *theme;
	gchar *tmp;

	/* lookup in cache */
	if (prelight && prelight_pixmaps_cache != NULL)
		pixmap = g_hash_table_lookup (prelight_pixmaps_cache, filename);
	else if (!prelight && pixmaps_cache != NULL)
		pixmap = g_hash_table_lookup (pixmaps_cache, filename);

	if (pixmap != NULL)
		return pixmap;

	theme = themes_get_theme_dir ();
	
	if (theme)
	{
		tmp = g_concat_dir_and_file (theme, filename);

		if (g_file_exists (tmp))
		{
			pixmap = misc_gui_pixmap_data_new_from_file (tmp,
								prelight);
		}

		g_free (tmp);
	}
	
	if (pixmap == NULL)
	{
		tmp = misc_general_user_file ((gchar *) filename, FALSE);

		if (tmp && g_file_exists (tmp))
		{
			pixmap = misc_gui_pixmap_data_new_from_file (tmp,
								prelight);
		}

		g_free (tmp);
	}

	/* if we didn't manage to load the icon, make an empty structure
	 * anyway, this allows the browser to run even when not fully 
	 * installed.... */
	if (pixmap == NULL)
	{
		pixmap = g_new0 (PixmapData, 1);
	}

	/* cache the created PixmapData */
	if (prelight)
	{
		if (prelight_pixmaps_cache == NULL)
		{
			prelight_pixmaps_cache = g_hash_table_new (g_str_hash,
							  	   g_str_equal);
		}

		g_hash_table_insert (prelight_pixmaps_cache,
				     g_strdup (filename), pixmap);
	}
	else
	{
		if (pixmaps_cache == NULL)
		{
			pixmaps_cache = g_hash_table_new (g_str_hash,
							  g_str_equal);
		}

		g_hash_table_insert (pixmaps_cache, g_strdup (filename),
				     pixmap);
	}

	return pixmap;
}

/**
 * themes_get_height: get the pixmap height of the current toolbar theme
 */
gint
themes_get_height ()
{
	gint icon_height;
	PixmapData *icon = themes_get_pixmap ("Refresh.png", FALSE);

	if (icon->mask)
	{
		gdk_window_get_size (icon->mask, NULL, &icon_height);
	}
	else
	{
		icon_height = 24; /* GNOME stock pixmap height */
	}

	return icon_height;
}

/**
 * themes_get_theme_dir: retreive current theme dir
 */
static const gchar *
themes_get_theme_dir ()
{
	if (themedir) return themedir;

	themedir = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);
	
	if (themedir && strcmp (themedir, "NAUTILUS_THEME") == 0)
	{
		g_free (themedir);
		themedir = themes_get_nautilus_theme_path ();
	}

	return themedir;
}
