/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "misc_general.h"
#include "misc_string.h"
#include "mozilla.h"
#include "eel-gconf-extensions.h"

/* system includes */
#include <string.h>
#include <glib.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <gnome-xml/xmlmemory.h>

/**
 * misc_general_user_file: returns the pathname of galeon shared files
 * (e.g., galeon.glade)
 *
 * fname: just the filename with no path information
 * critical: critical file? (halt if not found)
 */
gchar *
misc_general_user_file (gchar *fname, gboolean critical)
{
	static GHashTable *already_found = NULL;
	gchar *alternative[7];
	gchar *file;
	gint i;
	
	/* create cache hash table if it doesnt already exist */
	if (already_found == NULL)
	{
		already_found = g_hash_table_new (g_str_hash, g_str_equal);
	}

        /* Have we already found this? */
	file = g_hash_table_lookup (already_found, fname);
	if (file != NULL) 
	{
		return g_strdup (file);
	}

	/* try the default */
	file = g_strconcat (g_get_home_dir (), "/.galeon/", fname, NULL);
	
	/* success? */
	if (g_file_exists (file))
	{
		return file;
	}
	g_free(file);

	/* specify alternate locations in order of precedence */
	alternative[0] = g_strdup (fname);
	alternative[1] = g_strconcat ("../", fname, NULL);
	alternative[2] = g_strconcat ("ui/", fname, NULL);
	alternative[3] = g_strconcat ("../ui/", fname, NULL);
	alternative[4] = g_strconcat (SHARE_DIR "/", fname, NULL);
	alternative[5] = g_strconcat (GNOME_PIXMAPS_DIR "/", fname, NULL);
	alternative[6] = NULL;  /* NULL terminator needed */
	
	/* select one of the alternatives */
	file = NULL;
	for (i = 0; alternative[i] != NULL; i++)
	{
		if (file == NULL && g_file_exists (alternative[i])) 
		{
			file = alternative[i];
                        /* don't warn for the install default */
			if (i < 4)
			{
				g_message ("Using %s (usually OK)\n", file);
			}
		}
		else
		{
			/* free unused string */
			g_free (alternative[i]);
		}
	}

	/* check for success */
	if (file != NULL)
	{
		/* add it to the set of found files */
		g_hash_table_insert (already_found, g_strdup (fname), 
				     g_strdup (file));
	}
	/* if nothing then theres an error */
	else if (critical)
	{
		g_error(_("%s not found"), fname);
	}

	/* return result */
	return file;
}


/**
 * misc_general_read_line_from_file: reads a line from an opened file and
 * returns it in a new allocated string
 */
gchar *
misc_general_read_line_from_file (FILE *f)
{
	gchar *line = g_strdup ("");
	gchar *t;
	gchar *buf = g_new0 (gchar, 256);
	while ( ! ( strchr (buf, '\n') || feof (f) ) ) {
		fgets(buf, 256, f);
		t = line;
		line = g_strconcat (line, buf, NULL);
		g_free (t);
	}
	g_free (buf);
	return line;
}

/**
 * misc_general_launch_external_viewer: launches the external source viewer
 * to view the given file.
 */
void
misc_general_launch_external_viewer (const gchar *filename)
{
	gchar *prog, *command;
	int pid;

	g_return_if_fail(filename != NULL);

	/* mark the file to be deleted upon galeon exit. */
	mozilla_delete_temp_file_on_exit (filename);

	/* get the program to execute */
	prog = eel_gconf_get_string (CONF_PROGRAMS_EXTERNAL_SOURCE_VIEWER);
	if (!prog) return;

	command = g_strconcat (prog," ", filename, NULL);

	/* execute the viewer */
	pid = gnome_execute_shell (NULL, command);

	/* free allocated string */
	g_free (command);
	g_free (prog);
}

/**
 * misc_general_key_state_to_link_state:
 */
LinkState
misc_general_key_state_to_link_state (guint evstate)
{
	return ((evstate & GDK_CONTROL_MASK) ? 1 : 0) |
		((evstate & GDK_SHIFT_MASK) ? LINKSTATE_SHIFTED : 0);

}

/**
 * misc_general_mouse_state_to_link_state:
 */
LinkState
misc_general_mouse_state_to_link_state (guint button, guint evstate)
{
	return ((button - 1) |
		((evstate & GDK_SHIFT_MASK)   ? LINKSTATE_SHIFTED : 0) |
		((evstate & GDK_CONTROL_MASK) ? LINKSTATE_CTRLED  : 0) |
		((evstate & GDK_MOD1_MASK)    ? LINKSTATE_ALTED   : 0));
}

/**
 * xmlSetRawProp: set a property in an XML file which has
 * value encoded.  This works around the newline-losingness and broken
 * entitiy handling of libxml1. (see bug #56487)
 */
xmlAttrPtr
xmlSetRawProp (xmlNodePtr node, const xmlChar *name, const xmlChar *value)
{
	xmlAttrPtr attr;
	gchar *escaped = misc_string_escape_xml_prop (value);
	attr = xmlSetProp (node, name, escaped);
	g_free (escaped);
	return attr;
}
/**
 * xmlGetRawProp: get a property in an XML file which has been encoded This
 * works around the newline-losingness and broken entitiy handling of
 * libxml1. (see bug #56487)
 */
xmlChar *
xmlGetRawProp (xmlNodePtr node, const xmlChar *name)
{
	xmlChar *value;
	gchar *unescaped;
	value = xmlGetProp (node, name);
	if (value == NULL)
	{
		return NULL;
	}
	unescaped = misc_string_unescape_hexed_string (value);
	xmlFree (value);
	return unescaped;
}

/**
 * xmlGetIntProp:
 */
gint
xmlGetIntProp (xmlNodePtr node, const gchar *attribute)
{
	gchar *string_value;
	gint value;

	/* get the attribute as a string */
	string_value = xmlGetProp (node, attribute);

	/* convert to integer */
	/* FIXME: atoi is crap */
	value = atoi (string_value);

	/* free allocated string */
	xmlFree (string_value);

	/* return discovered value */
	return value;
}
