/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "autocompletion.h"
#include "prefs.h"
#include "history.h" /* to lookup for titles */

#include <string.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtkrc.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <gdk/gdkkeysyms.h>

/**
 * Total number of completions to offer
 */
#define MAX_ALTERNATIVES 200

/**
 * Avaible completions
 */
GCompletion *URLCompletion = NULL;

/** 
 * List of possible completions for the current entry
 */
GList *possible_completions = NULL;

/**
 * The alternatives window
 */ 
GtkWidget *alternatives_window = NULL;

/**
 * Common autocompletion prefixes
 * Please note that the order of this list is important
 * for simplifying urls : 
 * when we have http://www.foo.bar, 
 * we want to get foo.bar as simplified url 
 * so http://www. must come before http://
 */
const gchar *common_prefixes[] = 
{ 
	"", 
	"http://www.", 
	"http://", 
	"https://www.", 
	"https://", 
	"file://", 
	"www.", 
	NULL             /* terminator, must be last */
};

/* Private functions */
static GList *auto_completion_insert_sorted_nodup (GList *l, char *url);
static GList *auto_completion_concat (GList *l1, GList *l2);
static gchar *auto_completion_suggest (gchar *prefix);
static gboolean alternatives_win_button_press_cb (GtkWidget *widget,
						  GdkEventButton *event,
						  gpointer data); 
static gboolean alternatives_win_key_press_cb (GtkWidget *widget,
					       GdkEventKey *event,
					       GtkWidget *location_entry);
static gboolean alternatives_clist_key_press_cb (GtkWidget *widget,
						 GdkEventKey *event,
						 GtkWidget *location_entry);
static gboolean alternatives_clist_button_press_cb (GtkWidget *widget,
						    GdkEventButton *event,
						    GtkWidget *location_entry);
static void alternatives_jump_to_clist_selection (GtkWidget *entry, 
						  GtkCList *clist);
static GList *auto_completion_url_abbrev_list (gchar *url);
static gchar *auto_completion_get_page_title (const gchar *abrev_url);



/**
 * Adds an url and all it's possible abbreviations to the list of completions
 */
void
auto_completion_add_url (gchar *url)
{
	GList *list;

	/* build the completion structure if it doesn't yet exist */
	if (URLCompletion == NULL)
	{
		URLCompletion = g_completion_new (NULL);
	}
	if (url == NULL) return;

	/* get the abreviated urls */
	list = auto_completion_url_abbrev_list (url);
	if (!list) return;

	/* add each item from thelist to the global */
	g_completion_add_items (URLCompletion, list);

	/* free the local list */
	g_list_free (list);
}

/**
 * Removes an URL and its possible abbreviations from the completion list
 */
void
auto_completion_remove_url (gchar *url)
{
	GList *list;

	/* get the abreviated urls */
	list = auto_completion_url_abbrev_list (url);
	if (!list) return;

	/* add each item from thelist to the global */
	if (list)
		g_completion_remove_items (URLCompletion, list);

	/* free the local list */
	g_list_free (list);
}

/**
 * Returns a list of abbreviations of the url than need to be added to the 
 * competion list. The pointers of the list point inside the passed string!
 */
static GList *
auto_completion_url_abbrev_list (gchar *url)
{
	gint len = 0;
	GList *list = NULL;
	gint i;

	/* skip anything resembling a smart bookmark or NULL */
	if ((url == NULL) || strstr (url, "%s")) return NULL;
       
	/* add all completions to a list */
	for (i = 0; common_prefixes[i] != NULL; i++)
	{
		/* add with stripped prefix */
		len = strlen (common_prefixes[i]);
		if (strncmp(url, common_prefixes[i], len) == 0)
		{
		        list = g_list_prepend(list, url + len);
		}
	}

	return list;
}

/**
 * Returns the page title of a URL that may have been simplified.
 * Returns "" if not found.
 */
static gchar *
auto_completion_get_page_title (const gchar *abrev_url)
{
	int i;
	for (i = 0; common_prefixes[i] != NULL; i++)
	{
		gchar *full_url = g_strconcat (common_prefixes[i], abrev_url,
					       NULL);
		const gchar *title = history_get_page_title (full_url);
		g_free (full_url);
		if (title) return g_strdup (title);
	}
	return g_strdup ("");
}

/**
 * Destroys the alternatives window and reset the list of _current_ possible 
 * completions. Does not clear the whole completions list.
 */
void 
auto_completion_reset (void)
{
	g_list_free(possible_completions);
	possible_completions = NULL;

	if (alternatives_window != NULL)
	{
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (alternatives_window);
		gtk_widget_destroy (alternatives_window);
	}
	alternatives_window = NULL;
}

/** 
 * Tries to complete an url 
 * Returns a possible completion of the url being typed
 * You mustn't free the returned string
 */
gchar *
auto_completion_complete_url (gchar *prefix)
{
	gchar *common;
	GList *tmp = NULL;

	/* check we have a completion structure */
	if (URLCompletion == NULL)
	{
		return NULL;
	}

	/* try to complete */
	tmp = g_completion_complete (URLCompletion, prefix, &common);

	/* this might be NULL */
	return common;
}

/**
  * Concatenate the two GLists l1 and a copy of l2
  */
static GList *auto_completion_concat(GList *l1, GList *l2)
{
	while (l2!=NULL) {
		l1 = g_list_prepend (l1, l2->data);
		l2 = g_list_next (l2);
	}
	return l1;
}

/** 
 * Like auto_completion_complete_url but tries to prepend "http://" and similar
 * returns the longest common prefix of all the possible completions
 * You must free the returned string
 */
gchar *
auto_completion_complete_url_extended (gchar *prefix, gboolean suggest)
{
	gchar *common = NULL;
	GList *tmp = NULL;
	gchar *common_prefix = NULL;
	gchar *extended_prefix;
	int i;

	/* free existing list of completions */
	g_list_free (possible_completions);
	possible_completions = NULL;

	/* check prefix is valid and completions are available */
	if (prefix == NULL || strlen (prefix) == 0 || URLCompletion == NULL) 
	{
		return NULL;	
	}

	for (i = 0; common_prefixes[i] != NULL; i++)
	{
		/* prefix the "prefix" with some standard prefixes :-) */
		extended_prefix = g_strconcat (common_prefixes[i],
					       prefix, NULL);

		/* try to find a common completion */
		tmp = g_completion_complete (URLCompletion, extended_prefix,
					     &common);

		/* add to the list if found */
		if (tmp != NULL)
		{
			possible_completions = auto_completion_concat
				(possible_completions, tmp);
		}

		/* store this common prefix or free it if we've got one */
		if (common != NULL)
		{
			if (!common_prefix)
			{
				common_prefix = common;
			}
			else 
			{
			       	g_free(common);
			}
		}

		g_free (extended_prefix);
		common = NULL;
	}

	/* see if the common prefix we obtained is just what we started with */
	if ((common_prefix != NULL) && (strcmp (common_prefix, prefix) == 0)
	    && (suggest == TRUE))
	{
		/* yes, use more advanced strategy */
		g_free (common_prefix);
		common_prefix = auto_completion_suggest (prefix);
	}

	return common_prefix;
}


/**
 * auto_completion_suggest: this is a desperation strategy designed
 * to help when the user hits tab but the current text is the maximal
 * completion. At the moment this is quite severely rubbish, this 
 * needs a bit more thought -- MattA
 */
static gchar *
auto_completion_suggest (gchar *prefix)
{
	gchar *extended_prefix;
	gchar *common_prefix;
	gint next_letter[256];
	GList *l;
	gint i, n;
	gchar best;
	gint max;

	/* zero the totals */
	for (i = 0; i < 256; i++)
	{
		next_letter[i] = 0;
	}
	
	/* find the most popular next letter */
	best = 0;
	max = 0;
	n = strlen (prefix);
	l = g_completion_complete (URLCompletion, prefix, NULL);
	for (; l != NULL; l = g_list_next (l))
	{
		gchar *text = (gchar *)(l->data);

		/* valid to test? */
		if ((gint) strlen (text) > n)
		{
			gint letter = text[n];

			next_letter[letter]++;

			/* new best? */
			if (next_letter[letter] >= max)
			{
				max = next_letter[letter];
				best = letter;
			}
		}
	}

	/* no hoper */
	if (best == 0)
	{
		return g_strdup (prefix);
	}

	/* tack it onto the end of the word and and complete */
	extended_prefix = g_strdup_printf ("%s%c", prefix, best);
	g_completion_complete (URLCompletion, extended_prefix, 
			       &common_prefix);

	/* return the completed best guess */
	return common_prefix;
}

/**
 * insert the string data into the given glist 
 * the list returned is sorted and without duplicates 
 * type indicate whether we should consider full urls or
 * simplified urls
 */

/* FIXME FIXME FIXME pleeease.... */
static GList *
auto_completion_insert_sorted_nodup (GList *l, char *url)
{
	GList *tmp = l;
	int i = 0;
	int pos = 0;
	gchar *data = NULL;

	/** Inserts url into the GList l */
	while (tmp != NULL) {
		data = tmp->data;
		i = strcmp (url, data);
		if (i == 0) return l;
		if (i < 0) {
			break;
		} else {
			tmp = g_list_next (tmp);
			pos++;
		}
	}
	l = g_list_insert (l, url, pos);
	return l;
}


/** 
 * displays an completed url in the location bar
 * according to the preferences, appends a selected autocompleted 
 * string to the end of the url
 */
void
auto_completion_display (GaleonWindow *window, GtkWidget *entry)
{
	gchar *text;
	gint text_len;
	gchar *completion;

	/* check to see if autocompletion is enabled */
	if (!eel_gconf_get_boolean (CONF_HISTORY_AUTOCOMP_ENABLE))
	{
		return;
	}

	/* if so, get the text and try to compute a completion */
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	completion = auto_completion_complete_url (text);
	if (completion != NULL)
	{
		/* check original length */
		text_len = strlen (text);

		/* set entry to completed text */
		gtk_entry_set_text (GTK_ENTRY (entry), completion);

		/* move selection appropriately */
		gtk_editable_select_region (GTK_EDITABLE(entry), text_len, -1);
		gtk_editable_set_position (GTK_EDITABLE(entry),  -1);
		g_free (completion);
	}

	/* free allocated string */
	g_free (text);
}

static void
alternatives_jump_to_clist_selection (GtkWidget *entry, GtkCList *clist)
{
	if (GTK_IS_ENTRY (entry)
	    && clist->selection) {
		GdkEventKey tmp_event;
		gchar *text;
		int row = GPOINTER_TO_INT (clist->selection->data);
	
		gtk_clist_get_text (clist, row, 0, &text);
		gtk_entry_set_text (GTK_ENTRY (entry), text);
		auto_completion_reset();
		
		/* send a synthetic return keypress to the entry */
		tmp_event.type = GDK_KEY_PRESS;
		tmp_event.window = entry->window;
		tmp_event.send_event = TRUE;
		tmp_event.time = GDK_CURRENT_TIME;
		tmp_event.state = 0;
		tmp_event.keyval = GDK_Return;
		gtk_widget_event(entry, (GdkEvent *)&tmp_event);
	}
}

static gboolean 
alternatives_clist_button_press_cb (GtkWidget *widget, GdkEventButton  *event, 
				    GtkWidget *entry)
{
	GtkCList *clist = GTK_CLIST (widget);
	if (event->button == 1)
	{
		gint row, column;
		gtk_clist_get_selection_info (clist, event->x, event->y, 
					      &row, &column);
		gtk_clist_select_row (clist, row, 0);
		alternatives_jump_to_clist_selection (entry, clist);
	}
	return TRUE;
}

static gboolean
alternatives_clist_key_press_cb  (GtkWidget *widget, GdkEventKey *event,
				  GtkWidget *entry)
{
	GtkCList *clist = GTK_CLIST (widget);
	if (clist->focus_row >= 0)
	{
		gtk_clist_select_row (clist, clist->focus_row, 0);
	}

	if (((event->keyval == GDK_Return) 
	     || event->keyval == GDK_space)
	    && (clist->selection))
	{
		alternatives_jump_to_clist_selection (entry, clist);
	}
	return FALSE;
}


/**
 * Displays the list of possible completions under the location box of the
 * given browser
 */
void 
auto_completion_display_alternatives (GaleonWindow *window, GtkWidget *entry,
				      gboolean shortlist)
{
	gint x, y, height, width;
	gint opt0, opt1, totalwidth, maxwidth;
	GtkWidget *scroll;
	GtkWidget *clist;
	GtkWidget *old_alternatives_window;
	GtkRequisition r;
	GList *l;
	int count = 0;
	GList *labels = NULL;
	gint max = shortlist
		? eel_gconf_get_integer (CONF_COMPLETION_MAX_SHORT)
		: MAX_ALTERNATIVES;
	gboolean show_titles = eel_gconf_get_boolean 
		(CONF_COMPLETION_SHOW_TITLES);
	GtkStyle *rcstyle, *grey_style;
	GtkWidget *dummy;

	/* init style */
	dummy = gtk_label_new ("");
	rcstyle = gtk_rc_get_style (dummy);
	if (rcstyle == NULL)
		rcstyle = gtk_style_new ();
	grey_style = gtk_style_copy (rcstyle);
	gtk_widget_destroy (dummy);
	gdk_color_parse ("#808080", &(grey_style->fg[0]));
	
	if (possible_completions == NULL) 
	{
		/* hide the window */
		auto_completion_reset (); 
		return;
	}

	if (show_titles)
		clist = gtk_clist_new (2); 
	else
		clist = gtk_clist_new (1);

	/* hack to reduce flickering: don't destroy the old window until 
	 * the new one has been shown */	
	old_alternatives_window = alternatives_window;

	alternatives_window = gtk_window_new (GTK_WINDOW_POPUP);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
		       			GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (alternatives_window), scroll);
	gtk_container_add (GTK_CONTAINER (scroll), clist);

	for (l = possible_completions; (l != NULL) && (count < max);
	     l = g_list_next (l), count++) 
	{
		labels = auto_completion_insert_sorted_nodup (labels, l->data);
	}

	for (l =  labels; l != NULL; l = g_list_next (l)) 
		if (show_titles)
		{
			char *text[] = { l->data, "", NULL };
			char *title = auto_completion_get_page_title (l->data);
			gint row;
			text[1] = title;
			row = gtk_clist_append (GTK_CLIST (clist), text);
			gtk_clist_set_cell_style (GTK_CLIST (clist), row, 1,
						  grey_style);
			g_free (title);
		}
		else
		{
			char *text[] = { l->data, NULL };
			gtk_clist_append (GTK_CLIST (clist), text);
		}
	
	g_list_free(labels);
	gtk_clist_columns_autosize (GTK_CLIST (clist));

	/* Urgh. As of GTK+ 1.2.10, this isn't getting initialized correctly,
	   so we have to do it ourselves */
	GTK_CLIST (clist)->focus_row = -1;

	gdk_window_get_size (entry->window, &width, &height);
	gdk_window_get_deskrelative_origin (entry->window, &x, &y);
	y += height;
	gtk_widget_set_uposition (alternatives_window, x, y);

	/* calculate the best width */
	opt0 = gtk_clist_optimal_column_width (GTK_CLIST (clist), 0);
	opt1 = show_titles 
		? gtk_clist_optimal_column_width (GTK_CLIST (clist), 1)
		: 0;
	totalwidth = opt0 + opt1 + 40; /* FIXME: 40 is a magic number */
	if (eel_gconf_get_boolean (CONF_COMPLETION_HUGE_WINDOW))
	{
		maxwidth = gdk_screen_width () - x;
		if (totalwidth > width)
		{
			width = totalwidth > maxwidth
				? maxwidth
				: totalwidth;
		}
	}

	/* For some reason, the requisition height won't be correct until
	 * the clist has been shown. I get it twice and set the size of the 
	 * window twice to avoid flickering -- ricardo
	 */
       	gtk_widget_size_request (clist, &r);
	gtk_widget_set_usize (alternatives_window, width, r.height);
	gtk_widget_show_all (alternatives_window);       
	gtk_widget_size_request (clist, &r);

	/* set the width of the columns */
	if (show_titles)
	{
		if (totalwidth > width) 
		{
			gint width23 = width * 2 / 3;
			gint c0width = opt0 > width23
				? width23 
				: opt0;
			gtk_clist_set_column_width (GTK_CLIST (clist), 0, 
						    c0width);
			gtk_clist_set_column_width (GTK_CLIST (clist), 1, 
						    opt1);
		}
	}

	if ((y + r.height) > gdk_screen_height ()) {
		gtk_window_set_policy
			(GTK_WINDOW (alternatives_window), TRUE, FALSE, FALSE);
		gtk_widget_set_usize 
			(alternatives_window, width, gdk_screen_height () - y);
		
	}

	gtk_signal_connect (GTK_OBJECT(alternatives_window),
			    "button-press-event",
			    GTK_SIGNAL_FUNC(alternatives_win_button_press_cb),
			    alternatives_window);
	gtk_signal_connect (GTK_OBJECT(alternatives_window),
			    "key-press-event",
			    GTK_SIGNAL_FUNC(alternatives_win_key_press_cb),
			    entry);
	gtk_signal_connect_after (GTK_OBJECT (clist),
				  "key-press-event",
				  GTK_SIGNAL_FUNC
				  (alternatives_clist_key_press_cb),
				  entry);
	gtk_signal_connect (GTK_OBJECT (clist),
				  "button-press-event",
				  GTK_SIGNAL_FUNC
				  (alternatives_clist_button_press_cb),
				  entry);

	gtk_container_set_focus_child (GTK_CONTAINER (alternatives_window), 
				       GTK_WIDGET (clist));

	gdk_pointer_grab (alternatives_window->window, TRUE,
			  GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK,
			  NULL, NULL, GDK_CURRENT_TIME);
	gtk_grab_add (alternatives_window);

	/* now, destroy the old one */
	if (old_alternatives_window || GTK_IS_WIDGET (old_alternatives_window))
	{
		gtk_widget_destroy (old_alternatives_window);
	}

	gtk_style_unref (grey_style);
}

static gboolean
alternatives_win_button_press_cb (GtkWidget *widget,
				  GdkEventButton *event,
				  gpointer data)
{
	GtkWidget *event_widget;

	event_widget = gtk_get_event_widget ((GdkEvent *) event);

	/* Check to see if button press happened inside the alternatives
	   window.  If not, destroy the window. */
	if (event_widget != widget)
	{
		while (event_widget)
		{
			if (event_widget == widget)
				return FALSE;
			event_widget = event_widget->parent;
		}
	}
	auto_completion_reset();

	return TRUE;
}

static gboolean
alternatives_win_key_press_cb (GtkWidget *widget,
			       GdkEventKey *event,
			       GtkWidget *location_entry)
{
	GdkEventKey tmp_event;
	GtkWidget *dest_widget = location_entry;
	GtkCList *clist = GTK_CLIST (GTK_BIN (GTK_BIN (widget)->child)->child);

	/* allow keyboard navigation in the alternatives clist */
	if (event->keyval == GDK_Up || event->keyval == GDK_Down 
	    || event->keyval == GDK_Page_Up ||  event->keyval == GDK_Page_Down
	    || ((event->keyval == GDK_space || event->keyval == GDK_Return)
		&& clist->selection))
	{
		/* forward the event to the clist, instead of the entry */
		dest_widget = GTK_WIDGET (clist);
	}
	
	/* else send the key event to the location entry */
	tmp_event.type = event->type;
	tmp_event.window = location_entry->window;
	tmp_event.send_event = TRUE;
	tmp_event.time = event->time;
	tmp_event.state = event->state;
	tmp_event.keyval = event->keyval;
	tmp_event.length = event->length;
	tmp_event.string = event->string;
	gtk_widget_event (dest_widget, (GdkEvent *)&tmp_event);

	/* hmm... something seems to have changed in the GTK+ signal handling
	   code.  Returning true no longer seems to stop emission of the key
	   press signal in the clist window, so we have to do it manually */
	gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "key-press-event");

	return TRUE;
}

void
auto_completion_clear (void)
{
	g_completion_clear_items (URLCompletion);
}

