/*
 * man.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.  
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Some manpath, compression and locale related changes - aeb - 940320
 * Some suid related changes - aeb - 941008
 * Some more fixes, Pauline Middelink & aeb, Oct 1994
 * man -K: aeb, Jul 1995
 */

#define MAN_MAIN

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <locale.h>
#include "gripes.h"
#include "man.h"

#ifndef R_OK
#define R_OK 4
#endif

extern int ruid, euid, rgid, egid, suid;

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#else
extern char *getenv ();
extern void free ();
extern int strcmp ();
extern int strncmp ();
extern int exit ();
extern int fflush ();
extern int printf ();
extern FILE *fopen ();
extern int fclose ();
extern char *sprintf ();
#endif

extern char *rindex ();		/* not always in <string.h> */
extern char *my_malloc ();
extern char *get_expander ();
extern char *getval ();
extern char *convert_to_cat ();
extern char **glob_filename ();
extern int access ();
extern int unlink ();
extern int chmod ();
extern int is_newer ();
extern int do_system_command ();
extern int man_from_section ();
extern void gripe (), manpath ();
extern int global_apropos;

char *progname;
char *pager;
char *colon_sep_section_list;
char *roff_directive;
int apropos;
int whatis;
int nocats;
int findall;
int print_where;
int one_per_line;
int do_troff;
int debug;
int fsstnd;
struct dir srchlist;

static char *section;
static char **section_list;

#ifdef DO_COMPRESS
int do_compress = 1;
#else
int do_compress = 0;
#endif

#define BUFSIZE 4096

/*
 * Try to determine the line length to use.
 * Preferences: 1. MANWIDTH 2. ioctl 3. COLUMNS 4. 80
 *
 * joey, 950902
 */

#include <sys/ioctl.h>

int line_length = 80;
int ll = 0;

void
get_line_length(){
    char *cp;
    int width;

    if ((cp = getenv ("MANWIDTH")) != NULL && (width = atoi(cp)) > 0) {
	line_length = width;
	return;
    }
#ifdef TIOCGWINSZ
    if (isatty(0) && isatty(1)) { /* Jon Tombs */
	struct winsize wsz;

	if(ioctl(0, TIOCGWINSZ, &wsz))
	  perror("TIOCGWINSZ failed\n");
	else if(wsz.ws_col) {
	    line_length = wsz.ws_col;
	    return;
	}
    }
#endif
    if ((cp = getenv ("COLUMNS")) != NULL && (width = atoi(cp)) > 0)
      line_length = width;
    else
      line_length = 80;
}

int
setll(){
  return
    (!do_troff && (line_length < 66 || line_length > 80)) ?
	line_length*9/10 : 0;
}

/*
 * Check to see if the argument is a valid section number.  If the
 * first character of name is a numeral, or the name matches one of
 * the sections listed in section_list, we'll assume that it's a section.
 * The list of sections in config.h simply allows us to specify oddly
 * named directories like .../man3f.  Yuk. 
 */
char *
is_section (name)
     register char *name;
{
  register char **vs;

  for (vs = section_list; *vs != NULL; vs++)
    if ((strcmp (*vs, name) == 0)
	|| (isdigit (name[0]) && !isdigit (name[1])))
      return my_strdup (name);

  return NULL;
}


void
remove_file (file)
     char *file;
{
  int i;

  i = unlink (file);

  if (debug) {
      if (i)
	perror(file);
      else
	gripe (UNLINKED, file);
  }
}

void
remove_other_catfiles (catfile)
     char *catfile;
{
  char pathname[BUFSIZE];
  char *t;
  char **gf;
  int offset;

  strcpy (pathname, catfile);
  t = rindex(pathname, '.');
  if (t == NULL || strcmp(t, getval("COMPRESS_EXT")))
    return;
  offset = t - pathname;
  strcpy(t, "*");
  gf = glob_filename (pathname);

  if (gf != (char **) -1 && gf != NULL) {
      for ( ; *gf; gf++) {
	  /*
	   * Only remove files with a known extension, like .Z
	   * (otherwise we might kill a lot when called with
	   * catfile = ".gz" ...)
	   */
	  if (strlen (*gf) <= offset) {
	      if (strlen (*gf) == offset)  /* uncompressed version */
		  remove_file (*gf);
	      continue;
	  }

	  if (!strcmp (*gf + offset, getval("COMPRESS_EXT")))
	    continue;

	  if (get_expander (*gf) != NULL)
	    remove_file (*gf);
      }
  }
}

/*
 * Try to find the man page corresponding to the given name.  The
 * reason we do this with globbing is because some systems have man
 * page directories named man3 which contain files with names like
 * XtPopup.3Xt.  Rather than requiring that this program know about
 * all those possible names, we simply try to match things like
 * .../man[sect]/name[sect]*.
 */
char **
glob_for_file_ext (path, section, name, cat, ext)
     char *path, *section, *name, *ext;
     int cat;
{
  char pathname[BUFSIZE];
  char *p;

  if (cat) {
    sprintf (pathname, "%s/man%s/%s.%s*", path, section, name, ext);
    p = convert_to_cat(pathname,0,cat);
    if (p) {
	strcpy(pathname,p);
	free(p);
    }
    else sprintf (pathname, "%s/cat%s/%s.%s*", path, section, name, ext);
  }
  else
    sprintf (pathname, "%s/man%s/%s.%s*", path, section, name, ext);

  if (debug)
    gripe (GLOBBING, pathname);

  return glob_filename (pathname);
}

char **
glob_for_file (path, section, name, cat)
     register char *path;
     register char *section;
     register char *name;
     register int cat;
{
  char **gf;
  char ext[2];

  gf = glob_for_file_ext (path, section, name, cat, section);
  if ((gf == (char **) -1 || gf == NULL || *gf == NULL) &&
      isdigit (*section) && section[1] != 0) {
      ext[0] = *section;
      ext[1] = 0;
      gf = glob_for_file_ext (path, section, name, cat, ext);
  }
  if (gf == (char **) -1 || gf == NULL || *gf == NULL)
    gf = glob_for_file_ext (path, section, name, cat, "man");
  if (gf == (char **) -1)
    gf = NULL;
  return gf;
}

/*
 * Simply display the preformatted page.
 */
int
display_cat_file (file)
     register char *file;
{
  extern int different_cat_file();
  register int found;
  char command[BUFSIZE];

  found = 0;

  if (access (file, R_OK) == 0 && different_cat_file(file)) {
      char *expander = get_expander (file);

      if (expander != NULL && expander[0] != 0) {
	if (isatty(1))
	  sprintf (command, "%s %s | %s", expander, file, pager);
	else
	  sprintf (command, "%s %s", expander, file);
      } else {
	if (isatty(1))
	  sprintf (command, "%s %s", pager, file);
	else {
	    char *cat = getval("CAT");
	    sprintf (command, "%s %s", cat ? cat : "cat", file);
	}
      }
      found = !do_system_command (command, 0);
  }
  return found;
}

/*
 * Try to find the ultimate source file.  If the first line of the
 * current file is not of the form
 *
 *      .so man3/printf.3s
 *
 * the input file name is returned.
 */
static char *
ultimate_source (name0)
     char *name0;
{
  FILE *fp;
  char *name;
  char *expander;
  int expfl = 0;
  char *fgr;
  char *beg;
  char *end;
  char *cp;
  char buf[BUFSIZE];

/* TAMU FIX  -- this has to be static */
  static char ultname[BUFSIZE];

  name = my_strdup(name0);

 again:
  expander = get_expander (name);
  if (expander && *expander) {
      char command[BUFSIZE];

      sprintf (command, "%s %s", expander, name);
      fp = popen (command, "r");
      if (fp == NULL) {
	  perror("popen");
	  gripe (EXPANSION_FAILED, command);
	  return (NULL);
      }
      fgr = fgets (buf, BUFSIZE, fp);
      pclose (fp);
      expfl = 1;
  } else {
      fp = fopen (name, "r");
      if (fp == NULL && expfl) {
	  char *extp = rindex (name0, '.');
	  if (extp && *extp) {
	      strcat(name, extp);
	      fp = fopen (name, "r");
	  }
      }
      /*
       * Some people have compressed man pages, but uncompressed
       * .so files - we could glob for all possible extensions,
       * for now: only try .gz
       */
      else if (fp == NULL && get_expander(".gz")) {
	  strcat(name, ".gz");
	  fp = fopen (name, "r");
      }

      if (fp == NULL) {
	  perror("fopen");
	  gripe (OPEN_ERROR, name);
	  return (NULL);
      }
      fgr = fgets (buf, BUFSIZE, fp);
      fclose (fp);
  }

  if (fgr == NULL) {
      perror("fgets");
      gripe (READ_ERROR, name);
      return (NULL);
  }

  if (strncmp(buf, ".so", 3))
    return (name);

  beg = buf+3;
  while (*beg == ' ' || *beg == '\t')
    beg++;

  end = beg;
  while (*end != ' ' && *end != '\t' && *end != '\n' && *end != '\0')
    end++;

  *end = '\0';

  if (name != ultname) {
      strcpy(ultname, name);
      name = ultname;
  }

  /* If name ends in path/manx/foo.9x then use path, otherwise
     try same directory. */
  if ((cp = rindex(name, '/')) == NULL) /* very strange ... */
    return 0;
  *cp = 0;
  if((cp = rindex(name, '/')) != NULL && !strncmp(cp+1, "man", 3))
    strcpy(cp+1, beg);
  else if((cp = rindex(beg, '/')) != NULL)
    strcat(name, cp);
  else {
      strcat(name, "/");
      strcat(name, beg);
  }

  goto again;
}

static void
add_directive (d, file, buf)
     char *d;
     char *file;
     char *buf;
{
  if ((d = getval(d)) != 0 && *d)
    {
      if (*buf == 0)
	{
	  strcpy (buf, d);
	  strcat (buf, " ");
	  strcat (buf, file);
	}
      else
	{
	  strcat (buf, " | ");
	  strcat (buf, d);
	}
    }
}

static int
parse_roff_directive (cp, file, buf)
  char *cp;
  char *file;
  char *buf;
{
  char c;
  int tbl_found = 0;

  while ((c = *cp++) != '\0')
    {
      switch (c)
	{
	case 'e':

	  if (debug)
	    gripe (FOUND_EQN);
	  add_directive ((do_troff ? "EQN" : "NEQN"), file, buf);
	  break;

	case 'g':

	  if (debug)
	    gripe (FOUND_GRAP);
	  add_directive ("GRAP", file, buf);
	  break;

	case 'p':

	  if (debug)
	    gripe (FOUND_PIC);
	  add_directive ("PIC", file, buf);
	  break;

	case 't':

	  if (debug)
	    gripe (FOUND_TBL);
	  tbl_found++;
	  add_directive ("TBL", file, buf);
	  break;

	case 'v':

	  if (debug)
	    gripe (FOUND_VGRIND);
	  add_directive ("VGRIND", file, buf);
	  break;

	case 'r':

	  if (debug)
	    gripe (FOUND_REFER);
	  add_directive ("REFER", file, buf);
	  break;

	case ' ':
	case '\t':
	case '\n':

	  goto done;

	default:

	  return -1;
	}
    }

 done:

  if (*buf == 0)
    return 1;

  strcat (buf, " | ");
  strcat (buf, getval(do_troff ? "TROFF" : "NROFF"));

  if (tbl_found && !do_troff && *getval("COL")) {
      strcat (buf, " | ");
      strcat (buf, getval("COL"));
  }

  return 0;
}

char *
make_roff_command (file)
     char *file;
{
  FILE *fp;
  char line [BUFSIZE];
  static char buf [BUFSIZE];
  int status, ll;
  char *cp, *expander, *fgr;

  /* if window size differs much from 80, try to adapt */
  /* (but write only standard formatted files to the cat directory) */
  ll = setll();

  expander = get_expander (file);
  if (expander && *expander) {
      if (ll)
	sprintf (buf, "(echo \".ll %d.%di\"; %s %s)",
		 ll/10, ll%10, expander, file);
      else
	sprintf (buf, "%s %s", expander, file);
  } else {
      if (ll)
	sprintf (buf, "(echo \".ll %d.%di\"; cat %s)",
		 ll/10, ll%10, file);
      else
	buf[0] = 0;
  }

  if (roff_directive != NULL) {
      if (debug)
	gripe (ROFF_FROM_COMMAND_LINE);

      status = parse_roff_directive (roff_directive, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe (ROFF_CMD_FROM_COMMANDLINE_ERROR);
  }

  if (expander && *expander) {
      char command[BUFSIZE];

      sprintf (command, "%s %s", expander, file);
      fp = popen (command, "r");
      if (fp == NULL) {
	  perror("popen");
	  gripe (EXPANSION_FAILED, command);
	  return (NULL);
      }
      fgr = fgets (line, BUFSIZE, fp);
      pclose (fp);
  } else {
      fp = fopen (file, "r");
      if (fp == NULL) {
	  perror("fopen");
	  gripe (OPEN_ERROR, file);
	  return (NULL);
      }
      fgr = fgets (line, BUFSIZE, fp);
      fclose (fp);
  }

  if (fgr == NULL) {
      perror("fgets");
      gripe (READ_ERROR, file);
      return (NULL);
  }

  cp = &line[0];
  if (*cp++ == '\'' && *cp++ == '\\' && *cp++ == '"' && *cp++ == ' ') {
      if (debug)
	gripe (ROFF_FROM_FILE, file);

      status = parse_roff_directive (cp, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe (ROFF_CMD_FROM_FILE_ERROR, file);
  }

  if ((cp = getenv ("MANROFFSEQ")) != NULL) {
      if (debug)
	gripe (ROFF_FROM_ENV);

      status = parse_roff_directive (cp, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe (MANROFFSEQ_ERROR);
  }

  if (debug)
    gripe (USING_DEFAULT);

  (void) parse_roff_directive ("t", file, buf);

  return buf;
}

/*
 * Try to format the man page and create a new formatted file.  Return
 * 1 for success and 0 for failure.
 */
static int
make_cat_file (path, man_file, cat_file)
     register char *path;
     register char *man_file;
     register char *cat_file;
{
  int mode;
  FILE *fp;
  char *roff_command;
  char command[BUFSIZE];

      /* First make sure we can write the file; create an empty file. */
      if ((fp = fopen (cat_file, "w")) == NULL) {
	  if (errno == ENOENT)		/* directory does not exist */
	    return 0;

	  /* If we cannot write the file, maybe we can delete it */
	  if(unlink (cat_file) != 0 || (fp = fopen (cat_file, "w")) == NULL) {
	      if (errno == EROFS) 	/* possibly a CDROM */
		return 0;
	      if (debug)
		gripe (CAT_OPEN_ERROR, cat_file);
	      if (!suid)
		return 0;

	      /* maybe the real user can write it */
	      /* note: just doing "> %s" gives the wrong exit status */
	      sprintf(command, "cp /dev/null %s 2>/dev/null", cat_file);
	      if (do_system_command(command, 1)) {
		  if (debug)
		    gripe (USER_CANNOT_OPEN_CAT);
		  return 0;
	      }
	      if (debug)
		gripe (USER_CAN_OPEN_CAT);
	  }
      }
      fclose (fp);

      roff_command = make_roff_command (man_file);
      if (roff_command == NULL)
	return 0;
      else if (do_compress)
	  /* The cd is necessary, because of .so commands,
	     like .so man1/bash.1 in bash_builtins.1.
	     But it changes the meaning of man_file and cat_file,
	     if these are not absolute. */
	
	sprintf (command, "(cd %s ; %s | %s > %s)", path,
		 roff_command, getval("COMPRESS"), cat_file);
      else
        sprintf (command, "(cd %s ; %s > %s)", path,
		 roff_command, cat_file);

      /*
       * Don't let the user interrupt the system () call and screw up
       * the formmatted man page if we're not done yet.
       */
      signal (SIGINT, SIG_IGN);

      gripe (PLEASE_WAIT);

      if (suid)
	chmod (cat_file, 0666);
      if (!do_system_command (command, 0)) {
	  /* success */
	  mode = ((ruid != euid) ? 0644 : (rgid != egid) ? 0464 : 0444);
	  if(chmod (cat_file, mode) != 0 && suid) {
	      sprintf (command, "chmod 0%o %s", mode, cat_file);
	      (void) do_system_command (command, 1);
	  }
	  /* be silent about the success of chmod - it is not important */
	  if (debug)
	    gripe (CHANGED_MODE, cat_file, mode);
      } else {
	  /* something went wrong - remove garbage */
	  if(unlink(cat_file) != 0 && suid) {
	      sprintf (command, "rm %s", cat_file);
	      (void) do_system_command (command, 1);
	  }
      }

      signal (SIGINT, SIG_DFL);

      return 1;
}

static int
display_man_file(path, man_file)
     char *path, *man_file;
{
  extern int different_man_file();
  char *roff_command;
  char command[BUFSIZE];

  if (!different_man_file (man_file))
    return 0;
  roff_command = make_roff_command (man_file);
  if (roff_command == NULL)
    return 0;
  else if (do_troff)
    sprintf (command, "(cd %s ; %s)", path, roff_command);
  else
    sprintf (command, "(cd %s ; %s | %s)", path,
	     roff_command, pager);

  return !do_system_command (command, 0);
}

/*
 * make and display the cat file - return 0 if something went wrong
 */
int
make_and_display_cat_file (path, man_file)
     char *path, *man_file;
{
  char *cat_file;
  int status;

  if ((cat_file = convert_to_cat (man_file,1,CAT)) == NULL)
    return 0;

  if (debug)
    gripe (PROPOSED_CATFILE, cat_file);

  /*
   * If cat_file exists, check whether it is more recent.
   * Otherwise, check for other cat files (maybe there are
   * old .Z files that should be removed).
   */

  status = (nocats ? -2 : is_newer (man_file, cat_file));
  if (debug)
    gripe (IS_NEWER_RESULT, status);
  if (status == -1 || status == -3) {
      /* what? man_file does not exist anymore? */
      gripe (CANNOT_STAT, man_file);
      return(0);
  }

  if (status != 0 || access (cat_file, R_OK) != 0) {
      /*
       * Cat file is out of date (status = 1) or does not exist or is
       * empty (status = -2) or is unreadable.  Try to format and save it.
       */
      if (print_where) {
	  printf ("%s\n", man_file);
	  return 1;
      }

      if (!make_cat_file (path, man_file, cat_file))
	return 0;

      /*
       * If we just created this cat file, unlink any others.
       */
      if (status == -2 && do_compress)
	remove_other_catfiles(cat_file);
  } else {
      /*
       * Formatting not necessary.  Cat file is newer than source
       * file, or source file is not present but cat file is.
       */
      if (print_where) {
	  if (one_per_line) {
	      /* addition by marty leisner - leisner@sdsp.mc.xerox.com */
	      printf("%s\n", cat_file);
	      printf("%s\n", man_file);
	  } else
	    printf ("%s (<-- %s)\n", cat_file, man_file);
	  return 1;
      }
  }
  (void) display_cat_file (cat_file);
  return 1;
}

/*
 * Try to format the man page source and save it, then display it.  If
 * that's not possible, try to format the man page source and display
 * it directly.
 */
static int
format_and_display (path, man_file)
     char *path;
     char *man_file;
{
  char *roff_command;
  char command[BUFSIZE];

  if (access (man_file, R_OK) != 0)
    return 0;

  /* first test for contents  .so man1/xyzzy.1  */
  /* (in that case we do not want to make a cat file identical
     to cat1/xyzzy.1) */
  man_file = ultimate_source (man_file);
  if (man_file == NULL)
    return 0;

  if (do_troff) {
      roff_command = make_roff_command (man_file);
      if (roff_command == NULL)
	return 0;
      else
	sprintf (command, "(cd %s ; %s)", path, roff_command);

      return !do_system_command (command, 0);
  }

  if (setll()) {
      if (debug)
	gripe(NO_CAT_FOR_NONSTD_LL);
  } else if (make_and_display_cat_file (path, man_file))
      return 1;

  /* line length was wrong or could not display cat_file */
  if (print_where) {
      printf ("%s\n", man_file);
      return 1;
  }

  return display_man_file (path, man_file);
}

/*
 * See if the preformatted man page or the source exists in the given
 * section.
 */
int
try_section (path, section, name)
     char *path, *section, *name;
{
  register int found = 0;
  register char **names = NULL;
  register char **np;

  if (debug)
    gripe (TRYING_SECTION, section);

  /*
   * Look for man page source files if we know how to format them.
   */
  if (do_troff ? *getval("TROFF") : *getval("NROFF"))
      names = glob_for_file (path, section, name, MAN);

  if (names && *names) {
      if (debug) {
	  gripe (MAN_FOUND);
	  for (np = names; *np != NULL; np++)
	    fprintf(stderr, "  %s\n", *np);
      }
      for (np = names; *np != NULL; np++) {
	  found += format_and_display (path, *np);
	  if (found && !findall)
	    break;
      }
      if (found)
	return found;
  }

  /*
   * No files match.  See if there's a preformatted page
   * (`cat' or `stray cat') around that we can display. 
   * [Under FSSTND, cat pages live under /var/catman, but
   *  stray cats live in /usr/man/catx.]
   */

  if (do_troff)
    return 0;

  names = glob_for_file (path, section, name, CAT);
  if ((!names || !*names) && fsstnd)
    names = glob_for_file (path, section, name, SCAT);

  if (names != NULL) {
      for (np = names; *np != NULL; np++) {
	  if (print_where) {
	      printf ("%s\n", *np);
	      found++;
	  } else {
	      found += display_cat_file (*np);
	  }
	  if (found && !findall)
	    break;
      }
  }

  return found;
}

int
man_from_section (sec, name)
     char *sec, *name;
{
  register int found;
  register struct dir *dp;

  found = 0;

  for (dp = srchlist.nxt; dp; dp = dp->nxt) {
      if (debug)
	gripe (SEARCHING, dp->dir);

      found += try_section (dp->dir, sec, name);

      if (found && !findall)
	break;
  }
  return found;
}

/*
 * Search for manual pages.
 *
 * If preformatted manual pages are supported, look for the formatted
 * file first, then the man page source file.  If they both exist and
 * the man page source file is newer, or only the source file exists,
 * try to reformat it and write the results in the cat directory.  If
 * it is not possible to write the cat file, simply format and display
 * the man file.
 *
 * If preformatted pages are not supported, or the troff option is
 * being used, only look for the man page source file.
 *
 * Note that globbing is necessary also if the section is given,
 * since a preformatted man page might be compressed.
 *
 */
int
man (name)
     char *name;
{
  register int found;
  register char **sp;

  found = 0;

  /* allow  man ./manpage  for formatting explicitly given man pages */
  if (index(name, '/')) {
      char fullname[BUFSIZE];
      char fullpath[BUFSIZE];
      char *path;
      char *cp;
      FILE *fp = fopen(name, "r");

      if (!fp) {
	  perror(name);
	  return 0;
      }
      fclose (fp);
      if (*name != '/' && getcwd(fullname, sizeof(fullname))) {
	  strcat (fullname, "/");
	  strcat (fullname, name);
      } else
	strcpy (fullname, name);

      strcpy (fullpath, fullname);
      if ((cp = rindex(fullpath, '/')) != NULL) {
	  strcpy(cp+1, "..");
	  path = fullpath;
      } else
	path = ".";

      name = ultimate_source (fullname);
      if (!name)
	return 0;
      
      return display_man_file (path, name);
  }

  fflush (stdout);
  manpath();
  if (section != NULL)
    found = man_from_section (section, name);
  else {
      for (sp = section_list; *sp != NULL; sp++) {
	  found += man_from_section (*sp, name);
	  if (found && !findall)
	    break;
      }
  }
  return found;
}

char **
get_section_list ()
{
  int i;
  char *p;
  char *end;
  static char *tmp_section_list[100];

  if (colon_sep_section_list == NULL) {
      if ((p = getenv ("MANSECT")) == NULL)
	  p = getval ("MANSECT");
      colon_sep_section_list = my_strdup (p);
  }

  i = 0;
  for (p = colon_sep_section_list; ; p = end+1) {
      if ((end = strchr (p, ':')) != NULL)
	*end = '\0';

      tmp_section_list[i++] = my_strdup (p);

      if (end == NULL || i+1 == sizeof(tmp_section_list))
	break;
  }

  tmp_section_list [i] = NULL;
  return tmp_section_list;
}

void
do_global_apropos (name)
     char *name;
{
    struct dir *dp;
    char **gf;
    char pathname[BUFSIZE];
    char command[BUFSIZE];
    int res;

    manpath();
    for (dp = srchlist.nxt; dp; dp = dp->nxt) {
	if (debug)
	  gripe(SEARCHING, dp->dir);
	sprintf(pathname, "%s/man%s/*", dp->dir, section ? section : "*");
	gf = glob_filename (pathname);

	if (gf != (char **) -1 && gf != NULL) {
	    for( ; *gf; gf++) {
		char *expander = get_expander (*gf);
		if (expander)
		  sprintf(command, "%s %s | grep -%c '%s'", expander, *gf,
			  GREPSILENT, name);
		else
		  sprintf(command, "grep -%c '%s' %s", GREPSILENT, name, *gf);
		res = do_system_command (command, 1);
		if (res == 0) {
		    if (print_where)
		      printf("%s\n", *gf);
		    else {
			/* should read LOCALE, but libc 4.6.27 doesn't seem
			   to handle LC_RESPONSE yet */
			int answer, c;
			char path[BUFSIZE];

			printf("%s? [ynq] ", *gf);
			fflush(stdout);
			answer = c = getchar();
			while (c != '\n' && c != EOF)
			  c = getchar();
			if(index("QqXx", answer))
			  exit(0);
			if(index("YyJj", answer)) {
			    char *ri;

			    strcpy(path, *gf);
			    ri = rindex(path, '/');
			    if (ri)
			      *ri = 0;
			    format_and_display(path, *gf);
			}
		    }
		}
	    }
	}
    }
}

/*
 * Handle the apropos option.  Cheat by using another program.
 */
void
do_apropos (name)
     register char *name;
{
  register int len;
  register char *command;

  len = strlen (getval("APROPOS")) + strlen (name) + 2;
  command = my_malloc(len);
  sprintf (command, "%s %s", getval("APROPOS"), name);
  (void) do_system_command (command, 0);
  free (command);
}

/*
 * Handle the whatis option.  Cheat by using another program.
 */
void
do_whatis (name)
     register char *name;
{
  register int len;
  register char *command;

  len = strlen (getval("WHATIS")) + strlen (name) + 2;
  command = my_malloc(len);
  sprintf (command, "%s %s", getval("WHATIS"), name);
  (void) do_system_command (command, 0);
  free (command);
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  int status = 0;
  char *nextarg;
  char *tmp;
  extern int optind;

  extern char *mkprogname ();
  extern get_permissions ();
  extern void prmanpath ();
  char *is_section ();
  char **get_section_list ();
  void man_getopt ();

  setlocale(LC_MESSAGES, "");

  progname = mkprogname (argv[0]);

  get_permissions ();
  get_line_length();

  /*
   * read command line options and man.config
   */
  man_getopt (argc, argv);

  /*
   * manpath  or  man --path  or  man -w  will only print the manpath
   */
  if (!strcmp (progname, "manpath") || (optind == argc && print_where)) {
      manpath();
      prmanpath();
      exit(0);
  }

  if (optind == argc)
    gripe(NO_NAME_NO_SECTION);

  section_list = get_section_list ();

  while (optind < argc) {
      nextarg = argv[optind++];

      tmp = is_section (nextarg);
      if (tmp) {
	  if (optind < argc) {
	      section = tmp;
	      if (debug)
		gripe (SECTION, section);
	  } else {
	      gripe (NO_NAME_FROM_SECTION, tmp);
	  }
	  continue;
      }

      if (global_apropos)
	do_global_apropos (nextarg);
      else if (apropos)
	do_apropos (nextarg);
      else if (whatis)
	do_whatis (nextarg);
      else {
	  status = man (nextarg);

	  if (status == 0) {
	      if (section)
		gripe (NO_SUCH_ENTRY_IN_SECTION, nextarg, section);
	      else
		gripe (NO_SUCH_ENTRY, nextarg);
	  }
      }
  }
  return !status;
}
