/*
  Copyright 1999-2005 ImageMagick Studio LLC, a non-profit organization
  dedicated to making software imaging solutions freely available.
  
  You may not use this file except in compliance with the License.
  obtain a copy of the License at
  
    http://www.imagemagick.org/www/Copyright.html
  
  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  ImageMagick image color methods.
*/
#ifndef _MAGICK_COLOR_PRIVATE_H
#define _MAGICK_COLOR_PRIVATE_H

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

#include <magick/image.h>
#include <magick/color.h>
#include <magick/exception_private.h>

static inline IndexPacket ConstrainColormapIndex(Image *image,
  const unsigned long index)
{
  if (index >= image->colors)
    {
      (void) ThrowMagickException(&image->exception,GetMagickModule(),
        CorruptImageError,"InvalidColormapIndex",image->filename);
      return(0);
    }
  return((IndexPacket) index);
}

static inline MagickBooleanType IsMagickColorEqual(const MagickPixelPacket *p,
  const MagickPixelPacket *q)
{
  if (p->red != q->red)
    return(MagickFalse);
  if (p->green != q->green)
    return(MagickFalse);
  if (p->blue != q->blue)
    return(MagickFalse);
  if ((p->matte != MagickFalse) && (p->opacity != q->opacity))
    return(MagickFalse);
  if ((p->colorspace == CMYKColorspace) && (p->index != q->index))
    return(MagickFalse);
  return(MagickTrue);
}

static inline void SetMagickPixelPacket(const PixelPacket *color,
  const IndexPacket *index,MagickPixelPacket *pixel)
{
  pixel->red=(MagickRealType) color->red;
  pixel->green=(MagickRealType) color->green;
  pixel->blue=(MagickRealType) color->blue;
  pixel->opacity=(MagickRealType) (pixel->matte != MagickFalse ?
    color->opacity : OpaqueOpacity);
  pixel->index=(MagickRealType) ((pixel->colorspace == CMYKColorspace) &&
    (index != (const IndexPacket *) NULL) ? *index : 0);
}

static inline void SetPixelPacket(const MagickPixelPacket *pixel,
  PixelPacket *color,IndexPacket *index)
{
  color->red=RoundToQuantum(pixel->red);
  color->green=RoundToQuantum(pixel->green);
  color->blue=RoundToQuantum(pixel->blue);
  if (pixel->matte != MagickFalse)
    color->opacity=RoundToQuantum(pixel->opacity);
  if ((pixel->colorspace == CMYKColorspace) && (index != (IndexPacket *) NULL))
    *index=RoundToQuantum(pixel->index);
}

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif
