/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#include <gtk/gtk.h>
#include <libxfce4util/i18n.h>
#include <libxfce4util/util.h>
#include <libxfcegui4/dialogs.h>
#include <xfprinterlist.h>
#include <xfprintsettings.h>

#include "inline-icon.h"
#include "printdlg.h"
#include "input_page.h"
#include "head_page.h"
#include "pprint_page.h"
#include "sheets_page.h"
#include "vpages_page.h"

static void printdlg_class_init(PrintDlgClass *);
static void printdlg_init(PrintDlg *);
static void printdlg_finalize(GObject *);

/* signal handlers */
static void changed_cb(GtkEntry *, PrintDlg *);
static void clicked_cb(GtkWidget *, PrintDlg *);
static void toggled_cb(GtkWidget *, PrintDlg *);
static void update_image(GtkWidget *, PrintDlg *);

static GtkWidgetClass *parent_class = NULL;

GtkType
printdlg_get_type(void)
{
	static GtkType printdlg_type = 0;

	if (!printdlg_type) {
		static const GTypeInfo printdlg_info = {
			sizeof(PrintDlgClass),
			NULL,
			NULL,
			(GClassInitFunc)printdlg_class_init,
			NULL,
			NULL,
			sizeof(PrintDlg),
			0,
			(GInstanceInitFunc)printdlg_init
		};

		printdlg_type = g_type_register_static(GTK_TYPE_DIALOG,
		    "PrintDlg", &printdlg_info, 0);
	}

	return(printdlg_type);
}

static void
printdlg_class_init(PrintDlgClass *klass)
{
	GObjectClass *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = printdlg_finalize;

	parent_class = gtk_type_class(gtk_dialog_get_type());
}

static void
printdlg_init(PrintDlg *dlg)
{
	GtkWidget *frame;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *image;

	dlg->save = mixed_button_new(GTK_STOCK_SAVE, _("Save settings"));
	gtk_widget_show(dlg->save);
	g_signal_connect(G_OBJECT(dlg->save), "clicked",
		G_CALLBACK(clicked_cb), dlg);
	gtk_dialog_add_action_widget(GTK_DIALOG(dlg),
		dlg->save, GTK_RESPONSE_APPLY);
	gtk_button_box_set_child_secondary(
		GTK_BUTTON_BOX(GTK_DIALOG(dlg)->action_area), 
		dlg->save,
		TRUE);

	gtk_dialog_add_buttons(GTK_DIALOG(dlg),
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		GTK_STOCK_PRINT, GTK_RESPONSE_OK,
		NULL);

	gtk_window_set_default_size(GTK_WINDOW(dlg), 300, 10);
	gtk_window_set_title(GTK_WINDOW(dlg), _("XFPrint"));

	gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dlg)->vbox), 4);

	/* "Printer"-Frame: */
	frame = gtk_frame_new(_("Printer"));
	gtk_widget_show(frame);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), frame, FALSE,
		TRUE, 0);

	table = gtk_table_new(2, 3, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_widget_show(table);
	gtk_container_add(GTK_CONTAINER(frame), table);

	dlg->toname = gtk_radio_button_new_with_label(NULL, _("Name:"));
	gtk_widget_show(dlg->toname);
	g_signal_connect(G_OBJECT(dlg->toname), "toggled",
		G_CALLBACK(toggled_cb), dlg);
	g_signal_connect(G_OBJECT(dlg->toname), "toggled",
		G_CALLBACK(update_image), dlg);
	gtk_table_attach(GTK_TABLE(table), dlg->toname, 0, 1, 0, 1,
		GTK_FILL, 0, 0, 0);

	dlg->printer = gtk_combo_new();
	gtk_widget_show(dlg->printer);
	gtk_combo_set_value_in_list(GTK_COMBO(dlg->printer), TRUE, FALSE);
	gtk_editable_set_editable(GTK_EDITABLE(GTK_COMBO(dlg->printer)->entry),
		FALSE);
	g_signal_connect(G_OBJECT(GTK_COMBO(dlg->printer)->entry),
		"changed", G_CALLBACK(changed_cb), dlg);
	g_signal_connect(G_OBJECT(GTK_COMBO(dlg->printer)->entry),
		"changed", G_CALLBACK(update_image), dlg);
	gtk_table_attach(GTK_TABLE(table), dlg->printer, 1, 2, 0, 1,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	dlg->tofile = gtk_radio_button_new_with_label_from_widget(
		GTK_RADIO_BUTTON(dlg->toname), _("Print to file:"));
	gtk_widget_show(dlg->tofile);
	gtk_table_attach(GTK_TABLE(table), dlg->tofile, 0, 1, 1, 2,
		GTK_FILL, 0, 0, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 1, 2,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	dlg->file = gtk_entry_new();
	gtk_widget_show(dlg->file);
	gtk_box_pack_start(GTK_BOX(hbox), dlg->file, TRUE, TRUE, 0);

	dlg->select = gtk_button_new();
	gtk_widget_show(dlg->select);
	g_signal_connect(G_OBJECT(dlg->select), "clicked",
		G_CALLBACK(clicked_cb), dlg);
	gtk_box_pack_start(GTK_BOX(hbox), dlg->select, FALSE, TRUE, 0);

	image = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show(image);
	gtk_container_add(GTK_CONTAINER(dlg->select), image);
    
	dlg->image = gtk_image_new();
	gtk_widget_show(dlg->image);
	gtk_table_attach(GTK_TABLE(table), dlg->image, 2, 3, 0, 2,
		GTK_FILL, GTK_EXPAND | GTK_FILL, 20, 10);

	/* Notebook */
	dlg->notebook = gtk_notebook_new();
	gtk_widget_show(dlg->notebook);
	gtk_container_set_border_width(GTK_CONTAINER(dlg->notebook), 5);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), dlg->notebook,
		TRUE, TRUE, 0);

	dlg->sheets = sheets_page_new();
	gtk_widget_show(dlg->sheets);
	label = gtk_label_new(_("Sheets"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(dlg->notebook), dlg->sheets,
		label);

	dlg->vpages = vpages_page_new();
	gtk_widget_show(dlg->vpages);
	label = gtk_label_new(_("Virtual pages"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(dlg->notebook), dlg->vpages,
		label);

	dlg->pprint = pprint_page_new();
	gtk_widget_show(dlg->pprint);
	label = gtk_label_new(_("Pretty-printing"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(dlg->notebook), dlg->pprint,
		label);

	dlg->input = input_page_new();
	gtk_widget_show(dlg->input);
	label = gtk_label_new(_("Input"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(dlg->notebook), dlg->input,
		label);

	dlg->head = head_page_new();
	gtk_widget_show(dlg->head);
	label = gtk_label_new(_("Headings"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(dlg->notebook), dlg->head,
		label);

	/* number of copies */
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), hbox,
		FALSE, TRUE, 0);

	label = gtk_label_new(_("Copies:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	dlg->copies = gtk_spin_button_new_with_range(1.0, 1000.0, 1.0);
	gtk_widget_show(dlg->copies);
	gtk_box_pack_start(GTK_BOX(hbox), dlg->copies, FALSE, FALSE, 0);

	/* defaults to no printer selected */
	dlg->pr = NULL;
}

static void
printdlg_finalize(GObject *object)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(PRINT_IS_DLG(object));

	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
printdlg_new(const XfPrintSettings *settings)
{
	extern XfPrinterList *printers;
	XfPrinter *pr;
	PrintDlg *dlg;
	GList *list;
	gchar *path;

	dlg = PRINT_DLG(g_object_new(printdlg_get_type(), NULL));

	/* Check if there are any printers available */
	if (xfprinterlist_is_empty(printers)) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dlg->tofile),
			TRUE);
		gtk_widget_set_sensitive(dlg->toname, FALSE);
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dlg->printer)->entry),
				_("No printers available"));
	}
	else {
		list = xfprinterlist_get_aliases(printers);

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dlg->toname),
			TRUE);
		gtk_combo_set_popdown_strings(GTK_COMBO(dlg->printer), list);

		if ((pr = xfprinterlist_get_default(printers)) != NULL) {
			gtk_entry_set_text(GTK_ENTRY(
					GTK_COMBO(dlg->printer)->entry),
					pr->alias);
		}

		g_list_free(list);
	}

	/* set default output file name */
	path = xfce_get_homefile("print.ps", NULL);
	gtk_entry_set_text(GTK_ENTRY(dlg->file), path);
	g_free(path);

	/* load the settings */
	sheets_page_set_settings(SHEETS_PAGE(dlg->sheets), &settings->sheets);
	vpages_page_set_settings(VPAGES_PAGE(dlg->vpages), &settings->vpages);
	pprint_page_set_settings(PPRINT_PAGE(dlg->pprint), &settings->pprint);
	input_page_set_settings(INPUT_PAGE(dlg->input), &settings->input);
	head_page_set_settings(HEAD_PAGE(dlg->head), &settings->headings);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(dlg->copies),
		settings->copies);

	/* just to be sure */
	toggled_cb(dlg->tofile, dlg);

	return(GTK_WIDGET(dlg));
}

void
printdlg_get_settings(PrintDlg *dlg, XfPrintSettings *settings)
{
	g_return_if_fail(dlg != NULL);
	g_return_if_fail(settings != NULL);

	sheets_page_get_settings(SHEETS_PAGE(dlg->sheets), &settings->sheets);
	vpages_page_get_settings(VPAGES_PAGE(dlg->vpages), &settings->vpages);
	pprint_page_get_settings(PPRINT_PAGE(dlg->pprint), &settings->pprint);
	input_page_get_settings(INPUT_PAGE(dlg->input),	&settings->input);
	head_page_get_settings(HEAD_PAGE(dlg->head), &settings->headings);

	settings->copies = gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(dlg->copies));
}

XfPrinter *
printdlg_get_printer(PrintDlg *dlg)
{
	g_return_val_if_fail(dlg != NULL, NULL);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dlg->toname)))
		return(dlg->pr);
	else
		return(NULL);
}

const gchar *
printdlg_get_file(PrintDlg *dlg)
{
	g_return_val_if_fail(dlg != NULL, NULL);

	return(gtk_entry_get_text(GTK_ENTRY(dlg->file)));
}

static void
changed_cb(GtkEntry *entry, PrintDlg *dlg)
{
	extern XfPrinterList *printers;

	dlg->pr = xfprinterlist_lookup(printers, gtk_entry_get_text(entry));

	if (dlg->pr != NULL)
		gtk_widget_set_sensitive(dlg->notebook, dlg->pr->postscript);
}

static void
clicked_cb(GtkWidget *btn, PrintDlg *dlg)
{
	XfPrintSettings *settings;
	GtkWidget *sel;

	if (btn == dlg->save) {
		settings = xfprintsettings_defaults();

		printdlg_get_settings(dlg, settings);

		xfprintsettings_save(settings);
		xfprintsettings_free(settings);
	}
	else if (btn == dlg->select) {
		sel = gtk_file_selection_new(_("Select output file..."));

		gtk_file_selection_set_filename(GTK_FILE_SELECTION(sel),
			gtk_entry_get_text(GTK_ENTRY(dlg->file)));

		switch (gtk_dialog_run(GTK_DIALOG(sel))) {
		case GTK_RESPONSE_OK:
			gtk_entry_set_text(GTK_ENTRY(dlg->file),
				gtk_file_selection_get_filename(
					GTK_FILE_SELECTION(sel)));
			break;
		}

		gtk_widget_destroy(sel);
	}
}

static void
toggled_cb(GtkWidget *btn, PrintDlg *dlg)
{
	gboolean active;

	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dlg->tofile));

	gtk_widget_set_sensitive(dlg->printer, !active);
	gtk_widget_set_sensitive(dlg->file, active);
	gtk_widget_set_sensitive(dlg->select, active);
	gtk_widget_set_sensitive(dlg->copies, !active);

	gtk_widget_set_sensitive(dlg->notebook, active ||
			(dlg->pr && dlg->pr->postscript));
}

static void
update_image(GtkWidget *widget, PrintDlg *dlg)
{
	GdkPixbuf *icon;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dlg->toname))) {
		/*
		 * This is required to prevent xfprint4 from segfaulting on
		 * startup in some weird cases
		 */
		if (!dlg->pr)
			return;

		icon = xfprinter_icon(dlg->pr, FALSE);
	}
	else {
		icon = gdk_pixbuf_new_from_inline(-1, postscript_file, FALSE,
			NULL);
	}
	
	gtk_image_set_from_pixbuf(GTK_IMAGE(dlg->image), icon);
	g_object_unref(G_OBJECT(icon));
}


