/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     memory.c                                                       */
/*                                                                          */
/* description:  routines for memory handling                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg                                             */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

/*--------------------------------------------------------------------------*/
/*  some routines for allocation and deallocation                           */
/*--------------------------------------------------------------------------*/

/*
 * default value for increasing size of memory in an memoryadmin.
 *
 * see also setCapacityIncrement(...)
 */
#define DEFAULTCAPACITYINCREMENT 1000


#define ALIGNSIZE(size) ((size) + ((size) % sizeof(void*) == 0 ? 0 : sizeof(void*) - (size) % sizeof(void*)))

typedef struct FreeMemory FREEMEMORY;
struct FreeMemory {
    FREEMEMORY *next;
};

typedef struct Block BLOCK;
struct Block {
    BLOCK *next;
    void *end;
};

typedef struct MemoryAdmin MEMORYADMIN;
struct MemoryAdmin {
    unsigned int capacity;
    unsigned int capacityIncrement;
    size_t objectSize;
    BLOCK *blocks;
    FREEMEMORY *freeMem;
};

static int newBlock(MEMORYADMIN* ma, unsigned int capacityIncrement) {
    FUNCNAME("newBlock");
    BLOCK* block;
    FREEMEMORY *freeMem;
    void **i;
    size_t ma_objectSize_div_sizeofvoid;
    size_t sizeofBlock = ALIGNSIZE(sizeof(BLOCK));

    ma_objectSize_div_sizeofvoid = ma->objectSize / sizeof(void*);

    block = (BLOCK *)alberta_alloc(sizeofBlock + capacityIncrement * ma->objectSize, funcName, __FILE__, __LINE__);

    ma->capacity += capacityIncrement;
    block->next = ma->blocks;
    ma->blocks = block;
    freeMem = (FREEMEMORY*)((void**)block + sizeofBlock / sizeof(void*));
    block->end = ((void**)freeMem) + capacityIncrement * ma_objectSize_div_sizeofvoid;

    for(i = (void**)freeMem; i < (void**)(block->end) - ma_objectSize_div_sizeofvoid; i += ma_objectSize_div_sizeofvoid) {
	((FREEMEMORY*)i)->next = (FREEMEMORY*)(i + ma_objectSize_div_sizeofvoid);
    }
    ((FREEMEMORY*)i)->next = ma->freeMem;
    ma->freeMem = freeMem;

    return 0;
}

/*
 * newObject(..) returns a void-pointer to a memoryadmin for objects of size
 * objectSize. Use the returned value as parameter for getMemory.
 */
static void* newObject(size_t objectSize, unsigned int initialCapacity) {
    FUNCNAME("newObject");
    MEMORYADMIN *ma;

    ma = (MEMORYADMIN *)alberta_alloc(sizeof(MEMORYADMIN), funcName, __FILE__, __LINE__);

    ma->capacity = 0;
    ma->capacityIncrement = (initialCapacity ? initialCapacity : DEFAULTCAPACITYINCREMENT);

    TEST_EXIT(objectSize, "Attempted to allocate a zero length object!\n");

    ma->objectSize = ALIGNSIZE(objectSize);
    ma->blocks = nil;
    ma->freeMem = nil;
    if(initialCapacity)
	newBlock(ma, initialCapacity);

    return ma;
}

/*
 * A memoryadmin allocates new memory, whenever getMemory is called and there
 * is no more free memory already allocated. The size of the new memory is
 * given by objectSize * capacityIncrement. By default capacityIncrement ==
 * DEFAULTCAPACITYINCREMENT
 */
static void setCapacityIncrement(void* memoryAdmin, unsigned int capacityIncrement) {
    FUNCNAME("setCapacityIncrement");

    DEBUG_TEST_EXIT(memoryAdmin, "memoryAdmin == nil\n");

    ((MEMORYADMIN*)memoryAdmin)->capacityIncrement = capacityIncrement;

    if(!((MEMORYADMIN*)memoryAdmin)->capacityIncrement) {
      ERROR_EXIT("capacityIncrement == 0\n");
    }
    return;
}

/*
 * returns a pointer to an object of size objectsize given by creating the
 * memoryadmin.
 */
static void* getMemory(void* memoryAdmin)
{
    MEMORYADMIN *admin;
    FREEMEMORY  *tmp;

    admin = (MEMORYADMIN *)memoryAdmin;

    while(admin->capacity <= 0)
	newBlock(admin, admin->capacityIncrement);

    tmp = admin->freeMem;
    admin->freeMem = admin->freeMem->next;

    admin->capacity--;

#ifdef __GNUC__
    /* gcc has a bug when run with -O3 and the i386 default target; so
     * insert an optimization barrier here :(
     */
    __asm__ __volatile__("" ::: "memory");
#endif

    return (void *)tmp;
}

/*
 * returns memory at *object to memoryadmin. !!! object must be the result of
 * an call of getMemory(...) with the same memoryadmin. !!!
 */
static void freeMemory(void* object, void* memoryAdmin) {
    /* FUNCNAME("freeMemory"); */

    ((FREEMEMORY*)object)->next = ((MEMORYADMIN*)memoryAdmin)->freeMem;
    ((MEMORYADMIN*)memoryAdmin)->freeMem = (FREEMEMORY*)object;
    ((MEMORYADMIN*)memoryAdmin)->capacity++;
    return;
}

/*
 * the counterpart to newObject, frees all memory allocated with the given
 * memoryadmin.
 */
static void deleteObject(void* memoryAdmin) {
    FUNCNAME("deleteObject");
    BLOCK *b, *tmp;
    size_t sizeofBlock;

    DEBUG_TEST_EXIT(memoryAdmin, "memoryAdmin=nil\n");

    b = ((MEMORYADMIN*)memoryAdmin)->blocks;
    while(b != nil) {
	tmp = b->next;
        sizeofBlock = (size_t)b->end - (size_t)b;

	alberta_free(b, sizeofBlock);
	b = tmp;
    }
    alberta_free(memoryAdmin, sizeof(MEMORYADMIN));
    return;
}

typedef struct dof_admin_mem_info DOF_ADMIN_MEM_INFO;
struct dof_admin_mem_info
{
  void *dof_matrix;
  void *matrix_row;
  void *dof_dowb_matrix;
  void *dowb_matrix_row;
  void *dowb_matrix_row_s;
  void *dowb_matrix_row_d;
  void *dof_int_vec;
  void *dof_dof_vec;
  void *int_dof_vec;
  void *dof_uchar_vec;
  void *dof_schar_vec;
  void *dof_real_vec;
  void *dof_real_d_vec;
  void *dof_ptr_vec; 
};


/*--------------------------------------------------------------------------*/
/*  memory management for DOF admin structures                              */
/*--------------------------------------------------------------------------*/

static void add_dof_admin_to_mesh(DOF_ADMIN *admin, MESH *mesh)
{
  FUNCNAME("add_dof_admin_to_mesh");
  int i, n, dim = mesh->dim;

  admin->mesh = mesh;
  n = mesh->n_dof_admin;
  if ((n > 0) && (mesh->dof_admin == nil))
    ERROR_EXIT("no mesh->dof_admin but n_dof_admin=%d\n", n);
  if ((n <= 0) && (mesh->dof_admin != nil))
    ERROR_EXIT("found mesh->dof_admin but n_dof_admin=%d\n", n);

  for (i = 0; i < n; i++)
    if (mesh->dof_admin[i] == admin)
      ERROR_EXIT("admin %s is already associated to mesh %s\n",
		 NAME(admin), NAME(mesh));

  mesh->dof_admin = MEM_REALLOC(mesh->dof_admin, n, n+1, DOF_ADMIN *);
  n++;

  mesh->dof_admin[n-1] = admin;
  mesh->n_dof_admin = n;

  mesh->n_dof_el = 0;

  admin->n0_dof[VERTEX] = mesh->n_dof[VERTEX];
  mesh->n_dof[VERTEX]  += admin->n_dof[VERTEX];
  mesh->n_dof_el += N_VERTICES(dim) * mesh->n_dof[VERTEX];

  admin->n0_dof[CENTER] = mesh->n_dof[CENTER];
  mesh->n_dof[CENTER]  += admin->n_dof[CENTER];
  mesh->n_dof_el += mesh->n_dof[CENTER];

  if(dim > 1) {
    admin->n0_dof[EDGE]   = mesh->n_dof[EDGE];
    mesh->n_dof[EDGE]    += admin->n_dof[EDGE];
    mesh->n_dof_el += N_EDGES(dim) * mesh->n_dof[EDGE];
  }

  if(dim == 3) {
    admin->n0_dof[FACE] = mesh->n_dof[FACE];
    mesh->n_dof[FACE]  += admin->n_dof[FACE];
    mesh->n_dof_el     += N_FACES_3D * mesh->n_dof[FACE];
  }    

  mesh->node[VERTEX]  = 0;
  if (mesh->n_dof[VERTEX] > 0) mesh->n_node_el = N_VERTICES(dim);
  else                         mesh->n_node_el = 0;

  if(dim > 1) {
    mesh->node[EDGE]    = mesh->n_node_el;
    if (mesh->n_dof[EDGE] > 0) mesh->n_node_el += N_EDGES(dim);
  }

  if (dim == 3) {
    mesh->node[FACE]    = mesh->n_node_el;
    if (mesh->n_dof[FACE] > 0) mesh->n_node_el += N_FACES_3D;
  }

  mesh->node[CENTER]    = mesh->n_node_el;
  if (mesh->n_dof[CENTER] > 0) mesh->n_node_el += 1;

  return;
}

/*--------------------------------------------------------------------------*/

DOF_ADMIN *AI_get_dof_admin(MESH *mesh, const char *name, 
			    const int n_dof[N_NODE_TYPES])
{
  FUNCNAME("AI_get_dof_admin");
  DOF_ADMIN         *admin;
  int               i;
  DOF_ADMIN_MEM_INFO *mem_info;
  
  admin = MEM_CALLOC(1, DOF_ADMIN);
  admin->mesh = mesh;
  admin->name = name ? strdup(name) : nil;
  
  admin->dof_free         = nil;
  admin->dof_free_size    = admin->first_hole     = 0;
  
  TEST_EXIT((mesh->dim > 1) || (n_dof[EDGE]==0),
    "EDGE DOFs only make sense for mesh->dim > 1!\n");
  
  TEST_EXIT((mesh->dim == 3) || (n_dof[FACE]==0),
    "FACE DOFs only make sense for mesh->dim == 3!\n");
  
  for (i = 0; i < N_NODE_TYPES; i++) admin->n_dof[i] = n_dof[i];
  
  mem_info = (DOF_ADMIN_MEM_INFO *)
    (admin->mem_info = MEM_ALLOC(1, DOF_ADMIN_MEM_INFO));
  
  mem_info->dof_matrix = newObject(sizeof(DOF_MATRIX), 0); 
  setCapacityIncrement(mem_info->dof_matrix, 10);
  mem_info->matrix_row = newObject(sizeof(MATRIX_ROW), 0);
  setCapacityIncrement(mem_info->matrix_row, 100);
  mem_info->dof_dowb_matrix = newObject(sizeof(DOF_DOWB_MATRIX), 0); 
  setCapacityIncrement(mem_info->dof_dowb_matrix, 10);
  mem_info->dowb_matrix_row =
    newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_DD), 0);
  setCapacityIncrement(mem_info->dowb_matrix_row, 100);
  mem_info->dowb_matrix_row_s =
    newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_DDS), 0);
  setCapacityIncrement(mem_info->dowb_matrix_row_s, 100);
  mem_info->dowb_matrix_row_d =
    newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_D), 0);
  setCapacityIncrement(mem_info->dowb_matrix_row_d, 100);
  mem_info->dof_int_vec = newObject(sizeof(DOF_INT_VEC), 0); 
  setCapacityIncrement(mem_info->dof_int_vec, 10);
  mem_info->dof_dof_vec = newObject(sizeof(DOF_DOF_VEC), 0); 
  setCapacityIncrement(mem_info->dof_dof_vec, 10);
  mem_info->int_dof_vec = newObject(sizeof(DOF_DOF_VEC), 0);
  setCapacityIncrement(mem_info->int_dof_vec, 10); 
  mem_info->dof_uchar_vec = newObject(sizeof(DOF_UCHAR_VEC), 0); 
  setCapacityIncrement(mem_info->dof_uchar_vec, 10);
  mem_info->dof_schar_vec = newObject(sizeof(DOF_SCHAR_VEC), 0); 
  setCapacityIncrement(mem_info->dof_schar_vec, 10);
  mem_info->dof_real_vec = newObject(sizeof(DOF_REAL_VEC), 0); 
  setCapacityIncrement(mem_info->dof_real_vec, 10);
  mem_info->dof_real_d_vec = newObject(sizeof(DOF_REAL_D_VEC), 0); 
  setCapacityIncrement(mem_info->dof_real_d_vec, 10);
  mem_info->dof_ptr_vec = newObject(sizeof(DOF_PTR_VEC), 0); 
  setCapacityIncrement(mem_info->dof_ptr_vec, 10);
  
  add_dof_admin_to_mesh(admin, mesh);

  return(admin);
}


/*--------------------------------------------------------------------------*/
/*  Free all DOF_ADMINs in a rather brutal way, only makes sense when       */
/*  freeing an entire mesh.                                                 */
/*--------------------------------------------------------------------------*/

static void free_dof_admins(MESH *mesh)
{
  FUNCNAME("free_dof_admins");

  DOF_ADMIN **admin = mesh->dof_admin;
  int i, n;
  DOF_MATRIX       *dof_matrix,      *dof_matrix_next;
  DOF_DOWB_MATRIX  *dof_dowb_matrix, *dof_dowb_matrix_next;
  DOF_INT_VEC      *dof_int_vec,     *dof_int_vec_next;
  DOF_DOF_VEC      *dof_dof_vec,     *dof_dof_vec_next;
  DOF_DOF_VEC      *int_dof_vec,     *int_dof_vec_next;
  DOF_UCHAR_VEC    *dof_uchar_vec,   *dof_uchar_vec_next;
  DOF_SCHAR_VEC    *dof_schar_vec,   *dof_schar_vec_next;
  DOF_REAL_VEC     *dof_real_vec,    *dof_real_vec_next;
  DOF_REAL_D_VEC   *dof_real_d_vec,  *dof_real_d_vec_next;
  DOF_PTR_VEC      *dof_ptr_vec,     *dof_ptr_vec_next;

  n = mesh->n_dof_admin;
  if ((n > 0) && !admin)
    ERROR_EXIT("no mesh->dof_admin but n_dof_admin=%d\n", n);
  if ((n <= 0) && admin)
    ERROR_EXIT("found mesh->dof_admin but n_dof_admin=%d\n", n);

  for(i = 0; i < n; i++) {
#define FREE_DOF_OBJ(admin, obj)			\
    for(obj = admin[i]->obj; obj; obj = obj##_next) {	\
      obj##_next = obj->next;				\
      free_##obj(obj);					\
    }
    FREE_DOF_OBJ(admin, dof_matrix);
    FREE_DOF_OBJ(admin, dof_dowb_matrix);
    FREE_DOF_OBJ(admin, dof_int_vec);
    FREE_DOF_OBJ(admin, dof_dof_vec);
    FREE_DOF_OBJ(admin, int_dof_vec);
    FREE_DOF_OBJ(admin, dof_uchar_vec);
    FREE_DOF_OBJ(admin, dof_schar_vec);
    FREE_DOF_OBJ(admin, dof_real_vec);
    FREE_DOF_OBJ(admin, dof_real_d_vec);
    FREE_DOF_OBJ(admin, dof_ptr_vec);

    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_matrix);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->matrix_row);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_dowb_matrix);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dowb_matrix_row);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dowb_matrix_row_s);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dowb_matrix_row_d);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_int_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_dof_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->int_dof_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_uchar_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_schar_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_real_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_real_d_vec);
    deleteObject(((DOF_ADMIN_MEM_INFO *)admin[i]->mem_info)->dof_ptr_vec);

    MEM_FREE(admin[i]->mem_info, 1, DOF_ADMIN_MEM_INFO);
    MEM_FREE(admin[i]->dof_free, admin[i]->dof_free_size, DOF_FREE_UNIT);
  }

  return;
}


/*--------------------------------------------------------------------------*/
/*  allocate memory for a new mesh, and build a simply linked list of these */
/*--------------------------------------------------------------------------*/

typedef struct mesh_info MESH_INFO;

struct mesh_info
{
    MESH       *mesh;
    MESH_INFO  *next;
};

static MESH_INFO  *first_mesh = nil;


extern void AI_advance_cookies_rec(MESH *mesh)
{
  FUNCNAME("AI_advance_cookies_rec");
  int i;
  MESH_MEM_INFO *mem_info;

  TEST_EXIT(mesh,"Oops, did not get a mesh!\n");
  mem_info = (MESH_MEM_INFO *)mesh->mem_info;

  mesh->cookie += 1;

  for(i = 0; i < mem_info->n_slaves; i++)
    AI_advance_cookies_rec(mem_info->slaves[i]);

  return;
}

static MESH *get_mesh(int dim, const char *name,
		      const MACRO_DATA *macro_data,
		      NODE_PROJECTION *(*init_node_proj)
		      (MESH *, MACRO_EL *, int))
{
  FUNCNAME("get_mesh");
  MESH_INFO       *new_mesh;
  MESH            *mesh;
  MESH_MEM_INFO   *mem_info;
  HOOK_QUEUE_ENUM queue;

  new_mesh = MEM_CALLOC(1,MESH_INFO);
  mesh =     MEM_CALLOC(1,MESH);
  
  new_mesh->mesh = mesh;
  new_mesh->next = first_mesh;
  first_mesh = new_mesh;

  mesh->dim  = dim;
  mesh->name = name ? strdup(name) : nil;

  mem_info = (MESH_MEM_INFO *)
    (mesh->mem_info = MEM_CALLOC(1, MESH_MEM_INFO));
  
  mem_info->element = newObject(sizeof(EL), 0); 

  if(mesh->dim == 3)
    mem_info->rc_list = nil; 

  mem_info->real_d = newObject(sizeof(REAL_D), 0);

  mem_info->leaf_data = nil;

  if(macro_data)
    macro_data2mesh(mesh, macro_data, init_node_proj);

  for (queue = HOOK_QUEUE_0; queue <= HOOK_QUEUE_7; queue ++) {
    mesh->traverse_hooks[queue].next =
      mesh->traverse_hooks[queue].prev = &mesh->traverse_hooks[queue];
  }

#if ALBERTA_DEBUG
  srandom(13);
#else
  srandom((unsigned int) time(nil));
#endif

  mesh->cookie = random();

  check_mesh(mesh);
  
  return(mesh);
}

void add_traverse_hook(MESH *mesh,
		       TRAVERSE_HOOK *hook,
		       HOOK_QUEUE_ENUM queue)
{
  mesh->traverse_hooks[queue].next->prev = &hook->node;
  hook->node.next = mesh->traverse_hooks[queue].next;
  hook->node.prev = &mesh->traverse_hooks[queue];
  mesh->traverse_hooks[queue].next = &hook->node;

  mesh->active_hooks |= 1UL << (queue + HOOK_QUEUE_OFFSET);
}

void remove_traverse_hook(MESH *mesh, TRAVERSE_HOOK *hook, HOOK_QUEUE_ENUM queue)
{
  hook->node.next->prev = hook->node.prev;
  hook->node.prev->next = hook->node.next;

  if (mesh->traverse_hooks[queue].next == mesh->traverse_hooks[queue].prev) {
    mesh->active_hooks &= ~(1UL << (queue + HOOK_QUEUE_OFFSET));
  }
}

void free_mesh(MESH *mesh)
{
  FUNCNAME("free_mesh");
  MESH_INFO     *mesh_info, **prev;
  MESH_MEM_INFO *mem_info;
  int i;
  
  if(!mesh) {
    ERROR("No mesh specified!\n");
    return;
  }
  
  mem_info = (MESH_MEM_INFO *)mesh->mem_info;
  
  prev = &first_mesh;
  for (mesh_info = first_mesh; mesh_info; mesh_info = mesh_info->next)
  {
    if (mesh_info->mesh == mesh)
    {
      *prev = mesh_info->next;
      MEM_FREE(mesh_info, 1, MESH_INFO);
/* If this mesh is itself a slave mesh, then unchain it first.              */
      if(mem_info->master) unchain_submesh(mesh);
/* Free all slave meshes...                                                 */
      
      for(i = 0; i < mem_info->n_slaves; i++)
	unchain_submesh(mem_info->slaves[i]);
      
      MEM_FREE(mem_info->slaves, mem_info->n_slaves, MESH *);
      
      /* free all elements/dofs/... */
      deleteObject(mem_info->dof_ptrs);
      
      for(i = 0; i < N_NODE_TYPES; i++)
      {
	if(mem_info->dofs[i])  deleteObject(mem_info->dofs[i]);
      }
      
      deleteObject(mem_info->element);
      if(mem_info->rc_list)
	free_rc_list(mesh, mem_info->rc_list);
      
      deleteObject(mem_info->real_d);
      
      if (mem_info->leaf_data)
	deleteObject(mem_info->leaf_data);

      AI_free_dof_vec_list(mesh);

      MEM_FREE(mem_info->coords, mem_info->count, REAL_D);
      MEM_FREE(mem_info, 1, MESH_MEM_INFO);

      free_dof_admins(mesh);
      
      MEM_FREE(mesh->dof_admin, mesh->n_dof_admin, DOF_ADMIN);
	
      if(mesh->name) free((char *)mesh->name);
      
      MEM_FREE(mesh, 1, MESH);
      break;
    }
    prev = &(mesh_info->next);
  }
  TEST(mesh_info,"mesh not found in mesh_info list; mesh not freed\n");
  
  return;
}


extern MESH *check_and_get_mesh(int dim, int dow, int debug,
				const char *version, const char *name,
				const MACRO_DATA *macro_data,
				NODE_PROJECTION *(*init_node_proj)
				(MESH *, MACRO_EL *, int))
{
  FUNCNAME("check_and_get_mesh");
  int      error = 0;

  if (dow != DIM_OF_WORLD)
  {
    ERROR("%s = %d, but you are using a lib with %s = %d\n",
	  "DIM_OF_WORLD", dow, "DIM_OF_WORLD", DIM_OF_WORLD);
    error++;
  }
  if (dim > DIM_OF_WORLD)
  {
    ERROR("dim == %d > %d == DIM_OF_WORLD!\n",
	   dim, DIM_OF_WORLD);
    error++;
  }
  if (debug != ALBERTA_DEBUG)
  {
    ERROR("%s = %d, but you are using a lib with %s = %d\n",
	  "DEBUG", debug, "DEBUG", ALBERTA_DEBUG);
    error++;
  }
  if (strcmp(version,ALBERTA_VERSION))
  {
    ERROR("you are using %s but a lib with %s\n", version, ALBERTA_VERSION);
    error++;
  }
  if (error) ERROR_EXIT("Bye!\n");
  
  return get_mesh(dim, name, macro_data, init_node_proj);
}


/*--------------------------------------------------------------------------*/
/*  memory management for DOF pointers                                      */
/*--------------------------------------------------------------------------*/

static const int max_dof_ptrs[4] = {2, 4, 7, 15};

static DOF **get_dof_ptrs(MESH *mesh)
{
  FUNCNAME("get_dof_ptrs");
  int i, n;
  DOF **ptrs;
  MESH_MEM_INFO *mem_info;
  
  DEBUG_TEST_EXIT(mesh, "mesh=nil\n");
  DEBUG_TEST_EXIT(mesh->mem_info, "mesh \"%s\": mesh->mem_info=nil\n",
		  mesh->name);
  mem_info = (MESH_MEM_INFO*)(mesh->mem_info);
  n = mesh->n_node_el;
  if (n <= 0)
    return(nil);
  
  DEBUG_TEST_EXIT(n <= max_dof_ptrs[mesh->dim], 
		  "mesh \"%s\": too many nodes: %d > %d\n", 
		  mesh->name, n, max_dof_ptrs[mesh->dim]);
  DEBUG_TEST_EXIT(mem_info->dof_ptrs, 
		  "mesh \"%s\": mesh->mem_info->dof_ptrs=nil\n", mesh->name);
  
  ptrs = (DOF **)getMemory(mem_info->dof_ptrs);
  for (i = 0; i < n; i++)
    ptrs[i] = nil;
  
  return(ptrs);
}


static void free_dof_ptrs(DOF **ptrs, MESH *mesh)
{
  FUNCNAME("free_dof_ptrs");
  int      n;
  MESH_MEM_INFO *mem_info;
  
  DEBUG_TEST_EXIT(ptrs,"ptrs=nil\n");
  DEBUG_TEST_EXIT(mesh, "mesh=nil\n");
  DEBUG_TEST_EXIT(mesh->mem_info,
	      "mesh \"%s\": mesh->mem_info=nil\n", mesh->name);
  n = mesh->n_node_el;
  if (n <= 0)
    return;
  
  DEBUG_TEST_EXIT(n <= max_dof_ptrs[mesh->dim],
	      "mesh \"%s\": too many nodes: %d > %d\n",
	      mesh->name, n, max_dof_ptrs[mesh->dim]);
  
  mem_info = (MESH_MEM_INFO*)(mesh->mem_info);
  
  DEBUG_TEST_EXIT(mem_info->dof_ptrs,
	      "mesh \"%s\": mesh->mem_info->dof_ptrs=nil\n", mesh->name);
  
  freeMemory((void *)ptrs, mem_info->dof_ptrs);
  return;
}

void AI_get_dof_ptr_list(MESH *mesh)
{
  FUNCNAME("AI_get_dof_ptr_list");
  MESH_MEM_INFO *mem_info;

  DEBUG_TEST_EXIT(mesh, "No mesh given!\n");

  mem_info = (MESH_MEM_INFO*)mesh->mem_info;
  DEBUG_TEST_EXIT(mem_info, "No mesh memory info structure present!\n");

  mem_info->dof_ptrs = newObject(mesh->n_node_el * sizeof(DOF*), 1000);

  return;
}

/*--------------------------------------------------------------------------*/
/*  memory management for DOFs                                              */
/*--------------------------------------------------------------------------*/

#define DOF_BLOCK 1000

/****************************************************************************/
/* AI_reactivate_dof(mesh, el):                                             */
/* When coarsening the mesh, we must replace all -1 settings with new DOF   */
/* indices. We no longer allocate new memory for DOF pointers in the        */
/* el->dof[] vector - this memory was also freed in old versions depending  */
/* on the "mesh->preserve_coarse_dofs" setting.                             */
/****************************************************************************/

void AI_reactivate_dof(MESH *mesh, const EL *el)
{
  FUNCNAME("AI_reactivate_dof");
  DOF_ADMIN *admin;
  int     i, j, n, n0, node;
  MESH_MEM_INFO* mem_info;


  DEBUG_TEST_EXIT(mesh,"mesh=nil\n");
  DEBUG_TEST_EXIT(el, "el=nil\n");
  mem_info = (MESH_MEM_INFO *)mesh->mem_info;
  DEBUG_TEST_EXIT(mem_info, "mesh \"%s\": mesh->mem_info=nil\n", mesh->name);


  for (i = 0; i < mesh->n_dof_admin; i++) {
    admin = mesh->dof_admin[i];
    DEBUG_TEST_EXIT(admin, "mesh \"%s\": no dof_admin[%d]\n", mesh->name, i);

    if(mesh->n_dof[CENTER]) {
      node = mesh->node[CENTER];
      n = admin->n_dof[CENTER];
      
      if(n) {
	n0 = admin->n0_dof[CENTER];
      
	DEBUG_TEST_EXIT(n+n0 <= mesh->n_dof[CENTER],
	  "dof_admin \"%s\": n=%d, n0=%d too large: ndof[CENTER]=%d\n",
	   admin->name, n, n0, mesh->n_dof[CENTER]);
	
	if(el->dof[node][n0] == -1)
	  for (j = 0; j < n; j++)
	    el->dof[node][n0+j] = get_dof_index(admin);
      }
    }

#if DIM_OF_WORLD > 1
    if(mesh->n_dof[EDGE]) {
      int k;
      for(k = 0; k < N_EDGES(mesh->dim); k++) {
	node = mesh->node[EDGE] + k;
	n = admin->n_dof[EDGE];

	if(n) {
	  n0 = admin->n0_dof[EDGE];
      
	  DEBUG_TEST_EXIT(n+n0 <= mesh->n_dof[EDGE],
	    "dof_admin \"%s\": n=%d, n0=%d too large: ndof[EDGE]=%d\n",
	     admin->name, n, n0, mesh->n_dof[EDGE]);
	  
	  if(el->dof[node][n0] == -1)
	    for (j = 0; j < n; j++)
	      el->dof[node][n0+j] = get_dof_index(admin);
	}
      }
    }

#if DIM_OF_WORLD == 3
    if(mesh->n_dof[FACE]) {
      int k;
      for(k = 0; k < N_FACES_3D; k++) {
	node = mesh->node[FACE] + k;
	n = admin->n_dof[FACE];

	if(n) {
	  n0 = admin->n0_dof[FACE];
      
	  DEBUG_TEST_EXIT(n+n0 <= mesh->n_dof[FACE],
	    "dof_admin \"%s\": n=%d, n0=%d too large: ndof[FACE]=%d\n",
	     admin->name, n, n0, mesh->n_dof[FACE]);
	  
	  if(el->dof[node][n0] == -1)
	    for (j = 0; j < n; j++)
	      el->dof[node][n0+j] = get_dof_index(admin);
	}
      }
    }
#endif
#endif
  }

  return;
}

DOF *AI_get_dof_memory(MESH *mesh, int position)
{
  FUNCNAME("AI_get_dof_memory");
  int            ndof;
  MESH_MEM_INFO *mem_info;


  DEBUG_TEST_EXIT(mesh, "mesh=nil\n");
  mem_info = (MESH_MEM_INFO *)mesh->mem_info;
  DEBUG_TEST_EXIT(mem_info, "mesh \"%s\": mesh->mem_info=nil\n", mesh->name);

  DEBUG_TEST_EXIT(position >= 0 && position < N_NODE_TYPES,
    "mesh \"%s\": unknown position %d\n", mesh->name, position);

  ndof = mesh->n_dof[position];
  DEBUG_TEST_EXIT(ndof, "mesh->n_dof[%d] == 0!\n", position);

  return (DOF *)getMemory(mem_info->dofs[position]);
}


DOF *get_dof(MESH *mesh, int position)
{
  FUNCNAME("get_dof");
  DOF_ADMIN *admin;
  DOF     *dof;
  int     i, j, n, n0, ndof;
  
  ndof = mesh->n_dof[position];
  if (ndof <= 0) return(nil);

  dof = AI_get_dof_memory(mesh, position);
  
  for (i = 0; i < mesh->n_dof_admin; i++) {
    admin = mesh->dof_admin[i];
    DEBUG_TEST_EXIT(admin, "mesh \"%s\": no dof_admin[%d]\n", mesh->name, i);
    
    n  = admin->n_dof[position];
    n0 = admin->n0_dof[position];
    
    DEBUG_TEST_EXIT(n+n0 <= ndof,
		"dof_admin \"%s\": n=%d, n0=%d too large: ndof=%d\n",
		admin->name, n, n0, ndof);
    
    for (j = 0; j < n; j++)
      dof[n0+j] = get_dof_index(admin);
  }

  return(dof);
}

void free_dof(DOF *dof, MESH *mesh, int position, const int is_coarse_dof)
{
  FUNCNAME("free_dof");

  DOF_ADMIN *admin;
  int     i, j, n, n0, ndof;
  MESH_MEM_INFO* mem_info;
  
  DEBUG_TEST_EXIT(mesh, "mesh=nil\n");
  mem_info = (MESH_MEM_INFO*)mesh->mem_info;
  DEBUG_TEST_EXIT(mem_info,"mesh \"%s\": mesh->mem_info=nil\n", mesh->name);
  
  DEBUG_TEST_EXIT(position >= 0 && position < N_NODE_TYPES,
	      "mesh \"%s\": unknown position %d\n",mesh->name, position);
  
  ndof = mesh->n_dof[position];
  
  DEBUG_TEST_EXIT(!ndof || dof,
	      "dof = nil, but ndof=%d\n", ndof);
  DEBUG_TEST_EXIT(ndof || !dof,
	      "dof != nil, but ndof=0\n");
  
  DEBUG_TEST_EXIT(mem_info->dofs[position],
	      "mesh \"%s\": no memory management present for %d DOFs.",
	      mesh->name, position);
  
  for (i = 0; i < mesh->n_dof_admin; i++) {
    admin = mesh->dof_admin[i];
    DEBUG_TEST_EXIT(admin, "mesh \"%s\": no dof_admin[%d]\n", mesh->name, i);
    
    n  = admin->n_dof[position];
    n0 = admin->n0_dof[position];
    
    DEBUG_TEST_EXIT(n+n0 <= ndof,
		"dof_admin \"%s\": n=%d, n0=%d too large: ndof=%d\n",
		admin, n, n0, ndof);
    
    if(!(admin->preserve_coarse_dofs && is_coarse_dof))
      for (j = 0; j < n; j++) {
	free_dof_index(admin, dof[n0+j]);
	dof[n0+j] = -1;  /* Mark this DOF as unused!                        */
      }
  }

  if(!is_coarse_dof) {
    /* Also free the DOF pointer memory.                                    */
    freeMemory((void *)dof, mem_info->dofs[position]);
  }
  
  return;
}

/****************************************************************************/
/* AI_get_dof_list(mesh, position):                                         */
/* Allocate a memory management object to administrate DOFs. The el->dof[]  */
/* entries will point into this memory space. Please note that we will also */
/* allocate space for freed coarse DOFs - the allocated DOFs will have value*/
/* -1 to show that they are no longer being used.                           */
/*                                                                          */
/* This routine is also called from read_mesh.c.                            */
/****************************************************************************/

void AI_get_dof_list(MESH *mesh, int position)
{
  FUNCNAME("AI_get_dof_list");
  MESH_MEM_INFO *mem_info;

  DEBUG_TEST_EXIT(mesh, "No mesh given!\n");
  DEBUG_TEST_EXIT(position >=0 && position < N_NODE_TYPES, 
	      "Illegal position %d!\n", position);
  DEBUG_TEST_EXIT(mesh->n_dof[position], 
		  "Mesh has no DOFs on this position!\n");

  mem_info = (MESH_MEM_INFO*)mesh->mem_info;

  DEBUG_TEST_EXIT(mem_info, "No mesh memory info structure found!\n");

  mem_info->dofs[position] = 
    newObject(sizeof(DOF)*mesh->n_dof[position], DOF_BLOCK);

  return;
}


/*--------------------------------------------------------------------------*/
/*  memory management for FE_SPACE and DOF_ADMIN structures                 */
/*--------------------------------------------------------------------------*/

/****************************************************************************/
/* transfer_dofs(mesh, new_admin, old_dof, position, is_coarse_dof):        */
/* We allocate and return memory for a new dof pointer in an el->dof[]      */
/* entry. The field is filled either with new DOF indices for each admin or */
/* with -1 to mark it as unused.                                            */
/****************************************************************************/

static DOF *transfer_dofs(MESH *mesh, DOF_ADMIN *new_admin, 
			  DOF *old_dof, int position, int is_coarse_dof)
{
  /* FUNCNAME("transfer_dofs"); */
  DOF_ADMIN *admin;
  DOF       *new_dof = nil;
  int        i, j, n, n0, ndof = mesh->n_dof[position];
 
  if (ndof <= 0) return nil;

  new_dof = AI_get_dof_memory(mesh, position);

  for (i = 0; i < mesh->n_dof_admin; i++) {
    admin = mesh->dof_admin[i];
    
    n  = admin->n_dof[position];
    n0 = admin->n0_dof[position];
    
    for (j = 0; j < n; j++)
      if(admin == new_admin)
	if(!is_coarse_dof || admin->preserve_coarse_dofs)
	  new_dof[n0+j] = get_dof_index(admin);
	else
	  new_dof[n0+j] = -1;
      else
	if(old_dof)
	  new_dof[n0+j] = old_dof[n0 + j];
	else
	  new_dof[n0+j] = -1;
  }

  return new_dof;
}


#include "memory_0d.c"
#include "memory_1d.c"
#if DIM_OF_WORLD > 1
#include "memory_2d.c"
#if DIM_OF_WORLD > 2
#include "memory_3d.c"
#endif
#endif

/****************************************************************************/
/* get_fe_space(mesh,name,n_dof,bas_fcts,preserve_coarse_dofs): Unlike in   */
/* older versions of ALBERTA, this routine can be called at any time, even  */
/* after mesh refinement. As this is implemented at the price of temporary  */
/* memory usage and CPU time, it should still be avoided.                   */
/****************************************************************************/

const FE_SPACE *get_fe_space(MESH *mesh, const char *name,
			     const int n_dof[N_NODE_TYPES],
			     const BAS_FCTS *bas_fcts,
			     const U_CHAR preserve_coarse_dofs)
{
  FUNCNAME("get_fe_space");
  DOF_ADMIN         *admin = nil;
  FE_SPACE          *fe_space;
  int               i, j, good_admin;
  const int         *ndof;
  
  fe_space = MEM_ALLOC(1, FE_SPACE);
  fe_space->name = name ? strdup(name) : nil;
  
  if (bas_fcts)  ndof = bas_fcts->n_dof;
  else           ndof = n_dof;

  TEST_EXIT(ndof,"no n_dof or bas_fcts->n_dof\n");
  
  TEST_EXIT((bas_fcts==nil) || (bas_fcts->dim == mesh->dim),
    "Dimension of basis functions %d does not match mesh dimension %d!\n", 
     bas_fcts->dim, mesh->dim);
  
/****************************************************************************/
/* Search for a fitting DOF_ADMIN to serve the required n_dof field and     */
/* preserve_coarse_dofs status. Both have to match exactly!                 */
/****************************************************************************/
    for (i = 0; i < mesh->n_dof_admin; i++) {
      admin = mesh->dof_admin[i];
      good_admin = true;
      
      for (j = 0; j < N_NODE_TYPES; j++) {
	if (admin->n_dof[j] != ndof[j]) {
	  good_admin = false;
	  break;
	}
      }
      if (admin->preserve_coarse_dofs != preserve_coarse_dofs)
	good_admin = false;
      
      if(good_admin == true) break;
      
      admin = nil;
    }

    if (!admin) {
/****************************************************************************/
/* We did not find a fitting admin. In this case, we must adjust the        */
/* mem_info->dof_ptrs and mem_info->dofs[i] fields.                         */
/****************************************************************************/
      int            old_n_node_el;
      int            old_n_dof[N_NODE_TYPES];
      int            old_node[N_NODE_TYPES];
      void          *old_dof_ptrs;
      void          *old_dofs[N_NODE_TYPES];
      MESH_MEM_INFO *mem_info = (MESH_MEM_INFO *)mesh->mem_info;

      old_n_node_el            = mesh->n_node_el;
      old_dof_ptrs             = mem_info->dof_ptrs;
      for(i = 0; i < N_NODE_TYPES; i++) {
	old_n_dof[i] = mesh->n_dof[i];
	old_node[i]  = mesh->node[i];
	old_dofs[i]  = mem_info->dofs[i];
      }

      admin = AI_get_dof_admin(mesh, name, ndof);
      admin->preserve_coarse_dofs = (preserve_coarse_dofs ? 1 : 0);

/* We must now adjust the mem_info->dofs[i] memory management object.       */
      for(i = 0; i < N_NODE_TYPES; i++)
	if(ndof[i])
	  AI_get_dof_list(mesh, i);

/* Did mesh->n_node_el change while adding the new admin? If so, we have to */
/* adjust the mem_info->dof_ptrs memory management object.                  */
      if(old_n_node_el < mesh->n_node_el)
	AI_get_dof_ptr_list(mesh);

/* Now we magically adjust all pointers by traversing the mesh. :-)         */

      switch(mesh->dim) {
      case 0:
	adjust_dofs_and_dof_ptrs_0d(mesh, admin,
				    old_n_node_el, old_n_dof, old_node);
	break;
      case 1:
	adjust_dofs_and_dof_ptrs_1d(mesh, admin,
				    old_n_node_el, old_n_dof, old_node);
	break;
#if DIM_OF_WORLD > 1
      case 2:
	adjust_dofs_and_dof_ptrs_2d(mesh, admin,
				    old_n_node_el, old_n_dof, old_node);
	break;
#if DIM_OF_WORLD == 3
      case 3:
	adjust_dofs_and_dof_ptrs_3d(mesh, admin,
				    old_n_node_el, old_n_dof, old_node);
	break;
#endif
#endif
      default:
	ERROR_EXIT("Illegal mesh dimension!\n");
      }
/* Release old memory management objects.                                   */

      if(old_n_node_el < mesh->n_node_el && old_dof_ptrs)
	deleteObject(old_dof_ptrs);

      for(i = 0; i < N_NODE_TYPES; i++)
	if(ndof[i] && old_dofs[i])
	  deleteObject(old_dofs[i]);
    }

    fe_space->admin = admin;
    fe_space->bas_fcts = bas_fcts;
    fe_space->mesh = mesh;

    return(fe_space);
}

void free_fe_space(FE_SPACE *fe_space)
{
  FUNCNAME("free_fe_space");

  if(!fe_space) {
    ERROR("No fe_space specified!\n");
    return;
  }

  if(fe_space->name) free((char *)fe_space->name);
  MEM_FREE(fe_space, 1, FE_SPACE);

  return;
}


/****************************************************************************/
/* AI_fill_missing_dofs(mesh):                                              */
/* This routine allocates currently unused element DOF pointers since this  */
/* is not done during read_mesh(). These corresponding values pointed to    */
/* will be -1. The reason: write_mesh() does not save these pointers in     */
/* the file, since this information is unnecessary to recreate DOF_ADMINs.  */
/*                                                                          */
/* called by read_mesh().                                                   */
/****************************************************************************/

void AI_fill_missing_dofs(MESH *mesh)
{
  FUNCNAME("AI_fill_missing_dofs");

  DEBUG_TEST_EXIT(mesh, "Did not supply a mesh!\n");

  switch(mesh->dim) {
  case 0:
    break;
  case 1:
    fill_missing_dofs_1d(mesh);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    fill_missing_dofs_2d(mesh);
    break;
#if DIM_OF_WORLD == 3
  case 3:
    fill_missing_dofs_3d(mesh);
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal mesh dimension!\n");
  }

  return;
}


/*--------------------------------------------------------------------------*/
/*  memory management for leaf_data                                         */
/*--------------------------------------------------------------------------*/

#define LEAF_DATA_BLOCK  1000

void *AI_get_leaf_data(MESH *mesh)
{
  FUNCNAME("AI_get_leaf_data");
  MEMORYADMIN  *ldbi;

  DEBUG_TEST_EXIT(mesh, "pointer to mesh = nil\n");

  ldbi = (MEMORYADMIN *)((MESH_MEM_INFO*)(mesh->mem_info))->leaf_data;

  if (ldbi)
    return(getMemory(ldbi));
  else
    return(nil);
}

void AI_free_leaf_data(void *leaf_data, MESH *mesh)
{
    FUNCNAME("AI_free_leaf_data");
    MEMORYADMIN  *ldbi;

    if (!leaf_data)  return;

    DEBUG_TEST_EXIT(mesh, "pointer to mesh = nil\n");
    ldbi = (MEMORYADMIN *) ((MESH_MEM_INFO*)(mesh->mem_info))->leaf_data;

    if (!ldbi) return;

    freeMemory((void *)leaf_data, ldbi);
    return;
}


/****************************************************************************/
/* init_leaf_data(mesh, name, size, rld, cld):                              */
/* Initialize leaf data on the mesh.                                        */
/****************************************************************************/

size_t init_leaf_data(MESH *mesh, size_t size, 
		      void (*refine_leaf_data)(EL *parent, EL *child[2]),
		      void (*coarsen_leaf_data)(EL *parent, EL *child[2]))
{
  MESH_MEM_INFO    *mem_info;
  size_t            new_size;
  TRAVERSE_STACK   *stack = get_traverse_stack();
  const EL_INFO    *el_info = nil;

  TEST_EXIT(mesh,"No mesh specified!\n");
  TEST_EXIT(size, "size must be > 0!\n");
  TEST_EXIT(mesh->mem_info,"No memory management present for mesh!\n");

  mem_info = (MESH_MEM_INFO *)mesh->mem_info;

  TEST_EXIT(!mem_info->leaf_data,"Leaf data was already initialized!\n");
    
  new_size = ALIGNSIZE(size);

  if (new_size != size)
    WARNING("installing leafdata of size %d with aligned size %d\n", 
	      size, new_size);

  mem_info->leaf_data_info->leaf_data_size    = new_size;
  mem_info->leaf_data_info->refine_leaf_data  = refine_leaf_data;
  mem_info->leaf_data_info->coarsen_leaf_data = coarsen_leaf_data;
  
  mem_info->leaf_data = newObject(new_size, 0);
  
  el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL);

  while (el_info) {
    el_info->el->child[1] = (EL *) AI_get_leaf_data(mesh);
 
    el_info = traverse_next(stack, el_info);
  }

  free_traverse_stack(stack);

  return new_size;
}


#if 0  /*  not used at the moment              */
/*--------------------------------------------------------------------------*/
/*  memory management for quadratures                                       */
/*--------------------------------------------------------------------------*/

#define QUAD_VEC_BLOCK  100
#define MAX_QUAD_INFO     3

static MEM_BLOCK_INFO *quad_vec_block_infos[MAX_QUAD_INFO+1];

REAL *get_quad_vec(int dim)
{
    FUNCNAME("get_quad_vec");
    int             i, n;
    REAL            *quad_vec;
    MEM_BLOCK_INFO  *gvbi;

    DEBUG_TEST_EXIT(dim >= 1  &&  dim  <= 3, "dim = %d not allowed\n");

    gvbi = quad_vec_block_infos[dim];

    if (!gvbi)
    {
	gvbi = MEM_ALLOC(1, MEM_BLOCK_INFO);
	gvbi->free       = nil;
	gvbi->info       = nil;
	gvbi->free_count = 0;
	gvbi->used_count = 0;
	gvbi->unit_size  = get_max_no_quad_point(dim)*sizeof(REAL);
	gvbi->block_size = QUAD_VEC_BLOCK;

	quad_vec_block_infos[dim] = gvbi;
    }

    quad_vec = (REAL *) get_mem(gvbi);

    return(quad_vec);
}

void free_quad_vec(REAL *quad_vec, int dim)
{
    FUNCNAME("free_quad_vec");
    int             n;
    MEM_BLOCK_INFO  *gvbi;

    if (!quad_vec)  return;

    DEBUG_TEST_EXIT(dim >= 1  &&  dim  <= 3, "dim = %d not allowed\n");

    gvbi = quad_vec_block_infos[dim];

    if (!gvbi) return;

    free_mem((void *) quad_vec, gvbi);
    return;
}

static MEM_BLOCK_INFO *quad_vec_d_block_infos[MAX_QUAD_INFO+1];

REAL_D *get_quad_vec_d(int dim)
{
    FUNCNAME("get_quad_vec_d");
    int      i, n;
    REAL_D          *quad_vec_d;
    MEM_BLOCK_INFO  *gvdbi;

    DEBUG_TEST_EXIT(dim >= 1  &&  dim  <= 3, "dim = %d not allowed\n");

    gvdbi = quad_vec_d_block_infos[dim];

    if (!gvdbi)
    {
	gvdbi = MEM_ALLOC(1, MEM_BLOCK_INFO);
	gvdbi->free       = nil;
	gvdbi->info       = nil;
	gvdbi->free_count = 0;
	gvdbi->used_count = 0;
	gvdbi->unit_size  = get_max_no_quad_point(dim)*sizeof(REAL_D);
	gvdbi->block_size = QUAD_VEC_BLOCK;

	quad_vec_d_block_infos[dim] = gvdbi;
    }

    quad_vec_d = (REAL_D *) get_mem(gvdbi);

    return(quad_vec_d);
}

void free_quad_vec_d(REAL_D *quad_vec_d, int dim)
{
    FUNCNAME("free_quad_vec_d");
    int             n;
    MEM_BLOCK_INFO  *gvdbi;

    if (!quad_vec_d)  return;

    DEBUG_TEST_EXIT(dim >= 1  &&  dim  <= 3, "dim = %d not allowed\n");

    gvdbi = quad_vec_d_block_infos[dim];

    if (!gvdbi) return;

    free_mem((void *) quad_vec_d, gvdbi);
    return;
}
#endif

/*--------------------------------------------------------------------------*/
/*  memory management for elements                                          */
/*--------------------------------------------------------------------------*/

#define EL_BLOCK   1000

#if ALBERTA_DEBUG
static int el_index = 0;
#endif

EL *get_element(MESH *mesh) 
{
  FUNCNAME("get_element");
  EL *el;
  
  DEBUG_TEST_EXIT(mesh, "mesh == nil\n");
  DEBUG_TEST_EXIT(mesh->mem_info,
    "mesh \"%s\": no memory management present.\n", mesh->name);

  el = (EL *)getMemory(((MESH_MEM_INFO*)(mesh->mem_info))->element);
  el->child[0]  = nil;
  el->child[1]  = (EL *) AI_get_leaf_data(mesh);
  el->dof       = get_dof_ptrs(mesh);
#if ALBERTA_DEBUG
  el->index     = el_index++;
#endif
  el->mark      = 0;
  el->new_coord = nil;
  
  return el;
}

void free_element(EL *el, MESH *mesh)
{
    free_dof_ptrs(el->dof, mesh);
    if(mesh->dim > 1 && el->new_coord) {
      free_real_d(mesh, el->new_coord);
      el->new_coord = nil;
    }

    if (el->child[1]) AI_free_leaf_data((void *) el->child[1], mesh);

    freeMemory(el, ((MESH_MEM_INFO*)(mesh->mem_info))->element);
}


/*--------------------------------------------------------------------------*/
/*  some routines for allocation and deallocation of list for collecting    */
/*  neighbours at the refinement edge in 3 dimensions                       */
/*--------------------------------------------------------------------------*/

#define LIST_BLOCK 20

/* static MEM_BLOCK_INFO rc_list_block_info = {nil, nil, 0, 0,  0, LIST_BLOCK}; */

RC_LIST_EL *get_rc_list(MESH *mesh)
{
  FUNCNAME("get_rc_list");

  if (!((MESH_MEM_INFO*)mesh->mem_info)->rc_list) {
    ((MESH_MEM_INFO*)mesh->mem_info)->rc_list = 
      newObject(mesh->max_edge_neigh*sizeof(RC_LIST_EL), LIST_BLOCK);
  }
  else {
    DEBUG_TEST_EXIT(
   ((MEMORYADMIN *)((MESH_MEM_INFO*)mesh->mem_info)->rc_list)->objectSize == 
   (mesh->max_edge_neigh * sizeof(RC_LIST_EL)),
   "mesh \"%s\": mesh->max_edge_neigh changed\n", mesh->name);
  }

  return (RC_LIST_EL *)getMemory(((MESH_MEM_INFO*)mesh->mem_info)->rc_list);
}


void free_rc_list(MESH *mesh, RC_LIST_EL *list)
{
    freeMemory((void *)list, ((MESH_MEM_INFO *)(mesh->mem_info))->rc_list);
}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

REAL *get_real_d(MESH *mesh)
{
    FUNCNAME("get_real_d");

    DEBUG_TEST_EXIT(mesh, "mesh==nil\n");
    return (REAL *)getMemory(((MESH_MEM_INFO*)(mesh->mem_info))->real_d);
}

void free_real_d(MESH *mesh, REAL *ptr)
{
    FUNCNAME("free_real_d");

    DEBUG_TEST_EXIT(mesh, "mesh==nil\n");
    freeMemory((void *)ptr, ((MESH_MEM_INFO*)(mesh->mem_info))->real_d);
    return;
}

/*--------------------------------------------------------------------------*/
/*  memory management for matrix rows                                       */
/*  matrix rows not connected to a FE_SPACE, e.g. in multigrid.c are        */
/*  treated specially. Others are fixed to a FE_SPACE and thus to a unique  */
/*  DOF_ADMIN and MESH.                                                     */
/*--------------------------------------------------------------------------*/

static void *unconnected_rows = nil;

MATRIX_ROW  *get_matrix_row(const FE_SPACE *fe_space)
{
    MATRIX_ROW *row;
    static void *matrix_row_info = nil;
    int        j;

    if(!fe_space || !fe_space->admin) {
      if(!unconnected_rows)
	unconnected_rows = newObject(sizeof(MATRIX_ROW), 100); 
      matrix_row_info = unconnected_rows;
    }
    else
      matrix_row_info = ((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->matrix_row;

    row = (MATRIX_ROW *)getMemory(matrix_row_info);
    row->next = nil;
    for (j=0; j<ROW_LENGTH; j++) 
	row->col[j] = NO_MORE_ENTRIES;

    return(row);
}

void free_matrix_row(const FE_SPACE *fe_space, MATRIX_ROW *row)
{
  static void *matrix_row_info = nil;

  if(!fe_space || !fe_space->admin)
    matrix_row_info = unconnected_rows;
  else
    matrix_row_info = ((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->matrix_row;

  freeMemory((void *)row, matrix_row_info);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_rows(void))
{
  if(unconnected_rows) deleteObject(unconnected_rows);
  unconnected_rows = nil;
}

static void *unconnected_dowb_rows   = nil;
static void *unconnected_dowb_rows_s = nil;
static void *unconnected_dowb_rows_d = nil;

DOWB_MATRIX_ROW *get_dowb_matrix_row(const FE_SPACE *fe_space, DOWBM_TYPE type)
{
  DOWB_MATRIX_ROW *row;
  void *matrix_row_info = nil;
  int        j;

  switch (type) {
  case dowbm_full:
    if(!fe_space || !fe_space->admin) {
      if(!unconnected_dowb_rows)
	unconnected_dowb_rows =
	  newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_DD), 100); 
      matrix_row_info = unconnected_dowb_rows;
    } else {
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row;
    }
    break;
  case dowbm_symm:
    if(!fe_space || !fe_space->admin) {
      if(!unconnected_dowb_rows_s)
	unconnected_dowb_rows_s =
	  newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_DDS), 100); 
      matrix_row_info = unconnected_dowb_rows_s;
    } else {
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row_s;
    }
    break;
  case dowbm_diag:
    if(!fe_space || !fe_space->admin) {
      if(!unconnected_dowb_rows_d)
	unconnected_dowb_rows_d =
	  newObject(sizeof(DOWB_MATRIX_ROW)+ROW_LENGTH*sizeof(REAL_D), 100); 
      matrix_row_info = unconnected_dowb_rows_d;
    } else {
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row_d;
    }
    break;
  default:
    break;
  }
  
  row = (DOWB_MATRIX_ROW *)getMemory(matrix_row_info);
  row->next = nil;
  for (j=0; j<ROW_LENGTH; j++) {
    row->col[j] = NO_MORE_ENTRIES;
  }

  return(row);
}

void free_dowb_matrix_row(const FE_SPACE *fe_space, DOWBM_TYPE type, 
			  DOWB_MATRIX_ROW *row)
{
  void *matrix_row_info = nil;

  switch (type) {
  case dowbm_full:
    if(!fe_space || !fe_space->admin)
      matrix_row_info = unconnected_dowb_rows;
    else
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row;
    
    break;
  case dowbm_symm:
    if(!fe_space || !fe_space->admin)
      matrix_row_info = unconnected_dowb_rows_s;
    else
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row_s;
    
    break;
  case dowbm_diag:
    if(!fe_space || !fe_space->admin)
      matrix_row_info = unconnected_dowb_rows_d;
    else
      matrix_row_info =
	((DOF_ADMIN_MEM_INFO *)fe_space->admin->mem_info)->dowb_matrix_row_d;
    
    break;
  default:
    break;
  }

  freeMemory((void *)row, matrix_row_info);
}

#if 0
static void delete_unconnected_dowb_rows(void)
{
  if(unconnected_dowb_rows) deleteObject(unconnected_dowb_rows);
  unconnected_dowb_rows = nil;
  if(unconnected_dowb_rows) deleteObject(unconnected_dowb_rows_s);
  unconnected_dowb_rows_s = nil;
  if(unconnected_dowb_rows) deleteObject(unconnected_dowb_rows_d);
  unconnected_dowb_rows_d = nil;
}
#endif

/*--------------------------------------------------------------------------*/
/*  memory management for matrices                                          */
/*  unconnected_matrices: see unconnected rows above...                     */
/*--------------------------------------------------------------------------*/

static void *unconnected_matrices = nil;

DOF_MATRIX  *get_dof_matrix(const char *name, 
			    const FE_SPACE *row_fe_space,
			    const FE_SPACE *col_fe_space)
{
    DOF_MATRIX *mat;
    void *matrix_info = nil;

    if(!row_fe_space || !row_fe_space->admin) {
      if(!unconnected_matrices)
	unconnected_matrices = newObject(sizeof(DOF_MATRIX), 10); 
      matrix_info = unconnected_matrices;
    }
    else {
      matrix_info = 
	((DOF_ADMIN_MEM_INFO *)row_fe_space->admin->mem_info)->dof_matrix;

      if(!col_fe_space)
	col_fe_space = row_fe_space;
    }

    mat = (DOF_MATRIX *)getMemory(matrix_info);

    mat->next   = nil;
    mat->row_fe_space = row_fe_space;
    mat->col_fe_space = col_fe_space;
    mat->name   = name ? strdup(name) : nil;
    mat->matrix_row = nil;
    mat->size   = 0;

    mat->refine_interpol = mat->coarse_restrict = nil;

    mat->mem_info = nil;

    if (row_fe_space && row_fe_space->admin)
      AI_add_dof_matrix_to_admin(mat, (DOF_ADMIN *)row_fe_space->admin);

    return(mat);
}

void free_dof_matrix(DOF_MATRIX *mat)
{
  if (mat->row_fe_space && mat->row_fe_space->admin)
    AI_remove_dof_matrix_from_admin(mat);
  
  clear_dof_matrix(mat);  /* free all matrix_rows */
  

  MEM_FREE(mat->matrix_row, mat->size, MATRIX_ROW *);
  mat->matrix_row = nil;
  mat->size = 0;

  if(mat->row_fe_space && mat->row_fe_space->admin)
    freeMemory((void *)mat, 
     ((DOF_ADMIN_MEM_INFO*)(mat->row_fe_space->admin->mem_info))->dof_matrix);
  else
    freeMemory((void *)mat, unconnected_matrices);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_matrices(void))
{
  if(unconnected_matrices) deleteObject(unconnected_matrices);
  unconnected_matrices = nil;
}

static void *unconnected_dowb_matrices = nil;

DOF_DOWB_MATRIX  *get_dof_dowb_matrix(const char *name,
				      const FE_SPACE *row_fe_space,
				      const FE_SPACE *col_fe_space,
				      DOWBM_TYPE type)
{
  DOF_DOWB_MATRIX *mat;
  void *matrix_info = nil;

  if(!row_fe_space || !row_fe_space->admin) {
    if(!unconnected_dowb_matrices)
      unconnected_dowb_matrices = newObject(sizeof(DOF_DOWB_MATRIX), 10); 
    matrix_info = unconnected_dowb_matrices;
  }
  else {
    matrix_info =
      ((DOF_ADMIN_MEM_INFO *)row_fe_space->admin->mem_info)->dof_dowb_matrix;

    if(!col_fe_space)
      col_fe_space = row_fe_space;
  }

  mat = (DOF_DOWB_MATRIX *)getMemory(matrix_info);

  mat->next   = nil;
  mat->row_fe_space = row_fe_space;
  mat->col_fe_space = col_fe_space;
  mat->name   = name ? strdup(name) : nil;
  mat->matrix_row = nil;
  mat->size   = 0;
    
  mat->refine_interpol = mat->coarse_restrict = nil;

  mat->mem_info = nil;

  if (row_fe_space && row_fe_space->admin)
    AI_add_dof_dowb_matrix_to_admin(mat, (DOF_ADMIN *)row_fe_space->admin);

  mat->type = type;

  return(mat);
}

void free_dof_dowb_matrix(DOF_DOWB_MATRIX *mat)
{
  if (mat->row_fe_space && mat->row_fe_space->admin)
    AI_remove_dof_dowb_matrix_from_admin(mat);
  
  clear_dof_dowb_matrix(mat);  /* free all matrix_rows */
  

  MEM_FREE(mat->matrix_row, mat->size, MATRIX_ROW *);
  mat->matrix_row = nil;
  mat->size = 0;

  if(mat->row_fe_space && mat->row_fe_space->admin)
    freeMemory((void *)mat,
 ((DOF_ADMIN_MEM_INFO*)(mat->row_fe_space->admin->mem_info))->dof_dowb_matrix);
  else
    freeMemory((void *)mat, unconnected_dowb_matrices);
}

#if 0
static void delete_unconnected_dowb_matrices(void)
{
  if(unconnected_dowb_matrices) deleteObject(unconnected_dowb_matrices);
  unconnected_dowb_matrices = nil;
}
#endif

/*--------------------------------------------------------------------------*/
/*  memory management for dof_vectors                                       */
/*  unconnected dof_vectors are handled as above                            */
/*--------------------------------------------------------------------------*/

static void *unconnected_dof_int_vecs = nil;

DOF_INT_VEC  *get_dof_int_vec(const char *name, const FE_SPACE *fe_space)
{
    DOF_INT_VEC *vec;
    static void *vec_info = nil;


    if(!fe_space || !fe_space->admin) {
      if(!unconnected_dof_int_vecs)
	unconnected_dof_int_vecs = newObject(sizeof(DOF_INT_VEC), 10);
      vec_info = unconnected_dof_int_vecs;
    }
    else
      vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_int_vec;

    vec = (DOF_INT_VEC *)getMemory(vec_info);

    vec->next  = nil;
    vec->fe_space = fe_space;
    vec->name  = name ? strdup(name): nil;
    vec->size  = 0;
    vec->vec   = nil;
    vec->refine_interpol = vec->coarse_restrict = nil;
    vec->mem_info = nil;

    if (fe_space && fe_space->admin)
	AI_add_dof_int_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);

    return(vec);
}

void free_dof_int_vec(DOF_INT_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_int_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, int);
  vec->vec = nil;
  vec->size = 0;
  
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_int_vec);
  else
    freeMemory((void *)vec, unconnected_dof_int_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_int_vecs(void))
{
  if(unconnected_dof_int_vecs) deleteObject(unconnected_dof_int_vecs);
  unconnected_dof_int_vecs = nil;
}

static void *unconnected_dof_dof_vecs = nil;

DOF_DOF_VEC  *get_dof_dof_vec(const char *name, const FE_SPACE *fe_space)
{
    DOF_DOF_VEC *vec;
    static void *vec_info = nil;


    if(!fe_space || !fe_space->admin) {
      if(!unconnected_dof_dof_vecs)
	unconnected_dof_dof_vecs = newObject(sizeof(DOF_DOF_VEC), 10);
      vec_info = unconnected_dof_dof_vecs;
    }
    else
      vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_dof_vec;

    vec = (DOF_DOF_VEC *)getMemory(vec_info);

    vec->next  = nil;
    vec->fe_space = fe_space;
    vec->name  = name ? strdup(name) : nil;
    vec->size  = 0;
    vec->vec   = nil;
    vec->refine_interpol = vec->coarse_restrict = nil;
    vec->mem_info = nil;

    if (fe_space && fe_space->admin)
	AI_add_dof_dof_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);

    return(vec);
}

void free_dof_dof_vec(DOF_DOF_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_dof_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, int);
  vec->vec = nil;
  vec->size = 0;
  
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_dof_vec);
  else
    freeMemory((void *)vec, unconnected_dof_dof_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_dof_vecs(void))
{
  if(unconnected_dof_dof_vecs) deleteObject(unconnected_dof_dof_vecs);
  unconnected_dof_dof_vecs = nil;
}


static void *unconnected_int_dof_vecs = nil;

DOF_DOF_VEC  *get_int_dof_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_DOF_VEC *vec;
  static void *vec_info = nil;  
  
  if(!fe_space || !fe_space->admin) {
    if(!unconnected_int_dof_vecs)
      unconnected_int_dof_vecs = newObject(sizeof(DOF_DOF_VEC), 10);
    vec_info = unconnected_int_dof_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->int_dof_vec;
  
  vec = (DOF_DOF_VEC *)getMemory(vec_info);
  
  vec->next  = nil;
  vec->fe_space = fe_space;
  vec->name  = name ? strdup(name) : nil;
  vec->size  = 0;
  vec->vec   = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  
  if (fe_space && fe_space->admin)
    AI_add_int_dof_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);

  return(vec);
}

void free_int_dof_vec(DOF_DOF_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_int_dof_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, int);
  vec->vec = nil;
  vec->size = 0;

  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->int_dof_vec);
  else
    freeMemory((void *)vec, unconnected_int_dof_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_int_dof_vecs(void))
{
  if(unconnected_int_dof_vecs) deleteObject(unconnected_int_dof_vecs);
  unconnected_int_dof_vecs = nil;
}


static void *unconnected_dof_uchar_vecs = nil;

DOF_UCHAR_VEC  *get_dof_uchar_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_UCHAR_VEC *vec;
  static void *vec_info = nil;


  if(!fe_space || !fe_space->admin) {
    if(!unconnected_dof_uchar_vecs)
      unconnected_dof_uchar_vecs = newObject(sizeof(DOF_UCHAR_VEC), 10);
    vec_info = unconnected_dof_uchar_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_uchar_vec;
  
  vec = (DOF_UCHAR_VEC *)getMemory(vec_info);
  
  vec->next = nil;
  vec->fe_space = fe_space;
  vec->name = name ? strdup(name) : nil;
  vec->size = 0;
  vec->vec = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  vec->mem_info = nil;

  if (fe_space && fe_space->admin)
    AI_add_dof_uchar_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);
  
  return(vec);
}

void free_dof_uchar_vec(DOF_UCHAR_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_uchar_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, U_CHAR);
  vec->vec = nil;
  vec->size = 0;

  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_uchar_vec);
  else
    freeMemory((void *)vec, unconnected_dof_uchar_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_uchar_vecs(void))
{
  if(unconnected_dof_uchar_vecs) deleteObject(unconnected_dof_uchar_vecs);
  unconnected_dof_uchar_vecs = nil;
}


static void *unconnected_dof_schar_vecs = nil;

DOF_SCHAR_VEC  *get_dof_schar_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_SCHAR_VEC *vec;
  static void *vec_info = nil;


  if(!fe_space || !fe_space->admin) {
    if(!unconnected_dof_schar_vecs)
      unconnected_dof_schar_vecs = newObject(sizeof(DOF_SCHAR_VEC), 10);
    vec_info = unconnected_dof_schar_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_schar_vec;
  
  vec = (DOF_SCHAR_VEC *)getMemory(vec_info);

  vec->next = nil;
  vec->fe_space = fe_space;
  vec->name = name ? strdup(name) : nil;
  vec->size = 0;
  vec->vec = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  vec->mem_info = nil;

  if (fe_space && fe_space->admin)
    AI_add_dof_schar_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);
  
  return(vec);
}

void free_dof_schar_vec(DOF_SCHAR_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_schar_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, S_CHAR);
  vec->vec = nil;
  vec->size = 0;
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_schar_vec);
  else
    freeMemory((void *)vec, unconnected_dof_schar_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_schar_vecs(void))
{
  if(unconnected_dof_schar_vecs) deleteObject(unconnected_dof_schar_vecs);
  unconnected_dof_schar_vecs = nil;
}


static void *unconnected_dof_real_vecs = nil;

DOF_REAL_VEC  *get_dof_real_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_REAL_VEC *vec;
  static void *vec_info = nil;


  if(!fe_space || !fe_space->admin) {
    if(!unconnected_dof_real_vecs)
      unconnected_dof_real_vecs = newObject(sizeof(DOF_REAL_VEC), 10);
    vec_info = unconnected_dof_real_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_real_vec;
  
  vec = (DOF_REAL_VEC *)getMemory(vec_info);

  vec->next = nil;
  vec->fe_space = fe_space;
  vec->name = name ? strdup(name) : nil;
  vec->size = 0;
  vec->vec = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  vec->mem_info = nil;

  if (fe_space && fe_space->admin)
    AI_add_dof_real_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);
  
  return(vec);
}

void free_dof_real_vec(DOF_REAL_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_real_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, REAL);
  vec->vec = nil;
  vec->size = 0;
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_real_vec);
  else
    freeMemory((void *)vec, unconnected_dof_real_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_real_vecs(void))
{
  if(unconnected_dof_real_vecs) deleteObject(unconnected_dof_real_vecs);
  unconnected_dof_real_vecs = nil;
}


static void *unconnected_dof_real_d_vecs = nil;

DOF_REAL_D_VEC  *get_dof_real_d_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_REAL_D_VEC *vec;
  static void *vec_info = nil;


  if(!fe_space || !fe_space->admin) {
    if(!unconnected_dof_real_d_vecs)
      unconnected_dof_real_d_vecs = newObject(sizeof(DOF_REAL_D_VEC), 10);
    vec_info = unconnected_dof_real_d_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_real_d_vec;
  
  vec = (DOF_REAL_D_VEC *)getMemory(vec_info);

  vec->next = nil;
  vec->fe_space = fe_space;
  vec->name = name ? strdup(name) : nil;
  vec->size = 0;
  vec->vec = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  vec->mem_info = nil;

  if (fe_space && fe_space->admin)
    AI_add_dof_real_d_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);
  
  return(vec);
}

void free_dof_real_d_vec(DOF_REAL_D_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_real_d_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, REAL_D);
  vec->vec = nil;
  vec->size = 0;
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_real_d_vec);
  else
    freeMemory((void *)vec, unconnected_dof_real_d_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_real_d_vecs(void))
{
  if(unconnected_dof_real_d_vecs) deleteObject(unconnected_dof_real_d_vecs);
  unconnected_dof_real_d_vecs = nil;
}

static void *unconnected_dof_ptr_vecs = nil;

DOF_PTR_VEC  *get_dof_ptr_vec(const char *name, const FE_SPACE *fe_space)
{
  DOF_PTR_VEC *vec;
  static void *vec_info = nil;


  if(!fe_space || !fe_space->admin) {
    if(!unconnected_dof_ptr_vecs)
      unconnected_dof_ptr_vecs = newObject(sizeof(DOF_PTR_VEC), 10);
    vec_info = unconnected_dof_ptr_vecs;
  }
  else
    vec_info = ((DOF_ADMIN_MEM_INFO*)(fe_space->admin->mem_info))->dof_ptr_vec;
  
  vec = (DOF_PTR_VEC *)getMemory(vec_info);

  vec->next = nil;
  vec->fe_space = fe_space;
  vec->name = name ? strdup(name) : nil;
  vec->size = 0;
  vec->vec = nil;
  vec->refine_interpol = vec->coarse_restrict = nil;
  vec->mem_info = nil;

  if (fe_space && fe_space->admin)
    AI_add_dof_ptr_vec_to_admin(vec, (DOF_ADMIN *)fe_space->admin);
  
  return(vec);
}

void free_dof_ptr_vec(DOF_PTR_VEC *vec)
{
  if (vec->fe_space && vec->fe_space->admin)
    AI_remove_dof_ptr_vec_from_admin(vec);
  
  MEM_FREE(vec->vec, vec->size, void *);
  vec->vec = nil;
  vec->size = 0;
  if(vec->fe_space && vec->fe_space->admin)
    freeMemory((void *)vec, ((DOF_ADMIN_MEM_INFO*)(vec->fe_space->admin->mem_info))->dof_ptr_vec);
  else
    freeMemory((void *)vec, unconnected_dof_ptr_vecs);
}

ALBERTA_DEFUNUSED(static void delete_unconnected_dof_ptr_vecs(void))
{
  if(unconnected_dof_ptr_vecs) deleteObject(unconnected_dof_ptr_vecs);
  unconnected_dof_ptr_vecs = nil;
}

