#!/bin/sh
# Repackage upstream source to exclude non-distributable files
# should be called as "repack.sh --upstream-source <ver> <downloaded file>
# (for example, via uscan)

set -e
set -u

VER="$2"
FILE="$3"
PKG=$(dpkg-parsechangelog|sed 's/^Source: //p;d')

REPACK_DIR="$PKG-$VER.orig" # DevRef § 6.7.8.2

printf "\nRepackaging %s\n\n" "$FILE"

DIR=$(mktemp -d ./tmpRepackXXXXXX)
trap 'rm -rf $DIR' QUIT INT EXIT

# Create an extra directory to cope with rootless tarballs
UP_BASE="$DIR/unpack"
mkdir "$UP_BASE"
tar xf "$FILE" -C "$UP_BASE"

if [ "$(find "$UP_BASE" -mindepth 1 -maxdepth 1 | wc -l)" -eq 1 ]; then
	# Tarball does contain a root directory
	UP_BASE="$UP_BASE/$(ls -1 "$UP_BASE")"
fi

## Remove stuff

# There are some non-free DLLs under wxPython.  DLLs aren't useful for us
# so just nuke any regardless which protects us from any new DLLs which get
# added by upstream.
find "$UP_BASE" -iname '*.dll' -delete

# We don't use the built-in copy of expat and it contains an ancient copy
# of libtool which lintian warns about, so just delete it.  This also ensures
# that we don't accidentally start building against it in future.  By similar
# logic, remove other embedded copies of libraries we don't want to use, and
# which lintian might warn about in future.
rm -rf "$UP_BASE"/ext/wxWidgets/src/expat
rm -rf "$UP_BASE"/ext/wxWidgets/src/jpeg
rm -rf "$UP_BASE"/ext/wxWidgets/src/png
rm -rf "$UP_BASE"/ext/wxWidgets/src/tiff
rm -rf "$UP_BASE"/ext/wxWidgets/src/zlib

# Lacking (suitable) source code.  The only one in wxPython is only used
# by an ActiveX sample, so irrelevant for Debian anyway
# (http://trac.wxwidgets.org/ticket/15917).
find "$UP_BASE" -iname '*.swf' -delete

# Samples from TV shows (http://trac.wxwidgets.org/ticket/15917).
rm -f "$UP_BASE"/demo/data/anykey.wav
rm -f "$UP_BASE"/demo/data/plan.wav
rm -f "$UP_BASE"/demo/data/testmovie.mpg

# Seemingly only rebuildable with non-free tools.
rm -f "$UP_BASE"/ext/wxWidgets/samples/help/doc.chm

# The autotools-based build system for the embedded copy of wxWidgets is
# unused, but lintian moans about out of date config.* so just nuke them.
rm -f "$UP_BASE"/ext/wxWidgets/config.guess
rm -f "$UP_BASE"/ext/wxWidgets/config.sub

# The SIP generated files that we're going to regenerate anyway.
rm -f "$UP_BASE"/sip/cpp/*.cpp
rm -f "$UP_BASE"/sip/cpp/*.h
rm -f "$UP_BASE"/sip/cpp/*.sbf
rm -f "$UP_BASE"/sip/gen/*.sip
rm -f "$UP_BASE"/wx/*.pi
rm -f "$UP_BASE"/wx/*.py
rm -f "$UP_BASE"/wx/*.pyi

# Now rebuild the tarball.
mv "$UP_BASE" "$DIR/$REPACK_DIR"
# Using a pipe hides tar errors!
tar cfC "$DIR/repacked.tar" "$DIR" "$REPACK_DIR"
xz -9 < "$DIR/repacked.tar" > "$DIR/repacked.tar.xz"

FILE=$(dirname "$FILE")/${PKG}_${VER}+dfsg.orig.tar.xz
mv "$DIR/repacked.tar.xz" "$FILE"

echo "*** $FILE repackaged"
