/******************************************************************************
*       SOFA, Simulation Open-Framework Architecture, version 1.0 beta 4      *
*                (c) 2006-2009 MGH, INRIA, USTL, UJF, CNRS                    *
*                                                                             *
* This library is free software; you can redistribute it and/or modify it     *
* under the terms of the GNU Lesser General Public License as published by    *
* the Free Software Foundation; either version 2.1 of the License, or (at     *
* your option) any later version.                                             *
*                                                                             *
* This library is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License *
* for more details.                                                           *
*                                                                             *
* You should have received a copy of the GNU Lesser General Public License    *
* along with this library; if not, write to the Free Software Foundation,     *
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.          *
*******************************************************************************
*                               SOFA :: Modules                               *
*                                                                             *
* Authors: The SOFA Team and external contributors (see Authors.txt)          *
*                                                                             *
* Contact information: contact@sofa-framework.org                             *
******************************************************************************/
#ifndef SOFA_GPU_CUDA_CUDAVISUALMODEL_H
#define SOFA_GPU_CUDA_CUDAVISUALMODEL_H

#include <sofa/core/VisualModel.h>
#include <sofa/core/componentmodel/topology/BaseMeshTopology.h>
#include <sofa/core/componentmodel/behavior/State.h>
#include "CudaTypes.h"

namespace sofa
{

namespace gpu
{

namespace cuda
{

template<class DataTypes>
class CudaKernelsCudaVisualModel;

} // namespace cuda

} // namespace gpu


namespace component
{

namespace visualmodel
{

template <class TDataTypes>
class CudaVisualModel : public core::VisualModel
{
public:
    //typedef gpu::cuda::CudaVectorTypes<TCoord,TDeriv,TReal> DataTypes;
    typedef TDataTypes DataTypes;
    typedef typename DataTypes::VecCoord VecCoord;
    typedef typename DataTypes::VecDeriv VecDeriv;
    typedef typename DataTypes::Coord Coord;
    typedef typename DataTypes::Deriv Deriv;
    typedef typename DataTypes::Real Real;

    typedef core::componentmodel::topology::BaseMeshTopology::PointID Index;
    typedef core::componentmodel::topology::BaseMeshTopology::Triangle Triangle;
    typedef core::componentmodel::topology::BaseMeshTopology::SeqTriangles SeqTriangles;
    typedef core::componentmodel::topology::BaseMeshTopology::Quad Quad;
    typedef core::componentmodel::topology::BaseMeshTopology::SeqQuads SeqQuads;

    typedef core::componentmodel::behavior::State<DataTypes> TState;

    typedef gpu::cuda::CudaKernelsCudaVisualModel<DataTypes> Kernels;

    TState* state;
    core::componentmodel::topology::BaseMeshTopology* topology;
    bool needUpdateTopology;
    gpu::cuda::CudaVector<Triangle> triangles;
    gpu::cuda::CudaVector<Quad> quads;

    VecCoord fnormals;
    VecCoord vnormals;

    int nbElement; ///< number of elements
    int nbVertex; ///< number of vertices to process to compute all elements
    int nbElementPerVertex; ///< max number of elements connected to a vertex
    /// Index of elements attached to each points (layout per bloc of NBLOC vertices, with first element of each vertex, then second element, etc)
    gpu::cuda::CudaVector<int> velems;

    Data<defaulttype::Vec4f> matAmbient, matDiffuse, matSpecular, matEmissive;
    Data<float> matShininess;
    Data<bool> useVBO;
    Data<bool> computeNormals;

    CudaVisualModel()
    : state(NULL), topology(NULL), needUpdateTopology(true), nbElement(0), nbVertex(0), nbElementPerVertex(0)
      , matAmbient  ( initData( &matAmbient,   defaulttype::Vec4f(0.1f,0.1f,0.1f,0.0f), "ambient",   "material ambient color") )
      , matDiffuse  ( initData( &matDiffuse,   defaulttype::Vec4f(0.8f,0.8f,0.8f,1.0f), "diffuse",   "material diffuse color and alpha") )
      , matSpecular ( initData( &matSpecular,  defaulttype::Vec4f(1.0f,1.0f,1.0f,0.0f), "specular",  "material specular color") )
      , matEmissive ( initData( &matEmissive,  defaulttype::Vec4f(0.0f,0.0f,0.0f,0.0f), "emissive",  "material emissive color") )
      , matShininess( initData( &matShininess, 45.0f,                                   "shininess", "material specular shininess") )
      , useVBO( initData( &useVBO, false, "useVBO", "true to activate Vertex Buffer Object") )
      , computeNormals( initData( &computeNormals, false, "computeNormals", "true to compute smooth normals") )
    {}

    virtual void init();
    virtual void reinit();
    virtual void drawVisual();
    virtual void updateVisual();

protected:

    void updateTopology();
    void updateNormals();

    void initV(int nbe, int nbv, int nbelemperv)
    {
        nbElement = nbe;
        nbVertex = nbv;
        nbElementPerVertex = nbelemperv;
        int nbloc = (nbVertex+BSIZE-1)/BSIZE;
        velems.resize(nbloc*nbElementPerVertex*BSIZE);
        for (unsigned int i=0;i<velems.size();i++)
            velems[i] = 0;
    }

    void setV(int vertex, int num, int index)
    {
        int bloc = vertex/BSIZE;
        int b_x  = vertex%BSIZE;
        velems[ bloc*BSIZE*nbElementPerVertex // start of the bloc
              + num*BSIZE                     // offset to the element
              + b_x                           // offset to the vertex
              ] = index+1;
    }
};

} // namespace visualmodel

} // namespace component

} // namespace sofa

#endif
