use crate::types::temporal;

use super::*;
/// Takes in a temporal string and converts it into a potential `Temporal` value.
pub fn parse_temporal_value(temporal_value: &str) -> Result<temporal::Value, crate::Error> {
    temporal::Value::from_str(temporal_value).map_err(crate::Error::Temporal)
}

fn resolve_value_elements(node: Rc<Node>) -> Vec<Element> {
    node.children
        .borrow()
        .iter()
        .filter_map(|node| Element::try_from(Rc::clone(node)).ok())
        .filter(|element| element.has_class("value") || element.has_class("value-title"))
        .collect::<Vec<_>>()
}

fn extract_values(elements: Vec<Element>, base_url: Url) -> Vec<String> {
    elements
        .into_iter()
        .filter_map(|element| {
            if element.has_class("value") {
                if ["img", "area"].contains(&element.name.as_str()) {
                    element.attribute("alt")
                } else if &element.name == "data" {
                    element.attribute("value").or_else(|| {
                        extract_only_text(Rc::clone(&element.node), base_url.clone()).ok()
                    })
                } else if &element.name == "abbr" {
                    element.attribute("title").or_else(|| {
                        extract_only_text(Rc::clone(&element.node), base_url.clone()).ok()
                    })
                } else if ["time", "ins", "del"].contains(&element.name.as_str()) {
                    element.attribute("datetime").or_else(|| {
                        extract_only_text(Rc::clone(&element.node), base_url.clone()).ok()
                    })
                } else {
                    extract_only_text(Rc::clone(&element.node), base_url.clone()).ok()
                }
            } else if element.has_class("value-title") {
                element.attribute("title")
            } else {
                None
            }
        })
        .filter(|v| !v.is_empty())
        .map(|v| v.trim().to_string())
        .collect::<Vec<String>>()
}

pub fn compose_temporal_value(values: Vec<String>) -> Option<temporal::Value> {
    if let Some(iso8601_value) = values
        .first()
        .map(|s| s.as_str())
        .and_then(|s| temporal::Stamp::from_iso8601(s).ok())
        .filter(|_| values.len() == 1)
    {
        trace!(
            "The provided value-class value was a timestamp; returning {:#?}",
            iso8601_value
        );
        Some(temporal::Value::Timestamp(iso8601_value))
    } else if let Ok(value) = temporal::Duration::from_str(&values.join("")) {
        Some(temporal::Value::Duration(value))
    } else if values.is_empty() {
        trace!("No value was provided for temporal parsing.");
        None
    } else {
        trace!("Attempting to parse {:#?} into a temporal value.", values);
        let mut date = None;
        let mut time = None;
        let mut offset = None;

        for value in values.iter().cloned().flat_map(temporal::Stamp::parse) {
            if value.is_date() && date.is_none() {
                date = value.as_date()
            } else if value.is_time() && time.is_none() {
                time = value.as_time();
            } else if value.is_offset() && offset.is_none() {
                offset = value.as_offset();
            }
        }

        time = if let Some(mut t) = time {
            if t.prefix.is_none() {
                t.prefix = Some(' ');
            }
            Some(t)
        } else {
            time
        };
        Some(temporal::Value::Timestamp(temporal::Stamp::compose(
            date, time, offset,
        )))
    }
}

#[test]
fn compose_temporal_value_test() {
    assert_eq!(
        compose_temporal_value(vec!["2000-10-10".to_owned(), "10:00Z".to_owned()])
            .map(|t| t.to_string()),
        Some("2000-10-10 10:00Z".to_owned())
    );
    assert_eq!(
        compose_temporal_value(vec!["2000-100".to_owned(), "10:00Z".to_owned()])
            .map(|t| t.to_string()),
        Some("2000-100 10:00Z".to_owned())
    );
}

#[derive(Eq, PartialEq)]
pub enum TypeHint {
    Plain,
    Temporal,
}

pub fn parse(node: Rc<Node>, base_url: Url, type_hint: TypeHint) -> Option<PropertyValue> {
    let values = extract_values(resolve_value_elements(Rc::clone(&node)), base_url);

    if values.is_empty() {
        trace!("Nothing was parsed during value-class parsing.",);
        None
    } else {
        trace!("Resolved {:#?} as the value-class result", values);
        if type_hint == TypeHint::Temporal {
            compose_temporal_value(values).map(PropertyValue::Temporal)
        } else {
            Some(PropertyValue::Plain(values.join("")))
        }
    }
}
