/*
** Copyright 2001 Double Precision, Inc.
** See COPYING for distribution information.
**
*/

#include "config.h"
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <math.h>
#include <errno.h>
#include <libintl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <linux/fd.h>
#include <linux/kdev_t.h>
#include "superfloppy.h"

static const char rcsid[]="$Id: floppyfloppy.c,v 1.1 2001/02/13 00:15:38 mrsam Exp $";

#define SECSIZE 512

static const char * const drivetype[] = {
	"unknown CMOS type",
	"5.25\" DD",
	"5.25\" HD",
	"3.5\" DD",
	"3.5\" HD",
	"3.5\" EHD AMI BIOS",
	"3.5\" EHD"
} ;

static int do_format(const char *, int, int (*)(const char *, int), int);

struct fdinfo {
	const char *dev;
	unsigned char tracks;
	unsigned char heads;
	unsigned char sectors;
} ;

static const struct fdinfo floppy_type[32] = {
	{NULL,		0,0,0},
	{"d360",	40,2,9},
	{"h1200",	80,2,15},
	{"D360",	80,1,9},
	{"D720",	80,2,9},
	{"h360",	40,2,9},
	{"h720",	80,2,9},
	{"H1440",	80,2,18},
	{"u2880",	80,2,36},
	{"u3120",	80,2,39},
	{"h1440",	80,2,18},
	{"u1680",	80,2,21},
	{"h410",	41,2,10},
	{"u820",	82,2,10},
	{"h1476",	82,2,18},
	{"u1722",	82,2,21},
	{"h420",	42,2,10},
	{"u830",	83,2,10},
	{"h1494",	83,2,18},
	{"u1743",	83,2,21},
	{"h880",	80,2,11},
	{"u1040",	80,2,13},
	{"u1120",	80,2,14},
	{"h1660",	80,2,20},
	{"u1760",	80,2,22},
	{"u1920",	80,2,24},
	{"u3200",	80,2,40},
	{"u3520",	80,2,44},
	{"u3840",	80,2,48},
	{"u1840",	80,2,23},
	{"u800",	80,2,10},
	{"u1660",	80,2,20}
};

#define FLOPPY_TYPE_MASK 31

#define MINOR_DEV(n)  (((n) >> 2) & FLOPPY_TYPE_MASK)

int floppy_probe(const char *dev, int *foundflag)
{
	int fd;
	static struct floppy_drive_params fdp;
	struct stat stat_buf;

	if (foundflag)
		*foundflag=0;

	fd=open(dev, O_RDONLY | O_NONBLOCK);
	if (fd < 0)
		return (0);

	if (fstat(fd, &stat_buf) ||
	    !S_ISBLK(stat_buf.st_mode) ||
	    MINOR_DEV(stat_buf.st_rdev))
	{
		close(fd);
		return (0);
	}

	if (ioctl(fd, FDGETDRVPRM, &fdp) < 0)
	{
		close(fd);
                return (0);
	}

	if (fdp.cmos == 0)
	{
		close(fd);
		return (0);
	}

	if (foundflag)
	{
		*foundflag=1;
	}
	else
	{
		int n=fdp.cmos;

		if (n >= sizeof(drivetype)/sizeof(drivetype[0]))
			n=0;

		printf(_("floppy    %s: %s\n"), dev, drivetype[n]);
	}
	close(fd);
	return (0);
}

static struct floppy_drive_params fdp;

static int get_params(const char *dev)
{
	int fd;

	fd=open(dev, O_RDONLY | O_NONBLOCK);
	if (fd < 0)
	{
		perror(dev);
		return (1);
	}

	if (ioctl(fd, FDGETDRVPRM, &fdp) < 0)
	{
		perror(dev);
		close(fd);
                return (1);
	}
	close(fd);
	return (0);
}

int floppy_capacity(const char *dev, int (*fmt_func)(const char *, int),
		    int flags)
{
	int i;

	if (get_params(dev))
		return (1);

	if (flags & FLAGS_FORMAT)
	{
		int n=fdp.autodetect[0];

		if (n == 0)
			return (1);

		return (do_format(dev, n, fmt_func, flags));
	}

	for (i=0; i<sizeof(fdp.autodetect)/sizeof(fdp.autodetect[0]); i++)
	{
		int n=fdp.autodetect[i];
		char buf[200];

		if (n < 0 || n >= sizeof(floppy_type)/sizeof(floppy_type[0])
		    || floppy_type[n].dev == 0)
			continue;

		sprintf(buf, "%dx%dx%d", floppy_type[n].tracks,
			floppy_type[n].sectors * floppy_type[n].heads,
			SECSIZE);

                if (flags & FLAGS_BRIEF)
                        printf("%s\n", buf);
                else
                        printf("  %-15s (%s%s, %s)\n", buf,
			       dev,
			       floppy_type[n].dev,
                               fmtsize(
				       floppy_type[n].tracks
				       * floppy_type[n].sectors
				       * floppy_type[n].heads,
				       SECSIZE));
	}
	return (0);
}

int floppy_format(const char *dev, const char *fmt,
		  int (*fmt_func)(const char *, int), int flags)
{
	const char *p=strchr(fmt, 'x');
	int trk;
	int sec;

	if (p)
	{
		trk=atoi(fmt);
		sec=atoi(p+1);
		p=strchr(p+1, 'x');
		if (p && atoi(p+1) == SECSIZE)
		{
			int i;

			if (get_params(dev))
				return (1);

			for (i=0; i<sizeof(fdp.autodetect)/
				     sizeof(fdp.autodetect[0]); i++)
			{
				int n=fdp.autodetect[i];

				if (floppy_type[n].dev &&
				    floppy_type[n].tracks == trk &&
				    floppy_type[n].sectors
				    * floppy_type[n].heads
				    == sec)
					return (do_format(dev, n,
							  fmt_func, flags));
			}
		}
	}
	fprintf(stderr, _("%s: invalid format."), fmt);
	return (1);
}

static void print_format_progress(int nn, int flags)
{
	if (flags & FLAGS_BRIEF)
		printf("%d\n", nn);
	else
		printf("\b\b\b\b%3d%%", nn);
	fflush(stdout);
}

static int fmt_fd;

static RETSIGTYPE sighandler(int signum)
{
	if (fmt_fd >= 0)
	{
		ioctl(fmt_fd, FDFMTEND, NULL);
	}
	write(1, "\n", 1);
	_exit(0);
#if     RETSIGTYPE != void
        return (0);
#endif
}


static int do_format(const char *dev, int fmtnum,
		     int (*fmt_func)(const char *, int), int flags)
{
	int fd;
	struct floppy_struct geo;
	int t;
	int h;
	struct format_descr curtrack;
	int pct;
	struct stat stat_buf;

	int i, j;
	char *devname;

	if (fmtnum <= 0 ||
	    fmtnum >= sizeof(floppy_type)/sizeof(floppy_type[0]) ||
	    floppy_type[fmtnum].dev == 0)
	{
		errno=EINVAL;
		perror(dev);
		return (1);
	}

	devname=alloca(strlen(dev)+strlen(floppy_type[fmtnum].dev)+1);
	if (!devname)
	{
		perror("alloca");
		return (1);
	}

	strcat(strcpy(devname, dev), floppy_type[fmtnum].dev);

	fd=open(devname, O_WRONLY);
	if (fd < 0)
	{
		perror(devname);
		return (1);
	}

	if (fstat(fd, &stat_buf) ||
	    !S_ISBLK(stat_buf.st_mode) ||
	    MINOR_DEV(stat_buf.st_rdev) != fmtnum)
	{
		errno=EINVAL;
		perror(devname);
		close(fd);
		return (1);
	}

	if (ioctl(fd, FDGETPRM, &geo) < 0)
	{
		perror(devname);
		close(fd);
		return (1);
	}

	fmt_fd=fd;

	signal(SIGINT, sighandler);
        signal(SIGTERM, sighandler);
        signal(SIGHUP, sighandler);
        signal(SIGALRM, sighandler);

	if (ioctl(fd, FDFMTBEG, NULL) < 0)
	{
		perror(devname);
		close(fd);
                return (1);
	}

	if (!(flags & FLAGS_BRIEF))
		printf(_("Formatting %s...     "), fmtsize(geo.track *
							   geo.head *
							   geo.sect,
							   SECSIZE));
	fflush(stdout);

	pct=100;

	if (fmt_func)
		pct=95;

	if (flags & (FLAGS_FMTIDEVERIFY | FLAGS_FMTVERIFY))
		pct /= 2;

	i=0;
	j=geo.track * geo.head;

	for (t=0; t<geo.track; t++)
	{
		for (h=0; h<geo.head; h++)
		{
			curtrack.track=t;
			curtrack.head=h;
			if (ioctl(fd, FDFMTTRK, &curtrack) < 0)
			{
				perror(devname);
				ioctl(fd, FDFMTEND, NULL);
				close(fd);
				return (1);
			}

			print_format_progress (pct * ++i / j, flags);
		}
	}

	if (ioctl(fd, FDFMTEND, NULL) < 0)
	{
		perror(devname);
		fmt_fd= -1;
		close(fd);
		return (1);
	}
	signal(SIGINT, SIG_DFL);
        signal(SIGTERM, SIG_DFL);
        signal(SIGHUP, SIG_DFL);
        signal(SIGALRM, SIG_DFL);
	close(fd);

	if (flags & (FLAGS_FMTIDEVERIFY | FLAGS_FMTVERIFY))
	{
		char *buf;
		int todo;
		int chunksize;
		int blocks;

		fd=open(devname, O_RDONLY);
		if (fd < 0)
		{
			perror(devname);
			return (1);
		}

		chunksize= geo.head * geo.sect;

		buf=malloc(SECSIZE * chunksize);
		if (!buf)
		{
			close(fd);
			perror("malloc");
			return (1);
		}

		blocks=todo=geo.track * geo.head * geo.sect;
		while (todo > 0)
		{
			int n=chunksize;

			if (n > todo)
				n=todo;

			errno=EIO;
			n=read(fd, buf, n * 512);
			if (n <= 0 || (n % 512) != 0)
			{
				free(buf);
				perror("read");
				close(fd);
				return (1);
			}

			todo -= n / 512;

			n = pct * 2 - todo * pct / blocks;

			print_format_progress(n, flags);
		}
		free(buf);
		close(fd);
	}
	if (fmt_func)
	{
		i=(*fmt_func)(dev, flags);
		if (i)
		{
			printf("\n");
			return (i);
		}
	}
	print_format_progress(100, flags);
	printf("\n");
	return (0);
}
