/*
** passwd.c                           /etc/passwd access functions
**
** This file is part of the NYS Library.
**
**      Copyright (c) 1993 Signum Support AB
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include "misc.h"


#ifndef _PATH_PASSWD
#define _PATH_PASSWD "/etc/passwd"
#endif

static FILE *pfp = NULL;
static char *path_passwd = _PATH_PASSWD;

void _setpwent(void)
{
    if (pfp)
	rewind(pfp);
    else
	pfp = fopen(path_passwd, "r");
}


void _endpwent(void)
{
    if (pfp)
	fclose(pfp);
    pfp = NULL;
}


int setpwfile(char *file)
{
    if (pfp)
	_endpwent();
    
    if (file)
	path_passwd = file;
    else
	path_passwd = _PATH_PASSWD;

    return 0;
}




struct passwd *sgetpwent(char *buf)
{
    static struct passwd pwd;
    char *cp, *sp;
	

    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    pwd.pw_name = sp;

    
    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    pwd.pw_passwd = sp;

    
    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    if (*sp)
	pwd.pw_uid = atoi(sp);
    else
	pwd.pw_uid = -1;

    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    if (*sp)
	pwd.pw_gid = atoi(sp);
    else
	pwd.pw_gid = -1;

    
    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    pwd.pw_gecos = sp;

    
    sp = buf;
    cp = strchr(sp, ':');
    if (cp == NULL)
	return NULL;
    if (*cp)
	*cp++ = '\0';
    pwd.pw_dir = sp;

        
    pwd.pw_shell = cp;

    return &pwd;
}


struct passwd *fgetpwent(FILE *fp)
{
    static char *pw_buf = NULL;
    static int pw_size = 0;
    int len;

    
    len = _nsw_getline(&pw_buf, &pw_size, fp);
    if (len < 0)
	return NULL;
    
    return sgetpwent(pw_buf);
}


struct passwd *_getpwent(void)
{
    if (pfp == NULL)
	_setpwent();
    
    if (pfp == NULL)
	return NULL;

    return fgetpwent(pfp);
}


struct passwd *_getpwnam(char *name)
{
    struct passwd *pwp;

    _setpwent();
    
    while ((pwp = _getpwent()) != NULL &&
	   strcmp(pwp->pw_name, name) != 0)
	;

    _endpwent();
	      
    return pwp;
}

struct passwd *_getpwuid(int uid)
{
    struct passwd *pwp;


    _setpwent();
    
    while ((pwp = _getpwent()) != NULL &&
	   pwp->pw_uid != uid)
	;

    _endpwent();
    
    return pwp;
}
