/*
** yp_networks.c           NIS Version 2 Network map access routines
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Authors: Peter Eriksson <pen@signum.se>
**          Michael A. Griffith <grif@cs.ucr.edu>
*/

#include "config.h"

#ifdef ENABLE_YP


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "yp_misc.h"
#include "yp_networks.h"
#include "rpcsvc/ypclnt.h"


static struct netent *netent_parse(char *str, int len)
{
    static struct netent net;
    static char buf[2048], temp[2048], nname[YP_MAX_NET_NAME_LEN];
    static char *alias_list[YP_MAX_NET_ALIASES];
    static int i, j, shiftcount;
    static unsigned long addr;
            
    /* Can't handle this -> let's just forget it... */
    if (len > 2040)
	return NULL;

    strncpy(buf, str, len);

    /*
     * Copy out name
     */
    net.n_name = nname;
    for (i = 0 ; (!(isspace(buf[i])) && (i < len)) ; ++i)
    {
       net.n_name[i] = buf[i];
    }
    net.n_name[i] = '\0';
    
    
    /*
     * Eat white space
     */
    for ( ; ((isspace(buf[i])) && (i < len)) ; ++i)
       ;

#ifdef LSB
    /* 
     * Copy and build net address
     */
    addr       = 0;
    shiftcount = 24;
    while (1)
    {
       for (j = 0; ((isdigit(buf[i])) && (j < 3)) ; ++i, ++j)
	  temp[j] = buf[i];
       temp[j] = '\0';
       
       addr |= (atoi(temp) << shiftcount);

       if ((buf[i] != '.') || (shiftcount <= 0))
	  break;
       else
	  ++i;

       shiftcount -= 8;
    }

    net.n_net = htonl(addr);
#else
#error No MSB support, yet.
#endif

    net.n_addrtype = AF_INET;
    net.n_aliases  = alias_list;

    j = 0;
    net.n_aliases[j] = _yp_strip_names(_yp_xstrtok(buf+i, ' '));
    while ((net.n_aliases[j] = _yp_strip_names(_yp_xstrtok(NULL, ' '))))
    {
       if (strlen(net.n_aliases[j]) > 0)
	   ++j;
    }
    net.n_aliases[j] = 0;

    
    return &net;
    
}


static int rewind_flag = 1;
static char *savekey = NULL;
static int savekeylen = 0;


void yp_setnetent(int stayopen, void *opts)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


void yp_endnetent(void *opts)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


struct netent *yp_getnetent(void *opts)
{
    struct netent *net;
    char *map;
    char *domain;
    char *result;
    int len;
    char *outkey;
    int keylen;


    map = _ypopts_getmd(opts, "networks", ".byaddr", &domain);
    if (map == NULL)
	return NULL;

    net = NULL;
    
    if (rewind_flag)
    {
	if (yp_first(domain, map,
		     &outkey, &keylen,
		     &result, &len))
	    goto error;
	
	rewind_flag = 0;
	savekey = outkey;
	savekeylen = keylen;
    }
    else
    {
	if (yp_next(domain, map,
		    savekey, savekeylen, &outkey, &keylen,
		    &result, &len))
	    goto error;
	
	free(savekey);
	savekey = outkey;
	savekeylen = keylen;
    }

    net = netent_parse(result, len);
    free(result);

  error:
    free(map);
    free(domain);
    
    return net;
}


struct netent *yp_getnetbyaddr(long net, int type, void *opts)
{
    struct netent *netd;
    char *map;
    char *domain;
    char *result;
    int len;
    char buf[256];


    map = _ypopts_getmd(opts, "networks", ".byaddr", &domain);
    if (map == NULL)
	return NULL;

    sprintf(buf, "%d", net);

    net = 0;
        
    if (yp_match(domain, map, buf, strlen(buf), &result, &len) == 0)
    {
	netd = netent_parse(result, len);
	free(result);
    }

    free(map);
    free(domain);
    
    return netd;
}


struct netent *yp_getnetbyname(char *name, void *opts)
{
    struct netent *net;
    char *map;
    char *domain;
    char *result;
    int len;

    map = _ypopts_getmd(opts, "networks", ".byname", &domain);
    if (map == NULL)
	return NULL;

    net = NULL;
    if (yp_match(domain, map, name, strlen(name), &result, &len) == 0)
    {
	net = netent_parse(result, len);
	free(result);
    }

    free(map);
    free(domain);
    return net;
}

#endif /* ENABLE_YP */


