/*
 * arp		This file contains an implementation of the command
 *		that maintains the kernel's ARP cache.  It is derived
 *		from Berkeley UNIX arp(8), but cleaner and with sup-
 *		port for devices other than Ethernet.
 *
 * Usage:	arp [-v] [-t type] -a [hostname]
 *		arp [-v] -d hostname ...
 *		arp [-v] [-t type] -s hostname hw_addr
 *		arp [-v] -f filename
 *
 * Version:	@(#)arp.c	1.06	05/25/93
 *
 * Author: 	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/ddi.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <net/if.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include "pathnames.h"


/* This structure defines hardware protocols and their handlers. */
struct hwtype {
  char		*name;
  char		*title;
  int		type;
  int		hlen;
  char		*(*print)(unsigned char *);
  unsigned char	*(*input)(char *);
};


char *Version = "@(#) arp 1.06 (05/25/93)";


int opt_v = 0;				/* debugging output flag	*/
struct hwtype *hardware;		/* current hardware type	*/
int sockfd;				/* active socket descriptor	*/


/* Display an Ethernet address in readable format. */
char *pr_ether(unsigned char *ptr)
{
  static char buff[64];

  sprintf(buff, "%02X:%02X:%02X:%02X:%02X:%02X",
	(ptr[0] & 255), (ptr[1] & 255), (ptr[2] & 255),
	(ptr[3] & 255), (ptr[4] & 255), (ptr[5] & 255)
  );
  return(buff);
}


/* Input an Ethernet address and convert to binary. */
unsigned char *in_ether(char *ptr)
{
  static unsigned char buff[10];
  register const char *bufp;
  register int i, val;

  i = 0;
  bufp = ptr;
  while(*bufp != '\0') {
	val = 0;
	if (*bufp >= '0' && *bufp <= '9') val = *bufp - '0';
	  else if (*bufp >= 'a' && *bufp <= 'f') val = *bufp - 'a' + 10;
	  else if (*bufp >= 'A' && *bufp <= 'F') val = *bufp - 'A' + 10;
	  else {
		fprintf(stderr, "%s: invalid Ethernet address!\n", ptr);
		return((char *)NULL);
	}
	bufp++;
	val <<= 4;
	if (*bufp >= '0' && *bufp <= '9') val |= *bufp - '0';
	  else if (*bufp >= 'a' && *bufp <= 'f') val |= *bufp - 'a' + 10;
	  else if (*bufp >= 'A' && *bufp <= 'F') val |= *bufp - 'A' + 10;
	  else {
		fprintf(stderr, "%s: invalid Ethernet address!\n", ptr);
		return((char *)NULL);
	}
	buff[i++] = (char) (val & 255);
	bufp++;
	if ((i < 6) && (*bufp++ != ':')) {
		fprintf(stderr, "%s: invalid Ethernet address!\n", ptr);
		return((char *)NULL);
	}
  }

  if (opt_v) fprintf(stderr, "Ether: address is %s\n", pr_ether(buff));
  return(buff);
}


struct hwtype hwtypes[] = {
  { "ether",	"10Mbps Ethernet",	ARPHRD_ETHER,	6,
					pr_ether,	in_ether	},
#ifdef HAVE_ARPHRD_PRONET
  { "pronet",	"ProNET",		ARPHRD_PRONET,	2,
					pr_pronet,	in_pronet	},
#endif
#ifdef HAVE_ARPHRD_AX25
  { "ax25",	"AMPR AX.25",		ARPHRD_AX25,	12,
					pr_ax25,	in_ax25		},
#endif
  { NULL,	NULL,			-1,		0,
					NULL,		NULL		}
};


/* Check our hardware type table for this type. */
struct hwtype *get_type(char *name)
{
  register struct hwtype *hwp;

  hwp = hwtypes;
  while (hwp->name != NULL) {
	if (!strcmp(hwp->name, name)) return(hwp);
	hwp++;
  }
  return(NULL);
}


/* Check our hardware type table for this type. */
struct hwtype *get_ntype(int type)
{
  register struct hwtype *hwp;

  hwp = hwtypes;
  while (hwp->name != NULL) {
	if (hwp->type == type) return(hwp);
	hwp++;
  }
  return(NULL);
}


/* Delete an entry from the ARP cache. */
int arp_del(char **args)
{
  struct hostent *hp;
  struct arpreq req;
  struct sockaddr_in *si;
  register int i;
  int found;

  /* Resolve the host name. */
  if (*args == NULL) {
	fprintf(stderr, "arp: need host name\n");
	return(-1);
  }
  if ((hp = gethostbyname(*args)) == NULL) {
	fprintf(stderr, "arp: %s: unknown host\n", *args);
	return(-1);
  }

  /* The host can have more than one address, so we loop on them. */
  i = -1;
  found = 0;
  while (hp->h_addr_list[++i] != NULL) {
	memset((char *) &req, 0, sizeof(req));
	si = (struct sockaddr_in *) &req.arp_pa;
	si->sin_family = hp->h_addrtype;
	memcpy((char *) &si->sin_addr, hp->h_addr_list[i], hp->h_length);

	/* Call the kernel. */
	if (ioctl(sockfd, SIOCDARP, &req) < 0) {
		if (errno != ENXIO) {
			perror("SIOCDARP");
			return(-1);
		} else continue;
	}
	found++;
  }

  if (found == 0) printf("No ARP entry for %s\n", hp->h_name);
  return(0);
}


/* Set an entry in the ARP cache. */
int arp_set(char **args)
{
  struct hostent *hp;
  struct arpreq req;
  struct sockaddr_in *si;
  unsigned char *ha;

  /* Resolve the host name. */
  if (*args == NULL) {
	fprintf(stderr, "arp: need host name\n");
	return(-1);
  }
  if ((hp = gethostbyname(*args)) == NULL) {
	fprintf(stderr, "arp: %s: unknown host\n", *args);
	return(-1);
  }

  /* Fetch the hardware address. */
  if (*++args == NULL) {
	fprintf(stderr, "arp: need hardware address\n");
	return(-1);
  }
  if ((ha = hardware->input(*args)) == NULL) return(-1);

  /* Clear and fill in the request block. */
  memset((char *) &req, 0, sizeof(req));
  si = (struct sockaddr_in *) &req.arp_pa;
  si->sin_family = hp->h_addrtype;
  memcpy((char *) &si->sin_addr, hp->h_addr_list[0], hp->h_length);
  req.arp_ha.sa_family = hardware->type;
  memcpy(req.arp_ha.sa_data, ha, hardware->hlen);

  /* Call the kernel. */
  if (ioctl(sockfd, SIOCSARP, &req) < 0) {
	perror("SIOCSARP");
	return(-1);
  }
  return(0);
}


/* Process an EtherFile */
int arp_file(char *name)
{
  printf("Process EtherFile(%s)\n", name);
  return(0);
}


/* Print the contents of an ARP request block. */
void arp_disp(struct arpreq *req)
{
  static int title = 0;
  struct hwtype *hwp;

  /* Fetch the hardware type, which was given by the kernel. */
  hwp = get_ntype(req->arp_ha.sa_family);
  if (hwp == NULL) hwp = get_type("ether");

  if (title++ == 0) printf("IP address\t\tHW type\t\t\tHW address\n");
  printf("%-16s\t%-24s%s\n",
	inet_ntoa(*(struct in_addr *)req->arp_pa.sa_data),
	hwp->title,
	hwp->print(req->arp_ha.sa_data));
}


/* Display the contents of the ARP cache in the kernel. */
int pr_arps(void)
{
  struct arpreq req;
  register int fd;

  /* Open the PROCps kernel table. */
  if ((fd = open(_PATH_PROCNET_ARP, O_RDONLY)) < 0) {
	perror(_PATH_PROCNET_ARP);
	return(-1);
  }

  /* Read the ARP cache entries. */
  while(read(fd, (char *) &req, sizeof(req)) > 0) {
	(void) arp_disp(&req);
  }

  (void) close(fd);
  return(0);
}


/* Show one or more entries from the ARP cache. */
int arp_show(char *host)
{
  struct hostent *hp;
  struct arpreq req;
  struct sockaddr_in *si;
  register int i;
  int found;

  /* Do we have to show the whole table? */
  if (host == NULL) return(pr_arps());

  /* Nope.  Resolve the host name. */
  if ((hp = gethostbyname(host)) == NULL) {
	fprintf(stderr, "arp: %s: unknown host\n", host);
	return(-1);
  }

  /* The host can have more than one address, so we loop on them. */
  i = -1;
  found = 0;
  while (hp->h_addr_list[++i] != NULL) {
	memset((char *) &req, 0, sizeof(req));
	si = (struct sockaddr_in *) &req.arp_pa;
	si->sin_family = hp->h_addrtype;
	memcpy((char *) &si->sin_addr, hp->h_addr_list[i], hp->h_length);
	req.arp_ha.sa_family = AF_UNSPEC;	/* any type */

	/* Call the kernel. */
	if (ioctl(sockfd, SIOCGARP, &req) < 0) {
		if (errno != ENXIO) perror("SIOCGARP");
		continue;
	}
	found++;
 
  
	/* We found it.  Display the results. */
	arp_disp(&req);
  }

  if (found == 0) printf("No ARP entry for %s\n", hp->h_name);
  return(0);
}


static void usage(void)
{
  fprintf(stderr, "Usage: arp [-v] [-t type] -a [hostname]\n");
  fprintf(stderr, "       arp [-v] -d hostname ...\n");
  fprintf(stderr, "       arp [-v] [-t type] -s hostname hw_addr\n");
  fprintf(stderr, "       arp [-v] -f filename\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  register int c;
  int what;
  extern int getopt(), optind, opterr;
  extern char *optarg;

  /* Initialize variables... */
  hardware = get_type("ether");
  sockfd = -1;
  what = -1;

  /* Fetch the command-line arguments. */
  opterr = 0;
  while ((c = getopt(argc, argv, "adfst:v")) != EOF) switch(c) {
	case 'a':
		what = 1;
		break;
	case 'd':
		what = 3;
		break;
	case 'f':
		what = 2;
		break;
	case 's':
		what = 4;
		break;
	case 't':
		hardware = get_type(optarg);
		if (hardware->type == -1) {
			fprintf(stderr,
				"arp: %s: unknown hardware type.\n", optarg);
			exit(-1);
		}
		break;
	case 'v':
		opt_v = 1;
		break;
	default:
		usage();
  }

  /* Create an entry point into the ARP protocol. */
  if ((sockfd = open(_PATH_DEV_ARP, O_RDWR)) < 0) {
	perror(_PATH_DEV_ARP);
	exit(-1);
  }

  /* Now see what we have to do here... */
  switch(what) {
	case 1:		/* show an ARP entry in the cache */
		what = arp_show(argv[optind]);
		break;
	case 2:		/* process an EtherFile */
		what = arp_file(argv[optind]);
		break;
	case 3:		/* delete an ARP entry from the cache */
		what = arp_del(&argv[optind]);
		break;
	case 4:		/* set an ARP entry in the cache */
		what = arp_set(&argv[optind]);
		break;
	default:
		usage();
  }

  (void) close(sockfd);
  exit(what);
}
