package xDash::Logger::File;
# Copyright 2005 Jerzy Wachowiak

use strict;
use Carp;
use vars qw( $VERSION );

$VERSION = '1.00';

# PUBLIC METHODS (convention: capital first letter)

sub new {

    my $class = shift;
    my $self = {};
    $self->{VERSION} = $VERSION;
    bless ( $self, $class );
    return $self
}

sub Open {

#Contract: 
#	[1] Log file path in unix convention as input or
#	[2] Reserved words STDOUT or STDERR
#       [3] Open() can either suceed (true) or everything dies...

    my $self = shift;
    my $logfilepath = shift;
    
    if ( $logfilepath eq 'STDOUT' ) {
	$self->{filehandle} = *STDOUT;
    } 
    elsif ( $logfilepath eq 'STDERR' ) {
	$self->{filehandle} = *STDERR;
    }
    else {
	open( $self->{filehandle}, ">> $logfilepath")
	 or croak 'xDash can not open the logfile for '.ref( $self );
    }
    return 1
}

sub Log {

#Contract: 
#	[1] no line breaking character in input strings
#	[2] Log() can not bust the upper layers
#	[3] True or undef as return

    my $self = shift;
    my $who = shift;
    $who =~ s/\n//g;
    $who =~ s/\r//g;
    my $what = shift;
    $what =~ s/\n/\\n/g;
    $what =~ s/\r/\\r/g;
    my $logline = &linetime().": $who: $what\n";
    my $oldfilehandle = select( $self->{filehandle} );
    $|=1;
    my $succes = print $logline;
    select($oldfilehandle);
    return  $succes
}

sub Close {

    my $self = shift;
    return close $self->{filehandle}
}


# PRIVATE METHODS (convention: small first letter)
 
sub linetime {

    my ( $sec, $min, $hours, $day, $month, $year );
    ( $sec, $min, $hours, $day, $month, $year ) = (localtime)[0,1,2,3,4,5];
    $year = $year+1900 ;
    $month = format_time( $month+1 );
    $day = format_time( $day );
    $hours = format_time( $hours );
    $min = format_time( $min );
    $sec = format_time( $sec );
    return $year.$month.$day.'T'.$hours .':'.$min.':'.$sec
}
    
sub format_time{

    my $timevalue = shift;
    if ( $timevalue < 10 ) { $timevalue = '0'.$timevalue};
    return $timevalue
}

1;
__END__
######################## User Documentation ##################

=pod

=head1 NAME

xDash::Logger::File - Base class for EventLogger, ErrorLogger, MessageLogger and EmergencyLogger

=head1 SYNOPSIS

 package EventLogger;
 use base xDash::Logger::File;
 # Check the correct file path for logger and
 # Uncomment 1.line and comment out 2.line below after debugging,
 # sub Open { shift->SUPER::Open( '/home/xdash/sender/event.log' ) }
 sub Open { shift->SUPER::Open( STDOUT ) }

=head1 DESCRIPTION

=head2 USAGE

The module is developed in the object orientated way. It can be used as the 
base class for logging tasks, which have to implement a 
fixed set of methods called by the derived class C<EventLogger>, 
C<ErrorLogger>, C<MessageLogger> and C<EmergencyLogger>
hardcoded in C<xDash::Sender>, C<xDash::Receiver> and C<xDash::Archivist> 
(driver pattern). 
By deriving from the class, as the way of passing arguments, 
you have to implement explicit methods listed below.
The synopsis above is an example of the client script generated by the F<xdscr>.

=head2 METHODS

=over

=item Open( $file_path|STDOUT|STDERR )

Sets the logger file to $file_path and opens it. STDOUT and STDERR set the 
logging to standard output or standard error.

=back

=head1 BUGS

Any suggestions for improvement are welcomed!

If a bug is detected or nonconforming behavior, 
please send an error report to <jwach@cpan.org>.

=head1 COPYRIGHT

Copyright 2005 Jerzy Wachowiak <jwach@cpan.org>

This library is free software; you can redistribute it and/or modify
 it under the terms of the Apache 2.0 license attached to the module.

=head1 SEE ALSO

=over

=item L<xDash::Sender>

=item L<xDash::Receiver>

=item L<xDash::Archivist>

=back

=cut
