package PDL::Graphics::ColorSpace::RGBSpace;

use strict;
use warnings;

use Carp;

=head1 NAME

PDL::Graphics::ColorSpace::RGBSpace -- defines RGB space conversion parameters and white points.

=head1 DESCRIPTION

Sourced from Graphics::ColorObject (Izvorski & Reibenschuh, 2005).

=head1 Usage

    use Data::Dumper;

    print Dumper $PDL::Graphics::ColorSpace::RGBSpace::RGB_SPACE;

    print Dumper $PDL::Graphics::ColorSpace::RGBSpace::WHITE_POINT;

=cut

our $WHITE_POINT = {
          'D50' => [
                     '0.34567',
                     '0.3585'
                   ],
          'A' => [
                   '0.44757',
                   '0.40745'
                 ],
          'D75' => [
                     '0.29902',
                     '0.31485'
                   ],
          'D55' => [
                     '0.33242',
                     '0.34743'
                   ],
          'D65' => [
                     '0.312713',
                     '0.329016'
                   ],
          'E' => [
                   '0.333333',
                   '0.333333'
                 ],
          'B' => [
                   '0.34842',
                   '0.35161'
                 ],
          'F11' => [
                     '0.38054',
                     '0.37691'
                   ],
          'F2' => [
                    '0.37207',
                    '0.37512'
                  ],
          'C' => [
                   '0.310063',
                   '0.316158'
                 ],
          'D93' => [
                     '0.2848',
                     '0.2932'
                   ],
          'F7' => [
                    '0.31285',
                    '0.32918'
                  ]
};

our $RGB_SPACE = {
          'BruceRGB' => {
                          'gamma' => '2.2',
                          'mstar' => [
                                       [
                                         '2.74565437614039',
                                         '-0.969256810842655',
                                         '0.0112706581772173'
                                       ],
                                       [
                                         '-1.1358911781912',
                                         '1.87599300082369',
                                         '-0.113958877125197'
                                       ],
                                       [
                                         '-0.435056564214666',
                                         '0.0415556222493375',
                                         '1.01310694059653'
                                       ]
                                     ],
                          'm' => [
                                   [
                                     '0.467384242424242',
                                     '0.240995',
                                     '0.0219086363636363'
                                   ],
                                   [
                                     '0.294454030769231',
                                     '0.683554',
                                     '0.0736135076923076'
                                   ],
                                   [
                                     '0.18863',
                                     '0.075452',
                                     '0.993451333333334'
                                   ]
                                 ],
                          'white_point' => $WHITE_POINT->{'D65'},
                        },
          'Adobe RGB (1998)' => {
                                  'gamma' => '2.2',
                                  'mstar' => [
                                               [
                                                 '2.04147788287772',
                                                 '-0.969256870874686',
                                                 '0.0134454339800522'
                                               ],
                                               [
                                                 '-0.564976526119188',
                                                 '1.87599311701547',
                                                 '-0.118372546216537'
                                               ],
                                               [
                                                 '-0.34471277324621',
                                                 '0.0415556248231326',
                                                 '1.01526208347413'
                                               ]
                                             ],
                                  'm' => [
                                           [
                                             '0.576700121212121',
                                             '0.297361',
                                             '0.0270328181818181'
                                           ],
                                           [
                                             '0.185555704225352',
                                             '0.627355',
                                             '0.0706878873239437'
                                           ],
                                           [
                                             '0.1882125',
                                             '0.075285',
                                             '0.9912525'
                                           ]
                                         ],
                                  'white_point' => $WHITE_POINT->{'D65'},
                                },
          'WideGamut' => {
                           'gamma' => '2.2',
                           'mstar' => [
                                        [
                                          '1.46280876111587',
                                          '-0.521793192978599',
                                          '0.0349338148323482'
                                        ],
                                        [
                                          '-0.184062599070901',
                                          '1.44723772392177',
                                          '-0.0968919015161355'
                                        ],
                                        [
                                          '-0.274361028741716',
                                          '0.0677227300206644',
                                          '1.28839528723064'
                                        ]
                                      ],
                           'm' => [
                                    [
                                      '0.716103566037736',
                                      '0.258187',
                                      '0'
                                    ],
                                    [
                                      '0.100929624697337',
                                      '0.724938',
                                      '0.0517812857142858'
                                    ],
                                    [
                                      '0.1471875',
                                      '0.016875',
                                      '0.7734375'
                                    ]
                                  ],
                           'white_point' => $WHITE_POINT->{'D50'},
                         },
          'NTSC' => {
                      'gamma' => '2.2',
                      'mstar' => [
                                   [
                                     '1.91049094509024',
                                     '-0.984310618506658',
                                     '0.0583742441336926'
                                   ],
                                   [
                                     '-0.53259210489728',
                                     '1.99844883151352',
                                     '-0.118517404756285'
                                   ],
                                   [
                                     '-0.288283799898528',
                                     '-0.0282979742694222',
                                     '0.898609576361084'
                                   ]
                                 ],
                      'm' => [
                               [
                                 '0.606733727272727',
                                 '0.298839',
                                 '-1e-16'
                               ],
                               [
                                 '0.173563816901409',
                                 '0.586811',
                                 '0.0661195492957747'
                               ],
                               [
                                 '0.2001125',
                                 '0.11435',
                                 '1.1149125'
                               ]
                             ],
                      'white_point' => $WHITE_POINT->{'C'},
                    },
          'Ekta Space PS5' => {
                                'gamma' => '2.2',
                                'mstar' => [
                                             [
                                               '2.00437873609682',
                                               '-0.711029017049311',
                                               '0.0381257297502959'
                                             ],
                                             [
                                               '-0.730483256478366',
                                               '1.62021366180089',
                                               '-0.0868766628736253'
                                             ],
                                             [
                                               '-0.245004796257919',
                                               '0.0792227384931296',
                                               '1.27252435691152'
                                             ]
                                           ],
                                'm' => [
                                         [
                                           '0.59389231147541',
                                           '0.260629',
                                           '0'
                                         ],
                                         [
                                           '0.272979942857143',
                                           '0.734946',
                                           '0.0419969142857143'
                                         ],
                                         [
                                           '0.09735',
                                           '0.004425',
                                           '0.783225'
                                         ]
                                       ],
                                'white_point' => $WHITE_POINT->{'D50'},
                              },
          'PAL/SECAM' => {
                           'gamma' => '2.2',
                           'mstar' => [
                                        [
                                          '3.06313080780361',
                                          '-0.969257031353275',
                                          '0.0678676345258901'
                                        ],
                                        [
                                          '-1.3932854294802',
                                          '1.87599342762119',
                                          '-0.228821478155597'
                                        ],
                                        [
                                          '-0.475787968862948',
                                          '0.0415556317034429',
                                          '1.06919338982591'
                                        ]
                                      ],
                           'm' => [
                                    [
                                      '0.430586181818182',
                                      '0.222021',
                                      '0.0201837272727273'
                                    ],
                                    [
                                      '0.341545083333333',
                                      '0.706645',
                                      '0.129551583333333'
                                    ],
                                    [
                                      '0.178335',
                                      '0.071334',
                                      '0.939231'
                                    ]
                                  ],
                           'white_point' => $WHITE_POINT->{'D65'},
                         },
          'Apple RGB' => {
                           'gamma' => '1.8',
                           'mstar' => [
                                        [
                                          '2.95176033980206',
                                          '-1.08510012648728',
                                          '0.0854802409232915'
                                        ],
                                        [
                                          '-1.28950900724704',
                                          '1.9908397072633',
                                          '-0.2694550155056'
                                        ],
                                        [
                                          '-0.473880286660678',
                                          '0.0372022452865781',
                                          '1.09113013413848'
                                        ]
                                      ],
                           'm' => [
                                    [
                                      '0.449694852941176',
                                      '0.244634',
                                      '0.0251829117647059'
                                    ],
                                    [
                                      '0.316251294117647',
                                      '0.672034',
                                      '0.141183613445378'
                                    ],
                                    [
                                      '0.184520857142857',
                                      '0.083332',
                                      '0.922604285714286'
                                    ]
                                  ],
                           'white_point' => $WHITE_POINT->{'D65'},
                         },
          'sRGB' => {
                      'gamma' => '-1',     # mark it for special case
                      'mstar' => [
                                   [
                                     '3.24071094399417',
                                     '-0.969258109065483',
                                     '0.0556349466243886'
                                   ],
                                   [
                                     '-1.53726031958698',
                                     '1.87599551352921',
                                     '-0.203994804289425'
                                   ],
                                   [
                                     '-0.498570914460642',
                                     '0.0415556779089489',
                                     '1.05706398586338'
                                   ]
                                 ],
                      'm' => [
                               [
                                 '0.412423757575757',
                                 '0.212656',
                                 '0.0193323636363636'
                               ],
                               [
                                 '0.357579',
                                 '0.715158',
                                 '0.119193'
                               ],
                               [
                                 '0.180465',
                                 '0.072186',
                                 '0.950449'
                               ]
                             ],
                      'white_point' => $WHITE_POINT->{'D65'},
                    },
          'ColorMatch' => {
                            'gamma' => '1.8',
                            'mstar' => [
                                         [
                                           '2.64228725945873',
                                           '-1.11197540964573',
                                           '0.0821692807629542'
                                         ],
                                         [
                                           '-1.22342696462069',
                                           '2.05901666762151',
                                           '-0.280723441849461'
                                         ],
                                         [
                                           '-0.393014279448075',
                                           '0.0159613695164458',
                                           '1.45597744493852'
                                         ]
                                       ],
                            'm' => [
                                     [
                                       '0.509343882352941',
                                       '0.274884',
                                       '0.0242544705882353'
                                     ],
                                     [
                                       '0.320907338842975',
                                       '0.658132',
                                       '0.108782148760331'
                                     ],
                                     [
                                       '0.13397',
                                       '0.066985',
                                       '0.692178333333333'
                                     ]
                                   ],
                            'white_point' => $WHITE_POINT->{'D50'},
                          },
          'SMPTE-C' => {
                         'gamma' => '2.2',
                         'mstar' => [
                                      [
                                        '3.50569560396941',
                                        '-1.06906411585768',
                                        '0.056311654337365'
                                      ],
                                      [
                                        '-1.73963804628462',
                                        '1.97780951196929',
                                        '-0.196993365173273'
                                      ],
                                      [
                                        '-0.54401052306495',
                                        '0.0351719640259221',
                                        '1.0500467308791'
                                      ]
                                    ],
                         'm' => [
                                  [
                                    '0.393555441176471',
                                    '0.212395',
                                    '0.0187407352941176'
                                  ],
                                  [
                                    '0.365252420168067',
                                    '0.701049',
                                    '0.111932193277311'
                                  ],
                                  [
                                    '0.191659714285714',
                                    '0.086556',
                                    '0.958298571428571'
                                  ]
                                ],
                         'white_point' => $WHITE_POINT->{'D65'},
                       },
          'CIE' => {
                     'gamma' => '2.2',
                     'mstar' => [
                                  [
                                    '2.37068020229465',
                                    '-0.513884773083019',
                                    '0.0052981111618865'
                                  ],
                                  [
                                    '-0.900042762577686',
                                    '1.42530304987177',
                                    '-0.0146947611471193'
                                  ],
                                  [
                                    '-0.470634962281563',
                                    '0.088581346669925',
                                    '1.00938458712529'
                                  ]
                                ],
                     'm' => [
                              [
                                '0.488716754716981',
                                '0.176204',
                                '0'
                              ],
                              [
                                '0.310680460251046',
                                '0.812985',
                                '0.0102048326359833'
                              ],
                              [
                                '0.200604111111111',
                                '0.010811',
                                '0.989807111111111'
                              ]
                            ],
                     'white_point' => $WHITE_POINT->{'E'},
                   },
          'ProPhoto' => {
                          'gamma' => '1.8',
                          'mstar' => [
                                       [
                                         '1.3459444124134',
                                         '-0.544598943846181',
                                         '-0'
                                       ],
                                       [
                                         '-0.255607720396453',
                                         '1.50816752372329',
                                         '-0'
                                       ],
                                       [
                                         '-0.0511118080787822',
                                         '0.0205351443915685',
                                         '1.2070909349885'
                                       ]
                                     ],
                          'm' => [
                                   [
                                     '0.797674285714286',
                                     '0.28804',
                                     '0'
                                   ],
                                   [
                                     '0.135191683008091',
                                     '0.711874',
                                     '0'
                                   ],
                                   [
                                     '0.031476',
                                     '8.6e-05',
                                     '0.828438'
                                   ]
                                 ],
                          'white_point' => $WHITE_POINT->{'D50'},
                        },
          'BestRGB' => {
                         'gamma' => '2.2',
                         'mstar' => [
                                      [
                                        '1.75525888974901',
                                        '-0.544133847258114',
                                        '0.0063467101890703'
                                      ],
                                      [
                                        '-0.483678273936868',
                                        '1.50687952348487',
                                        '-0.0175760572028268'
                                      ],
                                      [
                                        '-0.252999899496505',
                                        '0.0215528345168675',
                                        '1.22569016415407'
                                      ]
                                    ],
                         'm' => [
                                  [
                                    '0.632670026008293',
                                    '0.228457',
                                    '0'
                                  ],
                                  [
                                    '0.204555716129032',
                                    '0.737352',
                                    '0.0095142193548387'
                                  ],
                                  [
                                    '0.126995142857143',
                                    '0.034191',
                                    '0.815699571428571'
                                  ]
                                ],
                         'white_point' => $WHITE_POINT->{'D50'},
                       },
          'DonRGB4' => {
                         'gamma' => '2.2',
                         'mstar' => [
                                      [
                                        '1.76038788466061',
                                        '-0.712628997581103',
                                        '0.0078207770365325'
                                      ],
                                      [
                                        '-0.488119149776404',
                                        '1.65274365376055',
                                        '-0.0347412748629646'
                                      ],
                                      [
                                        '-0.253612281154138',
                                        '0.0416715470705678',
                                        '1.24478041036567'
                                      ]
                                    ],
                         'm' => [
                                  [
                                    '0.645772',
                                    '0.27835',
                                    '0.0037113333333334'
                                  ],
                                  [
                                    '0.193351045751634',
                                    '0.68797',
                                    '0.0179861437908497'
                                  ],
                                  [
                                    '0.125097142857143',
                                    '0.03368',
                                    '0.803508571428572'
                                  ]
                                ],
                         'white_point' => $WHITE_POINT->{'D50'},
                       },
          'Beta RGB' => {
                          'gamma' => '2.2',
                          'mstar' => [
                                       [
                                         '1.68322461050127',
                                         '-0.771022999934446',
                                         '0.0400016919321019'
                                       ],
                                       [
                                         '-0.428235686922801',
                                         '1.70655733404514',
                                         '-0.0885384492378917'
                                       ],
                                       [
                                         '-0.236018152270938',
                                         '0.0446899574535591',
                                         '1.27237682509323'
                                       ]
                                     ],
                          'm' => [
                                   [
                                     '0.67125463496144',
                                     '0.303273',
                                     '1e-16'
                                   ],
                                   [
                                     '0.1745833659118',
                                     '0.663786',
                                     '0.0407009558998808'
                                   ],
                                   [
                                     '0.11838171875',
                                     '0.032941',
                                     '0.784501144886363'
                                   ]
                                 ],
                          'white_point' => $WHITE_POINT->{'D50'},
                        },
          'ECI' => {
                     'gamma' => '1.8',
                     'mstar' => [
                                  [
                                    '1.78276097904707',
                                    '-0.959362431268921',
                                    '0.0859317810050046'
                                  ],
                                  [
                                    '-0.496984518455576',
                                    '1.94779645136417',
                                    '-0.174467555373797'
                                  ],
                                  [
                                    '-0.269009968705312',
                                    '-0.0275807381172883',
                                    '1.32282862880431'
                                  ]
                                ],
                     'm' => [
                              [
                                '0.650204545454545',
                                '0.32025',
                                '-1e-16'
                              ],
                              [
                                '0.178077338028169',
                                '0.602071',
                                '0.067838985915493'
                              ],
                              [
                                '0.13593825',
                                '0.077679',
                                '0.75737025'
                              ]
                            ],
                     'white_point' => $WHITE_POINT->{'D50'},
                   },
};


# aliases
$RGB_SPACE->{'Adobe'} = $RGB_SPACE->{'Adobe RGB (1998)'};
$RGB_SPACE->{'601'}   = $RGB_SPACE->{'NTSC'};
$RGB_SPACE->{'Apple'} = $RGB_SPACE->{'Apple RGB'};
$RGB_SPACE->{'CIE ITU'} = $RGB_SPACE->{'PAL/SECAM'};
$RGB_SPACE->{'PAL'} = $RGB_SPACE->{'PAL/SECAM'};
$RGB_SPACE->{'709'} = $RGB_SPACE->{'sRGB'};
$RGB_SPACE->{'SMPTE'} = $RGB_SPACE->{'SMPTE-C'};
$RGB_SPACE->{'CIE Rec 709'} = $RGB_SPACE->{'sRGB'};
$RGB_SPACE->{'CIE Rec 601'} = $RGB_SPACE->{'NTSC'};

sub add_rgb_space {
	my ($new_space) = @_;

	while (my ($name, $profile) = each %$new_space) {
		croak "There is an existing RGB space definition with the same name: $name"
			if $RGB_SPACE->{$name};
		for (keys %{ $RGB_SPACE->{'sRGB'} }) {
			carp "Missing definition for custom RGB space $name: $_"
				unless defined $profile->{$_};
		}
		$RGB_SPACE->{$name} = $profile;
	}
}

1;
