## -*- Mode: CPerl -*-
## File: DTA::CAB::Chain::Tweet.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Description: robust analysis: tweet-munging chain

package DTA::CAB::Chain::Tweet;
use DTA::CAB::Datum ':all';
use DTA::CAB::Chain::Multi;
use IO::File;
use Carp;

use strict;

##==============================================================================
## Constants
##==============================================================================
our @ISA = qw(DTA::CAB::Chain::Multi);

##-- HACK: just inherit from DTA::CAB::Chain::DTA for now

##==============================================================================
## Constructors etc.
##==============================================================================

## $obj = CLASS_OR_OBJ->new(%args)
##  + object structure: HASH
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   ##-- analyzers
			   xlit  => DTA::CAB::Analyzer::Unicruft->new(),
			   lts   => DTA::CAB::Analyzer::LTS->new(),
			   ##
			   morph => DTA::CAB::Analyzer::Morph->new(),
			   #mlatin=> DTA::CAB::Analyzer::Morph::Latin->new(),
			   #msafe => DTA::CAB::Analyzer::MorphSafe->new(),
			   rw    => DTA::CAB::Analyzer::Rewrite->new(),
			   #rwsub => DTA::CAB::Analyzer::RewriteSub->new(),
			   ##
			   eqpho => DTA::CAB::Analyzer::EqPho->new(),       ##-- default (FST)
			   #eqrw  => DTA::CAB::Analyzer::EqRW->new(),        ##-- default (FST)
			   ##
			   ##
			   dmoot => DTA::CAB::Analyzer::Moot::DynLex->new(), ##-- moot n-gram disambiguator
			   #moot => DTA::CAB::Analyzer::Moot->new(),          ##-- moot tagger

			   ##-- user args
			   @_,

			   ##-- overrides
			   chains => undef, ##-- see setupChains() method
			   chain => undef, ##-- see setupChains() method
			  );

}

##==============================================================================
## Methods: Chain selection
##==============================================================================

## $ach = $ach->setupChains()
##  + setup default named sub-chains in $ach->{chains}
##  + override
sub setupChains {
  my $ach = shift;
  my @akeys = grep {UNIVERSAL::isa($ach->{$_},'DTA::CAB::Analyzer')} keys(%$ach);
  my $chains = $ach->{chains} =
    {
     (map {("sub.$_"=>[$ach->{$_}])} @akeys), ##-- sub.xlit, sub.lts, ...
     ##
     'default.xlit'  =>[@$ach{qw(xlit)}],
     'default.lts'   =>[@$ach{qw(xlit lts)}],
     'default.morph' =>[@$ach{qw(xlit morph)}],
     'default.rw'    =>[@$ach{qw(xlit rw)}],
     'default.base'     =>[@$ach{qw(xlit lts morph)}],
     'default.type'     =>[@$ach{qw(xlit lts morph rw)}],
     ##
     'noexpand'  =>[@$ach{qw(xlit lts morph rw)}],
     'expand'    =>[@$ach{qw(xlit lts morph rw eqpho)}],
     'default'   =>[@$ach{qw(xlit lts morph rw eqpho dmoot)}],
     'all'       =>[@$ach{qw(xlit lts morph rw eqpho dmoot)}],
    };

  ##-- sanitize chains
  foreach (values %{$ach->{chains}}) {
    @$_ = grep {ref($_)} @$_;
  }

  ##-- set default chain
  $ach->{chain} = $ach->{chains}{default};

  ##-- force default labels
  $ach->{$_}{label} = $_ foreach (grep {UNIVERSAL::isa($ach->{$_},'DTA::CAB::Analyzer')} keys(%$ach));
  return $ach;
}

##==============================================================================
## Analysis: Utilities
##==============================================================================

BEGIN { *_parseAnalysis = \&DTA::CAB::Analyzer::Moot::parseAnalysis; }

## @analyses = CLASS::dmootTagsGet($tok)
##  + for 'dmoot' analyzer (DTA::CAB::Analyzer::Moot::DynLex) 'analyzeTagsGet' pseudo-accessor
##  + utility for disambiguation using @$tok{qw(text xlit eqpho rw)} fields by default
##  + returns only $tok->{xlit} field if "$tok->{toktyp}" is true
sub dmootTagsGet {
  return
    ($_[0]{xlit} ? (_parseAnalysis($_[0]{xlit}{latin1Text},src=>'xlit')) : _parseAnalysis($_[0]{text},src=>'text'))
      if ($_[0]{toktyp});
  return
    (($_[0]{xlit}  ? (_parseAnalysis($_[0]{xlit}{latin1Text},src=>'xlit')) : _parseAnalysis($_[0]{text},src=>'text')),
     ($_[0]{eqpho} ? (map {_parseAnalysis($_,src=>'eqpho')} @{$_[0]{eqpho}}) : qw()),
     ($_[0]{rw}    ? (map {_parseAnalysis($_,src=>'rw')} @{$_[0]{rw}}) : qw()),
    );
}


1; ##-- be happy

__END__
##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl
=pod

=cut

##========================================================================
## NAME
=pod

=head1 NAME

DTA::CAB::Chain::Tweet - tweet-munging multi-chain

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 use DTA::CAB::Chain::Tweet;
 
 ##========================================================================
 ## Methods
 
 $obj = CLASS_OR_OBJ->new(%args);
 $ach = $ach->setupChains();
 @analyses = CLASS::dmootTagsGet($tok);
 

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

DTA::CAB::Chain::Tweet
is a L<DTA::CAB::Chain::Multi|DTA::CAB::Chain::Multi>
subclass for munging tweets,
to be used in conjunction with a slew of specialized analyzer data files.

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::CAB::Chain::Tweet: Constructors etc.
=pod

=head2 Methods

=over 4

=item new

 $obj = CLASS_OR_OBJ->new(%args);

object structure: HASH


=item setupChains

 $ach = $ach->setupChains();

Setup default named sub-chains in $ach-E<gt>{chains}.
override.

=item dmootTagsGet

 @analyses = CLASS::dmootTagsGet($tok);


For 'dmoot' analyzer (DTA::CAB::Analyzer::Moot::DynLex) 'analyzeTagsGet' pseudo-accessor.
Utility for disambiguation using @$tok{qw(text xlit eqpho rw)} fields by default.
Returns only $tok-E<gt>{xlit} field if "$tok-E<gt>{toktyp}" is true.

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl

##======================================================================
## Footer
##======================================================================
=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010-2019 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.24.1 or,
at your option, any later version of Perl 5 you may have available.

=head1 SEE ALSO

L<dta-cab-analyze.perl(1)|dta-cab-analyze.perl>,
L<DTA::CAB::Chain::Multi(3pm)|DTA::CAB::Chain::Multi>,
L<DTA::CAB::Chain(3pm)|DTA::CAB::Chain>,
L<DTA::CAB::Analyzer(3pm)|DTA::CAB::Analyzer>,
L<DTA::CAB(3pm)|DTA::CAB>,
L<perl(1)|perl>,
...



=cut
